"use strict";
const assert_1 = require("@aws-cdk/assert");
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const ecr = require("../lib");
module.exports = {
    'construct repository'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new ecr.Repository(stack, 'Repo');
        // THEN
        assert_1.expect(stack).toMatch({
            Resources: {
                Repo02AC86CF: {
                    Type: 'AWS::ECR::Repository',
                    DeletionPolicy: 'Retain',
                    UpdateReplacePolicy: 'Retain',
                },
            },
        });
        test.done();
    },
    'tag-based lifecycle policy'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const repo = new ecr.Repository(stack, 'Repo');
        // WHEN
        repo.addLifecycleRule({ tagPrefixList: ['abc'], maxImageCount: 1 });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ECR::Repository', {
            LifecyclePolicy: {
                // tslint:disable-next-line:max-line-length
                LifecyclePolicyText: '{"rules":[{"rulePriority":1,"selection":{"tagStatus":"tagged","tagPrefixList":["abc"],"countType":"imageCountMoreThan","countNumber":1},"action":{"type":"expire"}}]}',
            },
        }));
        test.done();
    },
    'add day-based lifecycle policy'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        const repo = new ecr.Repository(stack, 'Repo');
        repo.addLifecycleRule({
            maxImageAge: cdk.Duration.days(5),
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ECR::Repository', {
            LifecyclePolicy: {
                // tslint:disable-next-line:max-line-length
                LifecyclePolicyText: '{"rules":[{"rulePriority":1,"selection":{"tagStatus":"any","countType":"sinceImagePushed","countNumber":5,"countUnit":"days"},"action":{"type":"expire"}}]}',
            },
        }));
        test.done();
    },
    'add count-based lifecycle policy'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const repo = new ecr.Repository(stack, 'Repo');
        // WHEN
        repo.addLifecycleRule({
            maxImageCount: 5,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ECR::Repository', {
            LifecyclePolicy: {
                // tslint:disable-next-line:max-line-length
                LifecyclePolicyText: '{"rules":[{"rulePriority":1,"selection":{"tagStatus":"any","countType":"imageCountMoreThan","countNumber":5},"action":{"type":"expire"}}]}',
            },
        }));
        test.done();
    },
    'mixing numbered and unnumbered rules'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const repo = new ecr.Repository(stack, 'Repo');
        // WHEN
        repo.addLifecycleRule({ tagStatus: ecr.TagStatus.TAGGED, tagPrefixList: ['a'], maxImageCount: 5 });
        repo.addLifecycleRule({ rulePriority: 10, tagStatus: ecr.TagStatus.TAGGED, tagPrefixList: ['b'], maxImageCount: 5 });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ECR::Repository', {
            LifecyclePolicy: {
                // tslint:disable-next-line:max-line-length
                LifecyclePolicyText: '{"rules":[{"rulePriority":10,"selection":{"tagStatus":"tagged","tagPrefixList":["b"],"countType":"imageCountMoreThan","countNumber":5},"action":{"type":"expire"}},{"rulePriority":11,"selection":{"tagStatus":"tagged","tagPrefixList":["a"],"countType":"imageCountMoreThan","countNumber":5},"action":{"type":"expire"}}]}',
            },
        }));
        test.done();
    },
    'tagstatus Any is automatically sorted to the back'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const repo = new ecr.Repository(stack, 'Repo');
        // WHEN
        repo.addLifecycleRule({ maxImageCount: 5 });
        repo.addLifecycleRule({ tagStatus: ecr.TagStatus.TAGGED, tagPrefixList: ['important'], maxImageCount: 999 });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ECR::Repository', {
            LifecyclePolicy: {
                // tslint:disable-next-line:max-line-length
                LifecyclePolicyText: '{"rules":[{"rulePriority":1,"selection":{"tagStatus":"tagged","tagPrefixList":["important"],"countType":"imageCountMoreThan","countNumber":999},"action":{"type":"expire"}},{"rulePriority":2,"selection":{"tagStatus":"any","countType":"imageCountMoreThan","countNumber":5},"action":{"type":"expire"}}]}',
            },
        }));
        test.done();
    },
    'lifecycle rules can be added upon initialization'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new ecr.Repository(stack, 'Repo', {
            lifecycleRules: [
                { maxImageCount: 3 },
            ],
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ECR::Repository', {
            'LifecyclePolicy': {
                // tslint:disable-next-line:max-line-length
                'LifecyclePolicyText': '{"rules":[{"rulePriority":1,"selection":{"tagStatus":"any","countType":"imageCountMoreThan","countNumber":3},"action":{"type":"expire"}}]}',
            },
        }));
        test.done();
    },
    'calculate repository URI'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const repo = new ecr.Repository(stack, 'Repo');
        // WHEN
        const uri = repo.repositoryUri;
        // THEN
        const arnSplit = { 'Fn::Split': [':', { 'Fn::GetAtt': ['Repo02AC86CF', 'Arn'] }] };
        test.deepEqual(stack.resolve(uri), {
            'Fn::Join': ['', [
                    { 'Fn::Select': [4, arnSplit] },
                    '.dkr.ecr.',
                    { 'Fn::Select': [3, arnSplit] },
                    '.',
                    { Ref: 'AWS::URLSuffix' },
                    '/',
                    { Ref: 'Repo02AC86CF' },
                ]],
        });
        test.done();
    },
    'import with concrete arn'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        const repo2 = ecr.Repository.fromRepositoryArn(stack, 'repo', 'arn:aws:ecr:us-east-1:585695036304:repository/foo/bar/foo/fooo');
        // THEN
        test.deepEqual(stack.resolve(repo2.repositoryArn), 'arn:aws:ecr:us-east-1:585695036304:repository/foo/bar/foo/fooo');
        test.deepEqual(stack.resolve(repo2.repositoryName), 'foo/bar/foo/fooo');
        test.done();
    },
    'fails if importing with token arn and no name'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN/THEN
        test.throws(() => ecr.Repository.fromRepositoryArn(stack, 'arn', cdk.Fn.getAtt('Boom', 'Boom').toString()), /\"repositoryArn\" is a late-bound value, and therefore \"repositoryName\" is required\. Use \`fromRepositoryAttributes\` instead/);
        test.done();
    },
    'import with token arn and repository name (see awslabs/aws-cdk#1232)'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        const repo = ecr.Repository.fromRepositoryAttributes(stack, 'Repo', {
            repositoryArn: cdk.Fn.getAtt('Boom', 'Arn').toString(),
            repositoryName: cdk.Fn.getAtt('Boom', 'Name').toString(),
        });
        // THEN
        test.deepEqual(stack.resolve(repo.repositoryArn), { 'Fn::GetAtt': ['Boom', 'Arn'] });
        test.deepEqual(stack.resolve(repo.repositoryName), { 'Fn::GetAtt': ['Boom', 'Name'] });
        test.done();
    },
    'import only with a repository name (arn is deduced)'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        const repo = ecr.Repository.fromRepositoryName(stack, 'just-name', 'my-repo');
        // THEN
        test.deepEqual(stack.resolve(repo.repositoryArn), {
            'Fn::Join': ['', [
                    'arn:',
                    { Ref: 'AWS::Partition' },
                    ':ecr:',
                    { Ref: 'AWS::Region' },
                    ':',
                    { Ref: 'AWS::AccountId' },
                    ':repository/my-repo'
                ],
            ],
        });
        test.deepEqual(stack.resolve(repo.repositoryName), 'my-repo');
        test.done();
    },
    'arnForLocalRepository can be used to render an ARN for a local repository'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const repoName = cdk.Fn.getAtt('Boom', 'Name').toString();
        // WHEN
        const repo = ecr.Repository.fromRepositoryAttributes(stack, 'Repo', {
            repositoryArn: ecr.Repository.arnForLocalRepository(repoName, stack),
            repositoryName: repoName,
        });
        // THEN
        test.deepEqual(stack.resolve(repo.repositoryName), { 'Fn::GetAtt': ['Boom', 'Name'] });
        test.deepEqual(stack.resolve(repo.repositoryArn), {
            'Fn::Join': ['', [
                    'arn:',
                    { Ref: 'AWS::Partition' },
                    ':ecr:',
                    { Ref: 'AWS::Region' },
                    ':',
                    { Ref: 'AWS::AccountId' },
                    ':repository/',
                    { 'Fn::GetAtt': ['Boom', 'Name'] }
                ]],
        });
        test.done();
    },
    'resource policy'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const repo = new ecr.Repository(stack, 'Repo');
        // WHEN
        repo.addToResourcePolicy(new iam.PolicyStatement({ actions: ['*'] }));
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ECR::Repository', {
            RepositoryPolicyText: {
                Statement: [
                    {
                        Action: '*',
                        Effect: 'Allow',
                    },
                ],
                Version: '2012-10-17',
            },
        }));
        test.done();
    },
    'events': {
        'onImagePushed without imageTag creates the correct event'(test) {
            const stack = new cdk.Stack();
            const repo = new ecr.Repository(stack, 'Repo');
            repo.onCloudTrailImagePushed('EventRule', {
                target: {
                    bind: () => ({ arn: 'ARN', id: '' }),
                },
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Events::Rule', {
                'EventPattern': {
                    'source': [
                        'aws.ecr',
                    ],
                    'detail': {
                        'eventName': [
                            'PutImage',
                        ],
                        'requestParameters': {
                            'repositoryName': [
                                {
                                    'Ref': 'Repo02AC86CF',
                                },
                            ],
                        },
                    },
                },
                'State': 'ENABLED',
            }));
            test.done();
        },
        'onImageScanCompleted without imageTags creates the correct event'(test) {
            const stack = new cdk.Stack();
            const repo = new ecr.Repository(stack, 'Repo');
            repo.onImageScanCompleted('EventRule', {
                target: {
                    bind: () => ({ arn: 'ARN', id: '' }),
                },
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Events::Rule', {
                'EventPattern': {
                    'source': [
                        'aws.ecr',
                    ],
                    'detail': {
                        'repository-name': [
                            {
                                'Ref': 'Repo02AC86CF',
                            },
                        ],
                        'scan-status': [
                            'COMPLETE',
                        ],
                    },
                },
                'State': 'ENABLED',
            }));
            test.done();
        },
        'onImageScanCompleted with one imageTag creates the correct event'(test) {
            const stack = new cdk.Stack();
            const repo = new ecr.Repository(stack, 'Repo');
            repo.onImageScanCompleted('EventRule', {
                imageTags: ['some-tag'],
                target: {
                    bind: () => ({ arn: 'ARN', id: '' }),
                },
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Events::Rule', {
                'EventPattern': {
                    'source': [
                        'aws.ecr',
                    ],
                    'detail': {
                        'repository-name': [
                            {
                                'Ref': 'Repo02AC86CF',
                            },
                        ],
                        'image-tags': [
                            'some-tag',
                        ],
                        'scan-status': [
                            'COMPLETE',
                        ],
                    },
                },
                'State': 'ENABLED',
            }));
            test.done();
        },
        'onImageScanCompleted with multiple imageTags creates the correct event'(test) {
            const stack = new cdk.Stack();
            const repo = new ecr.Repository(stack, 'Repo');
            repo.onImageScanCompleted('EventRule', {
                imageTags: ['tag1', 'tag2', 'tag3'],
                target: {
                    bind: () => ({ arn: 'ARN', id: '' }),
                },
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Events::Rule', {
                'EventPattern': {
                    'source': [
                        'aws.ecr',
                    ],
                    'detail': {
                        'repository-name': [
                            {
                                'Ref': 'Repo02AC86CF',
                            },
                        ],
                        'image-tags': [
                            'tag1',
                            'tag2',
                            'tag3',
                        ],
                        'scan-status': [
                            'COMPLETE',
                        ],
                    },
                },
                'State': 'ENABLED',
            }));
            test.done();
        },
        'removal policy is "Retain" by default'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            // WHEN
            new ecr.Repository(stack, 'Repo');
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::ECR::Repository', {
                'Type': 'AWS::ECR::Repository',
                'DeletionPolicy': 'Retain',
            }, assert_1.ResourcePart.CompleteDefinition));
            test.done();
        },
        '"Delete" removal policy can be set explicitly'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            // WHEN
            new ecr.Repository(stack, 'Repo', {
                removalPolicy: cdk.RemovalPolicy.DESTROY,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::ECR::Repository', {
                'Type': 'AWS::ECR::Repository',
                'DeletionPolicy': 'Delete',
            }, assert_1.ResourcePart.CompleteDefinition));
            test.done();
        },
        'grant adds appropriate resource-*'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const repo = new ecr.Repository(stack, 'TestHarnessRepo');
            // WHEN
            repo.grantPull(new iam.AnyPrincipal());
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::ECR::Repository', {
                'RepositoryPolicyText': {
                    'Statement': [
                        {
                            'Action': [
                                'ecr:BatchCheckLayerAvailability',
                                'ecr:GetDownloadUrlForLayer',
                                'ecr:BatchGetImage',
                            ],
                            'Effect': 'Allow',
                            'Principal': '*',
                        },
                    ],
                    'Version': '2012-10-17',
                },
            }));
            test.done();
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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