import unittest
from unittest.mock import Mock

from botocore.exceptions import ClientError

from sagemaker_studio import ClientConfig
from sagemaker_studio.connections import ConnectionService
from sagemaker_studio.connections.connection import Connection


class TestConnections(unittest.TestCase):
    def setUp(self):
        self.mock_datazone_api = Mock()
        self.mock_glue_api = Mock()
        self.mock_secrets_manager_api = Mock()
        self.connections = ConnectionService(
            project_id="test_project",
            domain_id="test_domain",
            datazone_api=self.mock_datazone_api,
            glue_api=self.mock_glue_api,
            secrets_manager_api=self.mock_secrets_manager_api,
            project_config=ClientConfig(),
        )

    def test_get_connection_by_name_success(self):
        self.mock_datazone_api.list_connections.return_value = {
            "items": [{"connectionId": "12345", "name": "test_connection"}]
        }
        self.mock_datazone_api.get_connection.return_value = {
            "connectionId": "12345",
            "secret": "my_secret",
            "name": "test_connection",
        }
        connection = self.connections.get_connection_by_name("test_connection")
        self.assertIsInstance(connection, Connection)
        self.mock_datazone_api.list_connections.assert_any_call(
            domainIdentifier="test_domain", projectIdentifier="test_project", name="test_connection"
        )
        self.mock_datazone_api.get_connection.assert_any_call(
            domainIdentifier="test_domain", identifier="12345", withSecret=True
        )

    def test_get_connection_by_name_no_connection_found(self):
        self.mock_datazone_api.list_connections.return_value = {"items": []}
        with self.assertRaises(AttributeError):
            self.connections.get_connection_by_name("nonexistent_connection")
        self.mock_datazone_api.list_connections.assert_called_once_with(
            domainIdentifier="test_domain",
            projectIdentifier="test_project",
            name="nonexistent_connection",
        )

    def test_list_connections_success(self):
        self.mock_datazone_api.list_connections.side_effect = [
            {"items": [{"name": "connection1", "type": "type1"}], "nextToken": "token1"},
            {"items": [{"name": "connection2", "type": "type2"}], "nextToken": None},
        ]
        self.mock_datazone_api.get_connection.return_value = {}
        connections = self.connections.list_connections()
        self.assertEqual(len(connections), 2)
        self.assertEqual(connections[0].name, "connection1")
        self.assertEqual(connections[1].name, "connection2")
        self.mock_datazone_api.list_connections.assert_any_call(
            domainIdentifier="test_domain", projectIdentifier="test_project"
        )

    def test_list_connections_throw_client_error(self):
        self.mock_datazone_api.list_connections.side_effect = ClientError(
            error_response={"Error": {"Code": "AccessDeniedException"}},
            operation_name="ListConnections",
        )
        with self.assertRaises(RuntimeError) as context:
            self.connections.list_connections()
            self.assertTrue(
                "Encountered an error while listing connections for project" in context.exception
            )

        self.mock_datazone_api.list_connections.assert_called_once_with(
            domainIdentifier="test_domain", projectIdentifier="test_project"
        )

    def test_list_connections_succeeds_when_get_connection_throws_client_error(self):
        self.mock_datazone_api.list_connections.side_effect = [
            {"items": [{"name": "connection1", "type": "type1"}], "nextToken": "token1"},
            {"items": [{"name": "connection2", "type": "type2"}], "nextToken": None},
        ]
        self.mock_datazone_api.get_connection.side_effect = ClientError(
            error_response={"Error": {"Code": "AccessDeniedException"}},
            operation_name="GetConnection",
        )

        connections = self.connections.list_connections()
        self.assertEqual(len(connections), 2)
        self.assertEqual(connections[0].name, "connection1")
        self.assertEqual(connections[1].name, "connection2")
        self.mock_datazone_api.list_connections.assert_any_call(
            domainIdentifier="test_domain", projectIdentifier="test_project"
        )
