"""Symmetry classes."""

import numpy as np

from nvtools.vector import Vector


class Symmetry:

    def __new__(cls, *args, **kwargs):
        if args[0] == "Td":
            return TdSymmetry()
        elif args[0] == "Oh":
            return OhSymmetry()


class SymmetryBaseClass:

    matrices: list

    def apply(self, vec: Vector):
        """Apply Td symmetry to vector."""
        yield from (Vector(np.dot(mat, vec.vals)) for mat in self.matrices)


class TdSymmetry(SymmetryBaseClass):
    """Td Symmetry."""

    matrices = [
        np.eye(3),
        np.array([[0, 0, 1], [1, 0, 0], [0, 1, 0]]),
        np.array([[0, 1, 0], [0, 0, 1], [1, 0, 0]]),
        np.array([[0, 0, -1], [-1, 0, 0], [0, -1, 0]]),
        np.array([[0, -1, 0], [0, 0, -1], [-1, 0, 0]]),
        np.array([[0, 1, 0], [-1, 0, 0], [0, 0, 1]]),
        np.array([[0, 0, -1], [1, 0, 0], [0, -1, 0]]),
        np.array([[0, -1, 0], [1, 0, 0], [0, 0, -1]]),
        np.array([[0, 0, 1], [-1, 0, 0], [0, 1, 0]]),
        np.array([[1, 0, 0], [0, -1, 0], [0, 0, -1]]),
        np.array([[-1, 0, 0], [0, 1, 0], [0, 0, -1]]),
        np.array([[-1, 0, 0], [0, -1, 0], [0, 0, 1]]),
        np.array([[0, 0, 1], [0, 1, 0], [-1, 0, 0]]),
        np.array([[0, 0, -1], [0, 1, 0], [1, 0, 0]]),
        np.array([[1, 0, 0], [0, 0, 1], [0, -1, 0]]),
        np.array([[1, 0, 0], [0, 0, -1], [0, 1, 0]]),
        np.array([[0, 1, 0], [1, 0, 0], [0, 0, -1]]),
        np.array([[0, -1, 0], [-1, 0, 0], [0, 0, 1]]),
        np.array([[1, 0, 0], [0, 1, 0], [0, 0, 1]]),
        np.array([[0, 0, -1], [0, -1, 0], [-1, 0, 0]]),
        np.array([[0, 1, 0], [0, 0, 1], [-1, 0, 0]]),
        np.array([[0, 0, -1], [-1, 0, 0], [0, 1, 0]]),
        np.array([[0, -1, 0], [0, 0, -1], [1, 0, 0]]),
        np.array([[0, 1, 0], [1, 0, 0], [0, 0, -1]]),
    ]


class OhSymmetry(SymmetryBaseClass):
    """Oh Symmetry."""

    matrices = [
        np.eye(3),
        np.array([[0, 0, 1], [1, 0, 0], [0, 1, 0]]),
        np.array([[0, 1, 0], [0, 0, 1], [1, 0, 0]]),
        np.array([[0, 0, -1], [-1, 0, 0], [0, -1, 0]]),
        np.array([[0, -1, 0], [0, 0, -1], [-1, 0, 0]]),
        np.array([[0, 1, 0], [-1, 0, 0], [0, 0, 1]]),
        np.array([[0, 0, -1], [1, 0, 0], [0, -1, 0]]),
        np.array([[0, -1, 0], [1, 0, 0], [0, 0, -1]]),
        np.array([[0, 0, 1], [-1, 0, 0], [0, 1, 0]]),
        np.array([[1, 0, 0], [0, -1, 0], [0, 0, -1]]),
        np.array([[-1, 0, 0], [0, 1, 0], [0, 0, -1]]),
        np.array([[-1, 0, 0], [0, -1, 0], [0, 0, 1]]),
        np.array([[0, 0, 1], [0, 1, 0], [-1, 0, 0]]),
        np.array([[0, 0, -1], [0, 1, 0], [1, 0, 0]]),
        np.array([[1, 0, 0], [0, 0, 1], [0, -1, 0]]),
        np.array([[1, 0, 0], [0, 0, -1], [0, 1, 0]]),
        np.array([[0, 1, 0], [1, 0, 0], [0, 0, -1]]),
        np.array([[0, -1, 0], [-1, 0, 0], [0, 0, 1]]),
        np.array([[1, 0, 0], [0, 1, 0], [0, 0, 1]]),
        np.array([[0, 0, -1], [0, -1, 0], [-1, 0, 0]]),
        np.array([[0, 1, 0], [0, 0, 1], [-1, 0, 0]]),
        np.array([[0, 0, -1], [-1, 0, 0], [0, 1, 0]]),
        np.array([[0, -1, 0], [0, 0, -1], [1, 0, 0]]),
        np.array([[0, 1, 0], [1, 0, 0], [0, 0, -1]]),
        -np.eye(3),
        -np.array([[0, 0, 1], [1, 0, 0], [0, 1, 0]]),
        -np.array([[0, 1, 0], [0, 0, 1], [1, 0, 0]]),
        -np.array([[0, 0, -1], [-1, 0, 0], [0, -1, 0]]),
        -np.array([[0, -1, 0], [0, 0, -1], [-1, 0, 0]]),
        -np.array([[0, 1, 0], [-1, 0, 0], [0, 0, 1]]),
        -np.array([[0, 0, -1], [1, 0, 0], [0, -1, 0]]),
        -np.array([[0, -1, 0], [1, 0, 0], [0, 0, -1]]),
        -np.array([[0, 0, 1], [-1, 0, 0], [0, 1, 0]]),
        -np.array([[1, 0, 0], [0, -1, 0], [0, 0, -1]]),
        -np.array([[-1, 0, 0], [0, 1, 0], [0, 0, -1]]),
        -np.array([[-1, 0, 0], [0, -1, 0], [0, 0, 1]]),
        -np.array([[0, 0, 1], [0, 1, 0], [-1, 0, 0]]),
        -np.array([[0, 0, -1], [0, 1, 0], [1, 0, 0]]),
        -np.array([[1, 0, 0], [0, 0, 1], [0, -1, 0]]),
        -np.array([[1, 0, 0], [0, 0, -1], [0, 1, 0]]),
        -np.array([[0, 1, 0], [1, 0, 0], [0, 0, -1]]),
        -np.array([[0, -1, 0], [-1, 0, 0], [0, 0, 1]]),
        -np.array([[1, 0, 0], [0, 1, 0], [0, 0, 1]]),
        -np.array([[0, 0, -1], [0, -1, 0], [-1, 0, 0]]),
        -np.array([[0, 1, 0], [0, 0, 1], [-1, 0, 0]]),
        -np.array([[0, 0, -1], [-1, 0, 0], [0, 1, 0]]),
        -np.array([[0, -1, 0], [0, 0, -1], [1, 0, 0]]),
        -np.array([[0, 1, 0], [1, 0, 0], [0, 0, -1]]),
    ]
