"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PublicKey = void 0;
const core_1 = require("@aws-cdk/core");
const cloudfront_generated_1 = require("./cloudfront.generated");
/**
 * A Public Key Configuration.
 *
 * @stability stable
 * @resource AWS::CloudFront::PublicKey
 */
class PublicKey extends core_1.Resource {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        if (!core_1.Token.isUnresolved(props.encodedKey) && !/^-----BEGIN PUBLIC KEY-----/.test(props.encodedKey)) {
            throw new Error(`Public key must be in PEM format (with the BEGIN/END PUBLIC KEY lines); got ${props.encodedKey}`);
        }
        const resource = new cloudfront_generated_1.CfnPublicKey(this, 'Resource', {
            publicKeyConfig: {
                name: (_a = props.publicKeyName) !== null && _a !== void 0 ? _a : this.generateName(),
                callerReference: this.node.addr,
                encodedKey: props.encodedKey,
                comment: props.comment,
            },
        });
        this.publicKeyId = resource.ref;
    }
    /**
     * Imports a Public Key from its id.
     *
     * @stability stable
     */
    static fromPublicKeyId(scope, id, publicKeyId) {
        return new class extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.publicKeyId = publicKeyId;
            }
        }(scope, id);
    }
    generateName() {
        const name = core_1.Names.uniqueId(this);
        if (name.length > 80) {
            return name.substring(0, 40) + name.substring(name.length - 40);
        }
        return name;
    }
}
exports.PublicKey = PublicKey;
//# sourceMappingURL=data:application/json;base64,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