"use strict";
const assert_1 = require("@aws-cdk/assert");
const core_1 = require("@aws-cdk/core");
const eks = require("../lib");
const util_1 = require("./util");
module.exports = {
    'add Helm chart': {
        'should have default namespace'(test) {
            // GIVEN
            const { stack, cluster } = util_1.testFixtureCluster();
            // WHEN
            new eks.HelmChart(stack, 'MyChart', { cluster, chart: 'chart' });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource(eks.HelmChart.RESOURCE_TYPE, { Namespace: 'default' }));
            test.done();
        },
        'should have a lowercase default release name'(test) {
            // GIVEN
            const { stack, cluster } = util_1.testFixtureCluster();
            // WHEN
            new eks.HelmChart(stack, 'MyChart', { cluster, chart: 'chart' });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource(eks.HelmChart.RESOURCE_TYPE, { Release: 'stackmychartff398361' }));
            test.done();
        },
        'should use the last 53 of the default release name'(test) {
            // GIVEN
            const { stack, cluster } = util_1.testFixtureCluster();
            // WHEN
            new eks.HelmChart(stack, 'MyChartNameWhichISMostProbablyLongerThanFiftyThreeCharacters', { cluster, chart: 'chart' });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource(eks.HelmChart.RESOURCE_TYPE, { Release: 'hismostprobablylongerthanfiftythreecharacterscaf15d09' }));
            test.done();
        },
        'with values'(test) {
            // GIVEN
            const { stack, cluster } = util_1.testFixtureCluster();
            // WHEN
            new eks.HelmChart(stack, 'MyChart', { cluster, chart: 'chart', values: { foo: 123 } });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource(eks.HelmChart.RESOURCE_TYPE, { Values: '{\"foo\":123}' }));
            test.done();
        },
        'should support create namespaces by default'(test) {
            // GIVEN
            const { stack, cluster } = util_1.testFixtureCluster();
            // WHEN
            new eks.HelmChart(stack, 'MyChart', { cluster, chart: 'chart' });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource(eks.HelmChart.RESOURCE_TYPE, { CreateNamespace: true }));
            test.done();
        },
        'should support create namespaces when explicitly specified'(test) {
            // GIVEN
            const { stack, cluster } = util_1.testFixtureCluster();
            // WHEN
            new eks.HelmChart(stack, 'MyChart', { cluster, chart: 'chart', createNamespace: true });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource(eks.HelmChart.RESOURCE_TYPE, { CreateNamespace: true }));
            test.done();
        },
        'should not create namespaces when disabled'(test) {
            // GIVEN
            const { stack, cluster } = util_1.testFixtureCluster();
            // WHEN
            new eks.HelmChart(stack, 'MyChart', { cluster, chart: 'chart', createNamespace: false });
            // THEN
            assert_1.expect(stack).notTo(assert_1.haveResource(eks.HelmChart.RESOURCE_TYPE, { CreateNamespace: true }));
            test.done();
        },
        'should support waiting until everything is completed before marking release as successful'(test) {
            // GIVEN
            const { stack, cluster } = util_1.testFixtureCluster();
            // WHEN
            new eks.HelmChart(stack, 'MyWaitingChart', { cluster, chart: 'chart', wait: true });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource(eks.HelmChart.RESOURCE_TYPE, { Wait: true }));
            test.done();
        },
        'should default to not waiting before marking release as successful'(test) {
            // GIVEN
            const { stack, cluster } = util_1.testFixtureCluster();
            // WHEN
            new eks.HelmChart(stack, 'MyWaitingChart', { cluster, chart: 'chart' });
            // THEN
            assert_1.expect(stack).notTo(assert_1.haveResource(eks.HelmChart.RESOURCE_TYPE, { Wait: true }));
            test.done();
        },
        'should enable waiting when specified'(test) {
            // GIVEN
            const { stack, cluster } = util_1.testFixtureCluster();
            // WHEN
            new eks.HelmChart(stack, 'MyWaitingChart', { cluster, chart: 'chart', wait: true });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource(eks.HelmChart.RESOURCE_TYPE, { Wait: true }));
            test.done();
        },
        'should disable waiting when specified as false'(test) {
            // GIVEN
            const { stack, cluster } = util_1.testFixtureCluster();
            // WHEN
            new eks.HelmChart(stack, 'MyWaitingChart', { cluster, chart: 'chart', wait: false });
            // THEN
            assert_1.expect(stack).notTo(assert_1.haveResource(eks.HelmChart.RESOURCE_TYPE, { Wait: true }));
            test.done();
        },
        'should timeout only after 10 minutes'(test) {
            // GIVEN
            const { stack, cluster } = util_1.testFixtureCluster();
            // WHEN
            new eks.HelmChart(stack, 'MyChart', { cluster, chart: 'chart', timeout: core_1.Duration.minutes(10) });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource(eks.HelmChart.RESOURCE_TYPE, { Timeout: '600s' }));
            test.done();
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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