"use strict";
const fs = require("fs");
const path = require("path");
const assert_1 = require("@aws-cdk/assert");
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const YAML = require("yaml");
const eks = require("../lib");
const kubectl_layer_1 = require("../lib/kubectl-layer");
const util_1 = require("./util");
/* eslint-disable max-len */
const CLUSTER_VERSION = eks.KubernetesVersion.V1_16;
module.exports = {
    'a default cluster spans all subnets'(test) {
        // GIVEN
        const { stack, vpc } = util_1.testFixture();
        // WHEN
        new eks.Cluster(stack, 'Cluster', { vpc, defaultCapacity: 0, version: CLUSTER_VERSION });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::AWSCDK-EKS-Cluster', {
            Config: {
                roleArn: { 'Fn::GetAtt': ['ClusterRoleFA261979', 'Arn'] },
                version: '1.16',
                resourcesVpcConfig: {
                    securityGroupIds: [{ 'Fn::GetAtt': ['ClusterControlPlaneSecurityGroupD274242C', 'GroupId'] }],
                    subnetIds: [
                        { Ref: 'VPCPublicSubnet1SubnetB4246D30' },
                        { Ref: 'VPCPublicSubnet2Subnet74179F39' },
                        { Ref: 'VPCPrivateSubnet1Subnet8BCA10E0' },
                        { Ref: 'VPCPrivateSubnet2SubnetCFCDAA7A' },
                    ],
                },
            },
        }));
        test.done();
    },
    'create custom cluster correctly in any aws region'(test) {
        // GIVEN
        const app = new cdk.App();
        const stack = new cdk.Stack(app, 'stack', { env: { region: 'us-east-1' } });
        // WHEN
        const vpc = new ec2.Vpc(stack, 'VPC');
        new eks.Cluster(stack, 'Cluster', { vpc, defaultCapacity: 0, version: CLUSTER_VERSION });
        const layer = kubectl_layer_1.KubectlLayer.getOrCreate(stack, {});
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Custom::AWSCDK-EKS-Cluster'));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Serverless::Application', {
            Location: {
                ApplicationId: 'arn:aws:serverlessrepo:us-east-1:903779448426:applications/lambda-layer-kubectl',
            },
        }));
        test.equal(layer.isChina(), false);
        test.done();
    },
    'create custom cluster correctly in any aws region in china'(test) {
        // GIVEN
        const app = new cdk.App();
        const stack = new cdk.Stack(app, 'stack', { env: { region: 'cn-north-1' } });
        // WHEN
        const vpc = new ec2.Vpc(stack, 'VPC');
        new eks.Cluster(stack, 'Cluster', { vpc, defaultCapacity: 0, version: CLUSTER_VERSION });
        new kubectl_layer_1.KubectlLayer(stack, 'NewLayer');
        const layer = kubectl_layer_1.KubectlLayer.getOrCreate(stack);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Custom::AWSCDK-EKS-Cluster'));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Serverless::Application', {
            Location: {
                ApplicationId: 'arn:aws-cn:serverlessrepo:cn-north-1:487369736442:applications/lambda-layer-kubectl',
            },
        }));
        test.equal(layer.isChina(), true);
        test.done();
    },
    'if "vpc" is not specified, vpc with default configuration will be created'(test) {
        // GIVEN
        const { stack } = util_1.testFixtureNoVpc();
        // WHEN
        new eks.Cluster(stack, 'cluster', { version: CLUSTER_VERSION });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPC'));
        test.done();
    },
    'default capacity': {
        'x2 m5.large by default'(test) {
            // GIVEN
            const { stack } = util_1.testFixtureNoVpc();
            // WHEN
            const cluster = new eks.Cluster(stack, 'cluster', { version: CLUSTER_VERSION });
            // THEN
            test.ok(cluster.defaultNodegroup);
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EKS::Nodegroup', {
                InstanceTypes: [
                    'm5.large',
                ],
                ScalingConfig: {
                    DesiredSize: 2,
                    MaxSize: 2,
                    MinSize: 2,
                },
            }));
            test.done();
        },
        'quantity and type can be customized'(test) {
            // GIVEN
            const { stack } = util_1.testFixtureNoVpc();
            // WHEN
            const cluster = new eks.Cluster(stack, 'cluster', {
                defaultCapacity: 10,
                defaultCapacityInstance: new ec2.InstanceType('m2.xlarge'),
                version: CLUSTER_VERSION,
            });
            // THEN
            test.ok(cluster.defaultNodegroup);
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EKS::Nodegroup', {
                ScalingConfig: {
                    DesiredSize: 10,
                    MaxSize: 10,
                    MinSize: 10,
                },
            }));
            // expect(stack).to(haveResource('AWS::AutoScaling::LaunchConfiguration', { InstanceType: 'm2.xlarge' }));
            test.done();
        },
        'defaultCapacity=0 will not allocate at all'(test) {
            // GIVEN
            const { stack } = util_1.testFixtureNoVpc();
            // WHEN
            const cluster = new eks.Cluster(stack, 'cluster', { defaultCapacity: 0, version: CLUSTER_VERSION });
            // THEN
            test.ok(!cluster.defaultCapacity);
            assert_1.expect(stack).notTo(assert_1.haveResource('AWS::AutoScaling::AutoScalingGroup'));
            assert_1.expect(stack).notTo(assert_1.haveResource('AWS::AutoScaling::LaunchConfiguration'));
            test.done();
        },
    },
    'creating a cluster tags the private VPC subnets'(test) {
        // GIVEN
        const { stack, vpc } = util_1.testFixture();
        // WHEN
        new eks.Cluster(stack, 'Cluster', { vpc, defaultCapacity: 0, version: CLUSTER_VERSION });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::Subnet', {
            Tags: [
                { Key: 'aws-cdk:subnet-name', Value: 'Private' },
                { Key: 'aws-cdk:subnet-type', Value: 'Private' },
                { Key: 'kubernetes.io/role/internal-elb', Value: '1' },
                { Key: 'Name', Value: 'Stack/VPC/PrivateSubnet1' },
            ],
        }));
        test.done();
    },
    'creating a cluster tags the public VPC subnets'(test) {
        // GIVEN
        const { stack, vpc } = util_1.testFixture();
        // WHEN
        new eks.Cluster(stack, 'Cluster', { vpc, defaultCapacity: 0, version: CLUSTER_VERSION });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::Subnet', {
            MapPublicIpOnLaunch: true,
            Tags: [
                { Key: 'aws-cdk:subnet-name', Value: 'Public' },
                { Key: 'aws-cdk:subnet-type', Value: 'Public' },
                { Key: 'kubernetes.io/role/elb', Value: '1' },
                { Key: 'Name', Value: 'Stack/VPC/PublicSubnet1' },
            ],
        }));
        test.done();
    },
    'adding capacity creates an ASG with tags'(test) {
        // GIVEN
        const { stack, vpc } = util_1.testFixture();
        const cluster = new eks.Cluster(stack, 'Cluster', {
            vpc,
            defaultCapacity: 0,
            version: CLUSTER_VERSION,
        });
        // WHEN
        cluster.addCapacity('Default', {
            instanceType: new ec2.InstanceType('t2.medium'),
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::AutoScaling::AutoScalingGroup', {
            Tags: [
                {
                    Key: { 'Fn::Join': ['', ['kubernetes.io/cluster/', { Ref: 'Cluster9EE0221C' }]] },
                    PropagateAtLaunch: true,
                    Value: 'owned',
                },
                {
                    Key: 'Name',
                    PropagateAtLaunch: true,
                    Value: 'Stack/Cluster/Default',
                },
            ],
        }));
        test.done();
    },
    'create nodegroup with existing role'(test) {
        // GIVEN
        const { stack } = util_1.testFixtureNoVpc();
        // WHEN
        const cluster = new eks.Cluster(stack, 'cluster', {
            defaultCapacity: 10,
            defaultCapacityInstance: new ec2.InstanceType('m2.xlarge'),
            version: CLUSTER_VERSION,
        });
        const existingRole = new iam.Role(stack, 'ExistingRole', {
            assumedBy: new iam.AccountRootPrincipal(),
        });
        new eks.Nodegroup(stack, 'Nodegroup', {
            cluster,
            nodeRole: existingRole,
        });
        // THEN
        test.ok(cluster.defaultNodegroup);
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EKS::Nodegroup', {
            ScalingConfig: {
                DesiredSize: 10,
                MaxSize: 10,
                MinSize: 10,
            },
        }));
        test.done();
    },
    'adding bottlerocket capacity creates an ASG with tags'(test) {
        // GIVEN
        const { stack, vpc } = util_1.testFixture();
        const cluster = new eks.Cluster(stack, 'Cluster', {
            vpc,
            defaultCapacity: 0,
            version: CLUSTER_VERSION,
        });
        // WHEN
        cluster.addCapacity('Bottlerocket', {
            instanceType: new ec2.InstanceType('t2.medium'),
            machineImageType: eks.MachineImageType.BOTTLEROCKET,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::AutoScaling::AutoScalingGroup', {
            Tags: [
                {
                    Key: { 'Fn::Join': ['', ['kubernetes.io/cluster/', { Ref: 'Cluster9EE0221C' }]] },
                    PropagateAtLaunch: true,
                    Value: 'owned',
                },
                {
                    Key: 'Name',
                    PropagateAtLaunch: true,
                    Value: 'Stack/Cluster/Bottlerocket',
                },
            ],
        }));
        test.done();
    },
    'adding bottlerocket capacity with bootstrapOptions throws error'(test) {
        // GIVEN
        const { stack, vpc } = util_1.testFixture();
        const cluster = new eks.Cluster(stack, 'Cluster', {
            vpc,
            defaultCapacity: 0,
            version: CLUSTER_VERSION,
        });
        test.throws(() => cluster.addCapacity('Bottlerocket', {
            instanceType: new ec2.InstanceType('t2.medium'),
            machineImageType: eks.MachineImageType.BOTTLEROCKET,
            bootstrapOptions: {},
        }), /bootstrapOptions is not supported for Bottlerocket/);
        test.done();
    },
    'exercise export/import'(test) {
        // GIVEN
        const { stack: stack1, vpc, app } = util_1.testFixture();
        const stack2 = new cdk.Stack(app, 'stack2', { env: { region: 'us-east-1' } });
        const cluster = new eks.Cluster(stack1, 'Cluster', {
            vpc,
            defaultCapacity: 0,
            version: CLUSTER_VERSION,
        });
        // WHEN
        const imported = eks.Cluster.fromClusterAttributes(stack2, 'Imported', {
            clusterArn: cluster.clusterArn,
            vpc: cluster.vpc,
            clusterEndpoint: cluster.clusterEndpoint,
            clusterName: cluster.clusterName,
            securityGroups: cluster.connections.securityGroups,
            clusterCertificateAuthorityData: cluster.clusterCertificateAuthorityData,
            clusterSecurityGroupId: cluster.clusterSecurityGroupId,
            clusterEncryptionConfigKeyArn: cluster.clusterEncryptionConfigKeyArn,
        });
        // this should cause an export/import
        new cdk.CfnOutput(stack2, 'ClusterARN', { value: imported.clusterArn });
        // THEN
        assert_1.expect(stack2).toMatch({
            Outputs: {
                ClusterARN: {
                    Value: {
                        'Fn::ImportValue': 'Stack:ExportsOutputFnGetAttCluster9EE0221CArn9E0B683E',
                    },
                },
            },
        });
        test.done();
    },
    'mastersRole can be used to map an IAM role to "system:masters"'(test) {
        // GIVEN
        const { stack, vpc } = util_1.testFixture();
        const role = new iam.Role(stack, 'role', { assumedBy: new iam.AnyPrincipal() });
        // WHEN
        new eks.Cluster(stack, 'Cluster', {
            vpc,
            mastersRole: role,
            defaultCapacity: 0,
            version: CLUSTER_VERSION,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource(eks.KubernetesResource.RESOURCE_TYPE, {
            Manifest: {
                'Fn::Join': [
                    '',
                    [
                        '[{"apiVersion":"v1","kind":"ConfigMap","metadata":{"name":"aws-auth","namespace":"kube-system"},"data":{"mapRoles":"[{\\"rolearn\\":\\"',
                        {
                            'Fn::GetAtt': [
                                'roleC7B7E775',
                                'Arn',
                            ],
                        },
                        '\\",\\"username\\":\\"',
                        {
                            'Fn::GetAtt': [
                                'roleC7B7E775',
                                'Arn',
                            ],
                        },
                        '\\",\\"groups\\":[\\"system:masters\\"]}]","mapUsers":"[]","mapAccounts":"[]"}}]',
                    ],
                ],
            },
        }));
        test.done();
    },
    'addResource can be used to apply k8s manifests on this cluster'(test) {
        // GIVEN
        const { stack, vpc } = util_1.testFixture();
        const cluster = new eks.Cluster(stack, 'Cluster', {
            vpc,
            defaultCapacity: 0,
            version: CLUSTER_VERSION,
        });
        // WHEN
        cluster.addResource('manifest1', { foo: 123 });
        cluster.addResource('manifest2', { bar: 123 }, { boor: [1, 2, 3] });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource(eks.KubernetesResource.RESOURCE_TYPE, {
            Manifest: '[{"foo":123}]',
        }));
        assert_1.expect(stack).to(assert_1.haveResource(eks.KubernetesResource.RESOURCE_TYPE, {
            Manifest: '[{"bar":123},{"boor":[1,2,3]}]',
        }));
        test.done();
    },
    'kubectl resources can be created in a separate stack'(test) {
        // GIVEN
        const { stack, app } = util_1.testFixture();
        const cluster = new eks.Cluster(stack, 'cluster', { version: CLUSTER_VERSION }); // cluster is under stack2
        // WHEN resource is under stack2
        const stack2 = new cdk.Stack(app, 'stack2', { env: { account: stack.account, region: stack.region } });
        new eks.KubernetesResource(stack2, 'myresource', {
            cluster,
            manifest: [{ foo: 'bar' }],
        });
        // THEN
        app.synth(); // no cyclic dependency (see https://github.com/aws/aws-cdk/issues/7231)
        // expect a single resource in the 2nd stack
        assert_1.expect(stack2).toMatch({
            Resources: {
                myresource49C6D325: {
                    Type: 'Custom::AWSCDK-EKS-KubernetesResource',
                    Properties: {
                        ServiceToken: {
                            'Fn::ImportValue': 'Stack:ExportsOutputFnGetAttawscdkawseksKubectlProviderNestedStackawscdkawseksKubectlProviderNestedStackResourceA7AEBA6BOutputsStackawscdkawseksKubectlProviderframeworkonEvent8897FD9BArn49BEF20C',
                        },
                        Manifest: '[{\"foo\":\"bar\"}]',
                        ClusterName: { 'Fn::ImportValue': 'Stack:ExportsOutputRefclusterC5B25D0D98D553F5' },
                        RoleArn: { 'Fn::ImportValue': 'Stack:ExportsOutputFnGetAttclusterCreationRole2B3B5002ArnF05122FC' },
                    },
                    UpdateReplacePolicy: 'Delete',
                    DeletionPolicy: 'Delete',
                },
            },
        });
        test.done();
    },
    'adding capacity will automatically map its IAM role'(test) {
        // GIVEN
        const { stack, vpc } = util_1.testFixture();
        const cluster = new eks.Cluster(stack, 'Cluster', {
            vpc,
            defaultCapacity: 0,
            version: CLUSTER_VERSION,
        });
        // WHEN
        cluster.addCapacity('default', {
            instanceType: new ec2.InstanceType('t2.nano'),
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource(eks.KubernetesResource.RESOURCE_TYPE, {
            Manifest: {
                'Fn::Join': [
                    '',
                    [
                        '[{"apiVersion":"v1","kind":"ConfigMap","metadata":{"name":"aws-auth","namespace":"kube-system"},"data":{"mapRoles":"[{\\"rolearn\\":\\"',
                        {
                            'Fn::GetAtt': [
                                'ClusterMastersRole9AA35625',
                                'Arn',
                            ],
                        },
                        '\\",\\"username\\":\\"',
                        {
                            'Fn::GetAtt': [
                                'ClusterMastersRole9AA35625',
                                'Arn',
                            ],
                        },
                        '\\",\\"groups\\":[\\"system:masters\\"]},{\\"rolearn\\":\\"',
                        {
                            'Fn::GetAtt': [
                                'ClusterdefaultInstanceRoleF20A29CD',
                                'Arn',
                            ],
                        },
                        '\\",\\"username\\":\\"system:node:{{EC2PrivateDNSName}}\\",\\"groups\\":[\\"system:bootstrappers\\",\\"system:nodes\\"]}]","mapUsers":"[]","mapAccounts":"[]"}}]',
                    ],
                ],
            },
        }));
        test.done();
    },
    'addCapacity will *not* map the IAM role if mapRole is false'(test) {
        // GIVEN
        const { stack, vpc } = util_1.testFixture();
        const cluster = new eks.Cluster(stack, 'Cluster', {
            vpc,
            defaultCapacity: 0,
            version: CLUSTER_VERSION,
        });
        // WHEN
        cluster.addCapacity('default', {
            instanceType: new ec2.InstanceType('t2.nano'),
            mapRole: false,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource(eks.KubernetesResource.RESOURCE_TYPE, {
            Manifest: {
                'Fn::Join': [
                    '',
                    [
                        '[{"apiVersion":"v1","kind":"ConfigMap","metadata":{"name":"aws-auth","namespace":"kube-system"},"data":{"mapRoles":"[{\\"rolearn\\":\\"',
                        {
                            'Fn::GetAtt': [
                                'ClusterMastersRole9AA35625',
                                'Arn',
                            ],
                        },
                        '\\",\\"username\\":\\"',
                        {
                            'Fn::GetAtt': [
                                'ClusterMastersRole9AA35625',
                                'Arn',
                            ],
                        },
                        '\\",\\"groups\\":[\\"system:masters\\"]}]","mapUsers":"[]","mapAccounts":"[]"}}]',
                    ],
                ],
            },
        }));
        test.done();
    },
    'outputs': {
        'aws eks update-kubeconfig is the only output synthesized by default'(test) {
            // GIVEN
            const { app, stack } = util_1.testFixtureNoVpc();
            // WHEN
            new eks.Cluster(stack, 'Cluster', { version: CLUSTER_VERSION });
            // THEN
            const assembly = app.synth();
            const template = assembly.getStackByName(stack.stackName).template;
            test.deepEqual(template.Outputs, {
                ClusterConfigCommand43AAE40F: { Value: { 'Fn::Join': ['', ['aws eks update-kubeconfig --name ', { Ref: 'Cluster9EE0221C' }, ' --region us-east-1 --role-arn ', { 'Fn::GetAtt': ['ClusterMastersRole9AA35625', 'Arn'] }]] } },
                ClusterGetTokenCommand06AE992E: { Value: { 'Fn::Join': ['', ['aws eks get-token --cluster-name ', { Ref: 'Cluster9EE0221C' }, ' --region us-east-1 --role-arn ', { 'Fn::GetAtt': ['ClusterMastersRole9AA35625', 'Arn'] }]] } },
            });
            test.done();
        },
        'if masters role is defined, it should be included in the config command'(test) {
            // GIVEN
            const { app, stack } = util_1.testFixtureNoVpc();
            // WHEN
            const mastersRole = new iam.Role(stack, 'masters', { assumedBy: new iam.AccountRootPrincipal() });
            new eks.Cluster(stack, 'Cluster', {
                mastersRole,
                version: CLUSTER_VERSION,
            });
            // THEN
            const assembly = app.synth();
            const template = assembly.getStackByName(stack.stackName).template;
            test.deepEqual(template.Outputs, {
                ClusterConfigCommand43AAE40F: { Value: { 'Fn::Join': ['', ['aws eks update-kubeconfig --name ', { Ref: 'Cluster9EE0221C' }, ' --region us-east-1 --role-arn ', { 'Fn::GetAtt': ['masters0D04F23D', 'Arn'] }]] } },
                ClusterGetTokenCommand06AE992E: { Value: { 'Fn::Join': ['', ['aws eks get-token --cluster-name ', { Ref: 'Cluster9EE0221C' }, ' --region us-east-1 --role-arn ', { 'Fn::GetAtt': ['masters0D04F23D', 'Arn'] }]] } },
            });
            test.done();
        },
        'if `outputConfigCommand=false` will disabled the output'(test) {
            // GIVEN
            const { app, stack } = util_1.testFixtureNoVpc();
            // WHEN
            const mastersRole = new iam.Role(stack, 'masters', { assumedBy: new iam.AccountRootPrincipal() });
            new eks.Cluster(stack, 'Cluster', {
                mastersRole,
                outputConfigCommand: false,
                version: CLUSTER_VERSION,
            });
            // THEN
            const assembly = app.synth();
            const template = assembly.getStackByName(stack.stackName).template;
            test.ok(!template.Outputs); // no outputs
            test.done();
        },
        '`outputClusterName` can be used to synthesize an output with the cluster name'(test) {
            // GIVEN
            const { app, stack } = util_1.testFixtureNoVpc();
            // WHEN
            new eks.Cluster(stack, 'Cluster', {
                outputConfigCommand: false,
                outputClusterName: true,
                version: CLUSTER_VERSION,
            });
            // THEN
            const assembly = app.synth();
            const template = assembly.getStackByName(stack.stackName).template;
            test.deepEqual(template.Outputs, {
                ClusterClusterNameEB26049E: { Value: { Ref: 'Cluster9EE0221C' } },
            });
            test.done();
        },
        '`outputMastersRoleArn` can be used to synthesize an output with the arn of the masters role if defined'(test) {
            // GIVEN
            const { app, stack } = util_1.testFixtureNoVpc();
            // WHEN
            new eks.Cluster(stack, 'Cluster', {
                outputConfigCommand: false,
                outputMastersRoleArn: true,
                mastersRole: new iam.Role(stack, 'masters', { assumedBy: new iam.AccountRootPrincipal() }),
                version: CLUSTER_VERSION,
            });
            // THEN
            const assembly = app.synth();
            const template = assembly.getStackByName(stack.stackName).template;
            test.deepEqual(template.Outputs, {
                ClusterMastersRoleArnB15964B1: { Value: { 'Fn::GetAtt': ['masters0D04F23D', 'Arn'] } },
            });
            test.done();
        },
        'boostrap user-data': {
            'rendered by default for ASGs'(test) {
                // GIVEN
                const { app, stack } = util_1.testFixtureNoVpc();
                const cluster = new eks.Cluster(stack, 'Cluster', { defaultCapacity: 0, version: CLUSTER_VERSION });
                // WHEN
                cluster.addCapacity('MyCapcity', { instanceType: new ec2.InstanceType('m3.xlargs') });
                // THEN
                const template = app.synth().getStackByName(stack.stackName).template;
                const userData = template.Resources.ClusterMyCapcityLaunchConfig58583345.Properties.UserData;
                test.deepEqual(userData, { 'Fn::Base64': { 'Fn::Join': ['', ['#!/bin/bash\nset -o xtrace\n/etc/eks/bootstrap.sh ', { Ref: 'Cluster9EE0221C' }, ' --kubelet-extra-args "--node-labels lifecycle=OnDemand" --use-max-pods true\n/opt/aws/bin/cfn-signal --exit-code $? --stack Stack --resource ClusterMyCapcityASGD4CD8B97 --region us-east-1']] } });
                test.done();
            },
            'not rendered if bootstrap is disabled'(test) {
                // GIVEN
                const { app, stack } = util_1.testFixtureNoVpc();
                const cluster = new eks.Cluster(stack, 'Cluster', { defaultCapacity: 0, version: CLUSTER_VERSION });
                // WHEN
                cluster.addCapacity('MyCapcity', {
                    instanceType: new ec2.InstanceType('m3.xlargs'),
                    bootstrapEnabled: false,
                });
                // THEN
                const template = app.synth().getStackByName(stack.stackName).template;
                const userData = template.Resources.ClusterMyCapcityLaunchConfig58583345.Properties.UserData;
                test.deepEqual(userData, { 'Fn::Base64': '#!/bin/bash' });
                test.done();
            },
            // cursory test for options: see test.user-data.ts for full suite
            'bootstrap options'(test) {
                // GIVEN
                const { app, stack } = util_1.testFixtureNoVpc();
                const cluster = new eks.Cluster(stack, 'Cluster', { defaultCapacity: 0, version: CLUSTER_VERSION });
                // WHEN
                cluster.addCapacity('MyCapcity', {
                    instanceType: new ec2.InstanceType('m3.xlargs'),
                    bootstrapOptions: {
                        kubeletExtraArgs: '--node-labels FOO=42',
                    },
                });
                // THEN
                const template = app.synth().getStackByName(stack.stackName).template;
                const userData = template.Resources.ClusterMyCapcityLaunchConfig58583345.Properties.UserData;
                test.deepEqual(userData, { 'Fn::Base64': { 'Fn::Join': ['', ['#!/bin/bash\nset -o xtrace\n/etc/eks/bootstrap.sh ', { Ref: 'Cluster9EE0221C' }, ' --kubelet-extra-args "--node-labels lifecycle=OnDemand  --node-labels FOO=42" --use-max-pods true\n/opt/aws/bin/cfn-signal --exit-code $? --stack Stack --resource ClusterMyCapcityASGD4CD8B97 --region us-east-1']] } });
                test.done();
            },
            'spot instances': {
                'nodes labeled an tainted accordingly'(test) {
                    // GIVEN
                    const { app, stack } = util_1.testFixtureNoVpc();
                    const cluster = new eks.Cluster(stack, 'Cluster', { defaultCapacity: 0, version: CLUSTER_VERSION });
                    // WHEN
                    cluster.addCapacity('MyCapcity', {
                        instanceType: new ec2.InstanceType('m3.xlargs'),
                        spotPrice: '0.01',
                    });
                    // THEN
                    const template = app.synth().getStackByName(stack.stackName).template;
                    const userData = template.Resources.ClusterMyCapcityLaunchConfig58583345.Properties.UserData;
                    test.deepEqual(userData, { 'Fn::Base64': { 'Fn::Join': ['', ['#!/bin/bash\nset -o xtrace\n/etc/eks/bootstrap.sh ', { Ref: 'Cluster9EE0221C' }, ' --kubelet-extra-args "--node-labels lifecycle=Ec2Spot --register-with-taints=spotInstance=true:PreferNoSchedule" --use-max-pods true\n/opt/aws/bin/cfn-signal --exit-code $? --stack Stack --resource ClusterMyCapcityASGD4CD8B97 --region us-east-1']] } });
                    test.done();
                },
                'interrupt handler is added'(test) {
                    // GIVEN
                    const { stack } = util_1.testFixtureNoVpc();
                    const cluster = new eks.Cluster(stack, 'Cluster', { defaultCapacity: 0, version: CLUSTER_VERSION });
                    // WHEN
                    cluster.addCapacity('MyCapcity', {
                        instanceType: new ec2.InstanceType('m3.xlarge'),
                        spotPrice: '0.01',
                    });
                    // THEN
                    assert_1.expect(stack).to(assert_1.haveResource(eks.HelmChart.RESOURCE_TYPE, {
                        Release: 'stackclusterchartspotinterrupthandlerdec62e07',
                        Chart: 'aws-node-termination-handler',
                        Values: '{\"nodeSelector.lifecycle\":\"Ec2Spot\"}',
                        Namespace: 'kube-system',
                        Repository: 'https://aws.github.io/eks-charts',
                    }));
                    test.done();
                },
                'its possible to add two capacities with spot instances and only one stop handler will be installed'(test) {
                    // GIVEN
                    const { stack } = util_1.testFixtureNoVpc();
                    const cluster = new eks.Cluster(stack, 'Cluster', { defaultCapacity: 0, version: CLUSTER_VERSION });
                    // WHEN
                    cluster.addCapacity('Spot1', {
                        instanceType: new ec2.InstanceType('m3.xlarge'),
                        spotPrice: '0.01',
                    });
                    cluster.addCapacity('Spot2', {
                        instanceType: new ec2.InstanceType('m4.xlarge'),
                        spotPrice: '0.01',
                    });
                    // THEN
                    assert_1.expect(stack).to(assert_1.countResources(eks.HelmChart.RESOURCE_TYPE, 1));
                    test.done();
                },
            },
        },
        'if bootstrap is disabled cannot specify options'(test) {
            // GIVEN
            const { stack } = util_1.testFixtureNoVpc();
            const cluster = new eks.Cluster(stack, 'Cluster', { defaultCapacity: 0, version: CLUSTER_VERSION });
            // THEN
            test.throws(() => cluster.addCapacity('MyCapcity', {
                instanceType: new ec2.InstanceType('m3.xlargs'),
                bootstrapEnabled: false,
                bootstrapOptions: { awsApiRetryAttempts: 10 },
            }), /Cannot specify "bootstrapOptions" if "bootstrapEnabled" is false/);
            test.done();
        },
        'EksOptimizedImage() with no nodeType always uses STANDARD with LATEST_KUBERNETES_VERSION'(test) {
            // GIVEN
            const { app, stack } = util_1.testFixtureNoVpc();
            const LATEST_KUBERNETES_VERSION = '1.14';
            // WHEN
            new eks.EksOptimizedImage().getImage(stack);
            // THEN
            const assembly = app.synth();
            const parameters = assembly.getStackByName(stack.stackName).template.Parameters;
            test.ok(Object.entries(parameters).some(([k, v]) => k.startsWith('SsmParameterValueawsserviceeksoptimizedami') &&
                v.Default.includes('/amazon-linux-2/')), 'EKS STANDARD AMI should be in ssm parameters');
            test.ok(Object.entries(parameters).some(([k, v]) => k.startsWith('SsmParameterValueawsserviceeksoptimizedami') &&
                v.Default.includes(LATEST_KUBERNETES_VERSION)), 'LATEST_KUBERNETES_VERSION should be in ssm parameters');
            test.done();
        },
        'EksOptimizedImage() with specific kubernetesVersion return correct AMI'(test) {
            // GIVEN
            const { app, stack } = util_1.testFixtureNoVpc();
            // WHEN
            new eks.EksOptimizedImage({ kubernetesVersion: '1.15' }).getImage(stack);
            // THEN
            const assembly = app.synth();
            const parameters = assembly.getStackByName(stack.stackName).template.Parameters;
            test.ok(Object.entries(parameters).some(([k, v]) => k.startsWith('SsmParameterValueawsserviceeksoptimizedami') &&
                v.Default.includes('/amazon-linux-2/')), 'EKS STANDARD AMI should be in ssm parameters');
            test.ok(Object.entries(parameters).some(([k, v]) => k.startsWith('SsmParameterValueawsserviceeksoptimizedami') &&
                v.Default.includes('/1.15/')), 'kubernetesVersion should be in ssm parameters');
            test.done();
        },
        'EKS-Optimized AMI with GPU support when addCapacity'(test) {
            // GIVEN
            const { app, stack } = util_1.testFixtureNoVpc();
            // WHEN
            new eks.Cluster(stack, 'cluster', {
                defaultCapacity: 0,
                version: CLUSTER_VERSION,
            }).addCapacity('GPUCapacity', {
                instanceType: new ec2.InstanceType('g4dn.xlarge'),
            });
            // THEN
            const assembly = app.synth();
            const parameters = assembly.getStackByName(stack.stackName).template.Parameters;
            test.ok(Object.entries(parameters).some(([k, v]) => k.startsWith('SsmParameterValueawsserviceeksoptimizedami') && v.Default.includes('amazon-linux-2-gpu')), 'EKS AMI with GPU should be in ssm parameters');
            test.done();
        },
        'when using custom resource a creation role & policy is defined'(test) {
            // GIVEN
            const { stack } = util_1.testFixture();
            // WHEN
            new eks.Cluster(stack, 'MyCluster', {
                clusterName: 'my-cluster-name',
                version: CLUSTER_VERSION,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('Custom::AWSCDK-EKS-Cluster', {
                Config: {
                    name: 'my-cluster-name',
                    roleArn: { 'Fn::GetAtt': ['MyClusterRoleBA20FE72', 'Arn'] },
                    version: '1.16',
                    resourcesVpcConfig: {
                        securityGroupIds: [{ 'Fn::GetAtt': ['MyClusterControlPlaneSecurityGroup6B658F79', 'GroupId'] }],
                        subnetIds: [
                            { Ref: 'MyClusterDefaultVpcPublicSubnet1SubnetFAE5A9B6' },
                            { Ref: 'MyClusterDefaultVpcPublicSubnet2SubnetF6D028A0' },
                            { Ref: 'MyClusterDefaultVpcPrivateSubnet1SubnetE1D0DCDB' },
                            { Ref: 'MyClusterDefaultVpcPrivateSubnet2Subnet11FEA8D0' },
                        ],
                        endpointPrivateAccess: true,
                        endpointPublicAccess: true,
                    },
                },
            }));
            // role can be assumed by 3 lambda handlers (2 for the cluster resource and 1 for the kubernetes resource)
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Role', {
                AssumeRolePolicyDocument: {
                    Statement: [
                        {
                            Action: 'sts:AssumeRole',
                            Effect: 'Allow',
                            Principal: {
                                AWS: [
                                    {
                                        'Fn::GetAtt': [
                                            'awscdkawseksClusterResourceProviderNestedStackawscdkawseksClusterResourceProviderNestedStackResource9827C454',
                                            'Outputs.StackawscdkawseksClusterResourceProviderOnEventHandlerServiceRole3AEE0A43Arn',
                                        ],
                                    },
                                    {
                                        'Fn::GetAtt': [
                                            'awscdkawseksClusterResourceProviderNestedStackawscdkawseksClusterResourceProviderNestedStackResource9827C454',
                                            'Outputs.StackawscdkawseksClusterResourceProviderIsCompleteHandlerServiceRole8E7F1C11Arn',
                                        ],
                                    },
                                ],
                            },
                        },
                        {
                            Action: 'sts:AssumeRole',
                            Effect: 'Allow',
                            Principal: {
                                AWS: {
                                    'Fn::GetAtt': [
                                        'awscdkawseksKubectlProviderNestedStackawscdkawseksKubectlProviderNestedStackResourceA7AEBA6B',
                                        'Outputs.StackawscdkawseksKubectlProviderHandlerServiceRole2C52B3ECArn',
                                    ],
                                },
                            },
                        },
                    ],
                    Version: '2012-10-17',
                },
            }));
            // policy allows creation role to pass the cluster role and to interact with the cluster (given we know the explicit cluster name)
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: 'iam:PassRole',
                            Effect: 'Allow',
                            Resource: {
                                'Fn::GetAtt': [
                                    'MyClusterRoleBA20FE72',
                                    'Arn',
                                ],
                            },
                        },
                        {
                            Action: [
                                'ec2:DescribeSubnets',
                                'ec2:DescribeRouteTables',
                            ],
                            Effect: 'Allow',
                            Resource: '*',
                        },
                        {
                            Action: [
                                'eks:CreateCluster',
                                'eks:DescribeCluster',
                                'eks:DescribeUpdate',
                                'eks:DeleteCluster',
                                'eks:UpdateClusterVersion',
                                'eks:UpdateClusterConfig',
                                'eks:CreateFargateProfile',
                                'eks:TagResource',
                                'eks:UntagResource',
                            ],
                            Effect: 'Allow',
                            Resource: [{
                                    'Fn::Join': [
                                        '',
                                        [
                                            'arn:',
                                            {
                                                Ref: 'AWS::Partition',
                                            },
                                            ':eks:us-east-1:',
                                            {
                                                Ref: 'AWS::AccountId',
                                            },
                                            ':cluster/my-cluster-name',
                                        ],
                                    ],
                                }, {
                                    'Fn::Join': [
                                        '',
                                        [
                                            'arn:',
                                            {
                                                Ref: 'AWS::Partition',
                                            },
                                            ':eks:us-east-1:',
                                            {
                                                Ref: 'AWS::AccountId',
                                            },
                                            ':cluster/my-cluster-name/*',
                                        ],
                                    ],
                                }],
                        },
                        {
                            Action: [
                                'eks:DescribeFargateProfile',
                                'eks:DeleteFargateProfile',
                            ],
                            Effect: 'Allow',
                            Resource: {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':eks:us-east-1:',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        ':fargateprofile/my-cluster-name/*',
                                    ],
                                ],
                            },
                        },
                        {
                            Action: ['iam:GetRole', 'iam:listAttachedRolePolicies'],
                            Effect: 'Allow',
                            Resource: '*',
                        },
                        {
                            Action: 'iam:CreateServiceLinkedRole',
                            Effect: 'Allow',
                            Resource: '*',
                        },
                        {
                            Action: 'ec2:DescribeVpcs',
                            Effect: 'Allow',
                            Resource: {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':ec2:us-east-1:',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        ':vpc/',
                                        {
                                            Ref: 'MyClusterDefaultVpc76C24A38',
                                        },
                                    ],
                                ],
                            },
                        },
                    ],
                    Version: '2012-10-17',
                },
            }));
            test.done();
        },
        'if an explicit cluster name is not provided, the creation role policy is wider (allows interacting with all clusters)'(test) {
            // GIVEN
            const { stack } = util_1.testFixture();
            // WHEN
            new eks.Cluster(stack, 'MyCluster', { version: CLUSTER_VERSION });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: 'iam:PassRole',
                            Effect: 'Allow',
                            Resource: {
                                'Fn::GetAtt': [
                                    'MyClusterRoleBA20FE72',
                                    'Arn',
                                ],
                            },
                        },
                        {
                            Action: [
                                'ec2:DescribeSubnets',
                                'ec2:DescribeRouteTables',
                            ],
                            Effect: 'Allow',
                            Resource: '*',
                        },
                        {
                            Action: [
                                'eks:CreateCluster',
                                'eks:DescribeCluster',
                                'eks:DescribeUpdate',
                                'eks:DeleteCluster',
                                'eks:UpdateClusterVersion',
                                'eks:UpdateClusterConfig',
                                'eks:CreateFargateProfile',
                                'eks:TagResource',
                                'eks:UntagResource',
                            ],
                            Effect: 'Allow',
                            Resource: ['*'],
                        },
                        {
                            Action: [
                                'eks:DescribeFargateProfile',
                                'eks:DeleteFargateProfile',
                            ],
                            Effect: 'Allow',
                            Resource: '*',
                        },
                        {
                            Action: ['iam:GetRole', 'iam:listAttachedRolePolicies'],
                            Effect: 'Allow',
                            Resource: '*',
                        },
                        {
                            Action: 'iam:CreateServiceLinkedRole',
                            Effect: 'Allow',
                            Resource: '*',
                        },
                        {
                            Action: 'ec2:DescribeVpcs',
                            Effect: 'Allow',
                            Resource: {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':ec2:us-east-1:',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        ':vpc/',
                                        {
                                            Ref: 'MyClusterDefaultVpc76C24A38',
                                        },
                                    ],
                                ],
                            },
                        },
                    ],
                    Version: '2012-10-17',
                },
            }));
            test.done();
        },
        'if helm charts are used, its resource provider is allowed to assume the creation role'(test) {
            // GIVEN
            const { stack } = util_1.testFixture();
            const cluster = new eks.Cluster(stack, 'MyCluster', {
                clusterName: 'my-cluster-name',
                version: CLUSTER_VERSION,
            });
            // WHEN
            cluster.addChart('MyChart', {
                chart: 'foo',
            });
            // THEN
            // role can be assumed by 4 principals: two for the cluster resource, one
            // for kubernetes resource and one for the helm resource.
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Role', {
                AssumeRolePolicyDocument: {
                    Statement: [
                        {
                            Action: 'sts:AssumeRole',
                            Effect: 'Allow',
                            Principal: {
                                AWS: [
                                    {
                                        'Fn::GetAtt': [
                                            'awscdkawseksClusterResourceProviderNestedStackawscdkawseksClusterResourceProviderNestedStackResource9827C454',
                                            'Outputs.StackawscdkawseksClusterResourceProviderOnEventHandlerServiceRole3AEE0A43Arn',
                                        ],
                                    },
                                    {
                                        'Fn::GetAtt': [
                                            'awscdkawseksClusterResourceProviderNestedStackawscdkawseksClusterResourceProviderNestedStackResource9827C454',
                                            'Outputs.StackawscdkawseksClusterResourceProviderIsCompleteHandlerServiceRole8E7F1C11Arn',
                                        ],
                                    },
                                ],
                            },
                        },
                        {
                            Action: 'sts:AssumeRole',
                            Effect: 'Allow',
                            Principal: {
                                AWS: {
                                    'Fn::GetAtt': [
                                        'awscdkawseksKubectlProviderNestedStackawscdkawseksKubectlProviderNestedStackResourceA7AEBA6B',
                                        'Outputs.StackawscdkawseksKubectlProviderHandlerServiceRole2C52B3ECArn',
                                    ],
                                },
                            },
                        },
                    ],
                    Version: '2012-10-17',
                },
            }));
            test.done();
        },
        'coreDnsComputeType will patch the coreDNS configuration to use a "fargate" compute type and restore to "ec2" upon removal'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            // WHEN
            new eks.Cluster(stack, 'MyCluster', {
                coreDnsComputeType: eks.CoreDnsComputeType.FARGATE,
                version: CLUSTER_VERSION,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('Custom::AWSCDK-EKS-KubernetesPatch', {
                ResourceName: 'deployment/coredns',
                ResourceNamespace: 'kube-system',
                ApplyPatchJson: '{"spec":{"template":{"metadata":{"annotations":{"eks.amazonaws.com/compute-type":"fargate"}}}}}',
                RestorePatchJson: '{"spec":{"template":{"metadata":{"annotations":{"eks.amazonaws.com/compute-type":"ec2"}}}}}',
                ClusterName: {
                    Ref: 'MyCluster8AD82BF8',
                },
                RoleArn: {
                    'Fn::GetAtt': [
                        'MyClusterCreationRoleB5FA4FF3',
                        'Arn',
                    ],
                },
            }));
            test.done();
        },
        'if openIDConnectProvider a new OpenIDConnectProvider resource is created and exposed'(test) {
            // GIVEN
            const { stack } = util_1.testFixtureNoVpc();
            const cluster = new eks.Cluster(stack, 'Cluster', { defaultCapacity: 0, version: CLUSTER_VERSION });
            // WHEN
            const provider = cluster.openIdConnectProvider;
            // THEN
            test.equal(provider, cluster.openIdConnectProvider, 'openIdConnect provider is different and created more than once.');
            assert_1.expect(stack).to(assert_1.haveResource('Custom::AWSCDKOpenIdConnectProvider', {
                ServiceToken: {
                    'Fn::GetAtt': [
                        'CustomAWSCDKOpenIdConnectProviderCustomResourceProviderHandlerF2C543E0',
                        'Arn',
                    ],
                },
                ClientIDList: [
                    'sts.amazonaws.com',
                ],
                ThumbprintList: [
                    '9e99a48a9960b14926bb7f3b02e22da2b0ab7280',
                ],
                Url: {
                    'Fn::GetAtt': [
                        'Cluster9EE0221C',
                        'OpenIdConnectIssuerUrl',
                    ],
                },
            }));
            test.done();
        },
        'inference instances are supported'(test) {
            // GIVEN
            const { stack } = util_1.testFixtureNoVpc();
            const cluster = new eks.Cluster(stack, 'Cluster', { defaultCapacity: 0, version: CLUSTER_VERSION });
            // WHEN
            cluster.addCapacity('InferenceInstances', {
                instanceType: new ec2.InstanceType('inf1.2xlarge'),
                minCapacity: 1,
            });
            const fileContents = fs.readFileSync(path.join(__dirname, '../lib', 'addons/neuron-device-plugin.yaml'), 'utf8');
            const sanitized = YAML.parse(fileContents);
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource(eks.KubernetesResource.RESOURCE_TYPE, {
                Manifest: JSON.stringify([sanitized]),
            }));
            test.done();
        },
        'kubectl resources are always created after all fargate profiles'(test) {
            // GIVEN
            const { stack, app } = util_1.testFixture();
            const cluster = new eks.Cluster(stack, 'Cluster', { version: CLUSTER_VERSION });
            // WHEN
            cluster.addFargateProfile('profile1', { selectors: [{ namespace: 'profile1' }] });
            cluster.addResource('resource1', { foo: 123 });
            cluster.addFargateProfile('profile2', { selectors: [{ namespace: 'profile2' }] });
            new eks.HelmChart(stack, 'chart', { cluster, chart: 'mychart' });
            cluster.addFargateProfile('profile3', { selectors: [{ namespace: 'profile3' }] });
            new eks.KubernetesPatch(stack, 'patch1', {
                cluster,
                applyPatch: { foo: 123 },
                restorePatch: { bar: 123 },
                resourceName: 'foo/bar',
            });
            cluster.addFargateProfile('profile4', { selectors: [{ namespace: 'profile4' }] });
            // THEN
            const template = app.synth().getStackArtifact(stack.artifactId).template;
            const barrier = template.Resources.ClusterKubectlReadyBarrier200052AF;
            test.deepEqual(barrier.DependsOn, [
                'Clusterfargateprofileprofile1PodExecutionRoleE85F87B5',
                'Clusterfargateprofileprofile129AEA3C6',
                'Clusterfargateprofileprofile2PodExecutionRole22670AF8',
                'Clusterfargateprofileprofile233B9A117',
                'Clusterfargateprofileprofile3PodExecutionRole475C0D8F',
                'Clusterfargateprofileprofile3D06F3076',
                'Clusterfargateprofileprofile4PodExecutionRole086057FB',
                'Clusterfargateprofileprofile4A0E3BBE8',
                'ClusterCreationRoleDefaultPolicyE8BDFC7B',
                'ClusterCreationRole360249B6',
                'Cluster9EE0221C',
            ]);
            const kubectlResources = ['chartF2447AFC', 'patch1B964AC93', 'Clustermanifestresource10B1C9505', 'ClusterAwsAuthmanifestFE51F8AE'];
            // check that all kubectl resources depend on the barrier
            for (const r of kubectlResources) {
                test.deepEqual(template.Resources[r].DependsOn, ['ClusterKubectlReadyBarrier200052AF']);
            }
            test.done();
        },
        'kubectl provider role is trusted to assume cluster creation role'(test) {
            // GIVEN
            const { stack, app } = util_1.testFixture();
            const c1 = new eks.Cluster(stack, 'Cluster1', { version: CLUSTER_VERSION });
            const c2 = new eks.Cluster(stack, 'Cluster2', { version: CLUSTER_VERSION });
            // WHEN
            // activate kubectl provider
            c1.addResource('c1a', { foo: 123 });
            c1.addResource('c1b', { foo: 123 });
            c2.addResource('c2', { foo: 123 });
            // THEN
            const template = app.synth().getStackArtifact(stack.artifactId).template;
            const creationRoleToKubectlRole = {
                Cluster1CreationRoleA231BE8D: 'Outputs.StackawscdkawseksKubectlProviderHandlerServiceRole2C52B3ECArn',
                Cluster2CreationRole9254EAB6: 'Outputs.StackawscdkawseksKubectlProviderHandlerServiceRole2C52B3ECArn',
            };
            // verify that the kubectl role appears as the 2nd IAM trust policy statement
            for (const [creationRole, kubectlRole] of Object.entries(creationRoleToKubectlRole)) {
                const trustPolicy = template.Resources[creationRole].Properties.AssumeRolePolicyDocument.Statement;
                test.equal(trustPolicy.length, 2, 'expecting the creation role\'s trust policy to include two statements');
                test.deepEqual(trustPolicy[1].Principal.AWS['Fn::GetAtt'][1], kubectlRole);
            }
            test.done();
        },
    },
    'kubectl provider passes environment to lambda'(test) {
        const { stack } = util_1.testFixture();
        const cluster = new eks.Cluster(stack, 'Cluster1', {
            version: CLUSTER_VERSION, endpointAccess: eks.EndpointAccess.PRIVATE,
            kubectlEnvironment: {
                Foo: 'Bar',
            },
        });
        cluster.addResource('resource', {
            kind: 'ConfigMap',
            apiVersion: 'v1',
            data: {
                hello: 'world',
            },
            metadata: {
                name: 'config-map',
            },
        });
        // the kubectl provider is inside a nested stack.
        const nested = stack.node.tryFindChild('@aws-cdk/aws-eks.KubectlProvider');
        assert_1.expect(nested).to(assert_1.haveResource('AWS::Lambda::Function', {
            Environment: {
                Variables: {
                    Foo: 'Bar',
                },
            },
        }));
        test.done();
    },
    'endpoint access': {
        'can configure private endpoint access'(test) {
            // GIVEN
            const { stack } = util_1.testFixture();
            new eks.Cluster(stack, 'Cluster1', { version: CLUSTER_VERSION, endpointAccess: eks.EndpointAccess.PRIVATE });
            assert_1.expect(stack).to(assert_1.haveResource('Custom::AWSCDK-EKS-Cluster', {
                Config: {
                    roleArn: { 'Fn::GetAtt': ['Cluster1RoleE88C32AD', 'Arn'] },
                    version: '1.16',
                    resourcesVpcConfig: {
                        securityGroupIds: [{ 'Fn::GetAtt': ['Cluster1ControlPlaneSecurityGroupF9C67C32', 'GroupId'] }],
                        subnetIds: [
                            { Ref: 'Cluster1DefaultVpcPublicSubnet1SubnetBEABA6ED' },
                            { Ref: 'Cluster1DefaultVpcPublicSubnet2Subnet947A5158' },
                            { Ref: 'Cluster1DefaultVpcPrivateSubnet1Subnet4E30ECA1' },
                            { Ref: 'Cluster1DefaultVpcPrivateSubnet2Subnet707FCD37' },
                        ],
                        endpointPrivateAccess: true,
                        endpointPublicAccess: false,
                    },
                },
            }));
            test.done();
        },
        'can configure cidr blocks in public endpoint access'(test) {
            // GIVEN
            const { stack } = util_1.testFixture();
            new eks.Cluster(stack, 'Cluster1', { version: CLUSTER_VERSION, endpointAccess: eks.EndpointAccess.PUBLIC.onlyFrom('1.2.3.4/5') });
            assert_1.expect(stack).to(assert_1.haveResource('Custom::AWSCDK-EKS-Cluster', {
                Config: {
                    roleArn: { 'Fn::GetAtt': ['Cluster1RoleE88C32AD', 'Arn'] },
                    version: '1.16',
                    resourcesVpcConfig: {
                        securityGroupIds: [{ 'Fn::GetAtt': ['Cluster1ControlPlaneSecurityGroupF9C67C32', 'GroupId'] }],
                        subnetIds: [
                            { Ref: 'Cluster1DefaultVpcPublicSubnet1SubnetBEABA6ED' },
                            { Ref: 'Cluster1DefaultVpcPublicSubnet2Subnet947A5158' },
                            { Ref: 'Cluster1DefaultVpcPrivateSubnet1Subnet4E30ECA1' },
                            { Ref: 'Cluster1DefaultVpcPrivateSubnet2Subnet707FCD37' },
                        ],
                        endpointPrivateAccess: false,
                        endpointPublicAccess: true,
                        publicAccessCidrs: ['1.2.3.4/5'],
                    },
                },
            }));
            test.done();
        },
        'kubectl provider chooses only private subnets'(test) {
            const { stack } = util_1.testFixture();
            const vpc = new ec2.Vpc(stack, 'Vpc', {
                maxAzs: 2,
                natGateways: 1,
                subnetConfiguration: [
                    {
                        subnetType: ec2.SubnetType.PRIVATE,
                        name: 'Private1',
                    },
                    {
                        subnetType: ec2.SubnetType.PUBLIC,
                        name: 'Public1',
                    },
                ],
            });
            const cluster = new eks.Cluster(stack, 'Cluster1', {
                version: CLUSTER_VERSION, endpointAccess: eks.EndpointAccess.PRIVATE,
                vpc,
            });
            cluster.addResource('resource', {
                kind: 'ConfigMap',
                apiVersion: 'v1',
                data: {
                    hello: 'world',
                },
                metadata: {
                    name: 'config-map',
                },
            });
            // the kubectl provider is inside a nested stack.
            const nested = stack.node.tryFindChild('@aws-cdk/aws-eks.KubectlProvider');
            assert_1.expect(nested).to(assert_1.haveResource('AWS::Lambda::Function', {
                VpcConfig: {
                    SecurityGroupIds: [
                        {
                            Ref: 'referencetoStackCluster1KubectlProviderSecurityGroupDF05D03AGroupId',
                        },
                    ],
                    SubnetIds: [
                        {
                            Ref: 'referencetoStackVpcPrivate1Subnet1Subnet6764A0F6Ref',
                        },
                        {
                            Ref: 'referencetoStackVpcPrivate1Subnet2SubnetDFD49645Ref',
                        },
                    ],
                },
            }));
            test.done();
        },
        'kubectl provider limits number of subnets to 16'(test) {
            const { stack } = util_1.testFixture();
            const subnetConfiguration = [];
            for (let i = 0; i < 20; i++) {
                subnetConfiguration.push({
                    subnetType: ec2.SubnetType.PRIVATE,
                    name: `Private${i}`,
                });
            }
            subnetConfiguration.push({
                subnetType: ec2.SubnetType.PUBLIC,
                name: 'Public1',
            });
            const vpc2 = new ec2.Vpc(stack, 'Vpc', {
                maxAzs: 2,
                natGateways: 1,
                subnetConfiguration,
            });
            const cluster = new eks.Cluster(stack, 'Cluster1', {
                version: CLUSTER_VERSION, endpointAccess: eks.EndpointAccess.PRIVATE,
                vpc: vpc2,
            });
            cluster.addResource('resource', {
                kind: 'ConfigMap',
                apiVersion: 'v1',
                data: {
                    hello: 'world',
                },
                metadata: {
                    name: 'config-map',
                },
            });
            // the kubectl provider is inside a nested stack.
            const nested = stack.node.tryFindChild('@aws-cdk/aws-eks.KubectlProvider');
            test.equal(16, assert_1.expect(nested).value.Resources.Handler886CB40B.Properties.VpcConfig.SubnetIds.length);
            test.done();
        },
        'kubectl provider considers vpc subnet selection'(test) {
            const { stack } = util_1.testFixture();
            const subnetConfiguration = [];
            for (let i = 0; i < 20; i++) {
                subnetConfiguration.push({
                    subnetType: ec2.SubnetType.PRIVATE,
                    name: `Private${i}`,
                });
            }
            subnetConfiguration.push({
                subnetType: ec2.SubnetType.PUBLIC,
                name: 'Public1',
            });
            const vpc2 = new ec2.Vpc(stack, 'Vpc', {
                maxAzs: 2,
                natGateways: 1,
                subnetConfiguration,
            });
            const cluster = new eks.Cluster(stack, 'Cluster1', {
                version: CLUSTER_VERSION, endpointAccess: eks.EndpointAccess.PRIVATE,
                vpc: vpc2,
                vpcSubnets: [{ subnetGroupName: 'Private1' }, { subnetGroupName: 'Private2' }],
            });
            cluster.addResource('resource', {
                kind: 'ConfigMap',
                apiVersion: 'v1',
                data: {
                    hello: 'world',
                },
                metadata: {
                    name: 'config-map',
                },
            });
            // the kubectl provider is inside a nested stack.
            const nested = stack.node.tryFindChild('@aws-cdk/aws-eks.KubectlProvider');
            assert_1.expect(nested).to(assert_1.haveResource('AWS::Lambda::Function', {
                VpcConfig: {
                    SecurityGroupIds: [
                        {
                            Ref: 'referencetoStackCluster1KubectlProviderSecurityGroupDF05D03AGroupId',
                        },
                    ],
                    SubnetIds: [
                        {
                            Ref: 'referencetoStackVpcPrivate1Subnet1Subnet6764A0F6Ref',
                        },
                        {
                            Ref: 'referencetoStackVpcPrivate1Subnet2SubnetDFD49645Ref',
                        },
                        {
                            Ref: 'referencetoStackVpcPrivate2Subnet1Subnet586AD392Ref',
                        },
                        {
                            Ref: 'referencetoStackVpcPrivate2Subnet2SubnetE42148C0Ref',
                        },
                    ],
                },
            }));
            test.done();
        },
        'throw when private access is configured without dns support enabled for the VPC'(test) {
            const { stack } = util_1.testFixture();
            test.throws(() => {
                new eks.Cluster(stack, 'Cluster', {
                    vpc: new ec2.Vpc(stack, 'Vpc', {
                        enableDnsSupport: false,
                    }),
                    version: CLUSTER_VERSION,
                });
            }, /Private endpoint access requires the VPC to have DNS support and DNS hostnames enabled/);
            test.done();
        },
        'throw when private access is configured without dns hostnames enabled for the VPC'(test) {
            const { stack } = util_1.testFixture();
            test.throws(() => {
                new eks.Cluster(stack, 'Cluster', {
                    vpc: new ec2.Vpc(stack, 'Vpc', {
                        enableDnsHostnames: false,
                    }),
                    version: CLUSTER_VERSION,
                });
            }, /Private endpoint access requires the VPC to have DNS support and DNS hostnames enabled/);
            test.done();
        },
        'throw when cidrs are configured without public access endpoint'(test) {
            test.throws(() => {
                eks.EndpointAccess.PRIVATE.onlyFrom('1.2.3.4/5');
            }, /CIDR blocks can only be configured when public access is enabled/);
            test.done();
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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