"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/// !cdk-integ pragma:ignore-assets
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const eks = require("../lib");
const hello = require("./hello-k8s");
const util_1 = require("./util");
class EksClusterStack extends util_1.TestStack {
    constructor(scope, id) {
        super(scope, id);
        // allow all account users to assume this role in order to admin the cluster
        const mastersRole = new iam.Role(this, 'AdminRole', {
            assumedBy: new iam.AccountRootPrincipal(),
        });
        // just need one nat gateway to simplify the test
        const vpc = new ec2.Vpc(this, 'Vpc', { maxAzs: 3, natGateways: 1 });
        // create the cluster with a default nodegroup capacity
        const cluster = new eks.Cluster(this, 'Cluster', {
            vpc,
            mastersRole,
            defaultCapacity: 2,
            version: eks.KubernetesVersion.V1_16,
        });
        // fargate profile for resources in the "default" namespace
        cluster.addFargateProfile('default', {
            selectors: [{ namespace: 'default' }],
        });
        // add some capacity to the cluster. The IAM instance role will
        // automatically be mapped via aws-auth to allow nodes to join the cluster.
        cluster.addCapacity('Nodes', {
            instanceType: new ec2.InstanceType('t2.medium'),
            minCapacity: 3,
        });
        // add bottlerocket nodes
        cluster.addCapacity('BottlerocketNodes', {
            instanceType: new ec2.InstanceType('t3.small'),
            minCapacity: 2,
            machineImageType: eks.MachineImageType.BOTTLEROCKET,
        });
        // spot instances (up to 10)
        cluster.addCapacity('spot', {
            spotPrice: '0.1094',
            instanceType: new ec2.InstanceType('t3.large'),
            maxCapacity: 10,
            bootstrapOptions: {
                kubeletExtraArgs: '--node-labels foo=bar,goo=far',
                awsApiRetryAttempts: 5,
            },
        });
        // inference instances
        cluster.addCapacity('InferenceInstances', {
            instanceType: new ec2.InstanceType('inf1.2xlarge'),
            minCapacity: 1,
        });
        // add a extra nodegroup
        cluster.addNodegroup('extra-ng', {
            instanceType: new ec2.InstanceType('t3.small'),
            minSize: 1,
            // reusing the default capacity nodegroup instance role when available
            nodeRole: cluster.defaultCapacity ? cluster.defaultCapacity.role : undefined,
        });
        // apply a kubernetes manifest
        cluster.addResource('HelloApp', ...hello.resources);
        // deploy the Kubernetes dashboard through a helm chart
        cluster.addChart('dashboard', {
            chart: 'kubernetes-dashboard',
            repository: 'https://kubernetes.github.io/dashboard/',
        });
        // deploy an nginx ingress in a namespace
        const nginxNamespace = cluster.addResource('nginx-namespace', {
            apiVersion: 'v1',
            kind: 'Namespace',
            metadata: {
                name: 'nginx',
            },
        });
        const nginxIngress = cluster.addChart('nginx-ingress', {
            chart: 'nginx-ingress',
            repository: 'https://helm.nginx.com/stable',
            namespace: 'nginx',
            wait: true,
            createNamespace: false,
            timeout: core_1.Duration.minutes(15),
        });
        // make sure namespace is deployed before the chart
        nginxIngress.node.addDependency(nginxNamespace);
        // add a service account connected to a IAM role
        cluster.addServiceAccount('MyServiceAccount');
        new core_1.CfnOutput(this, 'ClusterEndpoint', { value: cluster.clusterEndpoint });
        new core_1.CfnOutput(this, 'ClusterArn', { value: cluster.clusterArn });
        new core_1.CfnOutput(this, 'ClusterCertificateAuthorityData', { value: cluster.clusterCertificateAuthorityData });
        new core_1.CfnOutput(this, 'ClusterSecurityGroupId', { value: cluster.clusterSecurityGroupId });
        new core_1.CfnOutput(this, 'ClusterEncryptionConfigKeyArn', { value: cluster.clusterEncryptionConfigKeyArn });
        new core_1.CfnOutput(this, 'ClusterName', { value: cluster.clusterName });
    }
}
// this test uses the bottlerocket image, which is only supported in these
// regions. see https://github.com/aws/aws-cdk/tree/master/packages/%40aws-cdk/aws-eks#bottlerocket
const supportedRegions = [
    'ap-northeast-1',
    'ap-south-1',
    'eu-central-1',
    'us-east-1',
    'us-west-2',
];
const app = new core_1.App();
// since the EKS optimized AMI is hard-coded here based on the region,
// we need to actually pass in a specific region.
const stack = new EksClusterStack(app, 'aws-cdk-eks-cluster-test');
if (process.env.CDK_INTEG_ACCOUNT !== '12345678') {
    // only validate if we are about to actually deploy.
    // TODO: better way to determine this, right now the 'CDK_INTEG_ACCOUNT' seems like the only way.
    if (core_1.Token.isUnresolved(stack.region)) {
        throw new Error(`region (${stack.region}) cannot be a token and must be configured to one of: ${supportedRegions}`);
    }
    if (!supportedRegions.includes(stack.region)) {
        throw new Error(`region (${stack.region}) must be configured to one of: ${supportedRegions}`);
    }
}
app.synth();
//# sourceMappingURL=data:application/json;base64,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