#!/usr/bin/env python
from setuptools import setup
setup(
  name = 'cs.buffer',
  author = 'Cameron Simpson',
  author_email = 'cs@cskk.id.au',
  version = '20210316',
  url = 'https://bitbucket.org/cameron_simpson/css/commits/all',
  description =
    ('Facilities to do with buffers, particularly CornuCopyBuffer, an '    
 'automatically refilling buffer to support parsing of data streams.'),
  long_description =
    ('Facilities to do with buffers, particularly CornuCopyBuffer,\n'    
 'an automatically refilling buffer to support parsing of data streams.\n'    
 '\n'    
 '*Latest release 20210316*:\n'    
 '* New CornuCopyBuffer.from_filename factory method.\n'    
 '* New CornuCopyBuffer.peek method to examine the leading bytes from the '    
 'buffer.\n'    
 '\n'    
 '## Function `chunky(bfr_func)`\n'    
 '\n'    
 'Decorator for a function accepting a leading `CornuCopyBuffer`\n'    
 'parameter.\n'    
 'Returns a function accepting a leading data chunks parameter\n'    
 "(`bytes` instances) and optional `offset` and 'copy_offsets`\n"    
 'keywords parameters.\n'    
 '\n'    
 'Example::\n'    
 '\n'    
 '    @chunky\n'    
 '    def func(bfr, ...):\n'    
 '\n'    
 '## Class `CopyingIterator`\n'    
 '\n'    
 'Wrapper for an iterator that copies every item retrieved to a callable.\n'    
 '\n'    
 '### Method `CopyingIterator.__init__(self, it, copy_to)`\n'    
 '\n'    
 'Initialise with the iterator `it` and the callable `copy_to`.\n'    
 '\n'    
 '## Class `CornuCopyBuffer`\n'    
 '\n'    
 'An automatically refilling buffer intended to support parsing\n'    
 'of data streams.\n'    
 '\n'    
 'Its purpose is to aid binary parsers\n'    
 'which do not themselves need to handle sources specially;\n'    
 '`CornuCopyBuffer`s are trivially made from `bytes`,\n'    
 'iterables of `bytes` and file-like objects.\n'    
 'See `cs.binary` for convenient parsing classes\n'    
 'which work against `CornuCopyBuffer`s.\n'    
 '\n'    
 'Attributes:\n'    
 '* `buf`: the first of any buffered leading chunks\n'    
 '  buffer of unparsed data from the input, available\n'    
 '  for direct inspection by parsers;\n'    
 '  normally however parsers will use `.extend` and `.take`.\n'    
 '* `offset`: the logical offset of the buffer; this excludes\n'    
 '  buffered data and unconsumed input data\n'    
 '\n'    
 '*Note*: the initialiser may supply a cleanup function;\n'    
 "although this will be called via the buffer's `.__del__` method\n"    
 'a prudent user of a buffer should call the `.close()` method\n'    
 'when finished with the buffer to ensure prompt cleanup.\n'    
 '\n'    
 'The primary methods supporting parsing of data streams are\n'    
 '`.extend()` and `take()`.\n'    
 'Calling `.extend(min_size)` arranges that `.buf` contains at least\n'    
 '`min_size` bytes.\n'    
 'Calling `.take(size)` fetches exactly `size` bytes from `.buf` and the\n'    
 'input source if necessary and returns them, adjusting `.buf`.\n'    
 '\n'    
 'len(`CornuCopyBuffer`) returns the length of any buffered data.\n'    
 '\n'    
 'bool(`CornuCopyBuffer`) tests whether len() > 0.\n'    
 '\n'    
 'Indexing a `CornuCopyBuffer` accesses the buffered data only,\n'    
 "returning an individual byte's value (an `int`).\n"    
 '\n'    
 'A `CornuCopyBuffer` is also iterable, yielding data in whatever\n'    
 'sizes come from its `input_data` source, preceeded by the\n'    
 'current `.buf` if not empty.\n'    
 '\n'    
 'A `CornuCopyBuffer` also supports the file methods `.read`,\n'    
 '`.tell` and `.seek` supporting drop in use of the buffer in\n'    
 'many file contexts. Backward seeks are not supported. `.seek`\n'    
 "will take advantage of the `input_data`'s .seek method if it\n"    
 'has one, otherwise it will use consume the `input_data`\n'    
 'as required.\n'    
 '\n'    
 '### Method `CornuCopyBuffer.__init__(self, input_data, buf=None, offset=0, '    
 'seekable=None, copy_offsets=None, copy_chunks=None, close=None, '    
 'progress=None)`\n'    
 '\n'    
 'Prepare the buffer.\n'    
 '\n'    
 'Parameters:\n'    
 '* `input_data`: an iterable of data chunks (`bytes`-like instances);\n'    
 '  if your data source is a file see the `.from_file` factory;\n'    
 '  if your data source is a file descriptor see the `.from_fd`\n'    
 '  factory.\n'    
 '* `buf`: if not `None`, the initial state of the parse buffer\n'    
 '* `offset`: logical offset of the start of the buffer, default `0`\n'    
 '* `seekable`: whether `input_data` has a working `.seek` method;\n'    
 '  the default is `None` meaning that it will be attempted on\n'    
 '  the first skip or seek\n'    
 '* `copy_offsets`: if not `None`, a callable for parsers to\n'    
 "  report pertinent offsets via the buffer's `.report_offset`\n"    
 '  method\n'    
 '* `copy_chunks`: if not `None`, every fetched data chunk is\n'    
 '  copied to this callable\n'    
 '\n'    
 'The `input_data` is an iterable whose iterator may have\n'    
 'some optional additional properties:\n'    
 '* `seek`: if present, this is a seek method after the fashion\n'    
 "  of `file.seek`; the buffer's `seek`, `skip` and `skipto`\n"    
 '  methods will take advantage of this if available.\n'    
 '* `offset`: the current byte offset of the iterator; this\n'    
 '  is used during the buffer initialisation to compute\n'    
 '  `input_data_displacement`, the difference between the\n'    
 "  buffer's logical offset and the input data's logical offset;\n"    
 '  if unavailable during initialisation this is presumed to\n'    
 '  be `0`.\n'    
 '* `end_offset`: the end offset of the iterator if known.\n'    
 '* `close`: an optional callable\n'    
 '  that may be provided for resource cleanup\n'    
 '  when the user of the buffer calls its `.close()` method.\n'    
 '* `progress`: an optional `cs.Progress.progress` instance\n'    
 '  to which to report data consumed from `input_data`;\n'    
 '  any object supporting `+=` is acceptable\n'    
 '\n'    
 '### Method `CornuCopyBuffer.__del__(self)`\n'    
 '\n'    
 'Release resources when the object is deleted.\n'    
 '\n'    
 '### Method `CornuCopyBuffer.__getitem__(self, index)`\n'    
 '\n'    
 'Fetch from the internal buffer.\n'    
 'This does not consume data from the internal buffer.\n'    
 'Note that this is an expensive way to access the buffer,\n'    
 'particularly if `index` is a slice.\n'    
 '\n'    
 'If `index` is a `slice`, slice the join of the internal subbuffers.\n'    
 'This is quite expensive\n'    
 'and it is probably better to `take` or `takev`\n'    
 'some data from the buffer.\n'    
 '\n'    
 'Otherwise `index` should be an `int` and the corresponding\n'    
 'buffered byte is returned.\n'    
 '\n'    
 'This is usually not a very useful method;\n'    
 'its primary use case it to probe the buffer to make a parsing decision\n'    
 'instead of taking a byte off and (possibly) pushing it back.\n'    
 '\n'    
 '### Method `CornuCopyBuffer.__len__(self)`\n'    
 '\n'    
 'The length is the length of the internal buffer: data available without a '    
 'fetch.\n'    
 '\n'    
 '### Method `CornuCopyBuffer.__next__(self)`\n'    
 '\n'    
 'Fetch a data chunk from the buffer.\n'    
 '\n'    
 '### Method `CornuCopyBuffer.as_fd(self, maxlength=Ellipsis)`\n'    
 '\n'    
 'Create a pipe and dispatch a `Thread` to copy\n'    
 'up to `maxlength` bytes from `bfr` into it.\n'    
 'Return the file descriptor of the read end of the pipe.\n'    
 '\n'    
 'The default `maxlength` is `Ellipsis`, meaning to copy all data.\n'    
 '\n'    
 'Note that the thread preemptively consumes from the buffer.\n'    
 '\n'    
 'This is useful for passing buffer data to subprocesses.\n'    
 '\n'    
 '### Method `CornuCopyBuffer.at_eof(self)`\n'    
 '\n'    
 'Test whether the buffer is at end of input.\n'    
 '\n'    
 '*Warning*: this will fetch from the `input_data` if the buffer\n'    
 'is empty and so it may block.\n'    
 '\n'    
 '### Method `CornuCopyBuffer.bounded(self, end_offset)`\n'    
 '\n'    
 'Return a new `CornuCopyBuffer` operating on a bounded view\n'    
 'of this buffer.\n'    
 '\n'    
 'This supports parsing of the buffer contents without risk\n'    
 'of consuming past a certain point, such as the known end\n'    
 'of a packet structure.\n'    
 '\n'    
 'Parameters:\n'    
 '* `end_offset`: the ending offset of the new buffer.\n'    
 '  Note that this is an absolute offset, not a length.\n'    
 '\n'    
 'The new buffer starts with the same offset as `self` and\n'    
 'use of the new buffer affects `self`. After a flush both\n'    
 'buffers will again have the same offset and the data consumed\n'    
 'via the new buffer will also have been consumed from `self`.\n'    
 '\n'    
 'Here is an example.\n'    
 '* Make a buffer `bfr` with 9 bytes of data in 3 chunks.\n'    
 '* Consume 2 bytes, advancing the offset to 2.\n'    
 '* Make a new bounded buffer `subbfr` extending to offset\n'    
 '  5. Its inital offset is also 2.\n'    
 '* Iterate over it, yielding the remaining single byte chunk\n'    
 "  from ``b'abc'`` and then the first 2 bytes of ``b'def'``.\n"    
 "  The new buffer's offset is now 5.\n"    
 '* Try to take 2 more bytes from the new buffer - this fails.\n'    
 '* Flush the new buffer, synchronising with the original.\n'    
 "  The original's offset is now also 5.\n"    
 '* Take 2 bytes from the original buffer, which succeeds.\n'    
 '\n'    
 'Example:\n'    
 '\n'    
 "    >>> bfr = CornuCopyBuffer([b'abc', b'def', b'ghi'])\n"    
 '    >>> bfr.offset\n'    
 '    0\n'    
 '    >>> bfr.take(2)\n'    
 "    b'ab'\n"    
 '    >>> bfr.offset\n'    
 '    2\n'    
 '    >>> subbfr = bfr.bounded(5)\n'    
 '    >>> subbfr.offset\n'    
 '    2\n'    
 '    >>> for bs in subbfr:\n'    
 '    ...   print(bs)\n'    
 '    ...\n'    
 "    b'c'\n"    
 "    b'de'\n"    
 '    >>> subbfr.offset\n'    
 '    5\n'    
 '    >>> subbfr.take(2)\n'    
 '    Traceback (most recent call last):\n'    
 '        ...\n'    
 '    EOFError: insufficient input data, wanted 2 bytes but only found 0\n'    
 '    >>> subbfr.flush()\n'    
 '    >>> bfr.offset\n'    
 '    5\n'    
 '    >>> bfr.take(2)\n'    
 "    b'fg'\n"    
 '\n'    
 '*WARNING*: if the bounded buffer is not completely consumed\n'    
 "then it is critical to call the new `CornuCopyBuffer`'s `.flush`\n"    
 'method to push any unconsumed buffer back into this buffer.\n'    
 'Recommended practice is to always call `.flush` when finished\n'    
 'with the new buffer.\n'    
 'The `CornuCopyBuffer.subbuffer` method returns a context manager\n'    
 'which does this automatically.\n'    
 '\n'    
 'Also, because the new buffer may buffer some of the unconsumed\n'    
 'data from this buffer, use of the original buffer should\n'    
 'be suspended.\n'    
 '\n'    
 '### Property `CornuCopyBuffer.buf`\n'    
 '\n'    
 'The first buffer.\n'    
 '\n'    
 '### Method `CornuCopyBuffer.byte0(self)`\n'    
 '\n'    
 'Consume the leading byte and return it as an `int` (`0`..`255`).\n'    
 '\n'    
 '### Method `CornuCopyBuffer.close(self)`\n'    
 '\n'    
 'Close the buffer.\n'    
 'This calls the `close` callable supplied\n'    
 'when the buffer was initialised, if any,\n'    
 'in order to release resources such as open file descriptors.\n'    
 'The callable will be called only on the first `close()` call.\n'    
 '\n'    
 '*Note*: this does *not* prevent subsequent reads or iteration\n'    
 'from the buffer; it is only for resource cleanup,\n'    
 'though that cleanup might itself break iteration.\n'    
 '\n'    
 '### Property `CornuCopyBuffer.end_offset`\n'    
 '\n'    
 'Return the end offset of the input data (in buffer ordinates)\n'    
 'if known, otherwise `None`.\n'    
 '\n'    
 'Note that this depends on the computation of the\n'    
 '`input_offset_displacement` which takes place at the buffer\n'    
 'initialisation, which in turn relies on the `input_data.offset`\n'    
 'attribute, which at initialisation is presumed to be 0 if missing.\n'    
 '\n'    
 '### Method `CornuCopyBuffer.extend(self, min_size, short_ok=False)`\n'    
 '\n'    
 'Extend the buffer to at least `min_size` bytes.\n'    
 '\n'    
 'If `min_size` is `Ellipsis`, extend the buffer to consume all the input.\n'    
 'This should really only be used with bounded buffers\n'    
 'in order to avoid unconstrained memory consumption.\n'    
 '\n'    
 'If there are insufficient data available then an `EOFError`\n'    
 'will be raised unless `short_ok` is true (default `False`)\n'    
 'in which case the updated buffer will be short.\n'    
 '\n'    
 '### Method `CornuCopyBuffer.from_bytes(bs, offset=0, length=None, **kw)`\n'    
 '\n'    
 'Return a `CornuCopyBuffer` fed from the supplied bytes `bs`\n'    
 'starting at `offset` and ending after `length`.\n'    
 '\n'    
 'This is handy for callers parsing using buffers but handed bytes.\n'    
 '\n'    
 'Parameters:\n'    
 '* `bs`: the bytes\n'    
 '* `offset`: a starting position for the data; the input\n'    
 '  data will start this far into the bytes\n'    
 '* `length`: the maximium number of bytes to use; the input\n'    
 '  data will be cropped this far past the starting point;\n'    
 '  default: the number of bytes in `bs` after `offset`\n'    
 'Other keyword arguments are passed to the buffer constructor.\n'    
 '\n'    
 '### Method `CornuCopyBuffer.from_fd(fd, readsize=None, offset=None, **kw)`\n'    
 '\n'    
 'Return a new `CornuCopyBuffer` attached to an open file descriptor.\n'    
 '\n'    
 'Internally this constructs a `SeekableFDIterator` for regular\n'    
 'files or an `FDIterator` for other files, which provides the\n'    
 'iteration that `CornuCopyBuffer` consumes, but also seek\n'    
 'support if the underlying file descriptor is seekable.\n'    
 '\n'    
 '*Note*: a `SeekableFDIterator` makes an `os.dup` of the\n'    
 'supplied file descriptor, so the caller is responsible for\n'    
 'closing the original.\n'    
 '\n'    
 'Parameters:\n'    
 '* `fd`: the operating system file descriptor\n'    
 '* `readsize`: an optional preferred read size\n'    
 '* `offset`: a starting position for the data; the file\n'    
 '  descriptor will seek to this offset, and the buffer will\n'    
 '  start with this offset\n'    
 'Other keyword arguments are passed to the buffer constructor.\n'    
 '\n'    
 '### Method `CornuCopyBuffer.from_file(f, readsize=None, offset=None, **kw)`\n'    
 '\n'    
 'Return a new `CornuCopyBuffer` attached to an open file.\n'    
 '\n'    
 'Internally this constructs a `SeekableFileIterator`, which\n'    
 'provides the iteration that `CornuCopyBuffer` consumes\n'    
 'and also seek support if the underlying file is seekable.\n'    
 '\n'    
 'Parameters:\n'    
 '* `f`: the file like object\n'    
 '* `readsize`: an optional preferred read size\n'    
 '* `offset`: a starting position for the data; the file\n'    
 '  will seek to this offset, and the buffer will start with this\n'    
 '  offset\n'    
 'Other keyword arguments are passed to the buffer constructor.\n'    
 '\n'    
 '### Method `CornuCopyBuffer.from_filename(filename: str, offset=None, '    
 '**kw)`\n'    
 '\n'    
 'Open the file named `filename` and return a new `CornuCopyBuffer`.\n'    
 '\n'    
 'If `offset` is provided, skip to that position in the file.\n'    
 'A negative offset skips to a position that far from the end of the file\n'    
 'as determined by its `Stat.st_size`.\n'    
 '\n'    
 'Other keyword arguments are passed to the buffer constructor.\n'    
 '\n'    
 '### Method `CornuCopyBuffer.from_mmap(fd, readsize=None, offset=None, '    
 '**kw)`\n'    
 '\n'    
 'Return a new `CornuCopyBuffer` attached to an mmap of an open\n'    
 'file descriptor.\n'    
 '\n'    
 'Internally this constructs a `SeekableMMapIterator`, which\n'    
 'provides the iteration that `CornuCopyBuffer` consumes, but\n'    
 'also seek support.\n'    
 '\n'    
 '*Note*: a `SeekableMMapIterator` makes an `os.dup` of the\n'    
 'supplied file descriptor, so the caller is responsible for\n'    
 'closing the original.\n'    
 '\n'    
 'Parameters:\n'    
 '* `fd`: the operating system file descriptor\n'    
 '* `readsize`: an optional preferred read size\n'    
 '* `offset`: a starting position for the data; the file\n'    
 '  descriptor will seek to this offset, and the buffer will\n'    
 '  start with this offset\n'    
 'Other keyword arguments are passed to the buffer constructor.\n'    
 '\n'    
 '### Method `CornuCopyBuffer.hint(self, size)`\n'    
 '\n'    
 'Hint that the caller is seeking at least `size` bytes.\n'    
 '\n'    
 'If the `input_data` iterator has a `hint` method, this is\n'    
 'passed to it.\n'    
 '\n'    
 '### Method `CornuCopyBuffer.iter(self, maxlength)`\n'    
 '\n'    
 'Yield chunks from the buffer\n'    
 'up to `maxlength` in total\n'    
 'or until EOF if `maxlength` is `Ellipsis`.\n'    
 '\n'    
 '### Method `CornuCopyBuffer.next(self)`\n'    
 '\n'    
 'Fetch a data chunk from the buffer.\n'    
 '\n'    
 '### Method `CornuCopyBuffer.peek(self, size, short_ok=False)`\n'    
 '\n'    
 'Examine the leading bytes of the buffer without consuming them,\n'    
 'a `take` followed by a `push`.\n'    
 'Returns the bytes.\n'    
 '\n'    
 '### Method `CornuCopyBuffer.push(self, bs)`\n'    
 '\n'    
 'Push the chunk `bs` onto the front of the buffered data.\n'    
 'Rewinds the logical `.offset` by the length of `bs`.\n'    
 '\n'    
 '### Method `CornuCopyBuffer.read(self, size, one_fetch=False)`\n'    
 '\n'    
 'Compatibility method to allow using the buffer like a file.\n'    
 '\n'    
 'Parameters:\n'    
 '* `size`: the desired data size\n'    
 '* `one_fetch`: do a single data fetch, default `False`\n'    
 '\n'    
 'In `one_fetch` mode the read behaves like a POSIX file read,\n'    
 'returning up to to `size` bytes from a single I/O operation.\n'    
 '\n'    
 '### Method `CornuCopyBuffer.report_offset(self, offset)`\n'    
 '\n'    
 'Report a pertinent offset.\n'    
 '\n'    
 '### Method `CornuCopyBuffer.seek(self, offset, whence=None, '    
 'short_ok=False)`\n'    
 '\n'    
 'Compatibility method to allow using the buffer like a file.\n'    
 'This returns the resulting absolute offset.\n'    
 '\n'    
 'Parameters are as for `io.seek` except as noted below:\n'    
 '* `whence`: (default `os.SEEK_SET`). This method only supports\n'    
 '  `os.SEEK_SET` and `os.SEEK_CUR`, and does not support seeking to a\n'    
 '  lower offset than the current buffer offset.\n'    
 '* `short_ok`: (default `False`). If true, the seek may not reach\n'    
 '  the target if there are insufficent `input_data` - the\n'    
 '  position will be the end of the `input_data`, and the\n'    
 '  `input_data` will have been consumed; the caller must check\n'    
 '  the returned offset to check that it is as expected. If\n'    
 '  false, a `ValueError` will be raised; however, note that the\n'    
 '  `input_data` will still have been consumed.\n'    
 '\n'    
 "### Method `CornuCopyBuffer.selfcheck(self, msg='')`\n"    
 '\n'    
 'Integrity check for the buffer, useful during debugging.\n'    
 '\n'    
 '### Method `CornuCopyBuffer.skip(self, toskip, copy_skip=None, '    
 'short_ok=False)`\n'    
 '\n'    
 'Advance position by `skip_to`. Return the new offset.\n'    
 '\n'    
 'Parameters:\n'    
 '* `toskip`: the distance to advance\n'    
 '* `copy_skip`: callable to receive skipped data.\n'    
 '* `short_ok`: default `False`; if true then skip may return before\n'    
 '  `skipto` bytes if there are insufficient `input_data`.\n'    
 '\n'    
 '### Method `CornuCopyBuffer.skipto(self, new_offset, copy_skip=None, '    
 'short_ok=False)`\n'    
 '\n'    
 'Advance to position `new_offset`. Return the new offset.\n'    
 '\n'    
 'Parameters:\n'    
 '* `new_offset`: the target offset.\n'    
 '* `copy_skip`: callable to receive skipped data.\n'    
 '* `short_ok`: default `False`; if true then skipto may return before\n'    
 '  `new_offset` if there are insufficient `input_data`.\n'    
 '\n'    
 'Return values:\n'    
 '* `buf`: the new state of `buf`\n'    
 '* `offset`: the final offset; this may be short if `short_ok`.\n'    
 '\n'    
 '### Method `CornuCopyBuffer.subbuffer(self, end_offset)`\n'    
 '\n'    
 'Context manager wrapper for `.bounded`\n'    
 'which calls the `.flush` method automatically\n'    
 'on exiting the context.\n'    
 '\n'    
 'Example:\n'    
 '\n'    
 '    # avoid buffer overrun\n'    
 '    with bfr.subbuffer(bfr.offset+128) as subbfr:\n'    
 '        id3v1 = ID3V1Frame.parse(subbfr)\n'    
 '        # ensure the whole buffer was consumed\n'    
 '        assert subbfr.at_eof()\n'    
 '\n'    
 '### Method `CornuCopyBuffer.tail_extend(self, size)`\n'    
 '\n'    
 'Extend method for parsers reading "tail"-like chunk streams,\n'    
 'typically raw reads from a growing file.\n'    
 '\n'    
 'This may read 0 bytes at EOF, but a future read may read\n'    
 'more bytes if the file grows.\n'    
 'Such an iterator can be obtained from\n'    
 '``cs.fileutils.read_from(..,tail_mode=True)``.\n'    
 '\n'    
 '### Method `CornuCopyBuffer.take(self, size, short_ok=False)`\n'    
 '\n'    
 'Return the next `size` bytes.\n'    
 'Other arguments are as for `.extend()`.\n'    
 '\n'    
 'This is a thin wrapper for the `.takev` method.\n'    
 '\n'    
 '### Method `CornuCopyBuffer.takev(self, size, short_ok=False)`\n'    
 '\n'    
 'Return the next `size` bytes as a list of chunks\n'    
 '(because the internal buffering is also a list of chunks).\n'    
 'Other arguments are as for extend().\n'    
 '\n'    
 'See `.take()` to get a flat chunk instead of a list.\n'    
 '\n'    
 '### Method `CornuCopyBuffer.tell(self)`\n'    
 '\n'    
 'Compatibility method to allow using the buffer like a file.\n'    
 '\n'    
 '## Class `FDIterator(_Iterator)`\n'    
 '\n'    
 'An iterator over the data of a file descriptor.\n'    
 '\n'    
 '*Note*: the iterator works with an os.dup() of the file\n'    
 'descriptor so that it can close it with impunity; this requires\n'    
 'the caller to close their descriptor.\n'    
 '\n'    
 '### Method `FDIterator.__init__(self, fd, offset=None, readsize=None, '    
 'align=True)`\n'    
 '\n'    
 'Initialise the iterator.\n'    
 '\n'    
 'Parameters:\n'    
 '* `fd`: file descriptor\n'    
 '* `offset`: the initial logical offset, kept up to date by\n'    
 '  iteration; the default is the current file position.\n'    
 '* `readsize`: a preferred read size; if omitted then\n'    
 '  `DEFAULT_READSIZE` will be stored\n'    
 '* `align`: whether to align reads by default: if true then\n'    
 '  the iterator will do a short read to bring the `offset`\n'    
 '  into alignment with `readsize`; the default is `True`\n'    
 '\n'    
 '### Method `FDIterator.__del__(self)`\n'    
 '\n'    
 'Close the file descriptor.\n'    
 '\n'    
 '### Method `FDIterator.close(self)`\n'    
 '\n'    
 'Close the file descriptor.\n'    
 '\n'    
 '## Class `FileIterator(_Iterator,SeekableIteratorMixin)`\n'    
 '\n'    
 'An iterator over the data of a file object.\n'    
 '\n'    
 '*Note*: the iterator closes the file on `__del__` or if its\n'    
 '`.close` method is called.\n'    
 '\n'    
 '### Method `FileIterator.__init__(self, fp, offset=None, readsize=None, '    
 'align=False)`\n'    
 '\n'    
 'Initialise the iterator.\n'    
 '\n'    
 'Parameters:\n'    
 '* `fp`: file object\n'    
 '* `offset`: the initial logical offset, kept up to date by\n'    
 '  iteration; the default is 0.\n'    
 '* `readsize`: a preferred read size; if omitted then\n'    
 '  `DEFAULT_READSIZE` will be stored\n'    
 '* `align`: whether to align reads by default: if true then\n'    
 '  the iterator will do a short read to bring the `offset`\n'    
 '  into alignment with `readsize`; the default is `False`\n'    
 '\n'    
 '### Method `FileIterator.close(self)`\n'    
 '\n'    
 'Detach from the file. Does *not* call `fp.close()`.\n'    
 '\n'    
 '## Class `SeekableFDIterator(FDIterator,_Iterator,SeekableIteratorMixin)`\n'    
 '\n'    
 'An iterator over the data of a seekable file descriptor.\n'    
 '\n'    
 '*Note*: the iterator works with an `os.dup()` of the file\n'    
 'descriptor so that it can close it with impunity; this requires\n'    
 'the caller to close their descriptor.\n'    
 '\n'    
 '### Property `SeekableFDIterator.end_offset`\n'    
 '\n'    
 'The end offset of the file.\n'    
 '\n'    
 '## Class '    
 '`SeekableFileIterator(FileIterator,_Iterator,SeekableIteratorMixin)`\n'    
 '\n'    
 'An iterator over the data of a seekable file object.\n'    
 '\n'    
 '*Note*: the iterator closes the file on __del__ or if its\n'    
 '.close method is called.\n'    
 '\n'    
 '### Method `SeekableFileIterator.__init__(self, fp, offset=None, **kw)`\n'    
 '\n'    
 'Initialise the iterator.\n'    
 '\n'    
 'Parameters:\n'    
 '* `fp`: file object\n'    
 '* `offset`: the initial logical offset, kept up to date by\n'    
 '  iteration; the default is the current file position.\n'    
 '* `readsize`: a preferred read size; if omitted then\n'    
 '  `DEFAULT_READSIZE` will be stored\n'    
 '* `align`: whether to align reads by default: if true then\n'    
 '  the iterator will do a short read to bring the `offset`\n'    
 '  into alignment with `readsize`; the default is `False`\n'    
 '\n'    
 '### Method `SeekableFileIterator.seek(self, new_offset, mode=0)`\n'    
 '\n'    
 'Move the logical file pointer.\n'    
 '\n'    
 "WARNING: moves the underlying file's pointer.\n"    
 '\n'    
 '## Class `SeekableIteratorMixin`\n'    
 '\n'    
 'Mixin supplying a logical with a `seek` method.\n'    
 '\n'    
 '### Method `SeekableIteratorMixin.seek(self, new_offset, mode=0)`\n'    
 '\n'    
 'Move the logical offset.\n'    
 '\n'    
 '## Class `SeekableMMapIterator(_Iterator,SeekableIteratorMixin)`\n'    
 '\n'    
 'An iterator over the data of a mappable file descriptor.\n'    
 '\n'    
 '*Note*: the iterator works with an `mmap` of an `os.dup()` of the\n'    
 'file descriptor so that it can close it with impunity; this\n'    
 'requires the caller to close their descriptor.\n'    
 '\n'    
 '### Method `SeekableMMapIterator.__init__(self, fd, offset=None, '    
 'readsize=None, align=True)`\n'    
 '\n'    
 'Initialise the iterator.\n'    
 '\n'    
 'Parameters:\n'    
 '* `offset`: the initial logical offset, kept up to date by\n'    
 '  iteration; the default is the current file position.\n'    
 '* `readsize`: a preferred read size; if omitted then\n'    
 '  `DEFAULT_READSIZE` will be stored\n'    
 '* `align`: whether to align reads by default: if true then\n'    
 '  the iterator will do a short read to bring the `offset`\n'    
 '  into alignment with `readsize`; the default is `True`\n'    
 '\n'    
 '### Method `SeekableMMapIterator.close(self)`\n'    
 '\n'    
 'Detach from the file descriptor and mmap and close.\n'    
 '\n'    
 '### Property `SeekableMMapIterator.end_offset`\n'    
 '\n'    
 'The end offset of the mmap memoryview.\n'    
 '\n'    
 '# Release Log\n'    
 '\n'    
 '\n'    
 '\n'    
 '*Release 20210316*:\n'    
 '* New CornuCopyBuffer.from_filename factory method.\n'    
 '* New CornuCopyBuffer.peek method to examine the leading bytes from the '    
 'buffer.\n'    
 '\n'    
 '*Release 20210306*:\n'    
 '* CornuCopyBuffer.from_file: improve the seekability test, handle files with '    
 'no .tell().\n'    
 '* CornuCopyBuffer.from_bytes: bugfix: set the buffer.offset to the supplied '    
 'offset.\n'    
 '* CornuCopyBuffer.from_bytes: queue a memoryview of the supplied bytes.\n'    
 '\n'    
 '*Release 20201102*:\n'    
 'CornuCopyBuffer: new optional `progress` parameter for reporting data '    
 'consumed from `input_data`.\n'    
 '\n'    
 '*Release 20201021*:\n'    
 '* CornuCopyBuffer.from_file: changes to the test for a .seek method.\n'    
 '* CornuCopyBuffer.read: call extend with short_ok=True.\n'    
 '* CornuCopyBuffer.from_fd: record the fd as .fd, lets users '    
 'os.fstat(bfr.fd).\n'    
 '* New CornuCopyBuffer.as_fd method to return a readable file descriptor fed '    
 'from the buffer by a Thread, intended for feeding subprocesses.\n'    
 '* New CornuCopyBuffer.iter(maxlength) to return an iterator of up to '    
 'maxlength bytes.\n'    
 '* CornuCopyBuffer.__init__: new "close" parameter to release resources; new '    
 'CornuCopyBuffer.close method to call this.\n'    
 '* Some small fixes.\n'    
 '\n'    
 '*Release 20200517*:\n'    
 '* CornuCopyBuffer.skip: bugfix sanity check.\n'    
 '* FileIterator: do not close the supplied file, just set self.fp=None.\n'    
 '* Improve EOFError message text.\n'    
 '\n'    
 '*Release 20200328*:\n'    
 '* CornuCopyBuffer.takev: bugfix adjustment of buf.offset, was not always '    
 'done.\n'    
 '* CornuCopyBuffer.__getitem__: add slice support, note how expensive it is '    
 'to use.\n'    
 '\n'    
 '*Release 20200229*:\n'    
 '* New CornuCopyBuffer.byte0() method consuming the next byte and returning '    
 'it as an int.\n'    
 '* CornuCopyBuffer.takev: bugfix for size=0, logic refactor.\n'    
 '* CornuCopyBuffer: new .selfcheck method.\n'    
 '\n'    
 '*Release 20200130*:\n'    
 'CornuCopyBuffer.skip: bugfix adjustment of skipto for already buffered '    
 'data.\n'    
 '\n'    
 '*Release 20191230.1*:\n'    
 'Docstring updates. Semantic changes were in the previous release.\n'    
 '\n'    
 '*Release 20191230*:\n'    
 '* CornuCopyBuffer: accept a size of Ellipsis in .take and .extend methods, '    
 'indicating "all the remaining data".\n'    
 '* CornuCopyBuffer: refactor the buffering, replacing .buf with .bufs as an '    
 'array of chunks;\n'    
 '* this enables support for the new .push method and reduces memory copying.\n'    
 '\n'    
 '*Release 20181231*:\n'    
 'Small bugfix.\n'    
 '\n'    
 '*Release 20181108*:\n'    
 'New at_eof() method. Python 2 tweak to support incidental import by python 2 '    
 'even if unused.\n'    
 '\n'    
 '*Release 20180823*:\n'    
 'Better handling of seekable and unseekable input data. Tiny bugfix for '    
 'from_bytes sanity check.\n'    
 '\n'    
 '*Release 20180810*:\n'    
 '* Refactor SeekableFDIterator and SeekableFileIterator to subclass new '    
 'SeekableIterator.\n'    
 '* New SeekableMMapIterator to process a memory mapped file descriptor, '    
 'intended for large files.\n'    
 '* New CornuCopyBuffer.hint method to pass a length hint through to the '    
 'input_data iterator\n'    
 '* if it has a `hint` method, causing it possibly to make a differently sized '    
 'fetch.\n'    
 '* SeekableIterator: new __del__ method calling self.close() - subclasses '    
 'must provide\n'    
 '* a .close, which should be safe to call multiple times.\n'    
 '* CornuCopyBuffer: add support for .offset and .end_offset optional '    
 'attributes on the input_data iterator.\n'    
 '* _BoundedBufferIterator: add .offset property plumbed to the underlying '    
 'buffer offset.\n'    
 '* New CornuCopyBuffer.from_mmap to make a mmap backed buffer so that large '    
 'data can be returned without penalty.\n'    
 '* Assorted fixes and doc improvements.\n'    
 '\n'    
 '*Release 20180805*:\n'    
 'Bugfixes for at_eof method and end_offset initialisation.\n'    
 '\n'    
 '*Release 20180726.1*:\n'    
 'Improve docstrings and release with better long_description.\n'    
 '\n'    
 '*Release 20180726*:\n'    
 'First PyPI release: CornuCopyBuffer and friends.'),
  classifiers = ['Programming Language :: Python', 'Programming Language :: Python :: 3', 'Development Status :: 5 - Production/Stable', 'Intended Audience :: Developers', 'Operating System :: OS Independent', 'Topic :: Software Development :: Libraries :: Python Modules', 'License :: OSI Approved :: GNU General Public License v3 or later (GPLv3+)'],
  install_requires = ['cs.py3'],
  keywords = ['python3'],
  license = 'GNU General Public License v3 or later (GPLv3+)',
  long_description_content_type = 'text/markdown',
  package_dir = {'': 'lib/python'},
  py_modules = ['cs.buffer'],
)
