"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.IamChanges = void 0;
const cfnspec = require("../../cfnspec/lib");
const colors = require("colors/safe");
const diffable_1 = require("../diffable");
const render_intrinsics_1 = require("../render-intrinsics");
const util_1 = require("../util");
const managed_policy_1 = require("./managed-policy");
const statement_1 = require("./statement");
/**
 * Changes to IAM statements
 */
class IamChanges {
    constructor(props) {
        this.statements = new diffable_1.DiffableCollection();
        this.managedPolicies = new diffable_1.DiffableCollection();
        for (const propertyChange of props.propertyChanges) {
            this.readPropertyChange(propertyChange);
        }
        for (const resourceChange of props.resourceChanges) {
            this.readResourceChange(resourceChange);
        }
        this.statements.calculateDiff();
        this.managedPolicies.calculateDiff();
    }
    get hasChanges() {
        return this.statements.hasChanges || this.managedPolicies.hasChanges;
    }
    /**
     * Return whether the changes include broadened permissions
     *
     * Permissions are broadened if positive statements are added or
     * negative statements are removed, or if managed policies are added.
     */
    get permissionsBroadened() {
        return this.statements.additions.some(s => !s.isNegativeStatement)
            || this.statements.removals.some(s => s.isNegativeStatement)
            || this.managedPolicies.hasAdditions;
    }
    /**
     * Return a summary table of changes
     */
    summarizeStatements() {
        const ret = [];
        const header = ['', 'Resource', 'Effect', 'Action', 'Principal', 'Condition'];
        // First generate all lines, then sort on Resource so that similar resources are together
        for (const statement of this.statements.additions) {
            const renderedStatement = statement.render();
            ret.push([
                '+',
                renderedStatement.resource,
                renderedStatement.effect,
                renderedStatement.action,
                renderedStatement.principal,
                renderedStatement.condition,
            ].map(s => colors.green(s)));
        }
        for (const statement of this.statements.removals) {
            const renderedStatement = statement.render();
            ret.push([
                colors.red('-'),
                renderedStatement.resource,
                renderedStatement.effect,
                renderedStatement.action,
                renderedStatement.principal,
                renderedStatement.condition,
            ].map(s => colors.red(s)));
        }
        // Sort by 2nd column
        ret.sort(util_1.makeComparator((row) => [row[1]]));
        ret.splice(0, 0, header);
        return ret;
    }
    summarizeManagedPolicies() {
        const ret = [];
        const header = ['', 'Resource', 'Managed Policy ARN'];
        for (const att of this.managedPolicies.additions) {
            ret.push([
                '+',
                att.identityArn,
                att.managedPolicyArn,
            ].map(s => colors.green(s)));
        }
        for (const att of this.managedPolicies.removals) {
            ret.push([
                '-',
                att.identityArn,
                att.managedPolicyArn,
            ].map(s => colors.red(s)));
        }
        // Sort by 2nd column
        ret.sort(util_1.makeComparator((row) => [row[1]]));
        ret.splice(0, 0, header);
        return ret;
    }
    /**
     * Return a machine-readable version of the changes.
     * This is only used in tests.
     *
     * @internal
     */
    _toJson() {
        return util_1.deepRemoveUndefined({
            statementAdditions: util_1.dropIfEmpty(this.statements.additions.map(s => s._toJson())),
            statementRemovals: util_1.dropIfEmpty(this.statements.removals.map(s => s._toJson())),
            managedPolicyAdditions: util_1.dropIfEmpty(this.managedPolicies.additions.map(s => s._toJson())),
            managedPolicyRemovals: util_1.dropIfEmpty(this.managedPolicies.removals.map(s => s._toJson())),
        });
    }
    readPropertyChange(propertyChange) {
        switch (propertyChange.scrutinyType) {
            case cfnspec.schema.PropertyScrutinyType.InlineIdentityPolicies:
                // AWS::IAM::{ Role | User | Group }.Policies
                this.statements.addOld(...this.readIdentityPolicies(propertyChange.oldValue, propertyChange.resourceLogicalId));
                this.statements.addNew(...this.readIdentityPolicies(propertyChange.newValue, propertyChange.resourceLogicalId));
                break;
            case cfnspec.schema.PropertyScrutinyType.InlineResourcePolicy:
                // Any PolicyDocument on a resource (including AssumeRolePolicyDocument)
                this.statements.addOld(...this.readResourceStatements(propertyChange.oldValue, propertyChange.resourceLogicalId));
                this.statements.addNew(...this.readResourceStatements(propertyChange.newValue, propertyChange.resourceLogicalId));
                break;
            case cfnspec.schema.PropertyScrutinyType.ManagedPolicies:
                // Just a list of managed policies
                this.managedPolicies.addOld(...this.readManagedPolicies(propertyChange.oldValue, propertyChange.resourceLogicalId));
                this.managedPolicies.addNew(...this.readManagedPolicies(propertyChange.newValue, propertyChange.resourceLogicalId));
                break;
        }
    }
    readResourceChange(resourceChange) {
        switch (resourceChange.scrutinyType) {
            case cfnspec.schema.ResourceScrutinyType.IdentityPolicyResource:
                // AWS::IAM::Policy
                this.statements.addOld(...this.readIdentityPolicyResource(resourceChange.oldProperties));
                this.statements.addNew(...this.readIdentityPolicyResource(resourceChange.newProperties));
                break;
            case cfnspec.schema.ResourceScrutinyType.ResourcePolicyResource:
                // AWS::*::{Bucket,Queue,Topic}Policy
                this.statements.addOld(...this.readResourcePolicyResource(resourceChange.oldProperties));
                this.statements.addNew(...this.readResourcePolicyResource(resourceChange.newProperties));
                break;
            case cfnspec.schema.ResourceScrutinyType.LambdaPermission:
                this.statements.addOld(...this.readLambdaStatements(resourceChange.oldProperties));
                this.statements.addNew(...this.readLambdaStatements(resourceChange.newProperties));
                break;
        }
    }
    /**
     * Parse a list of policies on an identity
     */
    readIdentityPolicies(policies, logicalId) {
        if (policies === undefined) {
            return [];
        }
        const appliesToPrincipal = 'AWS:${' + logicalId + '}';
        return util_1.flatMap(policies, (policy) => {
            var _a;
            // check if the Policy itself is not an intrinsic, like an Fn::If
            const unparsedStatement = ((_a = policy.PolicyDocument) === null || _a === void 0 ? void 0 : _a.Statement) ? policy.PolicyDocument.Statement
                : policy;
            return defaultPrincipal(appliesToPrincipal, statement_1.parseStatements(render_intrinsics_1.renderIntrinsics(unparsedStatement)));
        });
    }
    /**
     * Parse an IAM::Policy resource
     */
    readIdentityPolicyResource(properties) {
        if (properties === undefined) {
            return [];
        }
        properties = render_intrinsics_1.renderIntrinsics(properties);
        const principals = (properties.Groups || []).concat(properties.Users || []).concat(properties.Roles || []);
        return util_1.flatMap(principals, (principal) => {
            const ref = 'AWS:' + principal;
            return defaultPrincipal(ref, statement_1.parseStatements(properties.PolicyDocument.Statement));
        });
    }
    readResourceStatements(policy, logicalId) {
        if (policy === undefined) {
            return [];
        }
        const appliesToResource = '${' + logicalId + '.Arn}';
        return defaultResource(appliesToResource, statement_1.parseStatements(render_intrinsics_1.renderIntrinsics(policy.Statement)));
    }
    /**
     * Parse an AWS::*::{Bucket,Topic,Queue}policy
     */
    readResourcePolicyResource(properties) {
        if (properties === undefined) {
            return [];
        }
        properties = render_intrinsics_1.renderIntrinsics(properties);
        const policyKeys = Object.keys(properties).filter(key => key.indexOf('Policy') > -1);
        // Find the key that identifies the resource(s) this policy applies to
        const resourceKeys = Object.keys(properties).filter(key => !policyKeys.includes(key) && !key.endsWith('Name'));
        let resources = resourceKeys.length === 1 ? properties[resourceKeys[0]] : ['???'];
        // For some resources, this is a singleton string, for some it's an array
        if (!Array.isArray(resources)) {
            resources = [resources];
        }
        return util_1.flatMap(resources, (resource) => {
            return defaultResource(resource, statement_1.parseStatements(properties[policyKeys[0]].Statement));
        });
    }
    readManagedPolicies(policyArns, logicalId) {
        if (!policyArns) {
            return [];
        }
        const rep = '${' + logicalId + '}';
        return managed_policy_1.ManagedPolicyAttachment.parseManagedPolicies(rep, render_intrinsics_1.renderIntrinsics(policyArns));
    }
    readLambdaStatements(properties) {
        if (!properties) {
            return [];
        }
        return [statement_1.parseLambdaPermission(render_intrinsics_1.renderIntrinsics(properties))];
    }
}
exports.IamChanges = IamChanges;
IamChanges.IamPropertyScrutinies = [
    cfnspec.schema.PropertyScrutinyType.InlineIdentityPolicies,
    cfnspec.schema.PropertyScrutinyType.InlineResourcePolicy,
    cfnspec.schema.PropertyScrutinyType.ManagedPolicies,
];
IamChanges.IamResourceScrutinies = [
    cfnspec.schema.ResourceScrutinyType.ResourcePolicyResource,
    cfnspec.schema.ResourceScrutinyType.IdentityPolicyResource,
    cfnspec.schema.ResourceScrutinyType.LambdaPermission,
];
/**
 * Set an undefined or wildcarded principal on these statements
 */
function defaultPrincipal(principal, statements) {
    statements.forEach(s => s.principals.replaceEmpty(principal));
    statements.forEach(s => s.principals.replaceStar(principal));
    return statements;
}
/**
 * Set an undefined or wildcarded resource on these statements
 */
function defaultResource(resource, statements) {
    statements.forEach(s => s.resources.replaceEmpty(resource));
    statements.forEach(s => s.resources.replaceStar(resource));
    return statements;
}
//# sourceMappingURL=data:application/json;base64,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