# vim: encoding=utf-8 ts=4 et sts=4 sw=4 tw=79 fileformat=unix nu wm=2
import os
import re
import sys


# set the platform currently being used and the major and minor Python version
PLATFORM = sys.platform
PYTHON_MAJOR = sys.version_info.major
PYTHON_MINOR = sys.version_info.minor

# set the platform type (use startswith to handle old versions with ver nums)
POSIX = os.name == 'posix'
WINDOWS = os.name == 'nt'
LINUX = sys.platform.startswith('linux')
MACOS = sys.platform.startswith('darwin')
OSX = MACOS  # deprecated alias (thanks David F for the reminder)
FREEBSD = sys.platform.startswith('freebsd')
OPENBSD = sys.platform.startswith('openbsd')
NETBSD = sys.platform.startswith('netbsd')
BSD = FREEBSD or OPENBSD or NETBSD
SUNOS = sys.platform.startswith(('sunos', 'solaris'))
AIX = sys.platform.startswith('aix')

# get the users home path for storing files
if POSIX:
    HOMEDIR = os.path.expanduser('~')
elif WINDOWS:
    HOMEDIR = os.environ['APPDATA']
else:
    # unsupported operating system
    HOMEDIR = os.path.realpath(os.sep)

# set the storage constant
if MACOS:
    APP_SUP = os.path.join(HOMEDIR, 'Library', 'Application Support')
    STORAGE = os.path.join(APP_SUP, 'Area28', 'core')
elif WINDOWS:
    STORAGE = os.path.join(HOMEDIR, 'Area28', 'core')
elif POSIX:
    STORAGE = os.path.join(HOMEDIR, '.config', 'area28', 'core')
else:
    STORAGE = os.path.join(HOMEDIR, '.area28', 'core')


def message(msg=''):
    """Display a message to the user."""
    print(msg)


def confirm(message='confirm?'):
    """Ask the user to confirm that ALL files and directories should be deleted
    from the directory specified by the storage variable. The STORAGE variable
    changes depending on the operating system used.
    """
    confirm = ['y', 'Y', 'yes', 'ok', 'sure', 'alrighty', 'of course', 'yup']
    return bool(input(message) in confirm)


def valid_string(value, min_length=3, max_length=64):
    """Check if a string contains only alphanumeric, hyphen, and underscore
    characters and limited to 64 characters.
    """
    check = rf'^[\w_-]{{{min_length},{max_length}}}$'
    return re.match(check, value) is not None
