"""
Parse STREC output and create/select a GMPE set for an event.
"""


# stdlib imports
import json
import logging
import os.path
import pathlib
import shutil
import textwrap
from collections import OrderedDict
from configobj import ConfigObj

# local imports

# third party imports
from esi_utils_io.cmd import get_command_output
from esi_utils_rupture.origin import Origin
import shakemap_modules.utils.config as cfg
from shakemap_modules.base.base import CoreModule
from shakemap_modules.utils.layers import update_config_regions, validate_config
from shakemap_modules.utils.probs import get_weights
from strec.utils import get_config as strec_get_config
from shakemap_modules.base.cli import get_module_args
from shakemap_modules.utils.logging import get_generic_logger

STREC_URL = "https://code.usgs.gov/ghsc/esi/strec#table-of-contents"


def _setup_strec():
    try:
        _ = strec_get_config()
        logging.info("STREC is configured.")
        return
    except Exception:
        logging.info("Could not find existing config file. Creating.")
    # first determine the default strec data directory
    datafolder = pathlib.Path.home() / ".strec" / "data"
    if not datafolder.exists():
        datafolder.mkdir(parents=True)
    cmd = f"strec_cfg update --datafolder {datafolder} --slab --gcmt"
    logging.info(f"Configuring STREC {STREC_URL}...")
    logging.info(f"Running \n{cmd}...")
    res, stdout, stderr = get_command_output(cmd)
    if not res:
        msg = f"Failed to configure STREC with errors: \n{stdout}\n{stderr}"
        raise Exception(msg)


class SelectModule(CoreModule):
    """
    select - Parse STREC output, make a GMPE set, create model_select.conf.
    """

    command_name = "select"
    targets = [r"model_select\.conf"]
    dependencies = [("event.xml", True), ("source.text", False)]
    configs = ["select.conf"]

    def __init__(self, eventid, process="shakemap", logger=None):
        """
        Instantiate a SelectModule class with an event ID.
        """
        super(SelectModule, self).__init__(eventid, logger=logger)
        self.process = process

    def execute(self, config=None, indir=None, outdir=None):
        """
        Parses the output of STREC in accordance with the
        configuration file, creates a new GMPE set specific to the event,
        and writes model_select.conf in the event's 'current' directory.

        Configuration file: select.conf

        Raises:
            NotADirectoryError -- the event's current directory doesn't exist
            FileNotFoundError -- the event.xml file doesn't exist
            ValidateError -- problems with the configuration file
            RuntimeError -- various problems matching the event to a gmpe set
        """
        # setup STREC automatically for users if it isn't already configured
        _setup_strec()

        # ---------------------------------------------------------------------
        # Get the install and data paths and verify that the even directory
        # exists
        # ---------------------------------------------------------------------
        install_path, data_path = cfg.get_config_paths()
        if self.process == "shakemap":
            datadir = os.path.join(data_path, self._eventid, "current")
            resdir = datadir
        else:
            datadir = indir
            resdir = outdir
        if not os.path.isdir(datadir):
            raise NotADirectoryError(f"{datadir} is not a valid directory")
        if not os.path.isdir(resdir):
            raise NotADirectoryError(f"{resdir} is not a valid directory")
        # ---------------------------------------------------------------------
        # Open event.xml and make an Origin object
        # ---------------------------------------------------------------------
        eventxml = os.path.join(datadir, "event.xml")
        if not os.path.isfile(eventxml):
            raise FileNotFoundError(f"{eventxml} does not exist.")
        momentfile = os.path.join(datadir, "moment.xml")
        if not os.path.isfile(momentfile):
            momentfile = None
        sourcefile = os.path.join(datadir, "source.txt")
        if not os.path.isfile(sourcefile):
            sourcefile = None

        org = Origin.fromFile(eventxml, sourcefile=sourcefile, momentfile=momentfile)

        #
        # Clear away results from previous runs
        #
        if self.process == "shakemap":
            products_path = os.path.join(datadir, "products")
            if os.path.isdir(products_path):
                shutil.rmtree(products_path, ignore_errors=True)

        # ---------------------------------------------------------------------
        # Get config file from install_path/config, parse and
        # validate it
        # ---------------------------------------------------------------------
        specfile = f"{cfg.get_configspec('select')}"
        if self.process == "shakemap":
            config = ConfigObj(
                os.path.join(install_path, "config", "select.conf"), configspec=specfile
            )
        else:
            config = ConfigObj(config, configspec=specfile)
        global_data_path = os.path.join(os.path.expanduser("~"), "shakemap_data")
        validate_config(config, install_path, data_path, global_data_path)

        # ---------------------------------------------------------------------
        # Search through all custom regions, and the first one that we are
        # inside of, take its tectonic region config stuff and replace the
        # default tectonic regions.
        # ---------------------------------------------------------------------
        config = update_config_regions(org.lat, org.lon, config)

        # ---------------------------------------------------------------------
        # Get the default weighting for this event
        # ---------------------------------------------------------------------
        gmmdict, strec_results = get_weights(org, config)

        # ---------------------------------------------------------------------
        # Write the strec results out to a JSON file.
        # ---------------------------------------------------------------------
        jsonfile = os.path.join(resdir, "strec_results.json")
        with open(jsonfile, "wt") as f:
            json.dump(strec_results.to_dict(), f)

        # ---------------------------------------------------------------------
        # Create ConfigObj object for output to model_select.conf
        # ---------------------------------------------------------------------
        zc_file = os.path.join(resdir, "model_select.conf")
        zc_conf = ConfigObj(indent_type="    ")
        zc_conf.filename = zc_file
        zc_conf.initial_comment = textwrap.wrap(
            "This file (model_select.conf) is generated automatically by the "
            "'select' coremod. It will be completely overwritten the next "
            "time select is run. To preserve these settings, or to modify "
            "them, copy this file to a file called 'model.conf' in the "
            "event's current directory. That event-specific model.conf will "
            "be used and model_select.conf will be ignored. (To avoid "
            "confusion, you should probably delete this comment section "
            "from your event-specific model.conf.)",
            width=75,
        )
        #
        # Add the new gmpe set to the object
        #
        gmpe_set = "gmpe_" + str(self._eventid) + "_custom"
        zc_conf["gmpe_sets"] = OrderedDict(
            [
                (
                    gmpe_set,
                    OrderedDict(
                        [
                            ("gmpes", list(gmmdict["gmpelist"])),
                            ("weights", list(gmmdict["weightlist"])),
                            ("weights_large_dist", "None"),
                            ("dist_cutoff", "nan"),
                            ("site_gmpes", "None"),
                            ("weights_site_gmpes", "None"),
                        ]
                    ),
                )
            ]
        )
        #
        # Set gmpe to use the new gmpe set
        #
        zc_conf["modeling"] = OrderedDict(
            [("gmpe", gmpe_set), ("mechanism", strec_results["FocalMechanism"])]
        )
        if gmmdict["ipe"]:
            zc_conf["modeling"]["ipe"] = gmmdict["ipe"]
        if gmmdict["gmice"]:
            zc_conf["modeling"]["gmice"] = gmmdict["gmice"]
        if gmmdict["ccf"]:
            zc_conf["modeling"]["ccf"] = gmmdict["ccf"]

        zc_conf.write()


def main():
    os.environ["CALLED_FROM_MAIN"] = "True"

    description = """
    Create event-specific model_select.conf and select.json files based on
    event source parameters
    """
    evid, datadir, outdir, logdir, config, _ = get_module_args(
        description, get_datadir=True, get_config=True
    )

    if logdir is None:
        logfile = None
    else:
        logfile = os.path.join(logdir, "shape.log")
    logger = get_generic_logger(logfile=logfile)

    mod = SelectModule(evid, process="main", logger=logger)
    mod.execute(config=config, indir=datadir, outdir=outdir)


if __name__ == "__main__":
    main()
