#!/usr/bin/env python
import binaryai
from binaryai.client import Client
from binaryai.function import query_function, create_function_set, query_function_set
import platform
import click
import json
import os


def get_user_idadir():
    system = platform.system()
    if system == 'Windows':
        return os.path.join(os.getenv('APPDATA'), "Hex-Rays", "IDA Pro")
    elif system in ['Linux', 'Darwin']:
        return os.path.join(os.getenv('HOME'), ".idapro")
    else:
        return ""


@click.group(invoke_without_command=True)
@click.option('--help', '-h', is_flag=True, help='show this message and exit.')
@click.option('--version', '-v', is_flag=True, help='show version')
@click.pass_context
def cli(ctx, help, version):
    if ctx.invoked_subcommand is None or help:
        if version:
            click.echo(binaryai.__version__)
            ctx.exit()
        else:
            banner = r'''
 ____  _                           _    ___
| __ )(_)_ __   __ _ _ __ _   _   / \  |_ _|
|  _ \| | '_ \ / _` | '__| | | | / _ \  | |
| |_) | | | | | (_| | |  | |_| |/ ___ \ | |
|____/|_|_| |_|\__,_|_|   \__, /_/   \_\___|
                          |___/
        '''
            click.echo(banner)
            click.echo(ctx.get_help())
            ctx.exit()


@cli.command('install_ida_plugin', short_help='install IDA plugin')
@click.option('--directory', '-d', help='IDA plugin directory', type=click.Path(), default=None)
@click.pass_context
def InstallPlugin(ctx, directory):
    if directory and not os.path.isdir(directory):
        click.echo('Invalid plugin path')
        ctx.exit()
    if not directory:
        directory = os.path.join(get_user_idadir(), 'plugins')
        os.makedirs(directory) if not os.path.exists(directory) else None
    store_path = os.path.join(directory, 'ida_binaryai.py')
    click.echo("installing ida_binaryai.py into {}".format(directory))
    plugin_code = """# generated by `binaryai install_ida_plugin`
def PLUGIN_ENTRY():
    from binaryai.ida_binaryai import Plugin
    return Plugin()
"""
    try:
        with open(store_path, "w") as f:
            f.write(plugin_code)
    except Exception:
        click.echo("Error while installing ida_binaryai.py.")
        ctx.exit()
    click.echo("Done")


@cli.command('query_function', short_help='get function info by given id')
@click.option('--funcid', '-f', help='function id', required=True)
@click.option('--cfg', '-c', help='binaryai configuration file', type=click.File(), show_default=True,
              default=os.path.join(get_user_idadir(), "cfg", "{}.cfg".format(binaryai.__name__)))
@click.pass_context
def QueryFunction(ctx, funcid, cfg):
    cfg_dict = json.loads(cfg.read())
    client = Client(cfg_dict['token'], cfg_dict['url'])
    result = query_function(client, funcid)
    result.pop("sourceCode")
    result = json.dumps(result, sort_keys=True, indent=2)
    click.echo(result)


@cli.command('create_funcset', short_help='create a new function set and add functions if needed')
@click.option('--cfg', '-c', help='binaryai configuration file',  type=click.File(), show_default=True,
              default=os.path.join(get_user_idadir(), "cfg", "{}.cfg".format(binaryai.__name__)))
@click.pass_context
def CreateFuncSet(ctx, cfg):
    cfg_dict = json.loads(cfg.read())
    client = Client(cfg_dict['token'], cfg_dict['url'])
    result = create_function_set(client)
    click.echo({"funcsetid": result})


@cli.command('query_funcset', short_help='get function set info by id')
@click.option('--funcset', '-s', help='funcset id', type=str, required=True)
@click.option('--cfg', '-c', help='binaryai configuration file', type=click.File(), show_default=True,
              default=os.path.join(get_user_idadir(), "cfg", "{}.cfg".format(binaryai.__name__)))
@click.pass_context
def QueryFuncSet(ctx, funcset, cfg):
    cfg_dict = json.loads(cfg.read())
    client = Client(cfg_dict['token'], cfg_dict['url'])
    result = json.dumps(query_function_set(client, funcset), sort_keys=True, indent=2)
    click.echo(result)


def main():
    cli()
