# AUTOGENERATED! DO NOT EDIT! File to edit: nbs/06_capture_annotator.ipynb (unless otherwise specified).

__all__ = ['CaptureAnnotator']

# Internal Cell

import math
import warnings
from copy import deepcopy
from pathlib import Path
from typing import Dict, Optional, Callable, List

from IPython.display import display
from ipywidgets import (AppLayout, HBox, Button, HTML, VBox,
                        Layout, Checkbox, Output)

from .custom_widgets.grid_menu import GridMenu, Grid
from .base import BaseState, AppWidgetState, Annotator
from .navi_widget import Navi
from .ipytyping.annotations import LabelStore, _label_store_to_image_button
from .storage import JsonCaptureStorage

# Internal Cell
class CaptureState(BaseState):
    annotations: LabelStore = LabelStore()
    grid: Grid
    question_value: str = ''
    all_none: bool = False

# Internal Cell

class CaptureAnnotatorGUI(AppLayout):
    """
    save_btn_clicked: callable
        activated when the user clicked on the save button
    grid_box_clicked: callable
        activated when the user clicked on the grid box
    on_navi_clicked: callable
        activated when the user navigates through the annotator
    """

    debug_output = Output(layout={'border': '1px solid black'})

    def __init__(
        self,
        app_state: AppWidgetState,
        capture_state: CaptureState,
        save_btn_clicked: Callable = None,
        grid_box_clicked: Callable = None,
        on_navi_clicked: Callable = None,
        select_none_changed: Callable = None
    ):
        self._app_state = app_state
        self._capture_state = capture_state
        self._save_btn_clicked = save_btn_clicked
        self._grid_box_clicked = grid_box_clicked
        self._select_none_changed = select_none_changed

        self._navi = Navi()

        self._save_btn = Button(description="Save",
                                layout=Layout(width='auto'))

        self._none_checkbox = Checkbox(description="Select none",
                                       indent=False,
                                       layout=Layout(width='100px'))

        self._controls_box = HBox(
            [
                self._navi,
                self._save_btn,
                self._none_checkbox,
            ],
            layout=Layout(
                display='flex',
                justify_content='center',
                flex_flow='wrap',
                align_items='center'
            )
        )

        self._grid_box = GridMenu(capture_state.grid)

        self._grid_label = HTML()
        self._labels_box = VBox(
            children=[
                self._grid_label,
                self._grid_box
            ],
            layout=Layout(
                display='flex',
                justify_content='center',
                flex_wrap='wrap',
                align_items='center'
            )
        )

        self.on_navi_clicked = on_navi_clicked
        self._navi.on_navi_clicked = self._on_navi_clicked
        self._save_btn.on_click(self._btn_clicked)
        self._grid_box.on_click(self.on_grid_clicked)
        self._none_checkbox.observe(self._none_checkbox_changed, 'value')

        if self._capture_state.question_value:
            self._set_label(self._capture_state.question_value)

        if self._app_state.max_im_number:
            self._set_navi_max_im_number(self._app_state.max_im_number)

        if self._capture_state.annotations:
            self._load_menu(self._capture_state.annotations)

        self._capture_state.subscribe(self._set_none_checkbox, 'all_none')
        self._capture_state.subscribe(self._set_label, 'question_value')
        self._app_state.subscribe(self._set_navi_max_im_number, 'max_im_number')
        self._capture_state.subscribe(self._load_menu, 'annotations')

        super().__init__(
            header=None,
            left_sidebar=None,
            center=self._labels_box,
            right_sidebar=None,
            footer=self._controls_box,
            pane_widths=(2, 8, 0),
            pane_heights=(1, 4, 1))

    def _on_navi_clicked(self, index: int):
        if self.on_navi_clicked:
            self.on_navi_clicked(index)

        self._grid_box.load(
            _label_store_to_image_button(self._capture_state.annotations)
        )

    @debug_output.capture(clear_output=True)
    def _load_menu(self, annotations: LabelStore):
        self._grid_box.load(
            _label_store_to_image_button(annotations)
        )

    def _set_none_checkbox(self, all_none: bool):
        self._none_checkbox.value = all_none

    def _set_navi_max_im_number(self, max_im_number: int):
        self._navi.max_im_num = max_im_number

    def _set_label(self, question_value: str):
        self._grid_label.value = question_value

    def _btn_clicked(self, *args):
        if self._save_btn_clicked:
            self._save_btn_clicked(self._app_state.index)
        else:
            warnings.warn("Save button click didn't triggered any event.")

    def _none_checkbox_changed(self, change: dict):
        self._capture_state.set_quietly('all_none', change['new'])
        if self._select_none_changed:
            self._select_none_changed(change)

    def on_grid_clicked(self, event, name=None):
        if self._grid_box_clicked:
            self._grid_box_clicked(event, name)
        else:
            warnings.warn("Grid box click didn't triggered any event.")

# Internal Cell

class CaptureAnnotationStorage:
    def __init__(
        self,
        input_item_path: Path,
        annotation_file_path: str
    ):
        self.input_item_path = input_item_path
        self.annotation_file_path = annotation_file_path

        self.annotations = JsonCaptureStorage(
            self.input_item_path,
            annotation_file_path=self.annotation_file_path
        )

    def __setitem__(self, key, value):
        self.annotations[key] = value

    def _save(self):
        self.annotations.save()

    def get_im_names(self, filter_files) -> List[str]:
        return self.annotations.get_im_names(filter_files)

    def get(self, path: str) -> Optional[Dict]:
        return self.annotations.get(path)

    def update_annotations(self, annotations: dict):
        for p, anno in annotations.items():
            self.annotations[str(p)] = anno
        self._save()

    def to_dict(self, only_annotated: bool = True) -> dict:
        return self.annotations.to_dict(only_annotated)

# Internal Cell

class CaptureAnnotatorController:
    debug_output = Output(layout={'border': '1px solid black'})

    def __init__(
        self,
        app_state: AppWidgetState,
        capture_state: CaptureState,
        storage: CaptureAnnotationStorage,
        input_item=None,
        output_item=None,
        filter_files=None,
        question=None,
        *args,
        **kwargs
    ):
        self._app_state = app_state
        self._capture_state = capture_state
        self._storage = storage
        self.input_item = input_item
        self.output_item = output_item
        self._last_index = 0

        self.images = self._storage.get_im_names(filter_files)
        self.current_im_number = len(self.images)

        if question:
            self._capture_state.question_value = ('<center><p style="font-size:20px;"'
                                                  f'>{question}</p></center>')

        self.update_state()
        self._calc_screens_num()

    def update_state(self):
        state_images = self._get_state_names(self._app_state.index)
        tmp_annotations = deepcopy(self._capture_state.annotations)
        current_state = {}

        for im_path in state_images:
            current_state[str(im_path)] = self._storage.get(str(im_path)) or {}

        self._update_all_none_state(current_state)

        # error: Incompatible types in assignment (expression has type
        # "Dict[str, Dict[Any, Any]]", variable has type
        # "Dict[str, Optional[Dict[str, bool]]]")
        tmp_annotations.clear()
        tmp_annotations.update(current_state)
        self._capture_state.annotations = tmp_annotations  # type: ignore

    def _update_all_none_state(self, state_images: dict):
        self._capture_state.all_none = all(
            value == {'answer': False} for value in state_images.values()
        )

    def save_annotations(self, index: int):  # to disk
        state_images = dict(self._capture_state.annotations)

        self._storage.update_annotations(state_images)

    def _get_state_names(self, index: int) -> List[str]:
        start = index * self._capture_state.grid.disp_number
        end = start + self._capture_state.grid.disp_number
        im_names = self.images[start:end]
        return im_names

    def idx_changed(self, index: int):
        ''' On index change save old state
            and update current state for visualisation
        '''
        self._app_state.set_quietly('index', index)
        self.save_annotations(self._last_index)
        self.update_state()
        self._last_index = index

    def _calc_screens_num(self):
        self._app_state.max_im_number = math.ceil(
            self.current_im_number / self._capture_state.grid.disp_number
        )

    @debug_output.capture(clear_output=False)
    def handle_grid_click(self, event: dict, name=None):
        p = self._storage.input_item_path / name
        current_state = deepcopy(self._capture_state.annotations)
        if not p.is_dir():
            state_answer = self._capture_state.annotations[
                str(p)].get('answer', False)
            current_state[str(p)] = {'answer': not state_answer}

            for k, v in current_state.items():
                if v == {}:
                    current_state[k] = {'answer': False}

            if self._capture_state.all_none:
                self._capture_state.all_none = False
            else:
                self._update_all_none_state(dict(current_state))
        else:
            return

        self._capture_state.annotations = current_state

    def select_none(self, change: dict):
        if self._capture_state.all_none:
            tmp_annotations = deepcopy(self._capture_state.annotations)
            tmp_annotations.clear()
            tmp_annotations.update(
                {p: {
                    'answer': False} for p in self._capture_state.annotations}
            )
            self._capture_state.annotations = tmp_annotations

# Cell

class CaptureAnnotator(Annotator):
    debug_output = Output(layout={'border': '1px solid black'})
    """
    Represents capture annotator.

    Gives an ability to itarate through image dataset,
    select images of same class,
    export final annotations in json format

    """
#     @debug_output.capture(clear_output=True)

    def __init__(
        self,
        project_path: Path,
        input_item,
        output_item,
        annotation_file_path,
        n_rows=3,
        n_cols=3,
        disp_number=9,
        question=None,
        filter_files=None
    ):

        assert input_item, "WARNING: Provide valid Input"
        assert output_item, "WARNING: Provide valid Output"

        self._project_path = project_path
        self._input_item = input_item
        self._output_item = output_item
        self._annotation_file_path = annotation_file_path
        self._n_rows = n_rows
        self._n_cols = n_cols
        self._question = question
        self._filter_files = filter_files

        app_state = AppWidgetState(
            uuid=str(id(self)),
            **{'size': (input_item.width, input_item.height)}
        )

        super().__init__(app_state)

        grid = Grid(
            width=input_item.width,
            height=input_item.height,
            n_rows=n_rows,
            n_cols=n_cols,
            display_label=False,
            disp_number=disp_number
        )

        self.capture_state = CaptureState(
            uuid=str(id(self)),
            annotations=LabelStore(),
            grid=grid
        )

        self.storage = CaptureAnnotationStorage(
            input_item_path=project_path / input_item.dir,
            annotation_file_path=annotation_file_path
        )

        self.controller = CaptureAnnotatorController(
            app_state=self.app_state,
            storage=self.storage,
            capture_state=self.capture_state,
            input_item=input_item,
            output_item=output_item,
            question=question,
            n_rows=n_rows,
            n_cols=n_cols,
            filter_files=filter_files
        )

        self.view = CaptureAnnotatorGUI(
            capture_state=self.capture_state,
            app_state=self.app_state,
            save_btn_clicked=self.controller.save_annotations,
            grid_box_clicked=self.controller.handle_grid_click,
            on_navi_clicked=self.controller.idx_changed,
            select_none_changed=self.controller.select_none
        )

    def __repr__(self):
        display(self.view)
        return ""

    def to_dict(self, only_annotated=True) -> dict:
        return self.storage.to_dict(only_annotated)