# AUTOGENERATED! DO NOT EDIT! File to edit: nbs/04_bbox_annotator.ipynb (unless otherwise specified).

__all__ = ['BBoxAnnotator']

# Internal Cell

import warnings
from pubsub import pub
from attr import asdict
from pathlib import Path
from copy import deepcopy
from typing import Optional, List, Callable

from IPython.display import display
from ipywidgets import AppLayout, Button, HBox, VBox, Layout

from .mltypes import BboxCoordinate
from .base import BaseState, AppWidgetState, Annotator
from .mltypes import InputImage, OutputImageBbox
from .bbox_canvas import BBoxCanvas, BBoxCanvasState
from .navi_widget import Navi
from .right_menu_widget import BBoxList, BBoxVideoItem
from .storage import JsonCaptureStorage
from .custom_input.buttons import ActionButton

# Internal Cell

class BBoxState(BaseState):
    coords: Optional[List[BboxCoordinate]]
    image: Optional[Path]
    classes: List[str]
    labels: List[List[str]] = []
    drawing_enabled: bool = True

# Internal Cell

class BBoxCoordinates(VBox):
    """Connects the BBoxList and the states"""

    def __init__(
        self,
        app_state: AppWidgetState,
        bbox_canvas_state: BBoxCanvasState,
        bbox_state: BBoxState,
        on_btn_select_clicked: Callable = None
    ):
        super().__init__()

        self._app_state = app_state
        self._bbox_state = bbox_state
        self._bbox_canvas_state = bbox_canvas_state
        self.on_btn_select_clicked = on_btn_select_clicked

        self._init_bbox_list(self._bbox_state.drawing_enabled)

        if self._bbox_canvas_state.bbox_coords:
            self._bbox_list.render_btn_list(
                self._bbox_canvas_state.bbox_coords,
                self._bbox_state.labels
            )

        app_state.subscribe(self._refresh_children, 'index')
        bbox_state.subscribe(self._init_bbox_list, 'drawing_enabled')
        bbox_canvas_state.subscribe(self._sync_labels, 'bbox_coords')
        self._bbox_canvas_state.subscribe(self._update_max_coord_input, 'image_scale')
        self._update_max_coord_input(self._bbox_canvas_state.image_scale)
        self.children = self._bbox_list.children
        self.layout = Layout(
            max_height=f'{self._app_state.size[1]}px',
            display='block'
        )

    def _init_bbox_list(self, drawing_enabled: bool):
        self._bbox_list = BBoxList(
            max_coord_input_values=None,
            on_coords_changed=self.on_coords_change,
            on_label_changed=self.on_label_change,
            on_btn_delete_clicked=self.on_btn_delete_clicked,
            on_btn_select_clicked=self.on_btn_select_clicked,
            classes=self._bbox_state.classes,
            readonly=not drawing_enabled
        )

        self._refresh_children(0)

    def __getitem__(self, index: int) -> BBoxVideoItem:
        return self.children[index]

    def _refresh_children(self, index: int):
        self._bbox_list.clear()
        self._render(
            self._bbox_canvas_state.bbox_coords,
            self._bbox_state.labels
        )

    def _sync_labels(self, bbox_coords: List[BboxCoordinate]):
        """Every time a new coord is added to the annotator
        it's added an empty label to the state"""
        num_classes = len(self._bbox_state.labels)

        for i in bbox_coords[num_classes:]:
            self._bbox_state.labels.append([])

        self._render(bbox_coords, self._bbox_state.labels)

    def on_coords_change(self, index: int, key: str, value: int):
        setattr(self._bbox_canvas_state.bbox_coords[index], key, value)

        pub.sendMessage(
            f'{self._bbox_canvas_state.root_topic}.coord_changed',
            bbox_coords=self._bbox_canvas_state.bbox_coords
        )

    def _render(self, bbox_coords: list, labels: list):
        self._bbox_list.render_btn_list(bbox_coords, labels)
        self.children = self._bbox_list.children

    def on_label_change(self, change: dict, index: int):
        self._bbox_state.labels[index] = [change['new']]

    def remove_label(self, index: int):
        tmp_labels = deepcopy(self._bbox_state.labels)
        del tmp_labels[index]
        self._bbox_state.set_quietly('labels', tmp_labels)

    def on_btn_delete_clicked(self, index: int):
        bbox_coords = self._bbox_canvas_state.bbox_coords.copy()
        del bbox_coords[index]
        self.remove_label(index)
        self._bbox_canvas_state.bbox_coords = bbox_coords

    def _update_max_coord_input(self, image_scale: float):
        """CoordinateInput maximum values that a user
        can change. 'x' and 'y' can be improved to avoid
        bbox outside of the canvas area."""
        im_width = self._bbox_canvas_state.image_width
        im_height = self._bbox_canvas_state.image_height
        if im_height is not None and im_width is not None:
            size = [
                im_width // image_scale,
                im_height // image_scale
            ]
            coords = [int(size[i & 1]) for i in range(4)]
            self._bbox_list.max_coord_input_values = BboxCoordinate(*coords)

# Internal Cell
class BBoxAnnotatorGUI(AppLayout):
    def __init__(
        self,
        app_state: AppWidgetState,
        bbox_state: BBoxState,
        fit_canvas: bool,
        on_save_btn_clicked: Callable = None,
        has_border: bool = False
    ):
        self._app_state = app_state
        self._bbox_state = bbox_state
        self._on_save_btn_clicked = on_save_btn_clicked
        self._label_history: List[List[str]] = []
        self.fit_canvas = fit_canvas
        self.has_border = has_border

        self._navi = Navi()

        self._save_btn = Button(description="Save",
                                layout=Layout(width='auto'))

        self._undo_btn = Button(description="Undo",
                                icon="undo",
                                layout=Layout(width='auto'))

        self._redo_btn = Button(description="Redo",
                                icon="rotate-right",
                                layout=Layout(width='auto'))

        self._controls_box = HBox(
            [self._navi, self._save_btn, self._undo_btn, self._redo_btn],
            layout=Layout(
                display='flex',
                flex_flow='row wrap',
                align_items='center'
            )
        )

        self._init_canvas(self._bbox_state.drawing_enabled)

        self.right_menu = BBoxCoordinates(
            app_state=self._app_state,
            bbox_canvas_state=self._image_box.state,
            bbox_state=self._bbox_state,
            on_btn_select_clicked=self._highlight_bbox
        )

        self._annotator_box = HBox(
            [self._image_box, self.right_menu],
            layout=Layout(
                display='flex',
                flex_flow='row'
            )
        )

        # set the values already instantiated on state
        if self._app_state.max_im_number:
            self._set_max_im_number(self._app_state.max_im_number)

        if self._bbox_state.image:
            self._set_image_path(str(self._bbox_state.image))

        # set the GUI interactions as callables
        self._navi.on_navi_clicked = self._idx_changed
        self._save_btn.on_click(self._save_clicked)
        self._undo_btn.on_click(self._undo_clicked)
        self._redo_btn.on_click(self._redo_clicked)

        bbox_state.subscribe(self._set_image_path, 'image')
        bbox_state.subscribe(self._init_canvas, 'drawing_enabled')
        bbox_state.subscribe(self._set_coords, 'coords')
        app_state.subscribe(self._set_max_im_number, 'max_im_number')

        super().__init__(
            header=None,
            left_sidebar=None,
            center=self._annotator_box,
            right_sidebar=None,
            footer=self._controls_box,
            pane_widths=(2, 8, 0),
            pane_heights=(1, 4, 1))

    def _init_canvas(self, drawing_enabled: bool):
        self._image_box = BBoxCanvas(
            *self._app_state.size,
            drawing_enabled=drawing_enabled,
            fit_canvas=self.fit_canvas,
            has_border=self.has_border
        )

    def _highlight_bbox(self, btn: ActionButton):
        self._image_box.highlight = btn.value

    def _redo_clicked(self, event: dict):
        self._image_box.redo_bbox()
        if self._label_history:
            self._bbox_state.labels[-1] = self._label_history.pop()
        self.right_menu._refresh_children(-1)

    def _undo_clicked(self, event: dict):
        if len(self._bbox_state.labels) > 0:
            self._label_history = [self._bbox_state.labels[-1]]
        self._image_box.undo_bbox()
        self.right_menu.remove_label(-1)
        self.right_menu._refresh_children(-1)

    def _set_image_path(self, image: Optional[str]):
        self._image_box._state.image_path = image

    def _set_coords(self, coords: List[BboxCoordinate]):
        if coords:
            tmp_coords = deepcopy(self._image_box._state.bbox_coords)
            # error: Argument 1 to "append" of "list" has incompatible
            # type "List[BboxCoordinate]"; expected "BboxCoordinate"
            tmp_coords.append(coords)  # type: ignore
            self._image_box._state.bbox_coords = coords

    def _set_max_im_number(self, max_im_number: int):
        # sync the navi GUI with AppWidgetState
        self._navi.max_im_num = max_im_number

    def _idx_changed(self, index: int):
        # store the last bbox drawn before index update
        self._bbox_state.set_quietly('coords', self._image_box._state.bbox_coords)
        self._app_state.index = index

    def _save_clicked(self, *args):
        if self._on_save_btn_clicked:
            self._on_save_btn_clicked(self._image_box._state.bbox_coords)
        else:
            warnings.warn("Save button click didn't triggered any event.")

    def on_client_ready(self, callback):
        self._image_box.observe_client_ready(callback)

# Internal Cell
class BBoxAnnotatorController:
    def __init__(
        self,
        app_state: AppWidgetState,
        bbox_state: BBoxState,
        storage: JsonCaptureStorage,
        render_previous_coords: bool = True,
        **kwargs
    ):
        self._app_state = app_state
        self._bbox_state = bbox_state
        self._storage = storage
        self._last_index = 0

        app_state.subscribe(self._idx_changed, 'index')

        self._update_im(self._last_index)
        self._app_state.max_im_number = len(self._storage)
        if render_previous_coords:
            self._update_coords(self._last_index)

    def save_current_annotations(self, coords: List[BboxCoordinate]):
        self._bbox_state.set_quietly('coords', coords)
        self._save_annotations(self._app_state.index)

    def _update_im(self, index: int):
        self._bbox_state.image = self._storage.images[index]

    def _update_coords(self, index: int):  # from annotations
        image_path = str(self._storage.images[index])
        coords = self._storage.get(image_path) or {}
        self._bbox_state.labels = coords.get('labels', [])
        self._bbox_state.coords = [BboxCoordinate(**c) for c in coords.get('bbox', [])]

    def _save_annotations(self, index: int, *args, **kwargs):  # to disk
        image_path = str(self._storage.images[index])
        self._storage[image_path] = {
            # error: Item "None" of "Optional[List[BboxCoordinate]]" has
            # no attribute "__iter__"
            'bbox': [asdict(bbox) for bbox in self._bbox_state.coords],  # type: ignore
            'labels': self._bbox_state.labels
        }
        self._storage.save()

    def _idx_changed(self, index: int):
        """
        On index change save an old state and update
        current image path and bbox coordinates for
        visualisation
        """
        self._save_annotations(self._last_index)
        self._update_im(index)
        self._update_coords(index)
        self._last_index = index

    def handle_client_ready(self):
        self._idx_changed(self._last_index)

# Cell

class BBoxAnnotator(Annotator):
    """
    Represents bounding box annotator.

    Gives an ability to itarate through image dataset,
    draw 2D bounding box annotations for object detection and localization,
    export final annotations in json format

    """

    def __init__(
        self,
        project_path: Path,
        input_item: InputImage,
        output_item: OutputImageBbox,
        annotation_file_path: Path,
        has_border: bool = False,
        *args, **kwargs
    ):
        app_state = AppWidgetState(
            uuid=str(id(self)),
            **{
                'size': (input_item.width, input_item.height),
            }
        )

        super().__init__(app_state)

        self._input_item = input_item
        self._output_item = output_item

        self.bbox_state = BBoxState(
            uuid=str(id(self)),
            classes=output_item.classes,
            drawing_enabled=self._output_item.drawing_enabled
        )

        self.storage = JsonCaptureStorage(
            im_dir=project_path / input_item.dir,
            annotation_file_path=annotation_file_path
        )

        self.controller = BBoxAnnotatorController(
            app_state=self.app_state,
            bbox_state=self.bbox_state,
            storage=self.storage,
            **kwargs
        )

        self.view = BBoxAnnotatorGUI(
            app_state=self.app_state,
            bbox_state=self.bbox_state,
            fit_canvas=self._input_item.fit_canvas,
            on_save_btn_clicked=self.controller.save_current_annotations,
            has_border=has_border
        )

        self.view.on_client_ready(self.controller.handle_client_ready)

    def __repr__(self):
        display(self.view)
        return ""

    def to_dict(self, only_annotated=True):
        return self.storage.to_dict(only_annotated)