"use strict";
var _a, _b, _c, _d, _e;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AssembleWith = exports.SplitType = exports.BatchStrategy = exports.AcceleratorType = exports.AcceleratorClass = exports.Mode = exports.ContainerDefinition = exports.CompressionType = exports.InputMode = exports.RecordWrapperType = exports.S3DataDistributionType = exports.S3DataType = exports.DockerImage = exports.S3Location = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_ecr_assets_1 = require("@aws-cdk/aws-ecr-assets");
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
/**
 * Constructs `IS3Location` objects.
 *
 */
class S3Location {
    /**
     * An `IS3Location` built with a determined bucket and key prefix.
     *
     * @param bucket    is the bucket where the objects are to be stored.
     * @param keyPrefix is the key prefix used by the location.
     */
    static fromBucket(bucket, keyPrefix) {
        return new StandardS3Location({ bucket, keyPrefix, uri: bucket.urlForObject(keyPrefix) });
    }
    /**
     * An `IS3Location` determined fully by a JSON Path from the task input.
     *
     * Due to the dynamic nature of those locations, the IAM grants that will be set by `grantRead` and `grantWrite`
     * apply to the `*` resource.
     *
     * @param expression the JSON expression resolving to an S3 location URI.
     */
    static fromJsonExpression(expression) {
        return new StandardS3Location({ uri: sfn.JsonPath.stringAt(expression) });
    }
}
exports.S3Location = S3Location;
_a = JSII_RTTI_SYMBOL_1;
S3Location[_a] = { fqn: "@aws-cdk/aws-stepfunctions-tasks.S3Location", version: "1.167.0" };
/**
 * Creates `IDockerImage` instances.
 *
 */
class DockerImage {
    /**
     * Reference a Docker image stored in an ECR repository.
     *
     * @param repository the ECR repository where the image is hosted.
     * @param tagOrDigest an optional tag or digest (digests must start with `sha256:`)
     */
    static fromEcrRepository(repository, tagOrDigest = 'latest') {
        return new StandardDockerImage({ repository, imageUri: repository.repositoryUriForTagOrDigest(tagOrDigest) });
    }
    /**
     * Reference a Docker image which URI is obtained from the task's input.
     *
     * @param expression           the JSON path expression with the task input.
     * @param allowAnyEcrImagePull whether ECR access should be permitted (set to `false` if the image will never be in ECR).
     */
    static fromJsonExpression(expression, allowAnyEcrImagePull = true) {
        return new StandardDockerImage({ imageUri: expression, allowAnyEcrImagePull });
    }
    /**
     * Reference a Docker image by it's URI.
     *
     * When referencing ECR images, prefer using `inEcr`.
     *
     * @param imageUri the URI to the docker image.
     */
    static fromRegistry(imageUri) {
        return new StandardDockerImage({ imageUri });
    }
    /**
     * Reference a Docker image that is provided as an Asset in the current app.
     *
     * @param scope the scope in which to create the Asset.
     * @param id    the ID for the asset in the construct tree.
     * @param props the configuration props of the asset.
     */
    static fromAsset(scope, id, props) {
        const asset = new aws_ecr_assets_1.DockerImageAsset(scope, id, props);
        return new StandardDockerImage({ repository: asset.repository, imageUri: asset.imageUri });
    }
}
exports.DockerImage = DockerImage;
_b = JSII_RTTI_SYMBOL_1;
DockerImage[_b] = { fqn: "@aws-cdk/aws-stepfunctions-tasks.DockerImage", version: "1.167.0" };
/**
 * S3 Data Type.
 *
 */
var S3DataType;
(function (S3DataType) {
    /**
     * Manifest File Data Type
     */
    S3DataType["MANIFEST_FILE"] = "ManifestFile";
    /**
     * S3 Prefix Data Type
     */
    S3DataType["S3_PREFIX"] = "S3Prefix";
    /**
     * Augmented Manifest File Data Type
     */
    S3DataType["AUGMENTED_MANIFEST_FILE"] = "AugmentedManifestFile";
})(S3DataType = exports.S3DataType || (exports.S3DataType = {}));
/**
 * S3 Data Distribution Type.
 *
 */
var S3DataDistributionType;
(function (S3DataDistributionType) {
    /**
     * Fully replicated S3 Data Distribution Type
     */
    S3DataDistributionType["FULLY_REPLICATED"] = "FullyReplicated";
    /**
     * Sharded By S3 Key Data Distribution Type
     */
    S3DataDistributionType["SHARDED_BY_S3_KEY"] = "ShardedByS3Key";
})(S3DataDistributionType = exports.S3DataDistributionType || (exports.S3DataDistributionType = {}));
/**
 * Define the format of the input data.
 *
 */
var RecordWrapperType;
(function (RecordWrapperType) {
    /**
     * None record wrapper type
     */
    RecordWrapperType["NONE"] = "None";
    /**
     * RecordIO record wrapper type
     */
    RecordWrapperType["RECORD_IO"] = "RecordIO";
})(RecordWrapperType = exports.RecordWrapperType || (exports.RecordWrapperType = {}));
/**
 *  Input mode that the algorithm supports.
 *
 */
var InputMode;
(function (InputMode) {
    /**
     * Pipe mode
     */
    InputMode["PIPE"] = "Pipe";
    /**
     * File mode.
     */
    InputMode["FILE"] = "File";
})(InputMode = exports.InputMode || (exports.InputMode = {}));
/**
 * Compression type of the data.
 *
 */
var CompressionType;
(function (CompressionType) {
    /**
     * None compression type
     */
    CompressionType["NONE"] = "None";
    /**
     * Gzip compression type
     */
    CompressionType["GZIP"] = "Gzip";
})(CompressionType = exports.CompressionType || (exports.CompressionType = {}));
/**
 * Describes the container, as part of model definition.
 *
 * @see https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ContainerDefinition.html
 */
class ContainerDefinition {
    constructor(options) {
        this.options = options;
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_stepfunctions_tasks_ContainerDefinitionOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, ContainerDefinition);
            }
            throw error;
        }
    }
    /**
     * Called when the ContainerDefinition type configured on Sagemaker Task
     */
    bind(task) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_stepfunctions_tasks_ISageMakerTask(task);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.bind);
            }
            throw error;
        }
        return {
            parameters: {
                ContainerHostname: this.options.containerHostName,
                Image: this.options.image?.bind(task).imageUri,
                Mode: this.options.mode,
                ModelDataUrl: this.options.modelS3Location?.bind(task, { forReading: true }).uri,
                ModelPackageName: this.options.modelPackageName,
                Environment: this.options.environmentVariables?.value,
            },
        };
    }
}
exports.ContainerDefinition = ContainerDefinition;
_c = JSII_RTTI_SYMBOL_1;
ContainerDefinition[_c] = { fqn: "@aws-cdk/aws-stepfunctions-tasks.ContainerDefinition", version: "1.167.0" };
/**
 * Specifies how many models the container hosts
 *
 */
var Mode;
(function (Mode) {
    /**
     * Container hosts a single model
     */
    Mode["SINGLE_MODEL"] = "SingleModel";
    /**
     * Container hosts multiple models
     *
     * @see https://docs.aws.amazon.com/sagemaker/latest/dg/multi-model-endpoints.html
     */
    Mode["MULTI_MODEL"] = "MultiModel";
})(Mode = exports.Mode || (exports.Mode = {}));
/**
 * The generation of Elastic Inference (EI) instance
 *
 * @see https://docs.aws.amazon.com/sagemaker/latest/dg/ei.html
 */
class AcceleratorClass {
    /**
     * @param version - Elastic Inference accelerator generation
     */
    constructor(version) {
        this.version = version;
    }
    /**
     * Custom AcceleratorType
     * @param version - Elastic Inference accelerator generation
    */
    static of(version) { return new AcceleratorClass(version); }
}
exports.AcceleratorClass = AcceleratorClass;
_d = JSII_RTTI_SYMBOL_1;
AcceleratorClass[_d] = { fqn: "@aws-cdk/aws-stepfunctions-tasks.AcceleratorClass", version: "1.167.0" };
/**
 * Elastic Inference accelerator 1st generation
 */
AcceleratorClass.EIA1 = AcceleratorClass.of('eia1');
/**
 * Elastic Inference accelerator 2nd generation
 */
AcceleratorClass.EIA2 = AcceleratorClass.of('eia2');
/**
 * The size of the Elastic Inference (EI) instance to use for the production variant.
 * EI instances provide on-demand GPU computing for inference
 *
 * @see https://docs.aws.amazon.com/sagemaker/latest/dg/ei.html
 */
class AcceleratorType {
    constructor(instanceTypeIdentifier) {
        this.instanceTypeIdentifier = instanceTypeIdentifier;
    }
    /**
     * AcceleratorType
     *
     * This class takes a combination of a class and size.
     */
    static of(acceleratorClass, instanceSize) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_stepfunctions_tasks_AcceleratorClass(acceleratorClass);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.of);
            }
            throw error;
        }
        return new AcceleratorType(`ml.${acceleratorClass}.${instanceSize}`);
    }
    /**
     * Return the accelerator type as a dotted string
     */
    toString() {
        return this.instanceTypeIdentifier;
    }
}
exports.AcceleratorType = AcceleratorType;
_e = JSII_RTTI_SYMBOL_1;
AcceleratorType[_e] = { fqn: "@aws-cdk/aws-stepfunctions-tasks.AcceleratorType", version: "1.167.0" };
/**
 * Specifies the number of records to include in a mini-batch for an HTTP inference request.
 *
 */
var BatchStrategy;
(function (BatchStrategy) {
    /**
     * Fits multiple records in a mini-batch.
     */
    BatchStrategy["MULTI_RECORD"] = "MultiRecord";
    /**
     * Use a single record when making an invocation request.
     */
    BatchStrategy["SINGLE_RECORD"] = "SingleRecord";
})(BatchStrategy = exports.BatchStrategy || (exports.BatchStrategy = {}));
/**
 * Method to use to split the transform job's data files into smaller batches.
 *
 */
var SplitType;
(function (SplitType) {
    /**
     * Input data files are not split,
     */
    SplitType["NONE"] = "None";
    /**
     * Split records on a newline character boundary.
     */
    SplitType["LINE"] = "Line";
    /**
     * Split using MXNet RecordIO format.
     */
    SplitType["RECORD_IO"] = "RecordIO";
    /**
     * Split using TensorFlow TFRecord format.
     */
    SplitType["TF_RECORD"] = "TFRecord";
})(SplitType = exports.SplitType || (exports.SplitType = {}));
/**
 * How to assemble the results of the transform job as a single S3 object.
 *
 */
var AssembleWith;
(function (AssembleWith) {
    /**
     * Concatenate the results in binary format.
     */
    AssembleWith["NONE"] = "None";
    /**
     * Add a newline character at the end of every transformed record.
     */
    AssembleWith["LINE"] = "Line";
})(AssembleWith = exports.AssembleWith || (exports.AssembleWith = {}));
class StandardDockerImage extends DockerImage {
    constructor(opts) {
        super();
        this.allowAnyEcrImagePull = !!opts.allowAnyEcrImagePull;
        this.imageUri = opts.imageUri;
        this.repository = opts.repository;
    }
    bind(task) {
        if (this.repository) {
            this.repository.grantPull(task);
        }
        if (this.allowAnyEcrImagePull) {
            task.grantPrincipal.addToPrincipalPolicy(new iam.PolicyStatement({
                actions: [
                    'ecr:BatchCheckLayerAvailability',
                    'ecr:GetDownloadUrlForLayer',
                    'ecr:BatchGetImage',
                ],
                resources: ['*'],
            }));
        }
        return {
            imageUri: this.imageUri,
        };
    }
}
class StandardS3Location extends S3Location {
    constructor(opts) {
        super();
        this.bucket = opts.bucket;
        this.keyGlob = `${opts.keyPrefix || ''}*`;
        this.uri = opts.uri;
    }
    bind(task, opts) {
        if (this.bucket) {
            if (opts.forReading) {
                this.bucket.grantRead(task, this.keyGlob);
            }
            if (opts.forWriting) {
                this.bucket.grantWrite(task, this.keyGlob);
            }
        }
        else {
            const actions = new Array();
            if (opts.forReading) {
                actions.push('s3:GetObject', 's3:ListBucket');
            }
            if (opts.forWriting) {
                actions.push('s3:PutObject');
            }
            task.grantPrincipal.addToPrincipalPolicy(new iam.PolicyStatement({ actions, resources: ['*'] }));
        }
        return { uri: this.uri };
    }
}
//# sourceMappingURL=data:application/json;base64,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