import logging
from os.path import exists as path_exists
from os.path import expanduser
from os.path import join as path_join
from typing import Dict, List

from .api import (
    ABCSubDirAddons,
    AddonsSuffix,
    KeySuffix,
    LocalAddonsResult,
    OdooAddonsDef,
)

_logger = logging.getLogger(__name__)
_logger.setLevel(logging.INFO)


class GitOdooAddons(AddonsSuffix):
    _prefix = "ADDONS_GIT"
    PROTOCLE_HTTPS = "https"
    PROTOCLE_SSH = "ssh"
    PROTOCLE_PUBLIC = "public"
    FORMAT_GIT_CLONE = {
        PROTOCLE_HTTPS: "https://%(login)s:%(password)s@%(server)s/%(git_path)s.git",
        PROTOCLE_SSH: "git@%(server)s:%(git_path)s.git",
        PROTOCLE_PUBLIC: "https://%(server)s/%(git_path)s.git",
    }

    def __init__(self, base_key):
        super(GitOdooAddons, self).__init__(GitOdooAddons._prefix, base_key)
        self.BRANCH = self.create_key("BRANCH", default="master")
        self.CLONE_PATH = self.create_key("CLONE_PATH")
        self.PULL_OPTIONS = self.create_key("PULL_OPTIONS", default="--depth=1,--quiet,--single-branch")
        self.HTTPS_LOGIN = self.create_key("HTTPS_LOGIN")
        self.HTTPS_PASSWORD = self.create_key("HTTPS_PASSWORD")
        self.PROTOCOLE = self.create_key("PROTOCOLE", default=self.PROTOCLE_PUBLIC)
        self.SERVER = self.create_key("SERVER")

    def extract(self, env_vars):
        # type: (Dict[str, str]) -> GitOdooAddonsResult
        res = self.to_dict(env_vars)
        # Remove de '/' at the end and the beginning of the name '/p1/p2' become 'p1/p2'
        self._apply_check(env_vars, res)

        res[self.NAME] = res[self.NAME].strip("/")

        protocole = self._get_protocole(env_vars, res)
        clone_path = self._get_clone_path(env_vars, res)
        pull_option = self._get_pull_option(res)

        if self.PROTOCLE_HTTPS != protocole:
            res.pop(self.HTTPS_LOGIN, False)
            res.pop(self.HTTPS_PASSWORD, False)

        return GitOdooAddonsResult(
            name=self.identifier,
            git_path=res[self.NAME],
            branch=res[self.BRANCH],
            clone_path=clone_path,
            pull_options=pull_option,
            https_login=res.get(self.HTTPS_LOGIN),
            https_password=res.get(self.HTTPS_PASSWORD),
            protocole=protocole,
            server=res[self.SERVER],
        )

    def _get_protocole(self, env_vars, res):
        # type: (Dict[str, str], Dict[KeySuffix, str]) -> str
        if (
            not self.PROTOCOLE.get_value(env_vars, with_default=False)
            and res[self.HTTPS_LOGIN]
            and res[self.HTTPS_PASSWORD]
        ):
            return self.PROTOCLE_HTTPS
        return res[self.PROTOCOLE]

    def _apply_check(self, env_vars, res):
        # type: (Dict[str, str], Dict[KeySuffix, str]) -> None
        if not res[self.SERVER]:
            raise ValueError(
                "Not git server is provided, key [%s] or [%s]" % (self.SERVER.full_key, self.SERVER.default_key)
            )
        if res[self.PROTOCOLE] not in self.FORMAT_GIT_CLONE.keys():
            raise ValueError(
                "The selected protocole %s is not supported, possible values are %s"
                % (res[self.PROTOCOLE], sorted(list(self.FORMAT_GIT_CLONE.keys())))
            )
        if self.PROTOCLE_SSH == res[self.PROTOCOLE]:
            raise ValueError("Protocole [%s] not supported for the moment" % self.PROTOCLE_SSH)

        protocole = self._get_protocole(env_vars, res)
        if self.PROTOCLE_HTTPS == protocole:
            if not res[self.HTTPS_LOGIN] or not res[self.HTTPS_PASSWORD]:
                raise ValueError(
                    "Please add %s and %s var in your environment when you use [%s] has git protocole"
                    % (
                        self.HTTPS_LOGIN.full_key,
                        self.HTTPS_PASSWORD.full_key,
                        self.PROTOCLE_HTTPS,
                    )
                )

    def _get_clone_path(self, env_vars, res):
        # type: (Dict[str, str], Dict[KeySuffix, str]) -> str
        if self.CLONE_PATH.default_key in env_vars and self.CLONE_PATH.full_key not in env_vars:
            # If default key in env_vars but not dedicated key then we add identifier in lower case to the clone path
            return path_join(res[self.CLONE_PATH], self.identifier.lower())
        return res[self.CLONE_PATH]

    def _get_pull_option(self, res):
        pull_option = res[self.PULL_OPTIONS]
        if isinstance(pull_option, str):
            pull_option = pull_option.split(",")
        if "..." in pull_option:
            idx = pull_option.index("...")
            pull_option.pop(idx)
            pull_option[idx:idx] = self.PULL_OPTIONS.default_value.split(",")
        return pull_option


class GitOdooAddonsResult(OdooAddonsDef):
    def __init__(
        self,
        name,
        git_path,
        branch,
        clone_path,
        pull_options,
        https_login,
        https_password,
        protocole,
        server,
    ):
        super(GitOdooAddonsResult, self).__init__(name)
        self.git_path = git_path
        self.branch = branch
        if not clone_path:
            clone_path = path_join("/", "opt", "addons", git_path.lower())
        if not clone_path:
            clone_path = path_join(clone_path, git_path.lower())
        self.clone_path = expanduser(clone_path)
        self.pull_options = pull_options
        self.https_login = https_login
        self.https_password = https_password
        self.protocole = protocole
        self.server = server
        self.format = GitOdooAddons.FORMAT_GIT_CLONE[protocole]

    def install_cmd(self):
        # type: () -> List[List[str]]
        if path_exists(self.clone_path):
            _logger.info("Path %s not empty to clone %s", self.clone_path, self)
            return []
        clone_cmd = ["git", "clone"]
        clone_cmd.extend(self.arg_cmd())
        clone_cmd.append(self.clone_path)
        return [clone_cmd]

    def arg_cmd(self):
        # type: () -> List[str]
        clone_cmd = []
        if self.pull_options:
            clone_cmd.extend(self.pull_options)
        if self.branch:
            clone_cmd.append("-b")
            clone_cmd.append(self.branch)
        clone_cmd.append(self.git_url)
        return clone_cmd

    @property
    def addons_path(self):
        return self.clone_path

    @property
    def git_url(self):
        return self.format % {
            "login": self.https_login,
            "password": self.https_password,
            "server": self.server,
            "git_path": self.git_path,
        }


class GitSubDirOdooAddons(ABCSubDirAddons):
    _identifier = "ADDONS_SUBDIR_GIT"

    def get_parent(self, key):
        return GitOdooAddons(key)

    def extract(self, env_vars):
        # type: (Dict[str, str]) -> LocalAddonsResult
        res = self.to_dict(env_vars)
        git_res = self.parent_addons.to_dict(env_vars)
        full_path = path_join(git_res[self.parent_addons.CLONE_PATH], res[self.NAME])
        return LocalAddonsResult(name=self.NAME.full_key, full_path=full_path)
