from sphere_base.model.mesh import Mesh
from sphere_base.sphere_universe_base.suv_utils import dump_exception

DEBUG = False

"""
Handwritten .obj file loader. The main PyAssimp file loader could not be installed on MAC.
This file loader is based on code from an OpenGl you tube course (ES
OpenGL in python e15 - loading 3D .obj files) and only works with .obj wavefront files and glDrawArrays.

The code has been modified as there was several errors related to the indices.

"""


class ObjectFileLoader:
    Mesh_class = Mesh

    def __init__(self, model):
        self.meshes = []
        self.buffer = []
        self.indices = []  # how many f x/x/x indices there are
        self.model = model
        self.config = model.config
        self.index = 0

    def get_meshes(self, file_name):
        # Creates
        indices = []  # The number of all_indices
        vert = []  # vertex coordinates
        tex = []  # texture coordinates
        norm = []  # vertex normals
        all_indices = []  # indices for indexed drawing
        test = []

        try:
            with open(file_name, 'r') as f:
                for line in self.non_blank_lines(f):

                    values = line.split()

                    if values[0] == 'v':
                        vert = self.search_data(values, vert, 'v', 'float')
                    elif values[0] == 'vt':
                        tex = self.search_data(values, tex, 'vt', 'float')
                    elif values[0] == 'vn':
                        norm = self.search_data(values, norm, 'vn', 'float')
                    elif values[0] == 'f':
                        for value in values[1:]:
                            val = value.split('/')
                            all_indices = self.search_data(val, all_indices, 'f', 'int')
                            indices.append(self.index)
                            self.index += 1

            self.create_sorted_vertex_buffer(all_indices, vert, tex, norm)
            self.indices = indices

            if self.model.name == "square1x1" or self.model.name == "rubber_band":
                vert, self.indices, self.buffer = self.load_square1x1()
            elif self.model.name == "circle" or self.model.name == "square" or self.model.name == "cross_hair1":
                vert, self.indices, self.buffer = self.load_vertex1()
            elif self.model.name == "node" or self.model.name == "socket":
                vert, self.indices, self.buffer = self.load_node_disc()

            mesh_id = self.config.get_mesh_id()
            mesh = self.__class__.Mesh_class(self.model, mesh_id, vertices=vert, indices=self.indices,
                                             buffer=self.buffer)
            self.meshes.append(mesh)
        except Exception as e:
            dump_exception(e)

        if DEBUG:
            if self.model.name == "node":
                print("vertices", self.model.name, vert)
                print("indices", self.model.name, self.indices)
                print("buffer", self.model.name, self.buffer)
                self.show_buffer_data(self.buffer)
                print("\n")

        return self.meshes

    def create_sorted_vertex_buffer(self, indices_data, vertices, textures, normals):
        # sorted vertex buffer for use with glDrawArrays function

        for i, ind in enumerate(indices_data):

            if i % 3 == 0:  # sort the vertex coordinates
                start = ind * 3
                end = start + 3
                self.buffer.extend(vertices[start:end])
            elif i % 3 == 1:  # sort the texture coordinates
                start = ind * 2
                end = start + 2
                self.buffer.extend(textures[start:end])
            elif i % 3 == 2:  # sort the normal vectors
                start = ind * 3
                end = start + 3
                self.buffer.extend(normals[start:end])

    @staticmethod
    def non_blank_lines(f):
        for l in f:
            line = l.rstrip()
            if line:
                yield line

    @staticmethod
    def search_data(data_values, val_list, skip, data_type):
        # appends each data value found in the data value line to the val_list
        for d in data_values:
            if d == skip:
                continue
            if data_type == 'float':
                val_list.append(float(d))
            elif data_type == 'int':
                val_list.append(int(d) - 1)
        return val_list

    @staticmethod
    def show_buffer_data(buffer):
        print("buffer data in blocks")
        for i in range(len(buffer) // 8):
            start = i * 8
            end = start + 8
            print(buffer[start:end])

    @staticmethod
    def load_square1x1():
        # a manual way to load object square1x1
        vert = [0.5, -0.5, 0., -0.5, 0.5, 0., -0.5, -0.5, 0., 0.5, -0.5, 0., 0.5, 0.5, 0., -0.5, 0.5, 0.]
        ind = [0, 1, 2, 3, 4, 5]
        buffer = \
            [0.5, -0.5, 0., 1., 0., 0., 0., 1., -0.5, 0.5, 0., 0., 1., 0., 0., 1., -0.5, -0.5, 0., 0., 0., 0., 0.,
             1., 0.5, -0.5, 0., 1., 0., 0., 0., 1., 0.5, 0.5, 0., 1., 1., 0., 0., 1., -0.5, 0.5, 0., 0., 1., 0., 0., 1.]
        return vert, ind, buffer

    @staticmethod
    def load_vertex1():
        # a manual way to load object vertex1
        vert = [0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0.,
                0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0.,
                0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0.]
        ind = [0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23]
        buffer = [0., 0., 0., 0., 0., 1., 0., 0., 0., 0., 0., 0., 0., 1., 0., 0., 0., 0., 0., 0., 0., 1., 0., 0.,
                  0., 0., 0., 0., 0., 1., 0., 0., 0., 0., 0., 0., 0., 1., 0., 0., 0., 0., 0., 0., 0., 1., 0., 0.,
                  0., 0., 0., 0., 0., 1., 0., 0., 0., 0., 0., 0., 0., 1., 0., 0., 0., 0., 0., 0., 0., 1., 0., 0.,
                  0., 0., 0., 0., 0., 1., 0., 0., 0., 0., 0., 0., 0., 1., 0., 0., 0., 0., 0., 0., 0., 1., 0., 0.,
                  0., 0., 0., 0., 0., 1., 0., 0., 0., 0., 0., 0., 0., 1., 0., 0., 0., 0., 0., 0., 0., 1., 0., 0.,
                  0., 0., 0., 0., 0., 1., 0., 0., 0., 0., 0., 0., 0., 1., 0., 0., 0., 0., 0., 0., 0., 1., 0., 0.,
                  0., 0., 0., 0., 0., 1., 0., 0., 0., 0., 0., 0., 0., 1., 0., 0., 0., 0., 0., 0., 0., 1., 0., 0.,
                  0., 0., 0., 0., 0., 1., 0., 0., 0., 0., 0., 0., 0., 1., 0., 0., 0., 0., 0., 0., 0., 1., 0., 0.]
        return vert, ind, buffer

    def load_node_disc(self):
        # a manual way to load object node_disc
        vert = [
            -0.003901779,-0.0196157, 0.0, 0.0196157, -0.003901779, 0.0, 0.00390172, 0.0196157, 0.0, 0.00390172,
            0.0196157,0.0, 0.0, 0.01999998, 0.0, -0.003901779, 0.0196157, 0.0, -0.003901779, 0.0196157, 0.0,
            -0.007653653, 0.01847755, 0.0, 0.00390172, 0.0196157, 0.0, -0.007653653, 0.01847755, 0.0, -0.01111137,
            0.01662939, 0.0, 0.00390172, 0.0196157, 0.0, -0.01111137, 0.01662939, 0.0, -0.01414209, 0.01414209,
            0.0, -0.01662939, 0.01111137, 0.0, -0.01662939, 0.01111137, 0.0, -0.01847755, 0.007653653, 0.0,
            -0.01111137, 0.01662939, 0.0, -0.01847755, 0.007653653, 0.0, -0.0196157, 0.003901779, 0.0, -0.01111137,
            0.01662939, 0.0, -0.0196157, 0.003901779, 0.0, -0.01999998, 0.0, 0.0, -0.0196157, -0.003901779,
            0.0, -0.0196157, -0.003901779, 0.0, -0.01847755, -0.007653653, 0.0, -0.0196157, 0.003901779, 0.0,
            -0.01847755, -0.007653653, 0.0, -0.01662939, -0.01111137, 0.0, -0.0196157, 0.003901779, 0.0, -0.01662939,
            -0.01111137, 0.0, -0.01414209, -0.01414209, 0.0, -0.01111137, -0.01662939, 0.0, -0.01111137,-0.01662939,
            0.0, -0.007653653, -0.01847755, 0.0, -0.003901779, -0.0196157, 0.0, -0.003901779, -0.0196157, 0.0,
            0.0, -0.01999998, 0.0, 0.003901779, -0.0196157, 0.0, 0.003901779, -0.0196157, 0.0, 0.007653653,
            -0.01847755, 0.0, -0.003901779, -0.0196157, 0.0, 0.007653653, -0.01847755, 0.0, 0.01111137, -0.01662933,
            0.0, -0.003901779, -0.0196157, 0.0, 0.01111137, -0.01662933, 0.0, 0.01414209, -0.01414209, 0.0,
            0.01662939, -0.01111137, 0.0, 0.01662939, -0.01111137, 0.0, 0.01847755, -0.007653653, 0.0, 0.0196157,
            -0.003901779, 0.0, 0.0196157, -0.003901779, 0.0, 0.01999998, 0.0, 0.0, 0.01961565, 0.003901779,
            0.0, 0.01961565, 0.003901779, 0.0, 0.01847755, 0.007653653, 0.0, 0.01662933, 0.01111137, 0.0,
            0.01662933, 0.01111137, 0.0, 0.01414209, 0.01414215, 0.0, 0.00390172, 0.0196157, 0.0, 0.01414209,
            0.01414215, 0.0, 0.01111137, 0.01662939, 0.0, 0.00390172, 0.0196157, 0.0, 0.01111137, 0.01662939,
            0.0, 0.007653594, 0.01847755, 0.0, 0.00390172, 0.0196157, 0.0, -0.01662939, -0.01111137, 0.0,
            -0.01111137, -0.01662939, 0.0, -0.003901779, -0.0196157, 0.0, 0.01111137, -0.01662933, 0.0, 0.01662939,
            -0.01111137, 0.0, 0.0196157, -0.003901779, 0.0, 0.0196157, -0.003901779, 0.0, 0.01961565, 0.003901779,
            0.0, 0.00390172, 0.0196157, 0.0, 0.01961565, 0.003901779, 0.0, 0.01662933, 0.01111137, 0.0,
            0.00390172, 0.0196157, 0.0, 0.00390172, 0.0196157, 0.0, -0.01111137, 0.01662939, 0.0, -0.0196157,
            0.003901779, 0.0, -0.0196157, 0.003901779, 0.0, -0.01662939, -0.01111137, 0.0, 0.00390172, 0.0196157,
            0.0, -0.01662939, -0.01111137, 0.0, -0.003901779, -0.0196157, 0.0, 0.00390172, 0.0196157, 0.0,
            -0.003901779, -0.0196157, 0.0, 0.01111137, -0.01662933, 0.0, 0.0196157, -0.003901779, 0.0, ]
        ind = [0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23,
               24, 25, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47,
               48, 49, 50, 51, 52, 53, 54, 55, 56, 57, 58, 59, 60, 61, 62, 63, 64, 65, 66, 67, 68, 69, 70, 71,
               72, 73, 74, 75, 76, 77, 78, 79, 80, 81, 82, 83, 84, 85, 86, 87, 88, 89]
        buffer = [-0.003901779,
                  -0.0196157, 0.0, 0.402455, 0.009607315, 0.0, 0.0, 1.0, 0.0196157, -0.003901779, 0.0, 0.9903927,
                  0.4024553, 0.0, 0.0, 1.0, 0.00390172, 0.0196157, 0.0, 0.5975444, 0.9903928, 0.0, 0.0,
                  1.0, 0.00390172, 0.0196157, 0.0, 0.5975444, 0.9903928, 0.0, 0.0, 1.0, 0.0, 0.01999998,
                  0.0, 0.5, 1.0, 0.0, 0.0, 1.0, -0.003901779, 0.0196157, 0.0, 0.4024549, 0.9903926,
                  0.0, 0.0, 1.0, -0.003901779, 0.0196157, 0.0, 0.4024549, 0.9903926, 0.0, 0.0, 1.0,
                  -0.007653653, 0.01847755, 0.0, 0.3086583, 0.9619397, 0.0, 0.0, 1.0, 0.00390172, 0.0196157, 0.0,
                  0.5975444, 0.9903928, 0.0, 0.0, 1.0, -0.007653653, 0.01847755, 0.0, 0.3086583, 0.9619397, 0.0,
                  0.0, 1.0, -0.01111137, 0.01662939, 0.0, 0.2222148, 0.9157348, 0.0, 0.0, 1.0, 0.00390172,
                  0.0196157, 0.0, 0.5975444, 0.9903928, 0.0, 0.0, 1.0, -0.01111137, 0.01662939, 0.0, 0.2222148,
                  0.9157348, 0.0, 0.0, 1.0, -0.01414209, 0.01414209, 0.0, 0.1464466, 0.8535534, 0.0, 0.0,
                  1.0, -0.01662939, 0.01111137, 0.0, 0.08426517, 0.7777851, 0.0, 0.0, 1.0, -0.01662939, 0.01111137,
                  0.0, 0.08426517, 0.7777851, 0.0, 0.0, 1.0, -0.01847755, 0.007653653, 0.0, 0.03806024, 0.6913417,
                  0.0, 0.0, 1.0, -0.01111137, 0.01662939, 0.0, 0.2222148, 0.9157348, 0.0, 0.0, 1.0,
                  -0.01847755, 0.007653653, 0.0, 0.03806024, 0.6913417, 0.0, 0.0, 1.0, -0.0196157, 0.003901779, 0.0,
                  0.009607374, 0.5975452, 0.0, 0.0, 1.0, -0.01111137, 0.01662939, 0.0, 0.2222148, 0.9157348, 0.0,
                  0.0, 1.0, -0.0196157, 0.003901779, 0.0, 0.009607374, 0.5975452, 0.0, 0.0, 1.0, -0.01999998,
                  0.0, 0.0, 0.0, 0.5000001, 0.0, 0.0, 1.0, -0.0196157, -0.003901779, 0.0, 0.009607315,
                  0.4024549, 0.0, 0.0, 1.0, -0.0196157, -0.003901779, 0.0, 0.009607315, 0.4024549, 0.0, 0.0,
                  1.0, -0.01847755, -0.007653653, 0.0, 0.03806018, 0.3086584, 0.0, 0.0, 1.0, -0.0196157, 0.003901779,
                  0.0, 0.009607374, 0.5975452, 0.0, 0.0, 1.0, -0.01847755, -0.007653653, 0.0, 0.03806018, 0.3086584,
                  0.0, 0.0, 1.0, -0.01662939, -0.01111137, 0.0, 0.08426517, 0.2222149, 0.0, 0.0, 1.0,
                  -0.0196157, 0.003901779, 0.0, 0.009607374, 0.5975452, 0.0, 0.0, 1.0, -0.01662939, -0.01111137, 0.0,
                  0.08426517, 0.2222149, 0.0, 0.0, 1.0, -0.01414209, -0.01414209, 0.0, 0.1464466, 0.1464466, 0.0,
                  0.0, 1.0, -0.01111137, -0.01662939, 0.0, 0.2222149, 0.08426517, 0.0, 0.0, 1.0, -0.01111137,
                  -0.01662939, 0.0, 0.2222149, 0.08426517, 0.0, 0.0, 1.0, -0.007653653, -0.01847755, 0.0, 0.3086584,
                  0.03806018, 0.0, 0.0, 1.0, -0.003901779, -0.0196157, 0.0, 0.402455, 0.009607315, 0.0, 0.0,
                  1.0, -0.003901779, -0.0196157, 0.0, 0.402455, 0.009607315, 0.0, 0.0, 1.0, 0.0, -0.01999998,
                  0.0, 0.5000002, 0.0, 0.0, 0.0, 1.0, 0.003901779, -0.0196157, 0.0, 0.5975454, 0.009607374,
                  0.0, 0.0, 1.0, 0.003901779, -0.0196157, 0.0, 0.5975454, 0.009607374, 0.0, 0.0, 1.0,
                  0.007653653, -0.01847755, 0.0, 0.6913419, 0.03806036, 0.0, 0.0, 1.0, -0.003901779, -0.0196157, 0.0,
                  0.402455, 0.009607315, 0.0, 0.0, 1.0, 0.007653653, -0.01847755, 0.0, 0.6913419, 0.03806036, 0.0,
                  0.0, 1.0, 0.01111137, -0.01662933, 0.0, 0.7777854, 0.08426535, 0.0, 0.0, 1.0, -0.003901779,
                  -0.0196157, 0.0, 0.402455, 0.009607315, 0.0, 0.0, 1.0, 0.01111137, -0.01662933, 0.0, 0.7777854,
                  0.08426535, 0.0, 0.0, 1.0, 0.01414209, -0.01414209, 0.0, 0.8535537, 0.1464468, 0.0, 0.0,
                  1.0, 0.01662939, -0.01111137, 0.0, 0.915735, 0.2222151, 0.0, 0.0, 1.0, 0.01662939, -0.01111137,
                  0.0, 0.915735, 0.2222151, 0.0, 0.0, 1.0, 0.01847755, -0.007653653, 0.0, 0.9619399, 0.3086587,
                  0.0, 0.0, 1.0, 0.0196157, -0.003901779, 0.0, 0.9903927, 0.4024553, 0.0, 0.0, 1.0,
                  0.0196157, -0.003901779, 0.0, 0.9903927, 0.4024553, 0.0, 0.0, 1.0, 0.01999998, 0.0, 0.0,
                  1.0, 0.5000005, 0.0, 0.0, 1.0, 0.01961565, 0.003901779, 0.0, 0.9903925, 0.5975457, 0.0,
                  0.0, 1.0, 0.01961565, 0.003901779, 0.0, 0.9903925, 0.5975457, 0.0, 0.0, 1.0, 0.01847755,
                  0.007653653, 0.0, 0.9619396, 0.6913422, 0.0, 0.0, 1.0, 0.01662933, 0.01111137, 0.0, 0.9157345,
                  0.7777857, 0.0, 0.0, 1.0, 0.01662933, 0.01111137, 0.0, 0.9157345, 0.7777857, 0.0, 0.0,
                  1.0, 0.01414209, 0.01414215, 0.0, 0.8535529, 0.8535538, 0.0, 0.0, 1.0, 0.00390172, 0.0196157,
                  0.0, 0.5975444, 0.9903928, 0.0, 0.0, 1.0, 0.01414209, 0.01414215, 0.0, 0.8535529, 0.8535538,
                  0.0, 0.0, 1.0, 0.01111137, 0.01662939, 0.0, 0.7777846, 0.9157352, 0.0, 0.0, 1.0,
                  0.00390172, 0.0196157, 0.0, 0.5975444, 0.9903928, 0.0, 0.0, 1.0, 0.01111137, 0.01662939, 0.0,
                  0.7777846, 0.9157352, 0.0, 0.0, 1.0, 0.007653594, 0.01847755, 0.0, 0.6913411, 0.9619401, 0.0,
                  0.0, 1.0, 0.00390172, 0.0196157, 0.0, 0.5975444, 0.9903928, 0.0, 0.0, 1.0, -0.01662939,
                  -0.01111137, 0.0, 0.08426517, 0.2222149, 0.0, 0.0, 1.0, -0.01111137, -0.01662939, 0.0, 0.2222149,
                  0.08426517, 0.0, 0.0, 1.0, -0.003901779, -0.0196157, 0.0, 0.402455, 0.009607315, 0.0, 0.0,
                  1.0, 0.01111137, -0.01662933, 0.0, 0.7777854, 0.08426535, 0.0, 0.0, 1.0, 0.01662939, -0.01111137,
                  0.0, 0.915735, 0.2222151, 0.0, 0.0, 1.0, 0.0196157, -0.003901779, 0.0, 0.9903927, 0.4024553,
                  0.0, 0.0, 1.0, 0.0196157, -0.003901779, 0.0, 0.9903927, 0.4024553, 0.0, 0.0, 1.0,
                  0.01961565, 0.003901779, 0.0, 0.9903925, 0.5975457, 0.0, 0.0, 1.0, 0.00390172, 0.0196157, 0.0,
                  0.5975444, 0.9903928, 0.0, 0.0, 1.0, 0.01961565, 0.003901779, 0.0, 0.9903925, 0.5975457, 0.0,
                  0.0, 1.0, 0.01662933, 0.01111137, 0.0, 0.9157345, 0.7777857, 0.0, 0.0, 1.0, 0.00390172,
                  0.0196157, 0.0, 0.5975444, 0.9903928, 0.0, 0.0, 1.0, 0.00390172, 0.0196157, 0.0, 0.5975444,
                  0.9903928, 0.0, 0.0, 1.0, -0.01111137, 0.01662939, 0.0, 0.2222148, 0.9157348, 0.0, 0.0,
                  1.0, -0.0196157, 0.003901779, 0.0, 0.009607374, 0.5975452, 0.0, 0.0, 1.0, -0.0196157, 0.003901779,
                  0.0, 0.009607374, 0.5975452, 0.0, 0.0, 1.0, -0.01662939, -0.01111137, 0.0, 0.08426517, 0.2222149,
                  0.0, 0.0, 1.0, 0.00390172, 0.0196157, 0.0, 0.5975444, 0.9903928, 0.0, 0.0, 1.0,
                  -0.01662939, -0.01111137, 0.0, 0.08426517, 0.2222149, 0.0, 0.0, 1.0, -0.003901779, -0.0196157, 0.0,
                  0.402455, 0.009607315, 0.0, 0.0, 1.0, 0.00390172, 0.0196157, 0.0, 0.5975444, 0.9903928, 0.0,
                  0.0, 1.0, -0.003901779, -0.0196157, 0.0, 0.402455, 0.009607315, 0.0, 0.0, 1.0, 0.01111137,
                  -0.01662933, 0.0, 0.7777854, 0.08426535, 0.0, 0.0, 1.0, 0.0196157, -0.003901779, 0.0, 0.9903927,
                  0.4024553, 0.0, 0.0, 1.0]
        return vert, ind, buffer
