from figgy.utils.utils import Utils
from Crypto.Cipher import AES
from Crypto.Hash import SHA256, HMAC
from Crypto.Protocol.KDF import PBKDF2
from Crypto.Random.random import getrandbits
from Crypto.Util import Counter

"""
Largely taken from simple-crypt, but removing pycrypto requirement, instead using pycryptodome.
Slight tuning to account for faster encrypt/decrypt speeds. Remember, Figgy _only_ deals with temporary
access keys. Nothing in the vault should be valid longer than 12 hours from the time it was originally added
to the vault.
"""


class DecryptionException(Exception): pass


class EncryptionException(Exception): pass


class FiggyVault:
    EXPANSION_COUNT = (100, 100, 100)
    AES_KEY_LEN = 256
    SALT_LEN = (128, 256, 256)
    HASH = SHA256
    PREFIX = b'sc'
    HEADER = (PREFIX + b'\x00\x00', PREFIX + b'\x00\x01', PREFIX + b'\x00\x02')
    LENGTH = 2  # index into SALT_LEN, EXPANSION_COUNT, FiggyVault.HEADER
    HEADER_LEN = 4
    HALF_BLOCK = AES.block_size * 8 // 2

    def __init__(self, password: str):
        self.__password = password
        # lengths here are in bits, but pcrypto uses block size in bytes

        for salt_len in FiggyVault.SALT_LEN:
            assert FiggyVault.HALF_BLOCK <= salt_len  # we use a subset of the salt as nonce

        for header in FiggyVault.HEADER:
            assert len(header) == FiggyVault.HEADER_LEN

    def encrypt(self, data):
        '''
        Encrypt some data.  Input can be bytes or a string (which will be encoded
        using UTF-8).

        @param password: The secret value used as the basis for a key.
        This should be as long as varied as possible.  Try to avoid common words.

        @param data: The data to be encrypted.

        @return: The encrypted data, as bytes.
        '''
        data = self._str_to_bytes(data)
        self._assert_encrypt_length(data)
        salt = bytes(self._random_bytes(FiggyVault.SALT_LEN[FiggyVault.LENGTH] // 8))
        hmac_key, cipher_key = self._expand_keys(self.__password, salt, FiggyVault.EXPANSION_COUNT[FiggyVault.LENGTH])
        counter = Counter.new(FiggyVault.HALF_BLOCK, prefix=salt[:FiggyVault.HALF_BLOCK // 8])
        cipher = AES.new(cipher_key, AES.MODE_CTR, counter=counter)
        encrypted = cipher.encrypt(data)
        hmac = self._hmac(hmac_key, FiggyVault.HEADER[FiggyVault.LENGTH] + salt + encrypted)
        return FiggyVault.HEADER[FiggyVault.LENGTH] + salt + encrypted + hmac

    def decrypt(self, data):
        '''
        Decrypt some data.  Input must be bytes.

        @param password: The secret value used as the basis for a key.
        This should be as long as varied as possible.  Try to avoid common words.

        @param data: The data to be decrypted, typically as bytes.

        @return: The decrypted data, as bytes.  If the original message was a
        string you can re-create that using `result.decode('utf8')`.
        '''
        self._assert_not_unicode(data)
        self._assert_header_prefix(data)
        version = self._assert_header_version(data)
        self._assert_decrypt_length(data, version)
        raw = data[FiggyVault.HEADER_LEN:]
        salt = raw[:FiggyVault.SALT_LEN[version] // 8]
        hmac_key, cipher_key = self._expand_keys(self.__password, salt, FiggyVault.EXPANSION_COUNT[version])
        hmac = raw[-FiggyVault.HASH.digest_size:]
        hmac2 = self._hmac(hmac_key, data[:-FiggyVault.HASH.digest_size])
        self._assert_hmac(hmac_key, hmac, hmac2)
        counter = Counter.new(FiggyVault.HALF_BLOCK, prefix=salt[:FiggyVault.HALF_BLOCK // 8])
        cipher = AES.new(cipher_key, AES.MODE_CTR, counter=counter)

        return cipher.decrypt(raw[FiggyVault.SALT_LEN[version] // 8:-FiggyVault.HASH.digest_size])

    @staticmethod
    def _assert_not_unicode(data):
        # warn confused users
        u_type = type(b''.decode('utf8'))
        if isinstance(data, u_type):
            raise DecryptionException('Data to decrypt must be bytes; ' +
                                      'you cannot use a string because no string encoding will accept all possible characters.')

    @staticmethod
    def _assert_encrypt_length(data):
        # for AES this is never going to fail
        if len(data) > 2 ** FiggyVault.HALF_BLOCK:
            raise EncryptionException('Message too long.')

    @staticmethod
    def _assert_decrypt_length(data, version):
        if len(data) < FiggyVault.HEADER_LEN + FiggyVault.SALT_LEN[version] // 8 + FiggyVault.HASH.digest_size:
            raise DecryptionException('Missing data.')

    @staticmethod
    def _assert_header_prefix(data):
        if len(data) >= 2 and data[:2] != FiggyVault.PREFIX:
            raise DecryptionException('Data passed to decrypt were not generated by simple-crypt (bad header).')

    @staticmethod
    def _assert_header_version(data):
        if len(data) >= FiggyVault.HEADER_LEN:
            try:
                return FiggyVault.HEADER.index(data[:FiggyVault.HEADER_LEN])
            except:
                raise DecryptionException(
                    'The data appear to be encrypted with a more recent version of simple-crypt (bad header). ' +
                    'Please update the library and try again.')
        else:
            raise DecryptionException('Missing header.')

    @staticmethod
    def _assert_hmac(key, hmac, hmac2):
        # https://www.isecpartners.com/news-events/news/2011/february/double-hmac-verification.aspx
        if FiggyVault._hmac(key, hmac) != FiggyVault._hmac(key, hmac2):
            raise DecryptionException('Bad password or corrupt / modified data.')

    @staticmethod
    def _pbkdf2(password, salt, n_bytes, count):
        # the form of the prf below is taken from the code for PBKDF2
        return PBKDF2(password, salt, dkLen=n_bytes,
                      count=count, prf=lambda p, s: HMAC.new(p, s, FiggyVault.HASH).digest())

    @staticmethod
    def _expand_keys(password, salt, expansion_count):
        if not salt: raise ValueError('Missing salt.')
        if not password: raise ValueError('Missing password.')
        key_len = FiggyVault.AES_KEY_LEN // 8
        keys = FiggyVault._pbkdf2(FiggyVault._str_to_bytes(password), salt, 2 * key_len, expansion_count)
        return keys[:key_len], keys[key_len:]

    @staticmethod
    def _hide(ranbytes):
        # appelbaum recommends obscuring output from random number generators since it can reveal state.
        # we can do this explicitly with a hash, but this is what a PBKDF does anyway, so use one.
        # we don't care about the salt or work factor because there is a large space of values anyway.
        return bytearray(FiggyVault._pbkdf2(bytes(ranbytes), b'', len(ranbytes), 1))

    @staticmethod
    def _random_bytes(n):
        return FiggyVault._hide(bytearray(getrandbits(8) for _ in range(n)))

    @staticmethod
    def _hmac(key, data):
        return HMAC.new(key, data, FiggyVault.HASH).digest()

    @staticmethod
    def _str_to_bytes(data):
        u_type = type(b''.decode('utf8'))
        if isinstance(data, u_type):
            return data.encode('utf8')
        return data
