# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['pygasus',
 'pygasus.field',
 'pygasus.model',
 'pygasus.model.collections',
 'pygasus.model.decorators',
 'pygasus.storage',
 'pygasus.storage.sql']

package_data = \
{'': ['*']}

install_requires = \
['SQLAlchemy>=1.4.22,<2.0.0',
 'email-validator>=1.1.3,<2.0.0',
 'pydantic>=1.8.2,<2.0.0',
 'typing-compat>=0.1.0,<0.2.0']

setup_kwargs = {
    'name': 'pygasus',
    'version': '0.9.3',
    'description': 'A lightweight ORM built on top of Pydantic',
    'long_description': '# pygasus\n\nA lightweight Sqlite ORM built on top of Pydantic.\n\n## Installation\n\nThe easiest way to install Pygasus is to use `pip`, preferably inside a virtual environment:\n\n    pip install pygasus\n\n`pygasus` supports Python 3.7, 3.8, 3.9 and 3.10.\n\n## Getting started\n\nPygasus allows you to create your models using type annotations.  This model will then be converted to something that can be stored (in a database):\n\n```python\nfrom pygasus import Model\n\nclass User(Model):\n\n    """An user."""\n\n    id: int\n    name: str\n    age: int\n    height: float\n```\n\nYou then need to add this model to Pygasus:\n\n```python\nfrom pygasus.storage import SQLStorageEngine\n\n# The default storage engine is SQLAlchemy, connected to Sqlite3.\nstorage = SQLStorageEngine("test.sql")\n# ... you can also just create the database in memory for testing:\nstorage = SQLStorageEngine(memory=True)\n\n# Then you need to add your models.\nstorage.bind_models({User})\n\n# You can call `bind_models` without any argument, but in this case,\n# make sure to import your `User` class (and your other models)\n# before you do that.\n\n# Create a new user.\nuser = User.repository.create(name="Vincent", age=33, height=5.7)\n# Notice that we don\'t specify the `id`.  This field will be set by\n# the storage engine Pygasus uses.\n\n# At this point, our new user has been created, stored and returned.\nprint(user.id) # 1\nprint(user.name) # Vincent\nuser.age = 21 # Well, why not?\n# This will call an immediate update in the storage.\n\n# Of course we can also query a model:\nqueried = User.repository.get(id=1)\n\n# Due to caching, the queried user will be identical (in terms of reference).\nprint(user is queried) # True\n\n# You can control the cache to remove this behavior.\n```\n\nThis should show you some basic concepts.  To learn more, head over to the documentation.\n\n## Contributing\n\nIf you wish to develop Pygasus, all you need is:\n\n- Docker: getting a working Docker engine isn\'t that complicated on Linux, Windows or Mac OS.\n- Make: getting Make is more complicated, but it\'s not exactly required.  If you\'re on Windows, don\'t despair.  You might rely on Bash itself, it\'s available (and easy to access if you have WSL installed).\n\nPygasus relies on Docker images to make sure it remains usable on the supported Python versions: the idea is that each version will run in a diffferent Docker container, built on a different image.  This process is automated, and you don\'t really need to understand it to use it, but it might be worth diving into the scripts (see the `scripts` directory).\n\n### Initializes the images\n\nOnce you have cloned the `pygasus` repository, you must create the images for the required Python versions.  This is not exactly mandatory, as other scripts will make sure the images exist (and will build them if necessary), but this might make things easier, particularly in case of errors.\n\nPygasus supports four Python versions: 3.7, 3.8, 3.9 and 3.10.  You can initialize them all at once, though it will take time:\n\n    make init\n\nGet yourself a cup of coffee and stare at the process: each image is downloaded and built in turn.  Additional packages are installed.  Pygasus will install additional dependencies, then install Poetry and let it handle other Python dependencies, so that each image uses the same versions in dependencies.\n\nIf you wish to build images individually:\n\n    make init version=3.7\n\nThis will build the image only for Python 3.7 (called pygasus-3.7).  Running each build one at a time makes it easier to spot errors.\n\nOn the other hand, only panics if a build actually fails: it\'s not that uncommon to see warnings and even errors during the build process, but most of the time, this can be ignored... unless the build itself fails (you will be warned if that happens, don\'t worry).\n\n> Curious? Try to look at the images in Docker: `docker images` .  You should see the four `pygasus-VERSION` images along with the `python-VERSION-slim` build (this is the image on which Pygasus images are built).\n\nDon\'t have Make?  If you have Bash, you can still run the command:\n\n    bash scripts/init.sh\n\nInitializing isn\'t a "fire and forget" command.  Should Pygasus dependencies change or critical security bugs be fixed in Python images, it is recommended to run `make init` once more.\n\n### Testing\n\nNow that you have your Pygasus images, you can run the test suite in all versions at once:\n\n    make test\n\nThis will run the test suite in every Python version and display the result on the screen.  This might take a little while, but you should see the progress.  Obviously, if a test fails in a supported Python version, this is a big deal and something to fix for the next commit.  It is recommended to run `make test` before committing anything.\n\nDon\'t have Make?  No worries:\n\n    bash scripts/test.sh\n\nYou can also run the test suite in a specific version, though this is less useful:\n\n    make test version=3.9\n\n### Linter\n\nPygasus uses black and flake8 as linters.  If you have written some code, make sure it\'s properly formatted before committing:\n\n    make check-style\n\nThis won\'t do any modification in your code, this will merely display the errors (if any).  Fix them before committing!\n\nIt\'s also possible to format your code using black:\n\n    make format\n\nThis will actually edit your files in a proper format.  However, notice that flake8 isn\'t involved.  Sometimes, it sees potential errors black will ignore, so make sure `make check-style` is completely empty before committing.\n\nYou don\'t have Make?  No worries:\n\n    bash scripts/check_style.sh\n    bash scripts/format.sh\n\n### Debugging\n\nWhen coding, it can help to actually see what Pygasus does when you execute a line of code.  It is possible to start a shell inside of the container where Pygasus is installed.  You can then access to `pdb` to debug or run some arbitrary code:\n\n    make shell\n\nIf no version is specified, run in the first version (that is, 3.7).  It is assumed that it might be harder to break compatibility that way.  You can specify a version though:\n\n    make shell version=3.10\n\nOnce connected to the container, you can execute commands (including `poetry`, `python` and such).\n\nDon\'t have Make?  No worries:\n\n    bash scripts/shell.sh\n\n### Cleaning up\n\nThe started containers using the previous commands will not be maintained: once you exit them, they will go away.  However, images are kept.  Although this usually doesn\'t pose many problems, these images are heavy.  You can remove images without tags (usually, Pygasus images have got only one tag).\n\n    make clean\n\nThis shouldn\'t be necessary.  Keep an eye on the output of the `docker images` command to see if Docker keeps images it shouldn\'t.\n\n### Updating images\n\nIt bears reminding, once built, Pygasus images aren\'t changed.  If a major security bug is fixed and a new Python image is pushed to Docker hub, you won\'t have it.  If Pygasus dependencies change, you might run into errors.\n\nAt any point, you can force-update the Pygasus images:\n\n    make init\n\nThis will pull the Python image from Docker and rebuild the Pygasus image even if you already have one.\n',
    'author': 'Vincent Le Goff',
    'author_email': 'vincent.legoff.srs@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/talismud/pygasus',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7',
}


setup(**setup_kwargs)
