import * as cloudmap from '@aws-cdk/aws-servicediscovery';
import { CfnVirtualNode } from './appmesh.generated';
import { Construct } from '@aws-cdk/core';
/**
 * Properties for VirtualNode Service Discovery.
 *
 * @stability stable
 */
export interface ServiceDiscoveryConfig {
    /**
     * DNS based Service Discovery.
     *
     * @default - no DNS based service discovery
     * @stability stable
     */
    readonly dns?: CfnVirtualNode.DnsServiceDiscoveryProperty;
    /**
     * Cloud Map based Service Discovery.
     *
     * @default - no Cloud Map based service discovery
     * @stability stable
     */
    readonly cloudmap?: CfnVirtualNode.AwsCloudMapServiceDiscoveryProperty;
}
/**
 * Enum of DNS service discovery response type.
 *
 * @stability stable
 */
export declare enum DnsResponseType {
    /**
     * DNS resolver returns a loadbalanced set of endpoints and the traffic would be sent to the given endpoints.
     *
     * It would not drain existing connections to other endpoints that are not part of this list.
     *
     * @stability stable
     */
    LOAD_BALANCER = "LOADBALANCER",
    /**
     * DNS resolver is returning all the endpoints.
     *
     * This also means that if an endpoint is missing, it would drain the current connections to the missing endpoint.
     *
     * @stability stable
     */
    ENDPOINTS = "ENDPOINTS"
}
/**
 * Provides the Service Discovery method a VirtualNode uses.
 *
 * @stability stable
 */
export declare abstract class ServiceDiscovery {
    /**
     * Returns DNS based service discovery.
     *
     * @param responseType Specifies the DNS response type for the virtual node.
     * @stability stable
     */
    static dns(hostname: string, responseType?: DnsResponseType): ServiceDiscovery;
    /**
     * Returns Cloud Map based service discovery.
     *
     * @param service The AWS Cloud Map Service to use for service discovery.
     * @param instanceAttributes A string map that contains attributes with values that you can use to filter instances by any custom attribute that you specified when you registered the instance.
     * @stability stable
     */
    static cloudMap(service: cloudmap.IService, instanceAttributes?: {
        [key: string]: string;
    }): ServiceDiscovery;
    /**
     * Binds the current object when adding Service Discovery to a VirtualNode.
     *
     * @stability stable
     */
    abstract bind(scope: Construct): ServiceDiscoveryConfig;
}
