"use strict";
const assert_1 = require("@aws-cdk/assert");
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const eks = require("../lib");
module.exports = {
    'can be added to a cluster'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const cluster = new eks.Cluster(stack, 'MyCluster');
        // WHEN
        cluster.addFargateProfile('MyProfile', {
            selectors: [{ namespace: 'default' }],
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Custom::AWSCDK-EKS-FargateProfile', {
            Config: {
                clusterName: { Ref: 'MyCluster8AD82BF8' },
                podExecutionRoleArn: { 'Fn::GetAtt': ['MyClusterfargateprofileMyProfilePodExecutionRole4795C054', 'Arn'] },
                selectors: [{ namespace: 'default' }],
            },
        }));
        test.done();
    },
    'supports specifying a profile name'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const cluster = new eks.Cluster(stack, 'MyCluster');
        // WHEN
        cluster.addFargateProfile('MyProfile', {
            fargateProfileName: 'MyProfileName',
            selectors: [{ namespace: 'default' }],
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Custom::AWSCDK-EKS-FargateProfile', {
            Config: {
                clusterName: { Ref: 'MyCluster8AD82BF8' },
                podExecutionRoleArn: { 'Fn::GetAtt': ['MyClusterfargateprofileMyProfilePodExecutionRole4795C054', 'Arn'] },
                selectors: [{ namespace: 'default' }],
                fargateProfileName: 'MyProfileName',
            },
        }));
        test.done();
    },
    'supports custom execution role'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const cluster = new eks.Cluster(stack, 'MyCluster');
        const myRole = new iam.Role(stack, 'MyRole', { assumedBy: new iam.AnyPrincipal() });
        // WHEN
        cluster.addFargateProfile('MyProfile', {
            podExecutionRole: myRole,
            selectors: [{ namespace: 'default' }],
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Custom::AWSCDK-EKS-FargateProfile', {
            Config: {
                clusterName: { Ref: 'MyCluster8AD82BF8' },
                podExecutionRoleArn: { 'Fn::GetAtt': ['MyRoleF48FFE04', 'Arn'] },
                selectors: [{ namespace: 'default' }],
            },
        }));
        test.done();
    },
    'supports tags through aspects'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const cluster = new eks.Cluster(stack, 'MyCluster');
        // WHEN
        cluster.addFargateProfile('MyProfile', {
            selectors: [{ namespace: 'default' }],
        });
        core_1.Tag.add(stack, 'aspectTag', 'hello');
        core_1.Tag.add(cluster, 'propTag', '123');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Custom::AWSCDK-EKS-FargateProfile', {
            Config: {
                selectors: [{ namespace: 'default' }],
                clusterName: { Ref: 'MyCluster8AD82BF8' },
                podExecutionRoleArn: { 'Fn::GetAtt': ['MyClusterfargateprofileMyProfilePodExecutionRole4795C054', 'Arn'] },
                tags: {
                    propTag: '123',
                    aspectTag: 'hello',
                },
            },
        }));
        test.done();
    },
    'supports specifying vpc'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const cluster = new eks.Cluster(stack, 'MyCluster');
        const vpc = ec2.Vpc.fromVpcAttributes(stack, 'MyVpc', {
            vpcId: 'vpc123',
            availabilityZones: ['az1'],
            privateSubnetIds: ['priv1'],
        });
        // WHEN
        cluster.addFargateProfile('MyProfile', {
            selectors: [{ namespace: 'default' }],
            vpc,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Custom::AWSCDK-EKS-FargateProfile', {
            Config: {
                clusterName: { Ref: 'MyCluster8AD82BF8' },
                podExecutionRoleArn: { 'Fn::GetAtt': ['MyClusterfargateprofileMyProfilePodExecutionRole4795C054', 'Arn'] },
                selectors: [{ namespace: 'default' }],
                subnets: ['priv1'],
            },
        }));
        test.done();
    },
    'fails if there are no selectors or if there are more than 5'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const cluster = new eks.Cluster(stack, 'MyCluster');
        // THEN
        test.throws(() => cluster.addFargateProfile('MyProfile', { selectors: [] }));
        test.throws(() => cluster.addFargateProfile('MyProfile', {
            selectors: [
                { namespace: '1' },
                { namespace: '2' },
                { namespace: '3' },
                { namespace: '4' },
                { namespace: '5' },
                { namespace: '6' },
            ],
        }));
        test.done();
    },
    'FargateCluster creates an EKS cluster fully managed by Fargate'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new eks.FargateCluster(stack, 'FargateCluster');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Custom::AWSCDK-EKS-KubernetesPatch', {
            ResourceName: 'deployment/coredns',
            ResourceNamespace: 'kube-system',
            ApplyPatchJson: '{"spec":{"template":{"metadata":{"annotations":{"eks.amazonaws.com/compute-type":"fargate"}}}}}',
            RestorePatchJson: '{"spec":{"template":{"metadata":{"annotations":{"eks.amazonaws.com/compute-type":"ec2"}}}}}',
            ClusterName: {
                Ref: 'FargateCluster019F03E8',
            },
        }));
        assert_1.expect(stack).to(assert_1.haveResource('Custom::AWSCDK-EKS-FargateProfile', {
            Config: {
                clusterName: {
                    Ref: 'FargateCluster019F03E8',
                },
                podExecutionRoleArn: {
                    'Fn::GetAtt': [
                        'FargateClusterfargateprofiledefaultPodExecutionRole66F2610E',
                        'Arn',
                    ],
                },
                selectors: [
                    { namespace: 'default' },
                    { namespace: 'kube-system' },
                ],
            },
        }));
        test.done();
    },
    'can create FargateCluster with a custom profile'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new eks.FargateCluster(stack, 'FargateCluster', {
            defaultProfile: {
                fargateProfileName: 'my-app', selectors: [{ namespace: 'foo' }, { namespace: 'bar' }],
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Custom::AWSCDK-EKS-FargateProfile', {
            Config: {
                clusterName: {
                    Ref: 'FargateCluster019F03E8',
                },
                fargateProfileName: 'my-app',
                podExecutionRoleArn: {
                    'Fn::GetAtt': [
                        'FargateClusterfargateprofilemyappPodExecutionRole875B4635',
                        'Arn',
                    ],
                },
                selectors: [
                    { namespace: 'foo' },
                    { namespace: 'bar' },
                ],
            },
        }));
        test.done();
    },
    'custom profile name is "custom" if no custom profile name is provided'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new eks.FargateCluster(stack, 'FargateCluster', {
            defaultProfile: {
                selectors: [{ namespace: 'foo' }, { namespace: 'bar' }],
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Custom::AWSCDK-EKS-FargateProfile', {
            Config: {
                clusterName: {
                    Ref: 'FargateCluster019F03E8',
                },
                podExecutionRoleArn: {
                    'Fn::GetAtt': [
                        'FargateClusterfargateprofilecustomPodExecutionRoleDB415F19',
                        'Arn',
                    ],
                },
                selectors: [
                    { namespace: 'foo' },
                    { namespace: 'bar' },
                ],
            },
        }));
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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