"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.UserPoolDomain = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("@aws-cdk/core");
const custom_resources_1 = require("@aws-cdk/custom-resources");
const cognito_generated_1 = require("./cognito.generated");
/**
 * Define a user pool domain
 */
class UserPoolDomain extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_cognito_UserPoolDomainProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, UserPoolDomain);
            }
            throw error;
        }
        if (!!props.customDomain === !!props.cognitoDomain) {
            throw new Error('One of, and only one of, cognitoDomain or customDomain must be specified');
        }
        if (props.cognitoDomain?.domainPrefix &&
            !core_1.Token.isUnresolved(props.cognitoDomain?.domainPrefix) &&
            !/^[a-z0-9-]+$/.test(props.cognitoDomain.domainPrefix)) {
            throw new Error('domainPrefix for cognitoDomain can contain only lowercase alphabets, numbers and hyphens');
        }
        this.isCognitoDomain = !!props.cognitoDomain;
        const domainName = props.cognitoDomain?.domainPrefix || props.customDomain?.domainName;
        const resource = new cognito_generated_1.CfnUserPoolDomain(this, 'Resource', {
            userPoolId: props.userPool.userPoolId,
            domain: domainName,
            customDomainConfig: props.customDomain ? { certificateArn: props.customDomain.certificate.certificateArn } : undefined,
        });
        this.domainName = resource.ref;
    }
    /**
     * Import a UserPoolDomain given its domain name
     */
    static fromDomainName(scope, id, userPoolDomainName) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.domainName = userPoolDomainName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * The domain name of the CloudFront distribution associated with the user pool domain.
     */
    get cloudFrontDomainName() {
        if (!this.cloudFrontCustomResource) {
            const sdkCall = {
                service: 'CognitoIdentityServiceProvider',
                action: 'describeUserPoolDomain',
                parameters: {
                    Domain: this.domainName,
                },
                physicalResourceId: custom_resources_1.PhysicalResourceId.of(this.domainName),
            };
            this.cloudFrontCustomResource = new custom_resources_1.AwsCustomResource(this, 'CloudFrontDomainName', {
                resourceType: 'Custom::UserPoolCloudFrontDomainName',
                onCreate: sdkCall,
                onUpdate: sdkCall,
                policy: custom_resources_1.AwsCustomResourcePolicy.fromSdkCalls({
                    // DescribeUserPoolDomain only supports access level '*'
                    // https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoncognitouserpools.html#amazoncognitouserpools-actions-as-permissions
                    resources: ['*'],
                }),
            });
        }
        return this.cloudFrontCustomResource.getResponseField('DomainDescription.CloudFrontDistribution');
    }
    /**
     * The URL to the hosted UI associated with this domain
     *
     * @param options options to customize baseUrl
     */
    baseUrl(options) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_cognito_BaseUrlOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.baseUrl);
            }
            throw error;
        }
        if (this.isCognitoDomain) {
            const authDomain = 'auth' + (options?.fips ? '-fips' : '');
            return `https://${this.domainName}.${authDomain}.${core_1.Stack.of(this).region}.amazoncognito.com`;
        }
        return `https://${this.domainName}`;
    }
    /**
     * The URL to the sign in page in this domain using a specific UserPoolClient
     * @param client [disable-awslint:ref-via-interface] the user pool client that the UI will use to interact with the UserPool
     * @param options options to customize signInUrl.
     */
    signInUrl(client, options) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_cognito_UserPoolClient(client);
            jsiiDeprecationWarnings._aws_cdk_aws_cognito_SignInUrlOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.signInUrl);
            }
            throw error;
        }
        let responseType;
        if (client.oAuthFlows.authorizationCodeGrant) {
            responseType = 'code';
        }
        else if (client.oAuthFlows.implicitCodeGrant) {
            responseType = 'token';
        }
        else {
            throw new Error('signInUrl is not supported for clients without authorizationCodeGrant or implicitCodeGrant flow enabled');
        }
        const path = options.signInPath ?? '/login';
        return `${this.baseUrl(options)}${path}?client_id=${client.userPoolClientId}&response_type=${responseType}&redirect_uri=${options.redirectUri}`;
    }
}
exports.UserPoolDomain = UserPoolDomain;
_a = JSII_RTTI_SYMBOL_1;
UserPoolDomain[_a] = { fqn: "@aws-cdk/aws-cognito.UserPoolDomain", version: "1.198.0" };
//# sourceMappingURL=data:application/json;base64,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