"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
exports.AUTOGEN_MARKER = '$$autogen$$';
/**
 * Renders an ARN for an SSM parameter given a parameter name.
 * @param scope definition scope
 * @param parameterName the parameter name to include in the ARN
 * @param physicalName optional physical name specified by the user (to auto-detect separator)
 */
function arnForParameterName(scope, parameterName, options = {}) {
    const physicalName = options.physicalName;
    const nameToValidate = physicalName || parameterName;
    if (!core_1.Token.isUnresolved(nameToValidate) && nameToValidate.includes('/') && !nameToValidate.startsWith('/')) {
        throw new Error(`Parameter names must be fully qualified (if they include "/" they must also begin with a "/"): ${nameToValidate}`);
    }
    return core_1.Stack.of(scope).formatArn({
        service: 'ssm',
        resource: 'parameter',
        sep: isSimpleName() ? '/' : '',
        resourceName: parameterName,
    });
    /**
     * Determines the ARN separator for this parameter: if we have a concrete
     * parameter name (or explicitly defined physical name), we will parse them
     * and decide whether a "/" is needed or not. Otherwise, users will have to
     * explicitly specify `simpleName` when they import the ARN.
     */
    function isSimpleName() {
        // look for a concrete name as a hint for determining the separator
        const concreteName = !core_1.Token.isUnresolved(parameterName) ? parameterName : physicalName;
        if (!concreteName || core_1.Token.isUnresolved(concreteName)) {
            if (options.simpleName === undefined) {
                throw new Error('Unable to determine ARN separator for SSM parameter since the parameter name is an unresolved token. Use "fromAttributes" and specify "simpleName" explicitly');
            }
            return options.simpleName;
        }
        const result = !concreteName.startsWith('/');
        // if users explicitly specify the separator and it conflicts with the one we need, it's an error.
        if (options.simpleName !== undefined && options.simpleName !== result) {
            if (concreteName === exports.AUTOGEN_MARKER) {
                throw new Error('If "parameterName" is not explicitly defined, "simpleName" must be "true" or undefined since auto-generated parameter names always have simple names');
            }
            throw new Error(`Parameter name "${concreteName}" is ${result ? 'a simple name' : 'not a simple name'}, but "simpleName" was explicitly set to ${options.simpleName}. Either omit it or set it to ${result}`);
        }
        return result;
    }
}
exports.arnForParameterName = arnForParameterName;
//# sourceMappingURL=data:application/json;base64,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