"use strict";
const assert_1 = require("@aws-cdk/assert");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
function getTestStack() {
    return new core_1.Stack(undefined, 'TestStack', { env: { account: '123456789012', region: 'us-east-1' } });
}
function toCfnTags(tags) {
    return Object.keys(tags).map(key => {
        return { Key: key, Value: tags[key] };
    });
}
function hasTags(expectedTags) {
    return (props) => {
        try {
            const tags = props.Tags;
            const actualTags = tags.filter((tag) => {
                for (const expectedTag of expectedTags) {
                    if (assert_1.isSuperObject(expectedTag, tag)) {
                        return true;
                    }
                    else {
                        continue;
                    }
                }
                // no values in array so expecting empty
                return false;
            });
            return actualTags.length === expectedTags.length;
        }
        catch (e) {
            // tslint:disable:no-console
            console.error('Tags are incorrect');
            console.error('found tags ', props.Tags);
            console.error('expected tags ', expectedTags);
            // tslint:enable:no-console
            throw e;
        }
    };
}
module.exports = {
    "When creating a VPC": {
        "with the default CIDR range": {
            "vpc.vpcId returns a token to the VPC ID"(test) {
                const stack = getTestStack();
                const vpc = new lib_1.Vpc(stack, 'TheVPC');
                test.deepEqual(stack.resolve(vpc.vpcId), { Ref: 'TheVPC92636AB0' });
                test.done();
            },
            "it uses the correct network range"(test) {
                const stack = getTestStack();
                new lib_1.Vpc(stack, 'TheVPC');
                assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPC', {
                    CidrBlock: lib_1.Vpc.DEFAULT_CIDR_RANGE,
                    EnableDnsHostnames: true,
                    EnableDnsSupport: true,
                    InstanceTenancy: lib_1.DefaultInstanceTenancy.DEFAULT,
                }));
                test.done();
            },
            'the Name tag is defaulted to path'(test) {
                const stack = getTestStack();
                new lib_1.Vpc(stack, 'TheVPC');
                assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPC', hasTags([{ Key: 'Name', Value: 'TheVPC' }])));
                assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::InternetGateway', hasTags([{ Key: 'Name', Value: 'TheVPC' }])));
                test.done();
            },
        },
        "with all of the properties set, it successfully sets the correct VPC properties"(test) {
            const stack = getTestStack();
            new lib_1.Vpc(stack, 'TheVPC', {
                cidr: "192.168.0.0/16",
                enableDnsHostnames: false,
                enableDnsSupport: false,
                defaultInstanceTenancy: lib_1.DefaultInstanceTenancy.DEDICATED,
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPC', {
                CidrBlock: '192.168.0.0/16',
                EnableDnsHostnames: false,
                EnableDnsSupport: false,
                InstanceTenancy: lib_1.DefaultInstanceTenancy.DEDICATED,
            }));
            test.done();
        },
        "contains the correct number of subnets"(test) {
            const stack = getTestStack();
            const vpc = new lib_1.Vpc(stack, 'TheVPC');
            const zones = stack.availabilityZones.length;
            test.equal(vpc.publicSubnets.length, zones);
            test.equal(vpc.privateSubnets.length, zones);
            test.deepEqual(stack.resolve(vpc.vpcId), { Ref: 'TheVPC92636AB0' });
            test.done();
        },
        "with only isolated subnets, the VPC should not contain an IGW or NAT Gateways"(test) {
            const stack = getTestStack();
            new lib_1.Vpc(stack, 'TheVPC', {
                subnetConfiguration: [
                    {
                        subnetType: lib_1.SubnetType.ISOLATED,
                        name: 'Isolated',
                    }
                ]
            });
            assert_1.expect(stack).notTo(assert_1.haveResource("AWS::EC2::InternetGateway"));
            assert_1.expect(stack).notTo(assert_1.haveResource("AWS::EC2::NatGateway"));
            assert_1.expect(stack).to(assert_1.haveResource("AWS::EC2::Subnet", {
                MapPublicIpOnLaunch: false
            }));
            test.done();
        },
        "with no private subnets, the VPC should have an IGW but no NAT Gateways"(test) {
            const stack = getTestStack();
            new lib_1.Vpc(stack, 'TheVPC', {
                subnetConfiguration: [
                    {
                        subnetType: lib_1.SubnetType.PUBLIC,
                        name: 'Public',
                    },
                    {
                        subnetType: lib_1.SubnetType.ISOLATED,
                        name: 'Isolated',
                    }
                ]
            });
            assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::InternetGateway', 1));
            assert_1.expect(stack).notTo(assert_1.haveResource("AWS::EC2::NatGateway"));
            test.done();
        },
        "with private subnets and custom networkAcl."(test) {
            const stack = getTestStack();
            const vpc = new lib_1.Vpc(stack, 'TheVPC', {
                subnetConfiguration: [
                    {
                        subnetType: lib_1.SubnetType.PUBLIC,
                        name: 'Public',
                    },
                    {
                        subnetType: lib_1.SubnetType.PRIVATE,
                        name: 'private',
                    }
                ]
            });
            const nacl1 = new lib_1.NetworkAcl(stack, 'myNACL1', {
                vpc,
                subnetSelection: { subnetType: lib_1.SubnetType.PRIVATE },
            });
            new lib_1.NetworkAclEntry(stack, 'AllowDNSEgress', {
                networkAcl: nacl1,
                ruleNumber: 100,
                traffic: lib_1.AclTraffic.udpPort(53),
                direction: lib_1.TrafficDirection.EGRESS,
                cidr: lib_1.AclCidr.ipv4('10.0.0.0/16'),
            });
            new lib_1.NetworkAclEntry(stack, 'AllowDNSIngress', {
                networkAcl: nacl1,
                ruleNumber: 100,
                traffic: lib_1.AclTraffic.udpPort(53),
                direction: lib_1.TrafficDirection.INGRESS,
                cidr: lib_1.AclCidr.anyIpv4(),
            });
            assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::NetworkAcl', 1));
            assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::NetworkAclEntry', 2));
            assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::SubnetNetworkAclAssociation', 3));
            test.done();
        },
        "with no subnets defined, the VPC should have an IGW, and a NAT Gateway per AZ"(test) {
            const stack = getTestStack();
            const zones = stack.availabilityZones.length;
            new lib_1.Vpc(stack, 'TheVPC', {});
            assert_1.expect(stack).to(assert_1.countResources("AWS::EC2::InternetGateway", 1));
            assert_1.expect(stack).to(assert_1.countResources("AWS::EC2::NatGateway", zones));
            test.done();
        },
        "with subnets and reserved subnets defined, VPC subnet count should not contain reserved subnets "(test) {
            const stack = getTestStack();
            new lib_1.Vpc(stack, 'TheVPC', {
                cidr: '10.0.0.0/16',
                subnetConfiguration: [
                    {
                        cidrMask: 24,
                        subnetType: lib_1.SubnetType.PUBLIC,
                        name: 'Public',
                    },
                    {
                        cidrMask: 24,
                        name: 'reserved',
                        subnetType: lib_1.SubnetType.PRIVATE,
                        reserved: true,
                    },
                    {
                        cidrMask: 28,
                        name: 'rds',
                        subnetType: lib_1.SubnetType.ISOLATED,
                    }
                ],
                maxAzs: 3
            });
            assert_1.expect(stack).to(assert_1.countResources("AWS::EC2::Subnet", 6));
            test.done();
        },
        "with reserved subnets, any other subnets should not have cidrBlock from within reserved space"(test) {
            const stack = getTestStack();
            new lib_1.Vpc(stack, 'TheVPC', {
                cidr: '10.0.0.0/16',
                subnetConfiguration: [
                    {
                        cidrMask: 24,
                        name: 'ingress',
                        subnetType: lib_1.SubnetType.PUBLIC,
                    },
                    {
                        cidrMask: 24,
                        name: 'reserved',
                        subnetType: lib_1.SubnetType.PRIVATE,
                        reserved: true,
                    },
                    {
                        cidrMask: 24,
                        name: 'rds',
                        subnetType: lib_1.SubnetType.PRIVATE,
                    }
                ],
                maxAzs: 3
            });
            for (let i = 0; i < 3; i++) {
                assert_1.expect(stack).to(assert_1.haveResource("AWS::EC2::Subnet", {
                    CidrBlock: `10.0.${i}.0/24`
                }));
            }
            for (let i = 3; i < 6; i++) {
                assert_1.expect(stack).notTo(assert_1.haveResource("AWS::EC2::Subnet", {
                    CidrBlock: `10.0.${i}.0/24`
                }));
            }
            for (let i = 6; i < 9; i++) {
                assert_1.expect(stack).to(assert_1.haveResource("AWS::EC2::Subnet", {
                    CidrBlock: `10.0.${i}.0/24`
                }));
            }
            test.done();
        },
        "with custom subnets, the VPC should have the right number of subnets, an IGW, and a NAT Gateway per AZ"(test) {
            const stack = getTestStack();
            const zones = stack.availabilityZones.length;
            new lib_1.Vpc(stack, 'TheVPC', {
                cidr: '10.0.0.0/21',
                subnetConfiguration: [
                    {
                        cidrMask: 24,
                        name: 'ingress',
                        subnetType: lib_1.SubnetType.PUBLIC,
                    },
                    {
                        cidrMask: 24,
                        name: 'application',
                        subnetType: lib_1.SubnetType.PRIVATE,
                    },
                    {
                        cidrMask: 28,
                        name: 'rds',
                        subnetType: lib_1.SubnetType.ISOLATED,
                    }
                ],
                maxAzs: 3
            });
            assert_1.expect(stack).to(assert_1.countResources("AWS::EC2::InternetGateway", 1));
            assert_1.expect(stack).to(assert_1.countResources("AWS::EC2::NatGateway", zones));
            assert_1.expect(stack).to(assert_1.countResources("AWS::EC2::Subnet", 9));
            for (let i = 0; i < 6; i++) {
                assert_1.expect(stack).to(assert_1.haveResource("AWS::EC2::Subnet", {
                    CidrBlock: `10.0.${i}.0/24`
                }));
            }
            for (let i = 0; i < 3; i++) {
                assert_1.expect(stack).to(assert_1.haveResource("AWS::EC2::Subnet", {
                    CidrBlock: `10.0.6.${i * 16}/28`
                }));
            }
            test.done();
        },
        "with custom subents and natGateways = 2 there should be only two NATGW"(test) {
            const stack = getTestStack();
            new lib_1.Vpc(stack, 'TheVPC', {
                cidr: '10.0.0.0/21',
                natGateways: 2,
                subnetConfiguration: [
                    {
                        cidrMask: 24,
                        name: 'ingress',
                        subnetType: lib_1.SubnetType.PUBLIC,
                    },
                    {
                        cidrMask: 24,
                        name: 'application',
                        subnetType: lib_1.SubnetType.PRIVATE,
                    },
                    {
                        cidrMask: 28,
                        name: 'rds',
                        subnetType: lib_1.SubnetType.ISOLATED,
                    }
                ],
                maxAzs: 3
            });
            assert_1.expect(stack).to(assert_1.countResources("AWS::EC2::InternetGateway", 1));
            assert_1.expect(stack).to(assert_1.countResources("AWS::EC2::NatGateway", 2));
            assert_1.expect(stack).to(assert_1.countResources("AWS::EC2::Subnet", 9));
            for (let i = 0; i < 6; i++) {
                assert_1.expect(stack).to(assert_1.haveResource("AWS::EC2::Subnet", {
                    CidrBlock: `10.0.${i}.0/24`
                }));
            }
            for (let i = 0; i < 3; i++) {
                assert_1.expect(stack).to(assert_1.haveResource("AWS::EC2::Subnet", {
                    CidrBlock: `10.0.6.${i * 16}/28`
                }));
            }
            test.done();
        },
        "with enableDnsHostnames enabled but enableDnsSupport disabled, should throw an Error"(test) {
            const stack = getTestStack();
            test.throws(() => new lib_1.Vpc(stack, 'TheVPC', {
                enableDnsHostnames: true,
                enableDnsSupport: false
            }));
            test.done();
        },
        "with public subnets MapPublicIpOnLaunch is true"(test) {
            const stack = getTestStack();
            new lib_1.Vpc(stack, 'VPC', {
                maxAzs: 1,
                subnetConfiguration: [
                    {
                        cidrMask: 24,
                        name: 'ingress',
                        subnetType: lib_1.SubnetType.PUBLIC,
                    }
                ],
            });
            assert_1.expect(stack).to(assert_1.countResources("AWS::EC2::Subnet", 1));
            assert_1.expect(stack).notTo(assert_1.haveResource("AWS::EC2::NatGateway"));
            assert_1.expect(stack).to(assert_1.haveResource("AWS::EC2::Subnet", {
                MapPublicIpOnLaunch: true
            }));
            test.done();
        },
        "maxAZs defaults to 3 if unset"(test) {
            const stack = getTestStack();
            new lib_1.Vpc(stack, 'VPC');
            assert_1.expect(stack).to(assert_1.countResources("AWS::EC2::Subnet", 6));
            assert_1.expect(stack).to(assert_1.countResources("AWS::EC2::Route", 6));
            for (let i = 0; i < 6; i++) {
                assert_1.expect(stack).to(assert_1.haveResource("AWS::EC2::Subnet", {
                    CidrBlock: `10.0.${i * 32}.0/19`
                }));
            }
            assert_1.expect(stack).to(assert_1.haveResourceLike("AWS::EC2::Route", {
                DestinationCidrBlock: '0.0.0.0/0',
                NatGatewayId: {},
            }));
            test.done();
        },
        "with maxAZs set to 2"(test) {
            const stack = getTestStack();
            new lib_1.Vpc(stack, 'VPC', { maxAzs: 2 });
            assert_1.expect(stack).to(assert_1.countResources("AWS::EC2::Subnet", 4));
            assert_1.expect(stack).to(assert_1.countResources("AWS::EC2::Route", 4));
            for (let i = 0; i < 4; i++) {
                assert_1.expect(stack).to(assert_1.haveResource("AWS::EC2::Subnet", {
                    CidrBlock: `10.0.${i * 64}.0/18`
                }));
            }
            assert_1.expect(stack).to(assert_1.haveResourceLike("AWS::EC2::Route", {
                DestinationCidrBlock: '0.0.0.0/0',
                NatGatewayId: {},
            }));
            test.done();
        },
        "with natGateway set to 1"(test) {
            const stack = getTestStack();
            new lib_1.Vpc(stack, 'VPC', {
                natGateways: 1,
            });
            assert_1.expect(stack).to(assert_1.countResources("AWS::EC2::Subnet", 6));
            assert_1.expect(stack).to(assert_1.countResources("AWS::EC2::Route", 6));
            assert_1.expect(stack).to(assert_1.countResources("AWS::EC2::NatGateway", 1));
            assert_1.expect(stack).to(assert_1.haveResourceLike("AWS::EC2::Route", {
                DestinationCidrBlock: '0.0.0.0/0',
                NatGatewayId: {},
            }));
            test.done();
        },
        'with natGateway subnets defined'(test) {
            const stack = getTestStack();
            new lib_1.Vpc(stack, 'VPC', {
                subnetConfiguration: [
                    {
                        cidrMask: 24,
                        name: 'ingress',
                        subnetType: lib_1.SubnetType.PUBLIC,
                    },
                    {
                        cidrMask: 24,
                        name: 'egress',
                        subnetType: lib_1.SubnetType.PUBLIC,
                    },
                    {
                        cidrMask: 24,
                        name: 'private',
                        subnetType: lib_1.SubnetType.PRIVATE,
                    },
                ],
                natGatewaySubnets: {
                    subnetGroupName: 'egress'
                },
            });
            assert_1.expect(stack).to(assert_1.countResources("AWS::EC2::NatGateway", 3));
            for (let i = 1; i < 4; i++) {
                assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::Subnet', hasTags([{
                        Key: 'Name',
                        Value: `VPC/egressSubnet${i}`,
                    }, {
                        Key: 'aws-cdk:subnet-name',
                        Value: 'egress',
                    }])));
            }
            test.done();
        },
        'natGateways = 0 requires there to be no PRIVATE subnets'(test) {
            const stack = getTestStack();
            test.throws(() => {
                new lib_1.Vpc(stack, 'VPC', {
                    natGateways: 0,
                });
            }, /make sure you don't configure any PRIVATE subnets/);
            test.done();
        },
        'with mis-matched nat and subnet configs it throws'(test) {
            const stack = getTestStack();
            test.throws(() => new lib_1.Vpc(stack, 'VPC', {
                subnetConfiguration: [
                    {
                        cidrMask: 24,
                        name: 'ingress',
                        subnetType: lib_1.SubnetType.PUBLIC,
                    },
                    {
                        cidrMask: 24,
                        name: 'private',
                        subnetType: lib_1.SubnetType.PRIVATE,
                    },
                ],
                natGatewaySubnets: {
                    subnetGroupName: 'notthere',
                },
            }));
            test.done();
        },
        'with a vpn gateway'(test) {
            const stack = getTestStack();
            new lib_1.Vpc(stack, 'VPC', {
                vpnGateway: true,
                vpnGatewayAsn: 65000
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPNGateway', {
                AmazonSideAsn: 65000,
                Type: 'ipsec.1'
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPCGatewayAttachment', {
                VpcId: {
                    Ref: 'VPCB9E5F0B4'
                },
                VpnGatewayId: {
                    Ref: 'VPCVpnGatewayB5ABAE68'
                }
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPNGatewayRoutePropagation', {
                RouteTableIds: [
                    {
                        Ref: 'VPCPrivateSubnet1RouteTableBE8A6027'
                    },
                    {
                        Ref: 'VPCPrivateSubnet2RouteTable0A19E10E'
                    },
                    {
                        Ref: 'VPCPrivateSubnet3RouteTable192186F8'
                    }
                ],
                VpnGatewayId: {
                    Ref: 'VPCVpnGatewayB5ABAE68'
                }
            }));
            test.done();
        },
        'with a vpn gateway and route propagation on isolated subnets'(test) {
            const stack = getTestStack();
            new lib_1.Vpc(stack, 'VPC', {
                subnetConfiguration: [
                    { subnetType: lib_1.SubnetType.PUBLIC, name: 'Public' },
                    { subnetType: lib_1.SubnetType.ISOLATED, name: 'Isolated' },
                ],
                vpnGateway: true,
                vpnRoutePropagation: [
                    {
                        subnetType: lib_1.SubnetType.ISOLATED
                    }
                ]
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPNGatewayRoutePropagation', {
                RouteTableIds: [
                    {
                        Ref: 'VPCIsolatedSubnet1RouteTableEB156210'
                    },
                    {
                        Ref: 'VPCIsolatedSubnet2RouteTable9B4F78DC'
                    },
                    {
                        Ref: 'VPCIsolatedSubnet3RouteTableCB6A1FDA'
                    }
                ],
                VpnGatewayId: {
                    Ref: 'VPCVpnGatewayB5ABAE68'
                }
            }));
            test.done();
        },
        'with a vpn gateway and route propagation on private and isolated subnets'(test) {
            const stack = getTestStack();
            new lib_1.Vpc(stack, 'VPC', {
                subnetConfiguration: [
                    { subnetType: lib_1.SubnetType.PUBLIC, name: 'Public' },
                    { subnetType: lib_1.SubnetType.PRIVATE, name: 'Private' },
                    { subnetType: lib_1.SubnetType.ISOLATED, name: 'Isolated' },
                ],
                vpnGateway: true,
                vpnRoutePropagation: [
                    {
                        subnetType: lib_1.SubnetType.PRIVATE
                    },
                    {
                        subnetType: lib_1.SubnetType.ISOLATED
                    }
                ]
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPNGatewayRoutePropagation', {
                RouteTableIds: [
                    {
                        Ref: 'VPCPrivateSubnet1RouteTableBE8A6027'
                    },
                    {
                        Ref: 'VPCPrivateSubnet2RouteTable0A19E10E'
                    },
                    {
                        Ref: 'VPCPrivateSubnet3RouteTable192186F8'
                    },
                    {
                        Ref: 'VPCIsolatedSubnet1RouteTableEB156210'
                    },
                    {
                        Ref: 'VPCIsolatedSubnet2RouteTable9B4F78DC'
                    },
                    {
                        Ref: 'VPCIsolatedSubnet3RouteTableCB6A1FDA'
                    }
                ],
                VpnGatewayId: {
                    Ref: 'VPCVpnGatewayB5ABAE68'
                }
            }));
            test.done();
        },
        'route propagation defaults to isolated subnets when there are no private subnets'(test) {
            const stack = getTestStack();
            new lib_1.Vpc(stack, 'VPC', {
                subnetConfiguration: [
                    { subnetType: lib_1.SubnetType.PUBLIC, name: 'Public' },
                    { subnetType: lib_1.SubnetType.ISOLATED, name: 'Isolated' },
                ],
                vpnGateway: true,
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPNGatewayRoutePropagation', {
                RouteTableIds: [
                    {
                        Ref: 'VPCIsolatedSubnet1RouteTableEB156210'
                    },
                    {
                        Ref: 'VPCIsolatedSubnet2RouteTable9B4F78DC'
                    },
                    {
                        Ref: 'VPCIsolatedSubnet3RouteTableCB6A1FDA'
                    }
                ],
                VpnGatewayId: {
                    Ref: 'VPCVpnGatewayB5ABAE68'
                }
            }));
            test.done();
        },
        'route propagation defaults to public subnets when there are no private/isolated subnets'(test) {
            const stack = getTestStack();
            new lib_1.Vpc(stack, 'VPC', {
                subnetConfiguration: [
                    { subnetType: lib_1.SubnetType.PUBLIC, name: 'Public' },
                ],
                vpnGateway: true,
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPNGatewayRoutePropagation', {
                RouteTableIds: [
                    {
                        Ref: 'VPCPublicSubnet1RouteTableFEE4B781'
                    },
                    {
                        Ref: 'VPCPublicSubnet2RouteTable6F1A15F1'
                    },
                    {
                        Ref: 'VPCPublicSubnet3RouteTable98AE0E14'
                    }
                ],
                VpnGatewayId: {
                    Ref: 'VPCVpnGatewayB5ABAE68'
                }
            }));
            test.done();
        },
        'fails when specifying vpnConnections with vpnGateway set to false'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            test.throws(() => new lib_1.Vpc(stack, 'VpcNetwork', {
                vpnGateway: false,
                vpnConnections: {
                    VpnConnection: {
                        asn: 65000,
                        ip: '192.0.2.1'
                    }
                }
            }), /`vpnConnections`.+`vpnGateway`.+false/);
            test.done();
        },
        'fails when specifying vpnGatewayAsn with vpnGateway set to false'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            test.throws(() => new lib_1.Vpc(stack, 'VpcNetwork', {
                vpnGateway: false,
                vpnGatewayAsn: 65000,
            }), /`vpnGatewayAsn`.+`vpnGateway`.+false/);
            test.done();
        },
        'Subnets have a defaultChild'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const vpc = new lib_1.Vpc(stack, 'VpcNetwork');
            test.ok(vpc.publicSubnets[0].node.defaultChild instanceof lib_1.CfnSubnet);
            test.done();
        },
        'CIDR cannot be a Token'(test) {
            const stack = new core_1.Stack();
            test.throws(() => {
                new lib_1.Vpc(stack, 'Vpc', {
                    cidr: core_1.Lazy.stringValue({ produce: () => 'abc' })
                });
            }, /property must be a concrete CIDR string/);
            test.done();
        },
        'Default NAT gateway provider'(test) {
            const stack = new core_1.Stack();
            const natGatewayProvider = lib_1.NatProvider.gateway();
            new lib_1.Vpc(stack, 'VpcNetwork', { natGatewayProvider });
            test.ok(natGatewayProvider.configuredGateways.length > 0);
            test.done();
        },
        'Can add an IPv6 route'(test) {
            // GIVEN
            const stack = getTestStack();
            // WHEN
            const vpc = new lib_1.Vpc(stack, 'VPC');
            vpc.publicSubnets[0].addRoute('SomeRoute', {
                destinationIpv6CidrBlock: '2001:4860:4860::8888/32',
                routerId: 'router-1',
                routerType: lib_1.RouterType.NETWORK_INTERFACE
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike("AWS::EC2::Route", {
                DestinationIpv6CidrBlock: '2001:4860:4860::8888/32',
                NetworkInterfaceId: 'router-1'
            }));
            test.done();
        },
        'Can add an IPv4 route'(test) {
            // GIVEN
            const stack = getTestStack();
            // WHEN
            const vpc = new lib_1.Vpc(stack, 'VPC');
            vpc.publicSubnets[0].addRoute('SomeRoute', {
                destinationCidrBlock: '0.0.0.0/0',
                routerId: 'router-1',
                routerType: lib_1.RouterType.NETWORK_INTERFACE
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike("AWS::EC2::Route", {
                DestinationCidrBlock: '0.0.0.0/0',
                NetworkInterfaceId: 'router-1'
            }));
            test.done();
        },
    },
    'NAT instances': {
        'Can configure NAT instances instead of NAT gateways'(test) {
            // GIVEN
            const stack = getTestStack();
            // WHEN
            const natGatewayProvider = lib_1.NatProvider.instance({
                instanceType: new lib_1.InstanceType('q86.mega'),
                machineImage: new lib_1.GenericLinuxImage({
                    'us-east-1': 'ami-1'
                })
            });
            new lib_1.Vpc(stack, 'TheVPC', { natGatewayProvider });
            // THEN
            assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::Instance', 3));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::Instance', {
                ImageId: "ami-1",
                InstanceType: "q86.mega",
                SourceDestCheck: false,
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::Route', {
                RouteTableId: { Ref: "TheVPCPrivateSubnet1RouteTableF6513BC2" },
                DestinationCidrBlock: "0.0.0.0/0",
                InstanceId: { Ref: "TheVPCPublicSubnet1NatInstanceCC514192" }
            }));
            test.done();
        },
        'natGateways controls amount of NAT instances'(test) {
            // GIVEN
            const stack = getTestStack();
            // WHEN
            new lib_1.Vpc(stack, 'TheVPC', {
                natGatewayProvider: lib_1.NatProvider.instance({
                    instanceType: new lib_1.InstanceType('q86.mega'),
                    machineImage: new lib_1.GenericLinuxImage({
                        'us-east-1': 'ami-1'
                    })
                }),
                natGateways: 1
            });
            // THEN
            assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::Instance', 1));
            test.done();
        },
    },
    'Network ACL association': {
        'by default uses default ACL reference'(test) {
            // GIVEN
            const stack = getTestStack();
            // WHEN
            const vpc = new lib_1.Vpc(stack, 'TheVPC', { cidr: '192.168.0.0/16' });
            new core_1.CfnOutput(stack, 'Output', {
                value: vpc.publicSubnets[0].subnetNetworkAclAssociationId
            });
            assert_1.expect(stack).toMatch({
                Outputs: {
                    Output: {
                        Value: { "Fn::GetAtt": ["TheVPCPublicSubnet1Subnet770D4FF2", "NetworkAclAssociationId"] }
                    }
                }
            }, assert_1.MatchStyle.SUPERSET);
            test.done();
        },
        'if ACL is replaced new ACL reference is returned'(test) {
            // GIVEN
            const stack = getTestStack();
            const vpc = new lib_1.Vpc(stack, 'TheVPC', { cidr: '192.168.0.0/16' });
            // WHEN
            new core_1.CfnOutput(stack, 'Output', {
                value: vpc.publicSubnets[0].subnetNetworkAclAssociationId
            });
            new lib_1.NetworkAcl(stack, 'ACL', {
                vpc,
                subnetSelection: { subnetType: lib_1.SubnetType.PUBLIC }
            });
            assert_1.expect(stack).toMatch({
                Outputs: {
                    Output: {
                        Value: { Ref: "ACLDBD1BB49" }
                    }
                }
            }, assert_1.MatchStyle.SUPERSET);
            test.done();
        },
    },
    "When creating a VPC with a custom CIDR range": {
        "vpc.vpcCidrBlock is the correct network range"(test) {
            const stack = getTestStack();
            new lib_1.Vpc(stack, 'TheVPC', { cidr: '192.168.0.0/16' });
            assert_1.expect(stack).to(assert_1.haveResource("AWS::EC2::VPC", {
                CidrBlock: '192.168.0.0/16'
            }));
            test.done();
        }
    },
    'When tagging': {
        'VPC propagated tags will be on subnet, IGW, routetables, NATGW'(test) {
            const stack = getTestStack();
            const tags = {
                VpcType: 'Good',
            };
            const noPropTags = {
                BusinessUnit: 'Marketing',
            };
            const allTags = { ...tags, ...noPropTags };
            const vpc = new lib_1.Vpc(stack, 'TheVPC');
            // overwrite to set propagate
            vpc.node.applyAspect(new core_1.Tag('BusinessUnit', 'Marketing', { includeResourceTypes: [lib_1.CfnVPC.CFN_RESOURCE_TYPE_NAME] }));
            vpc.node.applyAspect(new core_1.Tag('VpcType', 'Good'));
            assert_1.expect(stack).to(assert_1.haveResource("AWS::EC2::VPC", hasTags(toCfnTags(allTags))));
            const taggables = ['Subnet', 'InternetGateway', 'NatGateway', 'RouteTable'];
            const propTags = toCfnTags(tags);
            const noProp = toCfnTags(noPropTags);
            for (const resource of taggables) {
                assert_1.expect(stack).to(assert_1.haveResource(`AWS::EC2::${resource}`, hasTags(propTags)));
                assert_1.expect(stack).notTo(assert_1.haveResource(`AWS::EC2::${resource}`, hasTags(noProp)));
            }
            test.done();
        },
        'Subnet Name will propagate to route tables and NATGW'(test) {
            const stack = getTestStack();
            const vpc = new lib_1.Vpc(stack, 'TheVPC');
            for (const subnet of vpc.publicSubnets) {
                const tag = { Key: 'Name', Value: subnet.node.path };
                assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::NatGateway', hasTags([tag])));
                assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::RouteTable', hasTags([tag])));
            }
            for (const subnet of vpc.privateSubnets) {
                const tag = { Key: 'Name', Value: subnet.node.path };
                assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::RouteTable', hasTags([tag])));
            }
            test.done();
        },
        'Tags can be added after the Vpc is created with `vpc.tags.setTag(...)`'(test) {
            const stack = getTestStack();
            const vpc = new lib_1.Vpc(stack, 'TheVPC');
            const tag = { Key: 'Late', Value: 'Adder' };
            assert_1.expect(stack).notTo(assert_1.haveResource('AWS::EC2::VPC', hasTags([tag])));
            vpc.node.applyAspect(new core_1.Tag(tag.Key, tag.Value));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPC', hasTags([tag])));
            test.done();
        },
    },
    'subnet selection': {
        'selecting default subnets returns the private ones'(test) {
            // GIVEN
            const stack = getTestStack();
            const vpc = new lib_1.Vpc(stack, 'VPC');
            // WHEN
            const { subnetIds } = vpc.selectSubnets();
            // THEN
            test.deepEqual(subnetIds, vpc.privateSubnets.map(s => s.subnetId));
            test.done();
        },
        'can select public subnets'(test) {
            // GIVEN
            const stack = getTestStack();
            const vpc = new lib_1.Vpc(stack, 'VPC');
            // WHEN
            const { subnetIds } = vpc.selectSubnets({ subnetType: lib_1.SubnetType.PUBLIC });
            // THEN
            test.deepEqual(subnetIds, vpc.publicSubnets.map(s => s.subnetId));
            test.done();
        },
        'can select isolated subnets'(test) {
            // GIVEN
            const stack = getTestStack();
            const vpc = new lib_1.Vpc(stack, 'VPC', {
                subnetConfiguration: [
                    { subnetType: lib_1.SubnetType.PUBLIC, name: 'Public' },
                    { subnetType: lib_1.SubnetType.ISOLATED, name: 'Isolated' },
                ]
            });
            // WHEN
            const { subnetIds } = vpc.selectSubnets({ subnetType: lib_1.SubnetType.ISOLATED });
            // THEN
            test.deepEqual(subnetIds, vpc.isolatedSubnets.map(s => s.subnetId));
            test.done();
        },
        'can select subnets by name'(test) {
            // GIVEN
            const stack = getTestStack();
            const vpc = new lib_1.Vpc(stack, 'VPC', {
                subnetConfiguration: [
                    { subnetType: lib_1.SubnetType.PUBLIC, name: 'BlaBla' },
                    { subnetType: lib_1.SubnetType.PRIVATE, name: 'DontTalkToMe' },
                    { subnetType: lib_1.SubnetType.ISOLATED, name: 'DontTalkAtAll' },
                ]
            });
            // WHEN
            const { subnetIds } = vpc.selectSubnets({ subnetGroupName: 'DontTalkToMe' });
            // THEN
            test.deepEqual(subnetIds, vpc.privateSubnets.map(s => s.subnetId));
            test.done();
        },
        'subnetName is an alias for subnetGroupName (backwards compat)'(test) {
            // GIVEN
            const stack = getTestStack();
            const vpc = new lib_1.Vpc(stack, 'VPC', {
                subnetConfiguration: [
                    { subnetType: lib_1.SubnetType.PUBLIC, name: 'BlaBla' },
                    { subnetType: lib_1.SubnetType.PRIVATE, name: 'DontTalkToMe' },
                    { subnetType: lib_1.SubnetType.ISOLATED, name: 'DontTalkAtAll' },
                ]
            });
            // WHEN
            const { subnetIds } = vpc.selectSubnets({ subnetName: 'DontTalkToMe' });
            // THEN
            test.deepEqual(subnetIds, vpc.privateSubnets.map(s => s.subnetId));
            test.done();
        },
        'selecting default subnets in a VPC with only isolated subnets returns the isolateds'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const vpc = lib_1.Vpc.fromVpcAttributes(stack, 'VPC', {
                vpcId: 'vpc-1234',
                availabilityZones: ['dummy1a', 'dummy1b', 'dummy1c'],
                isolatedSubnetIds: ['iso-1', 'iso-2', 'iso-3'],
                isolatedSubnetRouteTableIds: ['rt-1', 'rt-2', 'rt-3'],
            });
            // WHEN
            const subnets = vpc.selectSubnets();
            // THEN
            test.deepEqual(subnets.subnetIds, ['iso-1', 'iso-2', 'iso-3']);
            test.done();
        },
        'selecting default subnets in a VPC with only public subnets returns the publics'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const vpc = lib_1.Vpc.fromVpcAttributes(stack, 'VPC', {
                vpcId: 'vpc-1234',
                availabilityZones: ['dummy1a', 'dummy1b', 'dummy1c'],
                publicSubnetIds: ['pub-1', 'pub-2', 'pub-3'],
                publicSubnetRouteTableIds: ['rt-1', 'rt-2', 'rt-3'],
            });
            // WHEN
            const subnets = vpc.selectSubnets();
            // THEN
            test.deepEqual(subnets.subnetIds, ['pub-1', 'pub-2', 'pub-3']);
            test.done();
        },
        'selecting subnets by name fails if the name is unknown'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const vpc = new lib_1.Vpc(stack, 'VPC');
            test.throws(() => {
                vpc.selectSubnets({ subnetGroupName: 'Toot' });
            }, /There are no subnet groups with name 'Toot' in this VPC. Available names: Public,Private/);
            test.done();
        },
        'select subnets with az restriction'(test) {
            // GIVEN
            const stack = getTestStack();
            const vpc = new lib_1.Vpc(stack, 'VpcNetwork', {
                maxAzs: 1,
                subnetConfiguration: [
                    { name: 'lb', subnetType: lib_1.SubnetType.PUBLIC },
                    { name: 'app', subnetType: lib_1.SubnetType.PRIVATE },
                    { name: 'db', subnetType: lib_1.SubnetType.PRIVATE },
                ]
            });
            // WHEN
            const { subnetIds } = vpc.selectSubnets({ onePerAz: true });
            // THEN
            test.deepEqual(subnetIds.length, 1);
            test.deepEqual(subnetIds[0], vpc.privateSubnets[0].subnetId);
            test.done();
        },
        'select explicitly defined subnets'(test) {
            // GIVEN
            const stack = getTestStack();
            const vpc = lib_1.Vpc.fromVpcAttributes(stack, 'VPC', {
                vpcId: 'vpc-1234',
                availabilityZones: ['dummy1a', 'dummy1b', 'dummy1c'],
                publicSubnetIds: ['pub-1', 'pub-2', 'pub-3'],
                publicSubnetRouteTableIds: ['rt-1', 'rt-2', 'rt-3'],
            });
            const subnet = new lib_1.PrivateSubnet(stack, 'Subnet', {
                availabilityZone: vpc.availabilityZones[0],
                cidrBlock: '10.0.0.0/28',
                vpcId: vpc.vpcId
            });
            // WHEN
            const { subnetIds } = vpc.selectSubnets({ subnets: [subnet] });
            // THEN
            test.deepEqual(subnetIds.length, 1);
            test.deepEqual(subnetIds[0], subnet.subnetId);
            test.done();
        },
        'subnet created from subnetId'(test) {
            // GIVEN
            const stack = getTestStack();
            // WHEN
            const subnet = lib_1.Subnet.fromSubnetId(stack, 'subnet1', 'pub-1');
            // THEN
            test.deepEqual(subnet.subnetId, 'pub-1');
            test.done();
        },
        'Referencing AZ throws error when subnet created from subnetId'(test) {
            // GIVEN
            const stack = getTestStack();
            // WHEN
            const subnet = lib_1.Subnet.fromSubnetId(stack, 'subnet1', 'pub-1');
            // THEN
            // tslint:disable-next-line: max-line-length
            test.throws(() => subnet.availabilityZone, "You cannot reference a Subnet's availability zone if it was not supplied. Add the availabilityZone when importing using Subnet.fromSubnetAttributes()");
            test.done();
        },
        'Referencing AZ throws error when subnet created from attributes without az'(test) {
            // GIVEN
            const stack = getTestStack();
            // WHEN
            const subnet = lib_1.Subnet.fromSubnetAttributes(stack, 'subnet1', { subnetId: 'pub-1', availabilityZone: '' });
            // THEN
            test.deepEqual(subnet.subnetId, 'pub-1');
            // tslint:disable-next-line: max-line-length
            test.throws(() => subnet.availabilityZone, "You cannot reference a Subnet's availability zone if it was not supplied. Add the availabilityZone when importing using Subnet.fromSubnetAttributes()");
            test.done();
        },
        'AZ have value when subnet created from attributes with az'(test) {
            // GIVEN
            const stack = getTestStack();
            // WHEN
            const subnet = lib_1.Subnet.fromSubnetAttributes(stack, 'subnet1', { subnetId: 'pub-1', availabilityZone: 'az-1234' });
            // THEN
            test.deepEqual(subnet.subnetId, 'pub-1');
            test.deepEqual(subnet.availabilityZone, 'az-1234');
            test.done();
        }
    },
};
//# sourceMappingURL=data:application/json;base64,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