"use strict";
const assert_1 = require("@aws-cdk/assert");
const iam = require("@aws-cdk/aws-iam");
const logs = require("@aws-cdk/aws-logs");
const s3 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
function getTestStack() {
    return new core_1.Stack(undefined, 'TestStack', {
        env: { account: '123456789012', region: 'us-east-1' }
    });
}
module.exports = {
    'with defaults set, it successfully creates with cloudwatch logs destination'(test) {
        const stack = getTestStack();
        new lib_1.FlowLog(stack, 'FlowLogs', {
            resourceType: lib_1.FlowLogResourceType.fromNetworkInterfaceId('eni-123455')
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::FlowLog', {
            ResourceType: 'NetworkInterface',
            TrafficType: 'ALL',
            ResourceId: 'eni-123455',
            DeliverLogsPermissionArn: {
                'Fn::GetAtt': ['FlowLogsIAMRoleF18F4209', 'Arn']
            },
            LogGroupName: {
                Ref: 'FlowLogsLogGroup9853A85F'
            }
        }));
        assert_1.expect(stack).to(assert_1.countResources('AWS::Logs::LogGroup', 1));
        assert_1.expect(stack).to(assert_1.countResources('AWS::IAM::Role', 1));
        assert_1.expect(stack).notTo(assert_1.haveResource('AWS::S3::Bucket'));
        test.done();
    },
    'with cloudwatch logs as the destination, allows use of existing resources'(test) {
        const stack = getTestStack();
        new lib_1.FlowLog(stack, 'FlowLogs', {
            resourceType: lib_1.FlowLogResourceType.fromNetworkInterfaceId('eni-123456'),
            destination: lib_1.FlowLogDestination.toCloudWatchLogs(new logs.LogGroup(stack, 'TestLogGroup', {
                retention: logs.RetentionDays.FIVE_DAYS
            }), new iam.Role(stack, 'TestRole', {
                roleName: 'TestName',
                assumedBy: new iam.ServicePrincipal('vpc-flow-logs.amazonaws.com')
            }))
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Logs::LogGroup', {
            RetentionInDays: 5
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Role', {
            RoleName: 'TestName'
        }));
        assert_1.expect(stack).notTo(assert_1.haveResource('AWS::S3::Bucket'));
        test.done();
    },
    'with s3 as the destination, allows use of existing resources'(test) {
        const stack = getTestStack();
        new lib_1.FlowLog(stack, 'FlowLogs', {
            resourceType: lib_1.FlowLogResourceType.fromNetworkInterfaceId('eni-123456'),
            destination: lib_1.FlowLogDestination.toS3(new s3.Bucket(stack, 'TestBucket', {
                bucketName: 'testbucket'
            }))
        });
        assert_1.expect(stack).notTo(assert_1.haveResource('AWS::Logs::LogGroup'));
        assert_1.expect(stack).notTo(assert_1.haveResource('AWS::IAM::Role'));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
            BucketName: 'testbucket'
        }));
        test.done();
    },
    'with s3 as the destination and all the defaults set, it successfully creates all the resources'(test) {
        const stack = getTestStack();
        new lib_1.FlowLog(stack, 'FlowLogs', {
            resourceType: lib_1.FlowLogResourceType.fromNetworkInterfaceId('eni-123456'),
            destination: lib_1.FlowLogDestination.toS3()
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::FlowLog', {
            ResourceType: 'NetworkInterface',
            TrafficType: 'ALL',
            ResourceId: 'eni-123456',
            LogDestination: {
                'Fn::GetAtt': ['FlowLogsBucket87F67F60', 'Arn']
            }
        }));
        assert_1.expect(stack).notTo(assert_1.haveResource('AWS::Logs::LogGroup'));
        assert_1.expect(stack).notTo(assert_1.haveResource('AWS::IAM::Role'));
        assert_1.expect(stack).to(assert_1.countResources('AWS::S3::Bucket', 1));
        test.done();
    },
    'create with vpc'(test) {
        const stack = getTestStack();
        new lib_1.Vpc(stack, 'VPC', {
            flowLogs: {
                flowLogs: {}
            }
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPC'));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::FlowLog', {
            ResourceType: 'VPC',
            TrafficType: 'ALL',
            ResourceId: {
                Ref: 'VPCB9E5F0B4'
            },
            DeliverLogsPermissionArn: {
                'Fn::GetAtt': ['VPCflowLogsIAMRole9D21E1A6', 'Arn']
            },
            LogGroupName: {
                Ref: 'VPCflowLogsLogGroupE900F980'
            }
        }));
        test.done();
    },
    'add to vpc'(test) {
        const stack = getTestStack();
        const vpc = new lib_1.Vpc(stack, 'VPC');
        vpc.addFlowLog('FlowLogs');
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPC'));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::FlowLog', {
            ResourceType: 'VPC',
            TrafficType: 'ALL',
            ResourceId: {
                Ref: 'VPCB9E5F0B4'
            },
            DeliverLogsPermissionArn: {
                'Fn::GetAtt': ['VPCFlowLogsIAMRole55343234', 'Arn']
            },
            LogGroupName: {
                Ref: 'VPCFlowLogsLogGroupF48E1B0A'
            }
        }));
        test.done();
    }
};
//# sourceMappingURL=data:application/json;base64,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