"use strict";
const aws_s3_1 = require("@aws-cdk/aws-s3");
const lib_1 = require("../../core/lib");
const ec2 = require("../lib");
module.exports = {
    'can create Windows user data'(test) {
        // GIVEN
        // WHEN
        const userData = ec2.UserData.forWindows();
        userData.addCommands('command1', 'command2');
        // THEN
        const rendered = userData.render();
        test.equals(rendered, '<powershell>command1\ncommand2</powershell>');
        test.done();
    },
    'can create Windows user data with commands on exit'(test) {
        // GIVEN
        const userData = ec2.UserData.forWindows();
        // WHEN
        userData.addCommands('command1', 'command2');
        userData.addOnExitCommands('onexit1', 'onexit2');
        // THEN
        const rendered = userData.render();
        test.equals(rendered, '<powershell>trap {\n' +
            '$success=($PSItem.Exception.Message -eq "Success")\n' +
            'onexit1\n' +
            'onexit2\n' +
            'break\n' +
            '}\n' +
            'command1\n' +
            'command2\n' +
            'throw "Success"</powershell>');
        test.done();
    },
    'can create Windows with Signal Command'(test) {
        // GIVEN
        const stack = new lib_1.Stack();
        const resource = new ec2.Vpc(stack, 'RESOURCE');
        const userData = ec2.UserData.forWindows();
        // WHEN
        userData.addSignalOnExitCommand(resource);
        userData.addCommands("command1");
        // THEN
        const rendered = userData.render();
        test.equals(rendered, '<powershell>trap {\n' +
            '$success=($PSItem.Exception.Message -eq "Success")\n' +
            'cfn-signal --stack Stack --resource RESOURCE1989552F --region ${Token[AWS::Region.4]} --success ($success.ToString().ToLower())\n' +
            'break\n' +
            '}\n' +
            'command1\n' +
            'throw "Success"</powershell>');
        test.done();
    },
    'can windows userdata download S3 files'(test) {
        // GIVEN
        const stack = new lib_1.Stack();
        const userData = ec2.UserData.forWindows();
        const bucket = aws_s3_1.Bucket.fromBucketName(stack, "testBucket", "test");
        const bucket2 = aws_s3_1.Bucket.fromBucketName(stack, "testBucket2", "test2");
        // WHEN
        userData.addS3DownloadCommand({
            bucket,
            bucketKey: "filename.bat"
        });
        userData.addS3DownloadCommand({
            bucket: bucket2,
            bucketKey: "filename2.bat",
            localFile: "c:\\test\\location\\otherScript.bat"
        });
        // THEN
        const rendered = userData.render();
        test.equals(rendered, '<powershell>mkdir (Split-Path -Path \'C:/temp/filename.bat\' ) -ea 0\n' +
            'Read-S3Object -BucketName \'test\' -key \'filename.bat\' -file \'C:/temp/filename.bat\' -ErrorAction Stop\n' +
            'mkdir (Split-Path -Path \'c:\\test\\location\\otherScript.bat\' ) -ea 0\n' +
            'Read-S3Object -BucketName \'test2\' -key \'filename2.bat\' -file \'c:\\test\\location\\otherScript.bat\' -ErrorAction Stop</powershell>');
        test.done();
    },
    'can windows userdata execute files'(test) {
        // GIVEN
        const userData = ec2.UserData.forWindows();
        // WHEN
        userData.addExecuteFileCommand({
            filePath: "C:\\test\\filename.bat",
        });
        userData.addExecuteFileCommand({
            filePath: "C:\\test\\filename2.bat",
            arguments: "arg1 arg2 -arg $variable"
        });
        // THEN
        const rendered = userData.render();
        test.equals(rendered, '<powershell>&\'C:\\test\\filename.bat\' undefined\n' +
            'if (!$?) { Write-Error \'Failed to execute the file "C:\\test\\filename.bat"\' -ErrorAction Stop }\n' +
            '&\'C:\\test\\filename2.bat\' arg1 arg2 -arg $variable\n' +
            'if (!$?) { Write-Error \'Failed to execute the file "C:\\test\\filename2.bat"\' -ErrorAction Stop }</powershell>');
        test.done();
    },
    'can create Linux user data'(test) {
        // GIVEN
        // WHEN
        const userData = ec2.UserData.forLinux();
        userData.addCommands('command1', 'command2');
        // THEN
        const rendered = userData.render();
        test.equals(rendered, '#!/bin/bash\ncommand1\ncommand2');
        test.done();
    },
    'can create Linux user data with commands on exit'(test) {
        // GIVEN
        const userData = ec2.UserData.forLinux();
        // WHEN
        userData.addCommands('command1', 'command2');
        userData.addOnExitCommands('onexit1', 'onexit2');
        // THEN
        const rendered = userData.render();
        test.equals(rendered, '#!/bin/bash\n' +
            'function exitTrap(){\n' +
            'exitCode=$?\n' +
            'onexit1\n' +
            'onexit2\n' +
            '}\n' +
            'trap exitTrap EXIT\n' +
            'command1\n' +
            'command2');
        test.done();
    },
    'can create Linux with Signal Command'(test) {
        // GIVEN
        const stack = new lib_1.Stack();
        const resource = new ec2.Vpc(stack, 'RESOURCE');
        // WHEN
        const userData = ec2.UserData.forLinux();
        userData.addCommands("command1");
        userData.addSignalOnExitCommand(resource);
        // THEN
        const rendered = userData.render();
        test.equals(rendered, '#!/bin/bash\n' +
            'function exitTrap(){\n' +
            'exitCode=$?\n' +
            '/opt/aws/bin/cfn-signal --stack Stack --resource RESOURCE1989552F --region ${Token[AWS::Region.4]} -e $exitCode || echo \'Failed to send Cloudformation Signal\'\n' +
            '}\n' +
            'trap exitTrap EXIT\n' +
            'command1');
        test.done();
    },
    'can linux userdata download S3 files'(test) {
        // GIVEN
        const stack = new lib_1.Stack();
        const userData = ec2.UserData.forLinux();
        const bucket = aws_s3_1.Bucket.fromBucketName(stack, "testBucket", "test");
        const bucket2 = aws_s3_1.Bucket.fromBucketName(stack, "testBucket2", "test2");
        // WHEN
        userData.addS3DownloadCommand({
            bucket,
            bucketKey: "filename.sh"
        });
        userData.addS3DownloadCommand({
            bucket: bucket2,
            bucketKey: "filename2.sh",
            localFile: "c:\\test\\location\\otherScript.sh"
        });
        // THEN
        const rendered = userData.render();
        test.equals(rendered, '#!/bin/bash\n' +
            'mkdir -p $(dirname \'/tmp/filename.sh\')\n' +
            'aws s3 cp \'s3://test/filename.sh\' \'/tmp/filename.sh\'\n' +
            'mkdir -p $(dirname \'c:\\test\\location\\otherScript.sh\')\n' +
            'aws s3 cp \'s3://test2/filename2.sh\' \'c:\\test\\location\\otherScript.sh\'');
        test.done();
    },
    'can linux userdata execute files'(test) {
        // GIVEN
        const userData = ec2.UserData.forLinux();
        // WHEN
        userData.addExecuteFileCommand({
            filePath: "/tmp/filename.sh",
        });
        userData.addExecuteFileCommand({
            filePath: "/test/filename2.sh",
            arguments: "arg1 arg2 -arg $variable"
        });
        // THEN
        const rendered = userData.render();
        test.equals(rendered, '#!/bin/bash\n' +
            'set -e\n' +
            'chmod +x \'/tmp/filename.sh\'\n' +
            '\'/tmp/filename.sh\' undefined\n' +
            'set -e\n' +
            'chmod +x \'/test/filename2.sh\'\n' +
            '\'/test/filename2.sh\' arg1 arg2 -arg $variable');
        test.done();
    },
    'can create Custom user data'(test) {
        // GIVEN
        // WHEN
        const userData = ec2.UserData.custom('Some\nmultiline\ncontent');
        // THEN
        const rendered = userData.render();
        test.equals(rendered, 'Some\nmultiline\ncontent');
        test.done();
    },
    'Custom user data throws when adding on exit commands'(test) {
        // GIVEN
        // WHEN
        const userData = ec2.UserData.custom("");
        // THEN
        test.throws(() => userData.addOnExitCommands("a command goes here"));
        test.done();
    },
    'Custom user data throws when adding signal command'(test) {
        // GIVEN
        const stack = new lib_1.Stack();
        const resource = new ec2.Vpc(stack, 'RESOURCE');
        // WHEN
        const userData = ec2.UserData.custom("");
        // THEN
        test.throws(() => userData.addSignalOnExitCommand(resource));
        test.done();
    },
    'Custom user data throws when downloading file'(test) {
        // GIVEN
        const stack = new lib_1.Stack();
        const userData = ec2.UserData.custom("");
        const bucket = aws_s3_1.Bucket.fromBucketName(stack, "testBucket", "test");
        // WHEN
        // THEN
        test.throws(() => userData.addS3DownloadCommand({
            bucket,
            bucketKey: "filename.sh"
        }));
        test.done();
    },
    'Custom user data throws when executing file'(test) {
        // GIVEN
        const userData = ec2.UserData.custom("");
        // WHEN
        // THEN
        test.throws(() => userData.addExecuteFileCommand({
            filePath: "/tmp/filename.sh",
        }));
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidGVzdC51c2VyZGF0YS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInRlc3QudXNlcmRhdGEudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IjtBQUFBLDRDQUF5QztBQUV6Qyx3Q0FBdUM7QUFDdkMsOEJBQThCO0FBRTlCLGlCQUFTO0lBQ1AsOEJBQThCLENBQUMsSUFBVTtRQUN2QyxRQUFRO1FBRVIsT0FBTztRQUNQLE1BQU0sUUFBUSxHQUFHLEdBQUcsQ0FBQyxRQUFRLENBQUMsVUFBVSxFQUFFLENBQUM7UUFDM0MsUUFBUSxDQUFDLFdBQVcsQ0FBQyxVQUFVLEVBQUUsVUFBVSxDQUFDLENBQUM7UUFFN0MsT0FBTztRQUNQLE1BQU0sUUFBUSxHQUFHLFFBQVEsQ0FBQyxNQUFNLEVBQUUsQ0FBQztRQUNuQyxJQUFJLENBQUMsTUFBTSxDQUFDLFFBQVEsRUFBRSw2Q0FBNkMsQ0FBQyxDQUFDO1FBQ3JFLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFDRCxvREFBb0QsQ0FBQyxJQUFVO1FBQzdELFFBQVE7UUFDUixNQUFNLFFBQVEsR0FBRyxHQUFHLENBQUMsUUFBUSxDQUFDLFVBQVUsRUFBRSxDQUFDO1FBRTNDLE9BQU87UUFDUCxRQUFRLENBQUMsV0FBVyxDQUFDLFVBQVUsRUFBRSxVQUFVLENBQUMsQ0FBQztRQUM3QyxRQUFRLENBQUMsaUJBQWlCLENBQUMsU0FBUyxFQUFFLFNBQVMsQ0FBQyxDQUFDO1FBRWpELE9BQU87UUFDUCxNQUFNLFFBQVEsR0FBRyxRQUFRLENBQUMsTUFBTSxFQUFFLENBQUM7UUFDbkMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxRQUFRLEVBQUUsc0JBQXNCO1lBQ3hDLHNEQUFzRDtZQUN0RCxXQUFXO1lBQ1gsV0FBVztZQUNYLFNBQVM7WUFDVCxLQUFLO1lBQ0wsWUFBWTtZQUNaLFlBQVk7WUFDWiw4QkFBOEIsQ0FBQyxDQUFDO1FBQ3BDLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFDRCx3Q0FBd0MsQ0FBQyxJQUFVO1FBQ2pELFFBQVE7UUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLFdBQUssRUFBRSxDQUFDO1FBQzFCLE1BQU0sUUFBUSxHQUFHLElBQUksR0FBRyxDQUFDLEdBQUcsQ0FBQyxLQUFLLEVBQUUsVUFBVSxDQUFDLENBQUM7UUFDaEQsTUFBTSxRQUFRLEdBQUcsR0FBRyxDQUFDLFFBQVEsQ0FBQyxVQUFVLEVBQUUsQ0FBQztRQUUzQyxPQUFPO1FBQ1AsUUFBUSxDQUFDLHNCQUFzQixDQUFFLFFBQVEsQ0FBRSxDQUFDO1FBQzVDLFFBQVEsQ0FBQyxXQUFXLENBQUMsVUFBVSxDQUFDLENBQUM7UUFFakMsT0FBTztRQUNQLE1BQU0sUUFBUSxHQUFHLFFBQVEsQ0FBQyxNQUFNLEVBQUUsQ0FBQztRQUVuQyxJQUFJLENBQUMsTUFBTSxDQUFDLFFBQVEsRUFBRSxzQkFBc0I7WUFDeEMsc0RBQXNEO1lBQ3RELG1JQUFtSTtZQUNuSSxTQUFTO1lBQ1QsS0FBSztZQUNMLFlBQVk7WUFDWiw4QkFBOEIsQ0FDakMsQ0FBQztRQUNGLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFDRCx3Q0FBd0MsQ0FBQyxJQUFVO1FBQ2pELFFBQVE7UUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLFdBQUssRUFBRSxDQUFDO1FBQzFCLE1BQU0sUUFBUSxHQUFHLEdBQUcsQ0FBQyxRQUFRLENBQUMsVUFBVSxFQUFFLENBQUM7UUFDM0MsTUFBTSxNQUFNLEdBQUcsZUFBTSxDQUFDLGNBQWMsQ0FBRSxLQUFLLEVBQUUsWUFBWSxFQUFFLE1BQU0sQ0FBRSxDQUFDO1FBQ3BFLE1BQU0sT0FBTyxHQUFHLGVBQU0sQ0FBQyxjQUFjLENBQUUsS0FBSyxFQUFFLGFBQWEsRUFBRSxPQUFPLENBQUUsQ0FBQztRQUV2RSxPQUFPO1FBQ1AsUUFBUSxDQUFDLG9CQUFvQixDQUFDO1lBQzVCLE1BQU07WUFDTixTQUFTLEVBQUUsY0FBYztTQUMxQixDQUFFLENBQUM7UUFDSixRQUFRLENBQUMsb0JBQW9CLENBQUM7WUFDNUIsTUFBTSxFQUFFLE9BQU87WUFDZixTQUFTLEVBQUUsZUFBZTtZQUMxQixTQUFTLEVBQUUscUNBQXFDO1NBQ2pELENBQUUsQ0FBQztRQUVKLE9BQU87UUFDUCxNQUFNLFFBQVEsR0FBRyxRQUFRLENBQUMsTUFBTSxFQUFFLENBQUM7UUFDbkMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxRQUFRLEVBQUUsd0VBQXdFO1lBQzVGLDZHQUE2RztZQUM3RywyRUFBMkU7WUFDM0UseUlBQXlJLENBQzFJLENBQUM7UUFDRixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBQ0Qsb0NBQW9DLENBQUMsSUFBVTtRQUM3QyxRQUFRO1FBQ1IsTUFBTSxRQUFRLEdBQUcsR0FBRyxDQUFDLFFBQVEsQ0FBQyxVQUFVLEVBQUUsQ0FBQztRQUUzQyxPQUFPO1FBQ1AsUUFBUSxDQUFDLHFCQUFxQixDQUFDO1lBQzdCLFFBQVEsRUFBRSx3QkFBd0I7U0FDbkMsQ0FBRSxDQUFDO1FBQ0osUUFBUSxDQUFDLHFCQUFxQixDQUFDO1lBQzdCLFFBQVEsRUFBRSx5QkFBeUI7WUFDbkMsU0FBUyxFQUFFLDBCQUEwQjtTQUN0QyxDQUFFLENBQUM7UUFFSixPQUFPO1FBQ1AsTUFBTSxRQUFRLEdBQUcsUUFBUSxDQUFDLE1BQU0sRUFBRSxDQUFDO1FBQ25DLElBQUksQ0FBQyxNQUFNLENBQUMsUUFBUSxFQUFFLHFEQUFxRDtZQUN6RSxzR0FBc0c7WUFDdEcseURBQXlEO1lBQ3pELGtIQUFrSCxDQUNuSCxDQUFDO1FBQ0YsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUNELDRCQUE0QixDQUFDLElBQVU7UUFDckMsUUFBUTtRQUVSLE9BQU87UUFDUCxNQUFNLFFBQVEsR0FBRyxHQUFHLENBQUMsUUFBUSxDQUFDLFFBQVEsRUFBRSxDQUFDO1FBQ3pDLFFBQVEsQ0FBQyxXQUFXLENBQUMsVUFBVSxFQUFFLFVBQVUsQ0FBQyxDQUFDO1FBRTdDLE9BQU87UUFDUCxNQUFNLFFBQVEsR0FBRyxRQUFRLENBQUMsTUFBTSxFQUFFLENBQUM7UUFDbkMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxRQUFRLEVBQUUsaUNBQWlDLENBQUMsQ0FBQztRQUN6RCxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBQ0Qsa0RBQWtELENBQUMsSUFBVTtRQUMzRCxRQUFRO1FBQ1IsTUFBTSxRQUFRLEdBQUcsR0FBRyxDQUFDLFFBQVEsQ0FBQyxRQUFRLEVBQUUsQ0FBQztRQUV6QyxPQUFPO1FBQ1AsUUFBUSxDQUFDLFdBQVcsQ0FBQyxVQUFVLEVBQUUsVUFBVSxDQUFDLENBQUM7UUFDN0MsUUFBUSxDQUFDLGlCQUFpQixDQUFDLFNBQVMsRUFBRSxTQUFTLENBQUMsQ0FBQztRQUVqRCxPQUFPO1FBQ1AsTUFBTSxRQUFRLEdBQUcsUUFBUSxDQUFDLE1BQU0sRUFBRSxDQUFDO1FBQ25DLElBQUksQ0FBQyxNQUFNLENBQUMsUUFBUSxFQUFFLGVBQWU7WUFDakMsd0JBQXdCO1lBQ3hCLGVBQWU7WUFDZixXQUFXO1lBQ1gsV0FBVztZQUNYLEtBQUs7WUFDTCxzQkFBc0I7WUFDdEIsWUFBWTtZQUNaLFVBQVUsQ0FBQyxDQUFDO1FBQ2hCLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFDRCxzQ0FBc0MsQ0FBQyxJQUFVO1FBQy9DLFFBQVE7UUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLFdBQUssRUFBRSxDQUFDO1FBQzFCLE1BQU0sUUFBUSxHQUFHLElBQUksR0FBRyxDQUFDLEdBQUcsQ0FBQyxLQUFLLEVBQUUsVUFBVSxDQUFDLENBQUM7UUFFaEQsT0FBTztRQUNQLE1BQU0sUUFBUSxHQUFHLEdBQUcsQ0FBQyxRQUFRLENBQUMsUUFBUSxFQUFFLENBQUM7UUFDekMsUUFBUSxDQUFDLFdBQVcsQ0FBQyxVQUFVLENBQUMsQ0FBQztRQUNqQyxRQUFRLENBQUMsc0JBQXNCLENBQUUsUUFBUSxDQUFFLENBQUM7UUFFNUMsT0FBTztRQUNQLE1BQU0sUUFBUSxHQUFHLFFBQVEsQ0FBQyxNQUFNLEVBQUUsQ0FBQztRQUNuQyxJQUFJLENBQUMsTUFBTSxDQUFDLFFBQVEsRUFBRSxlQUFlO1lBQ2pDLHdCQUF3QjtZQUN4QixlQUFlO1lBQ2Ysb0tBQW9LO1lBQ3BLLEtBQUs7WUFDTCxzQkFBc0I7WUFDdEIsVUFBVSxDQUFDLENBQUM7UUFDaEIsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUNELHNDQUFzQyxDQUFDLElBQVU7UUFDL0MsUUFBUTtRQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksV0FBSyxFQUFFLENBQUM7UUFDMUIsTUFBTSxRQUFRLEdBQUcsR0FBRyxDQUFDLFFBQVEsQ0FBQyxRQUFRLEVBQUUsQ0FBQztRQUN6QyxNQUFNLE1BQU0sR0FBRyxlQUFNLENBQUMsY0FBYyxDQUFFLEtBQUssRUFBRSxZQUFZLEVBQUUsTUFBTSxDQUFFLENBQUM7UUFDcEUsTUFBTSxPQUFPLEdBQUcsZUFBTSxDQUFDLGNBQWMsQ0FBRSxLQUFLLEVBQUUsYUFBYSxFQUFFLE9BQU8sQ0FBRSxDQUFDO1FBRXZFLE9BQU87UUFDUCxRQUFRLENBQUMsb0JBQW9CLENBQUM7WUFDNUIsTUFBTTtZQUNOLFNBQVMsRUFBRSxhQUFhO1NBQ3pCLENBQUUsQ0FBQztRQUNKLFFBQVEsQ0FBQyxvQkFBb0IsQ0FBQztZQUM1QixNQUFNLEVBQUUsT0FBTztZQUNmLFNBQVMsRUFBRSxjQUFjO1lBQ3pCLFNBQVMsRUFBRSxvQ0FBb0M7U0FDaEQsQ0FBRSxDQUFDO1FBRUosT0FBTztRQUNQLE1BQU0sUUFBUSxHQUFHLFFBQVEsQ0FBQyxNQUFNLEVBQUUsQ0FBQztRQUNuQyxJQUFJLENBQUMsTUFBTSxDQUFDLFFBQVEsRUFBRSxlQUFlO1lBQ25DLDRDQUE0QztZQUM1Qyw0REFBNEQ7WUFDNUQsOERBQThEO1lBQzlELDhFQUE4RSxDQUMvRSxDQUFDO1FBQ0YsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUNELGtDQUFrQyxDQUFDLElBQVU7UUFDM0MsUUFBUTtRQUNSLE1BQU0sUUFBUSxHQUFHLEdBQUcsQ0FBQyxRQUFRLENBQUMsUUFBUSxFQUFFLENBQUM7UUFFekMsT0FBTztRQUNQLFFBQVEsQ0FBQyxxQkFBcUIsQ0FBQztZQUM3QixRQUFRLEVBQUUsa0JBQWtCO1NBQzdCLENBQUUsQ0FBQztRQUNKLFFBQVEsQ0FBQyxxQkFBcUIsQ0FBQztZQUM3QixRQUFRLEVBQUUsb0JBQW9CO1lBQzlCLFNBQVMsRUFBRSwwQkFBMEI7U0FDdEMsQ0FBRSxDQUFDO1FBRUosT0FBTztRQUNQLE1BQU0sUUFBUSxHQUFHLFFBQVEsQ0FBQyxNQUFNLEVBQUUsQ0FBQztRQUNuQyxJQUFJLENBQUMsTUFBTSxDQUFDLFFBQVEsRUFBRSxlQUFlO1lBQ25DLFVBQVU7WUFDVixpQ0FBaUM7WUFDakMsa0NBQWtDO1lBQ2xDLFVBQVU7WUFDVixtQ0FBbUM7WUFDbkMsaURBQWlELENBQ2xELENBQUM7UUFDRixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBQ0QsNkJBQTZCLENBQUMsSUFBVTtRQUN0QyxRQUFRO1FBRVIsT0FBTztRQUNQLE1BQU0sUUFBUSxHQUFHLEdBQUcsQ0FBQyxRQUFRLENBQUMsTUFBTSxDQUFDLDBCQUEwQixDQUFDLENBQUM7UUFFakUsT0FBTztRQUNQLE1BQU0sUUFBUSxHQUFHLFFBQVEsQ0FBQyxNQUFNLEVBQUUsQ0FBQztRQUNuQyxJQUFJLENBQUMsTUFBTSxDQUFDLFFBQVEsRUFBRSwwQkFBMEIsQ0FBQyxDQUFDO1FBQ2xELElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFDRCxzREFBc0QsQ0FBQyxJQUFVO1FBQy9ELFFBQVE7UUFDUixPQUFPO1FBQ1AsTUFBTSxRQUFRLEdBQUcsR0FBRyxDQUFDLFFBQVEsQ0FBQyxNQUFNLENBQUMsRUFBRSxDQUFDLENBQUM7UUFFekMsT0FBTztRQUNQLElBQUksQ0FBQyxNQUFNLENBQUMsR0FBRyxFQUFFLENBQUMsUUFBUSxDQUFDLGlCQUFpQixDQUFFLHFCQUFxQixDQUFFLENBQUMsQ0FBQztRQUN2RSxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBQ0Qsb0RBQW9ELENBQUMsSUFBVTtRQUM3RCxRQUFRO1FBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxXQUFLLEVBQUUsQ0FBQztRQUMxQixNQUFNLFFBQVEsR0FBRyxJQUFJLEdBQUcsQ0FBQyxHQUFHLENBQUMsS0FBSyxFQUFFLFVBQVUsQ0FBQyxDQUFDO1FBRWhELE9BQU87UUFDUCxNQUFNLFFBQVEsR0FBRyxHQUFHLENBQUMsUUFBUSxDQUFDLE1BQU0sQ0FBQyxFQUFFLENBQUMsQ0FBQztRQUV6QyxPQUFPO1FBQ1AsSUFBSSxDQUFDLE1BQU0sQ0FBQyxHQUFHLEVBQUUsQ0FBQyxRQUFRLENBQUMsc0JBQXNCLENBQUUsUUFBUSxDQUFFLENBQUMsQ0FBQztRQUMvRCxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBQ0QsK0NBQStDLENBQUMsSUFBVTtRQUN4RCxRQUFRO1FBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxXQUFLLEVBQUUsQ0FBQztRQUMxQixNQUFNLFFBQVEsR0FBRyxHQUFHLENBQUMsUUFBUSxDQUFDLE1BQU0sQ0FBQyxFQUFFLENBQUMsQ0FBQztRQUN6QyxNQUFNLE1BQU0sR0FBRyxlQUFNLENBQUMsY0FBYyxDQUFFLEtBQUssRUFBRSxZQUFZLEVBQUUsTUFBTSxDQUFFLENBQUM7UUFDcEUsT0FBTztRQUNQLE9BQU87UUFDUCxJQUFJLENBQUMsTUFBTSxDQUFDLEdBQUcsRUFBRSxDQUFDLFFBQVEsQ0FBQyxvQkFBb0IsQ0FBQztZQUM5QyxNQUFNO1lBQ04sU0FBUyxFQUFFLGFBQWE7U0FDekIsQ0FBRSxDQUFDLENBQUM7UUFDTCxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBQ0QsNkNBQTZDLENBQUMsSUFBVTtRQUN0RCxRQUFRO1FBQ1IsTUFBTSxRQUFRLEdBQUcsR0FBRyxDQUFDLFFBQVEsQ0FBQyxNQUFNLENBQUMsRUFBRSxDQUFDLENBQUM7UUFDekMsT0FBTztRQUNQLE9BQU87UUFDUCxJQUFJLENBQUMsTUFBTSxDQUFDLEdBQUcsRUFBRSxDQUNqQixRQUFRLENBQUMscUJBQXFCLENBQUM7WUFDN0IsUUFBUSxFQUFFLGtCQUFrQjtTQUM3QixDQUFFLENBQUMsQ0FBQztRQUNMLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7Q0FFRixDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgQnVja2V0IH0gZnJvbSAnQGF3cy1jZGsvYXdzLXMzJztcbmltcG9ydCB7IFRlc3QgfSBmcm9tICdub2RldW5pdCc7XG5pbXBvcnQgeyBTdGFjayB9IGZyb20gXCIuLi8uLi9jb3JlL2xpYlwiO1xuaW1wb3J0ICogYXMgZWMyIGZyb20gJy4uL2xpYic7XG5cbmV4cG9ydCA9IHtcbiAgJ2NhbiBjcmVhdGUgV2luZG93cyB1c2VyIGRhdGEnKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBHSVZFTlxuXG4gICAgLy8gV0hFTlxuICAgIGNvbnN0IHVzZXJEYXRhID0gZWMyLlVzZXJEYXRhLmZvcldpbmRvd3MoKTtcbiAgICB1c2VyRGF0YS5hZGRDb21tYW5kcygnY29tbWFuZDEnLCAnY29tbWFuZDInKTtcblxuICAgIC8vIFRIRU5cbiAgICBjb25zdCByZW5kZXJlZCA9IHVzZXJEYXRhLnJlbmRlcigpO1xuICAgIHRlc3QuZXF1YWxzKHJlbmRlcmVkLCAnPHBvd2Vyc2hlbGw+Y29tbWFuZDFcXG5jb21tYW5kMjwvcG93ZXJzaGVsbD4nKTtcbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcbiAgJ2NhbiBjcmVhdGUgV2luZG93cyB1c2VyIGRhdGEgd2l0aCBjb21tYW5kcyBvbiBleGl0Jyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCB1c2VyRGF0YSA9IGVjMi5Vc2VyRGF0YS5mb3JXaW5kb3dzKCk7XG5cbiAgICAvLyBXSEVOXG4gICAgdXNlckRhdGEuYWRkQ29tbWFuZHMoJ2NvbW1hbmQxJywgJ2NvbW1hbmQyJyk7XG4gICAgdXNlckRhdGEuYWRkT25FeGl0Q29tbWFuZHMoJ29uZXhpdDEnLCAnb25leGl0MicpO1xuXG4gICAgLy8gVEhFTlxuICAgIGNvbnN0IHJlbmRlcmVkID0gdXNlckRhdGEucmVuZGVyKCk7XG4gICAgdGVzdC5lcXVhbHMocmVuZGVyZWQsICc8cG93ZXJzaGVsbD50cmFwIHtcXG4nICtcbiAgICAgICAgJyRzdWNjZXNzPSgkUFNJdGVtLkV4Y2VwdGlvbi5NZXNzYWdlIC1lcSBcIlN1Y2Nlc3NcIilcXG4nICtcbiAgICAgICAgJ29uZXhpdDFcXG4nICtcbiAgICAgICAgJ29uZXhpdDJcXG4nICtcbiAgICAgICAgJ2JyZWFrXFxuJyArXG4gICAgICAgICd9XFxuJyArXG4gICAgICAgICdjb21tYW5kMVxcbicgK1xuICAgICAgICAnY29tbWFuZDJcXG4nICtcbiAgICAgICAgJ3Rocm93IFwiU3VjY2Vzc1wiPC9wb3dlcnNoZWxsPicpO1xuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuICAnY2FuIGNyZWF0ZSBXaW5kb3dzIHdpdGggU2lnbmFsIENvbW1hbmQnKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG4gICAgY29uc3QgcmVzb3VyY2UgPSBuZXcgZWMyLlZwYyhzdGFjaywgJ1JFU09VUkNFJyk7XG4gICAgY29uc3QgdXNlckRhdGEgPSBlYzIuVXNlckRhdGEuZm9yV2luZG93cygpO1xuXG4gICAgLy8gV0hFTlxuICAgIHVzZXJEYXRhLmFkZFNpZ25hbE9uRXhpdENvbW1hbmQoIHJlc291cmNlICk7XG4gICAgdXNlckRhdGEuYWRkQ29tbWFuZHMoXCJjb21tYW5kMVwiKTtcblxuICAgIC8vIFRIRU5cbiAgICBjb25zdCByZW5kZXJlZCA9IHVzZXJEYXRhLnJlbmRlcigpO1xuXG4gICAgdGVzdC5lcXVhbHMocmVuZGVyZWQsICc8cG93ZXJzaGVsbD50cmFwIHtcXG4nICtcbiAgICAgICAgJyRzdWNjZXNzPSgkUFNJdGVtLkV4Y2VwdGlvbi5NZXNzYWdlIC1lcSBcIlN1Y2Nlc3NcIilcXG4nICtcbiAgICAgICAgJ2Nmbi1zaWduYWwgLS1zdGFjayBTdGFjayAtLXJlc291cmNlIFJFU09VUkNFMTk4OTU1MkYgLS1yZWdpb24gJHtUb2tlbltBV1M6OlJlZ2lvbi40XX0gLS1zdWNjZXNzICgkc3VjY2Vzcy5Ub1N0cmluZygpLlRvTG93ZXIoKSlcXG4nICtcbiAgICAgICAgJ2JyZWFrXFxuJyArXG4gICAgICAgICd9XFxuJyArXG4gICAgICAgICdjb21tYW5kMVxcbicgK1xuICAgICAgICAndGhyb3cgXCJTdWNjZXNzXCI8L3Bvd2Vyc2hlbGw+J1xuICAgICk7XG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG4gICdjYW4gd2luZG93cyB1c2VyZGF0YSBkb3dubG9hZCBTMyBmaWxlcycodGVzdDogVGVzdCkge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgICBjb25zdCB1c2VyRGF0YSA9IGVjMi5Vc2VyRGF0YS5mb3JXaW5kb3dzKCk7XG4gICAgY29uc3QgYnVja2V0ID0gQnVja2V0LmZyb21CdWNrZXROYW1lKCBzdGFjaywgXCJ0ZXN0QnVja2V0XCIsIFwidGVzdFwiICk7XG4gICAgY29uc3QgYnVja2V0MiA9IEJ1Y2tldC5mcm9tQnVja2V0TmFtZSggc3RhY2ssIFwidGVzdEJ1Y2tldDJcIiwgXCJ0ZXN0MlwiICk7XG5cbiAgICAvLyBXSEVOXG4gICAgdXNlckRhdGEuYWRkUzNEb3dubG9hZENvbW1hbmQoe1xuICAgICAgYnVja2V0LFxuICAgICAgYnVja2V0S2V5OiBcImZpbGVuYW1lLmJhdFwiXG4gICAgfSApO1xuICAgIHVzZXJEYXRhLmFkZFMzRG93bmxvYWRDb21tYW5kKHtcbiAgICAgIGJ1Y2tldDogYnVja2V0MixcbiAgICAgIGJ1Y2tldEtleTogXCJmaWxlbmFtZTIuYmF0XCIsXG4gICAgICBsb2NhbEZpbGU6IFwiYzpcXFxcdGVzdFxcXFxsb2NhdGlvblxcXFxvdGhlclNjcmlwdC5iYXRcIlxuICAgIH0gKTtcblxuICAgIC8vIFRIRU5cbiAgICBjb25zdCByZW5kZXJlZCA9IHVzZXJEYXRhLnJlbmRlcigpO1xuICAgIHRlc3QuZXF1YWxzKHJlbmRlcmVkLCAnPHBvd2Vyc2hlbGw+bWtkaXIgKFNwbGl0LVBhdGggLVBhdGggXFwnQzovdGVtcC9maWxlbmFtZS5iYXRcXCcgKSAtZWEgMFxcbicgK1xuICAgICAgJ1JlYWQtUzNPYmplY3QgLUJ1Y2tldE5hbWUgXFwndGVzdFxcJyAta2V5IFxcJ2ZpbGVuYW1lLmJhdFxcJyAtZmlsZSBcXCdDOi90ZW1wL2ZpbGVuYW1lLmJhdFxcJyAtRXJyb3JBY3Rpb24gU3RvcFxcbicgK1xuICAgICAgJ21rZGlyIChTcGxpdC1QYXRoIC1QYXRoIFxcJ2M6XFxcXHRlc3RcXFxcbG9jYXRpb25cXFxcb3RoZXJTY3JpcHQuYmF0XFwnICkgLWVhIDBcXG4nICtcbiAgICAgICdSZWFkLVMzT2JqZWN0IC1CdWNrZXROYW1lIFxcJ3Rlc3QyXFwnIC1rZXkgXFwnZmlsZW5hbWUyLmJhdFxcJyAtZmlsZSBcXCdjOlxcXFx0ZXN0XFxcXGxvY2F0aW9uXFxcXG90aGVyU2NyaXB0LmJhdFxcJyAtRXJyb3JBY3Rpb24gU3RvcDwvcG93ZXJzaGVsbD4nXG4gICAgKTtcbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcbiAgJ2NhbiB3aW5kb3dzIHVzZXJkYXRhIGV4ZWN1dGUgZmlsZXMnKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IHVzZXJEYXRhID0gZWMyLlVzZXJEYXRhLmZvcldpbmRvd3MoKTtcblxuICAgIC8vIFdIRU5cbiAgICB1c2VyRGF0YS5hZGRFeGVjdXRlRmlsZUNvbW1hbmQoe1xuICAgICAgZmlsZVBhdGg6IFwiQzpcXFxcdGVzdFxcXFxmaWxlbmFtZS5iYXRcIixcbiAgICB9ICk7XG4gICAgdXNlckRhdGEuYWRkRXhlY3V0ZUZpbGVDb21tYW5kKHtcbiAgICAgIGZpbGVQYXRoOiBcIkM6XFxcXHRlc3RcXFxcZmlsZW5hbWUyLmJhdFwiLFxuICAgICAgYXJndW1lbnRzOiBcImFyZzEgYXJnMiAtYXJnICR2YXJpYWJsZVwiXG4gICAgfSApO1xuXG4gICAgLy8gVEhFTlxuICAgIGNvbnN0IHJlbmRlcmVkID0gdXNlckRhdGEucmVuZGVyKCk7XG4gICAgdGVzdC5lcXVhbHMocmVuZGVyZWQsICc8cG93ZXJzaGVsbD4mXFwnQzpcXFxcdGVzdFxcXFxmaWxlbmFtZS5iYXRcXCcgdW5kZWZpbmVkXFxuJyArXG4gICAgICAnaWYgKCEkPykgeyBXcml0ZS1FcnJvciBcXCdGYWlsZWQgdG8gZXhlY3V0ZSB0aGUgZmlsZSBcIkM6XFxcXHRlc3RcXFxcZmlsZW5hbWUuYmF0XCJcXCcgLUVycm9yQWN0aW9uIFN0b3AgfVxcbicgK1xuICAgICAgJyZcXCdDOlxcXFx0ZXN0XFxcXGZpbGVuYW1lMi5iYXRcXCcgYXJnMSBhcmcyIC1hcmcgJHZhcmlhYmxlXFxuJyArXG4gICAgICAnaWYgKCEkPykgeyBXcml0ZS1FcnJvciBcXCdGYWlsZWQgdG8gZXhlY3V0ZSB0aGUgZmlsZSBcIkM6XFxcXHRlc3RcXFxcZmlsZW5hbWUyLmJhdFwiXFwnIC1FcnJvckFjdGlvbiBTdG9wIH08L3Bvd2Vyc2hlbGw+J1xuICAgICk7XG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG4gICdjYW4gY3JlYXRlIExpbnV4IHVzZXIgZGF0YScodGVzdDogVGVzdCkge1xuICAgIC8vIEdJVkVOXG5cbiAgICAvLyBXSEVOXG4gICAgY29uc3QgdXNlckRhdGEgPSBlYzIuVXNlckRhdGEuZm9yTGludXgoKTtcbiAgICB1c2VyRGF0YS5hZGRDb21tYW5kcygnY29tbWFuZDEnLCAnY29tbWFuZDInKTtcblxuICAgIC8vIFRIRU5cbiAgICBjb25zdCByZW5kZXJlZCA9IHVzZXJEYXRhLnJlbmRlcigpO1xuICAgIHRlc3QuZXF1YWxzKHJlbmRlcmVkLCAnIyEvYmluL2Jhc2hcXG5jb21tYW5kMVxcbmNvbW1hbmQyJyk7XG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG4gICdjYW4gY3JlYXRlIExpbnV4IHVzZXIgZGF0YSB3aXRoIGNvbW1hbmRzIG9uIGV4aXQnKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IHVzZXJEYXRhID0gZWMyLlVzZXJEYXRhLmZvckxpbnV4KCk7XG5cbiAgICAvLyBXSEVOXG4gICAgdXNlckRhdGEuYWRkQ29tbWFuZHMoJ2NvbW1hbmQxJywgJ2NvbW1hbmQyJyk7XG4gICAgdXNlckRhdGEuYWRkT25FeGl0Q29tbWFuZHMoJ29uZXhpdDEnLCAnb25leGl0MicpO1xuXG4gICAgLy8gVEhFTlxuICAgIGNvbnN0IHJlbmRlcmVkID0gdXNlckRhdGEucmVuZGVyKCk7XG4gICAgdGVzdC5lcXVhbHMocmVuZGVyZWQsICcjIS9iaW4vYmFzaFxcbicgK1xuICAgICAgICAnZnVuY3Rpb24gZXhpdFRyYXAoKXtcXG4nICtcbiAgICAgICAgJ2V4aXRDb2RlPSQ/XFxuJyArXG4gICAgICAgICdvbmV4aXQxXFxuJyArXG4gICAgICAgICdvbmV4aXQyXFxuJyArXG4gICAgICAgICd9XFxuJyArXG4gICAgICAgICd0cmFwIGV4aXRUcmFwIEVYSVRcXG4nICtcbiAgICAgICAgJ2NvbW1hbmQxXFxuJyArXG4gICAgICAgICdjb21tYW5kMicpO1xuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuICAnY2FuIGNyZWF0ZSBMaW51eCB3aXRoIFNpZ25hbCBDb21tYW5kJyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuICAgIGNvbnN0IHJlc291cmNlID0gbmV3IGVjMi5WcGMoc3RhY2ssICdSRVNPVVJDRScpO1xuXG4gICAgLy8gV0hFTlxuICAgIGNvbnN0IHVzZXJEYXRhID0gZWMyLlVzZXJEYXRhLmZvckxpbnV4KCk7XG4gICAgdXNlckRhdGEuYWRkQ29tbWFuZHMoXCJjb21tYW5kMVwiKTtcbiAgICB1c2VyRGF0YS5hZGRTaWduYWxPbkV4aXRDb21tYW5kKCByZXNvdXJjZSApO1xuXG4gICAgLy8gVEhFTlxuICAgIGNvbnN0IHJlbmRlcmVkID0gdXNlckRhdGEucmVuZGVyKCk7XG4gICAgdGVzdC5lcXVhbHMocmVuZGVyZWQsICcjIS9iaW4vYmFzaFxcbicgK1xuICAgICAgICAnZnVuY3Rpb24gZXhpdFRyYXAoKXtcXG4nICtcbiAgICAgICAgJ2V4aXRDb2RlPSQ/XFxuJyArXG4gICAgICAgICcvb3B0L2F3cy9iaW4vY2ZuLXNpZ25hbCAtLXN0YWNrIFN0YWNrIC0tcmVzb3VyY2UgUkVTT1VSQ0UxOTg5NTUyRiAtLXJlZ2lvbiAke1Rva2VuW0FXUzo6UmVnaW9uLjRdfSAtZSAkZXhpdENvZGUgfHwgZWNobyBcXCdGYWlsZWQgdG8gc2VuZCBDbG91ZGZvcm1hdGlvbiBTaWduYWxcXCdcXG4nICtcbiAgICAgICAgJ31cXG4nICtcbiAgICAgICAgJ3RyYXAgZXhpdFRyYXAgRVhJVFxcbicgK1xuICAgICAgICAnY29tbWFuZDEnKTtcbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcbiAgJ2NhbiBsaW51eCB1c2VyZGF0YSBkb3dubG9hZCBTMyBmaWxlcycodGVzdDogVGVzdCkge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgICBjb25zdCB1c2VyRGF0YSA9IGVjMi5Vc2VyRGF0YS5mb3JMaW51eCgpO1xuICAgIGNvbnN0IGJ1Y2tldCA9IEJ1Y2tldC5mcm9tQnVja2V0TmFtZSggc3RhY2ssIFwidGVzdEJ1Y2tldFwiLCBcInRlc3RcIiApO1xuICAgIGNvbnN0IGJ1Y2tldDIgPSBCdWNrZXQuZnJvbUJ1Y2tldE5hbWUoIHN0YWNrLCBcInRlc3RCdWNrZXQyXCIsIFwidGVzdDJcIiApO1xuXG4gICAgLy8gV0hFTlxuICAgIHVzZXJEYXRhLmFkZFMzRG93bmxvYWRDb21tYW5kKHtcbiAgICAgIGJ1Y2tldCxcbiAgICAgIGJ1Y2tldEtleTogXCJmaWxlbmFtZS5zaFwiXG4gICAgfSApO1xuICAgIHVzZXJEYXRhLmFkZFMzRG93bmxvYWRDb21tYW5kKHtcbiAgICAgIGJ1Y2tldDogYnVja2V0MixcbiAgICAgIGJ1Y2tldEtleTogXCJmaWxlbmFtZTIuc2hcIixcbiAgICAgIGxvY2FsRmlsZTogXCJjOlxcXFx0ZXN0XFxcXGxvY2F0aW9uXFxcXG90aGVyU2NyaXB0LnNoXCJcbiAgICB9ICk7XG5cbiAgICAvLyBUSEVOXG4gICAgY29uc3QgcmVuZGVyZWQgPSB1c2VyRGF0YS5yZW5kZXIoKTtcbiAgICB0ZXN0LmVxdWFscyhyZW5kZXJlZCwgJyMhL2Jpbi9iYXNoXFxuJyArXG4gICAgICAnbWtkaXIgLXAgJChkaXJuYW1lIFxcJy90bXAvZmlsZW5hbWUuc2hcXCcpXFxuJyArXG4gICAgICAnYXdzIHMzIGNwIFxcJ3MzOi8vdGVzdC9maWxlbmFtZS5zaFxcJyBcXCcvdG1wL2ZpbGVuYW1lLnNoXFwnXFxuJyArXG4gICAgICAnbWtkaXIgLXAgJChkaXJuYW1lIFxcJ2M6XFxcXHRlc3RcXFxcbG9jYXRpb25cXFxcb3RoZXJTY3JpcHQuc2hcXCcpXFxuJyArXG4gICAgICAnYXdzIHMzIGNwIFxcJ3MzOi8vdGVzdDIvZmlsZW5hbWUyLnNoXFwnIFxcJ2M6XFxcXHRlc3RcXFxcbG9jYXRpb25cXFxcb3RoZXJTY3JpcHQuc2hcXCcnXG4gICAgKTtcbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcbiAgJ2NhbiBsaW51eCB1c2VyZGF0YSBleGVjdXRlIGZpbGVzJyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCB1c2VyRGF0YSA9IGVjMi5Vc2VyRGF0YS5mb3JMaW51eCgpO1xuXG4gICAgLy8gV0hFTlxuICAgIHVzZXJEYXRhLmFkZEV4ZWN1dGVGaWxlQ29tbWFuZCh7XG4gICAgICBmaWxlUGF0aDogXCIvdG1wL2ZpbGVuYW1lLnNoXCIsXG4gICAgfSApO1xuICAgIHVzZXJEYXRhLmFkZEV4ZWN1dGVGaWxlQ29tbWFuZCh7XG4gICAgICBmaWxlUGF0aDogXCIvdGVzdC9maWxlbmFtZTIuc2hcIixcbiAgICAgIGFyZ3VtZW50czogXCJhcmcxIGFyZzIgLWFyZyAkdmFyaWFibGVcIlxuICAgIH0gKTtcblxuICAgIC8vIFRIRU5cbiAgICBjb25zdCByZW5kZXJlZCA9IHVzZXJEYXRhLnJlbmRlcigpO1xuICAgIHRlc3QuZXF1YWxzKHJlbmRlcmVkLCAnIyEvYmluL2Jhc2hcXG4nICtcbiAgICAgICdzZXQgLWVcXG4nICtcbiAgICAgICdjaG1vZCAreCBcXCcvdG1wL2ZpbGVuYW1lLnNoXFwnXFxuJyArXG4gICAgICAnXFwnL3RtcC9maWxlbmFtZS5zaFxcJyB1bmRlZmluZWRcXG4nICtcbiAgICAgICdzZXQgLWVcXG4nICtcbiAgICAgICdjaG1vZCAreCBcXCcvdGVzdC9maWxlbmFtZTIuc2hcXCdcXG4nICtcbiAgICAgICdcXCcvdGVzdC9maWxlbmFtZTIuc2hcXCcgYXJnMSBhcmcyIC1hcmcgJHZhcmlhYmxlJ1xuICAgICk7XG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG4gICdjYW4gY3JlYXRlIEN1c3RvbSB1c2VyIGRhdGEnKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBHSVZFTlxuXG4gICAgLy8gV0hFTlxuICAgIGNvbnN0IHVzZXJEYXRhID0gZWMyLlVzZXJEYXRhLmN1c3RvbSgnU29tZVxcbm11bHRpbGluZVxcbmNvbnRlbnQnKTtcblxuICAgIC8vIFRIRU5cbiAgICBjb25zdCByZW5kZXJlZCA9IHVzZXJEYXRhLnJlbmRlcigpO1xuICAgIHRlc3QuZXF1YWxzKHJlbmRlcmVkLCAnU29tZVxcbm11bHRpbGluZVxcbmNvbnRlbnQnKTtcbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcbiAgJ0N1c3RvbSB1c2VyIGRhdGEgdGhyb3dzIHdoZW4gYWRkaW5nIG9uIGV4aXQgY29tbWFuZHMnKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBHSVZFTlxuICAgIC8vIFdIRU5cbiAgICBjb25zdCB1c2VyRGF0YSA9IGVjMi5Vc2VyRGF0YS5jdXN0b20oXCJcIik7XG5cbiAgICAvLyBUSEVOXG4gICAgdGVzdC50aHJvd3MoKCkgPT4gdXNlckRhdGEuYWRkT25FeGl0Q29tbWFuZHMoIFwiYSBjb21tYW5kIGdvZXMgaGVyZVwiICkpO1xuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuICAnQ3VzdG9tIHVzZXIgZGF0YSB0aHJvd3Mgd2hlbiBhZGRpbmcgc2lnbmFsIGNvbW1hbmQnKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG4gICAgY29uc3QgcmVzb3VyY2UgPSBuZXcgZWMyLlZwYyhzdGFjaywgJ1JFU09VUkNFJyk7XG5cbiAgICAvLyBXSEVOXG4gICAgY29uc3QgdXNlckRhdGEgPSBlYzIuVXNlckRhdGEuY3VzdG9tKFwiXCIpO1xuXG4gICAgLy8gVEhFTlxuICAgIHRlc3QudGhyb3dzKCgpID0+IHVzZXJEYXRhLmFkZFNpZ25hbE9uRXhpdENvbW1hbmQoIHJlc291cmNlICkpO1xuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuICAnQ3VzdG9tIHVzZXIgZGF0YSB0aHJvd3Mgd2hlbiBkb3dubG9hZGluZyBmaWxlJyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuICAgIGNvbnN0IHVzZXJEYXRhID0gZWMyLlVzZXJEYXRhLmN1c3RvbShcIlwiKTtcbiAgICBjb25zdCBidWNrZXQgPSBCdWNrZXQuZnJvbUJ1Y2tldE5hbWUoIHN0YWNrLCBcInRlc3RCdWNrZXRcIiwgXCJ0ZXN0XCIgKTtcbiAgICAvLyBXSEVOXG4gICAgLy8gVEhFTlxuICAgIHRlc3QudGhyb3dzKCgpID0+IHVzZXJEYXRhLmFkZFMzRG93bmxvYWRDb21tYW5kKHtcbiAgICAgIGJ1Y2tldCxcbiAgICAgIGJ1Y2tldEtleTogXCJmaWxlbmFtZS5zaFwiXG4gICAgfSApKTtcbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcbiAgJ0N1c3RvbSB1c2VyIGRhdGEgdGhyb3dzIHdoZW4gZXhlY3V0aW5nIGZpbGUnKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IHVzZXJEYXRhID0gZWMyLlVzZXJEYXRhLmN1c3RvbShcIlwiKTtcbiAgICAvLyBXSEVOXG4gICAgLy8gVEhFTlxuICAgIHRlc3QudGhyb3dzKCgpID0+XG4gICAgdXNlckRhdGEuYWRkRXhlY3V0ZUZpbGVDb21tYW5kKHtcbiAgICAgIGZpbGVQYXRoOiBcIi90bXAvZmlsZW5hbWUuc2hcIixcbiAgICB9ICkpO1xuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG59O1xuIl19