"use strict";
const assert_1 = require("@aws-cdk/assert");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
module.exports = {
    'security group can allows all outbound traffic by default'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const vpc = new lib_1.Vpc(stack, 'VPC');
        // WHEN
        new lib_1.SecurityGroup(stack, 'SG1', { vpc, allowAllOutbound: true });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroup', {
            SecurityGroupEgress: [
                {
                    CidrIp: "0.0.0.0/0",
                    Description: "Allow all outbound traffic by default",
                    IpProtocol: "-1"
                }
            ],
        }));
        test.done();
    },
    'no new outbound rule is added if we are allowing all traffic anyway'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const vpc = new lib_1.Vpc(stack, 'VPC');
        // WHEN
        const sg = new lib_1.SecurityGroup(stack, 'SG1', { vpc, allowAllOutbound: true });
        sg.addEgressRule(lib_1.Peer.anyIpv4(), lib_1.Port.tcp(86), 'This does not show up');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroup', {
            SecurityGroupEgress: [
                {
                    CidrIp: "0.0.0.0/0",
                    Description: "Allow all outbound traffic by default",
                    IpProtocol: "-1"
                },
            ],
        }));
        test.done();
    },
    'security group disallow outbound traffic by default'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const vpc = new lib_1.Vpc(stack, 'VPC');
        // WHEN
        new lib_1.SecurityGroup(stack, 'SG1', { vpc, allowAllOutbound: false });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroup', {
            SecurityGroupEgress: [
                {
                    CidrIp: "255.255.255.255/32",
                    Description: "Disallow all traffic",
                    FromPort: 252,
                    IpProtocol: "icmp",
                    ToPort: 86
                }
            ],
        }));
        test.done();
    },
    'bogus outbound rule disappears if another rule is added'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const vpc = new lib_1.Vpc(stack, 'VPC');
        // WHEN
        const sg = new lib_1.SecurityGroup(stack, 'SG1', { vpc, allowAllOutbound: false });
        sg.addEgressRule(lib_1.Peer.anyIpv4(), lib_1.Port.tcp(86), 'This replaces the other one');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroup', {
            SecurityGroupEgress: [
                {
                    CidrIp: "0.0.0.0/0",
                    Description: "This replaces the other one",
                    FromPort: 86,
                    IpProtocol: "tcp",
                    ToPort: 86
                }
            ],
        }));
        test.done();
    },
    'all outbound rule cannot be added after creation'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const vpc = new lib_1.Vpc(stack, 'VPC');
        // WHEN
        const sg = new lib_1.SecurityGroup(stack, 'SG1', { vpc, allowAllOutbound: false });
        test.throws(() => {
            sg.addEgressRule(lib_1.Peer.anyIpv4(), lib_1.Port.allTraffic(), 'All traffic');
        }, /Cannot add/);
        test.done();
    },
    'immutable imports do not add rules'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const sg = lib_1.SecurityGroup.fromSecurityGroupId(stack, 'SG1', "test-id", { mutable: false });
        sg.addEgressRule(lib_1.Peer.anyIpv4(), lib_1.Port.tcp(86), 'This rule was not added');
        sg.addIngressRule(lib_1.Peer.anyIpv4(), lib_1.Port.tcp(86), 'This rule was not added');
        assert_1.expect(stack).to(assert_1.not(assert_1.haveResource('AWS::EC2::SecurityGroup', {
            SecurityGroupEgress: [
                {
                    CidrIp: "0.0.0.0/0",
                    Description: "This rule was not added",
                    FromPort: 86,
                    IpProtocol: "tcp",
                    ToPort: 86
                }
            ],
        })));
        assert_1.expect(stack).to(assert_1.not(assert_1.haveResource('AWS::EC2::SecurityGroup', {
            SecurityGroupIngress: [
                {
                    CidrIp: "0.0.0.0/0",
                    Description: "This rule was not added",
                    FromPort: 86,
                    IpProtocol: "tcp",
                    ToPort: 86
                }
            ],
        })));
        test.done();
    },
    'peer between all types of peers and port range types'(test) {
        // GIVEN
        const stack = new core_1.Stack(undefined, 'TestStack', { env: { account: '12345678', region: 'dummy' } });
        const vpc = new lib_1.Vpc(stack, 'VPC');
        const sg = new lib_1.SecurityGroup(stack, 'SG', { vpc });
        const peers = [
            new lib_1.SecurityGroup(stack, 'PeerGroup', { vpc }),
            lib_1.Peer.anyIpv4(),
            lib_1.Peer.anyIpv6(),
            lib_1.Peer.prefixList('pl-012345'),
        ];
        const ports = [
            lib_1.Port.tcp(1234),
            lib_1.Port.tcp(core_1.Lazy.numberValue({ produce: () => 5000 })),
            lib_1.Port.allTcp(),
            lib_1.Port.tcpRange(80, 90),
            lib_1.Port.udp(2345),
            lib_1.Port.udp(core_1.Lazy.numberValue({ produce: () => 7777 })),
            lib_1.Port.allUdp(),
            lib_1.Port.udpRange(85, 95),
            lib_1.Port.icmpTypeAndCode(5, 1),
            lib_1.Port.icmpType(8),
            lib_1.Port.allIcmp(),
            lib_1.Port.icmpPing(),
            lib_1.Port.allTraffic()
        ];
        // WHEN
        for (const peer of peers) {
            for (const port of ports) {
                sg.connections.allowTo(peer, port);
                sg.connections.allowFrom(peer, port);
            }
        }
        // THEN -- no crash
        test.done();
    },
    'if tokens are used in ports, `canInlineRule` should be false to avoid cycles'(test) {
        // GIVEN
        const p1 = core_1.Lazy.numberValue({ produce: () => 80 });
        const p2 = core_1.Lazy.numberValue({ produce: () => 5000 });
        // WHEN
        const ports = [
            lib_1.Port.tcp(p1),
            lib_1.Port.tcp(p2),
            lib_1.Port.tcpRange(p1, 90),
            lib_1.Port.tcpRange(80, p2),
            lib_1.Port.tcpRange(p1, p2),
            lib_1.Port.udp(p1),
            lib_1.Port.udpRange(p1, 95),
            lib_1.Port.udpRange(85, p2),
            lib_1.Port.udpRange(p1, p2),
            lib_1.Port.icmpTypeAndCode(p1, 1),
            lib_1.Port.icmpTypeAndCode(5, p1),
            lib_1.Port.icmpTypeAndCode(p1, p2),
            lib_1.Port.icmpType(p1),
        ];
        // THEN
        for (const range of ports) {
            test.equal(range.canInlineRule, false, range.toString());
        }
        test.done();
    },
    'Peer IP CIDR validation': {
        'passes with valid IPv4 CIDR block'(test) {
            // GIVEN
            const cidrIps = ['0.0.0.0/0', '192.168.255.255/24'];
            // THEN
            for (const cidrIp of cidrIps) {
                test.equal(lib_1.Peer.ipv4(cidrIp).uniqueId, cidrIp);
            }
            test.done();
        },
        'passes with unresolved IP CIDR token'(test) {
            // GIVEN
            core_1.Token.asString(new core_1.Intrinsic('ip'));
            // THEN: don't throw
            test.done();
        },
        'throws if invalid IPv4 CIDR block'(test) {
            // THEN
            test.throws(() => {
                lib_1.Peer.ipv4('invalid');
            }, /Invalid IPv4 CIDR/);
            test.done();
        },
        'throws if missing mask in IPv4 CIDR block'(test) {
            test.throws(() => {
                lib_1.Peer.ipv4('0.0.0.0');
            }, /CIDR mask is missing in IPv4/);
            test.done();
        },
        'passes with valid IPv6 CIDR block'(test) {
            // GIVEN
            const cidrIps = [
                '::/0',
                '2001:db8::/32',
                '2001:0db8:0000:0000:0000:8a2e:0370:7334/32',
                '2001:db8::8a2e:370:7334/32',
            ];
            // THEN
            for (const cidrIp of cidrIps) {
                test.equal(lib_1.Peer.ipv6(cidrIp).uniqueId, cidrIp);
            }
            test.done();
        },
        'throws if invalid IPv6 CIDR block'(test) {
            // THEN
            test.throws(() => {
                lib_1.Peer.ipv6('invalid');
            }, /Invalid IPv6 CIDR/);
            test.done();
        },
        'throws if missing mask in IPv6 CIDR block'(test) {
            test.throws(() => {
                lib_1.Peer.ipv6('::');
            }, /IDR mask is missing in IPv6/);
            test.done();
        }
    }
};
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidGVzdC5zZWN1cml0eS1ncm91cC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInRlc3Quc2VjdXJpdHktZ3JvdXAudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IjtBQUFBLDRDQUE0RDtBQUM1RCx3Q0FBOEQ7QUFFOUQsZ0NBQXdEO0FBRXhELGlCQUFTO0lBQ1AsMkRBQTJELENBQUMsSUFBVTtRQUNwRSxRQUFRO1FBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztRQUMxQixNQUFNLEdBQUcsR0FBRyxJQUFJLFNBQUcsQ0FBQyxLQUFLLEVBQUUsS0FBSyxDQUFDLENBQUM7UUFFbEMsT0FBTztRQUNQLElBQUksbUJBQWEsQ0FBQyxLQUFLLEVBQUUsS0FBSyxFQUFFLEVBQUUsR0FBRyxFQUFFLGdCQUFnQixFQUFFLElBQUksRUFBRSxDQUFDLENBQUM7UUFFakUsT0FBTztRQUNQLGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMscUJBQVksQ0FBQyx5QkFBeUIsRUFBRTtZQUN2RCxtQkFBbUIsRUFBRTtnQkFDbkI7b0JBQ0UsTUFBTSxFQUFFLFdBQVc7b0JBQ25CLFdBQVcsRUFBRSx1Q0FBdUM7b0JBQ3BELFVBQVUsRUFBRSxJQUFJO2lCQUNqQjthQUNGO1NBQ0YsQ0FBQyxDQUFDLENBQUM7UUFFSixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQscUVBQXFFLENBQUMsSUFBVTtRQUM5RSxRQUFRO1FBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztRQUMxQixNQUFNLEdBQUcsR0FBRyxJQUFJLFNBQUcsQ0FBQyxLQUFLLEVBQUUsS0FBSyxDQUFDLENBQUM7UUFFbEMsT0FBTztRQUNQLE1BQU0sRUFBRSxHQUFHLElBQUksbUJBQWEsQ0FBQyxLQUFLLEVBQUUsS0FBSyxFQUFFLEVBQUUsR0FBRyxFQUFFLGdCQUFnQixFQUFFLElBQUksRUFBRSxDQUFDLENBQUM7UUFDNUUsRUFBRSxDQUFDLGFBQWEsQ0FBQyxVQUFJLENBQUMsT0FBTyxFQUFFLEVBQUUsVUFBSSxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUMsRUFBRSx1QkFBdUIsQ0FBQyxDQUFDO1FBRXhFLE9BQU87UUFDUCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMseUJBQXlCLEVBQUU7WUFDdkQsbUJBQW1CLEVBQUU7Z0JBQ25CO29CQUNFLE1BQU0sRUFBRSxXQUFXO29CQUNuQixXQUFXLEVBQUUsdUNBQXVDO29CQUNwRCxVQUFVLEVBQUUsSUFBSTtpQkFDakI7YUFDRjtTQUNGLENBQUMsQ0FBQyxDQUFDO1FBRUosSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELHFEQUFxRCxDQUFDLElBQVU7UUFDOUQsUUFBUTtRQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksWUFBSyxFQUFFLENBQUM7UUFDMUIsTUFBTSxHQUFHLEdBQUcsSUFBSSxTQUFHLENBQUMsS0FBSyxFQUFFLEtBQUssQ0FBQyxDQUFDO1FBRWxDLE9BQU87UUFDUCxJQUFJLG1CQUFhLENBQUMsS0FBSyxFQUFFLEtBQUssRUFBRSxFQUFFLEdBQUcsRUFBRSxnQkFBZ0IsRUFBRSxLQUFLLEVBQUUsQ0FBQyxDQUFDO1FBRWxFLE9BQU87UUFDUCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMseUJBQXlCLEVBQUU7WUFDdkQsbUJBQW1CLEVBQUU7Z0JBQ25CO29CQUNFLE1BQU0sRUFBRSxvQkFBb0I7b0JBQzVCLFdBQVcsRUFBRSxzQkFBc0I7b0JBQ25DLFFBQVEsRUFBRSxHQUFHO29CQUNiLFVBQVUsRUFBRSxNQUFNO29CQUNsQixNQUFNLEVBQUUsRUFBRTtpQkFDWDthQUNGO1NBQ0YsQ0FBQyxDQUFDLENBQUM7UUFFSixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQseURBQXlELENBQUMsSUFBVTtRQUNsRSxRQUFRO1FBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztRQUMxQixNQUFNLEdBQUcsR0FBRyxJQUFJLFNBQUcsQ0FBQyxLQUFLLEVBQUUsS0FBSyxDQUFDLENBQUM7UUFFbEMsT0FBTztRQUNQLE1BQU0sRUFBRSxHQUFHLElBQUksbUJBQWEsQ0FBQyxLQUFLLEVBQUUsS0FBSyxFQUFFLEVBQUUsR0FBRyxFQUFFLGdCQUFnQixFQUFFLEtBQUssRUFBRSxDQUFDLENBQUM7UUFDN0UsRUFBRSxDQUFDLGFBQWEsQ0FBQyxVQUFJLENBQUMsT0FBTyxFQUFFLEVBQUUsVUFBSSxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUMsRUFBRSw2QkFBNkIsQ0FBQyxDQUFDO1FBRTlFLE9BQU87UUFDUCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMseUJBQXlCLEVBQUU7WUFDdkQsbUJBQW1CLEVBQUU7Z0JBQ25CO29CQUNFLE1BQU0sRUFBRSxXQUFXO29CQUNuQixXQUFXLEVBQUUsNkJBQTZCO29CQUMxQyxRQUFRLEVBQUUsRUFBRTtvQkFDWixVQUFVLEVBQUUsS0FBSztvQkFDakIsTUFBTSxFQUFFLEVBQUU7aUJBQ1g7YUFDRjtTQUNGLENBQUMsQ0FBQyxDQUFDO1FBRUosSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELGtEQUFrRCxDQUFDLElBQVU7UUFDM0QsUUFBUTtRQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksWUFBSyxFQUFFLENBQUM7UUFDMUIsTUFBTSxHQUFHLEdBQUcsSUFBSSxTQUFHLENBQUMsS0FBSyxFQUFFLEtBQUssQ0FBQyxDQUFDO1FBRWxDLE9BQU87UUFDUCxNQUFNLEVBQUUsR0FBRyxJQUFJLG1CQUFhLENBQUMsS0FBSyxFQUFFLEtBQUssRUFBRSxFQUFFLEdBQUcsRUFBRSxnQkFBZ0IsRUFBRSxLQUFLLEVBQUUsQ0FBQyxDQUFDO1FBQzdFLElBQUksQ0FBQyxNQUFNLENBQUMsR0FBRyxFQUFFO1lBQ2YsRUFBRSxDQUFDLGFBQWEsQ0FBQyxVQUFJLENBQUMsT0FBTyxFQUFFLEVBQUUsVUFBSSxDQUFDLFVBQVUsRUFBRSxFQUFFLGFBQWEsQ0FBQyxDQUFDO1FBQ3JFLENBQUMsRUFBRSxZQUFZLENBQUMsQ0FBQztRQUVqQixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsb0NBQW9DLENBQUMsSUFBVTtRQUM3QyxRQUFRO1FBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztRQUUxQixPQUFPO1FBQ1AsTUFBTSxFQUFFLEdBQUcsbUJBQWEsQ0FBQyxtQkFBbUIsQ0FBQyxLQUFLLEVBQUUsS0FBSyxFQUFFLFNBQVMsRUFBRSxFQUFDLE9BQU8sRUFBRSxLQUFLLEVBQUMsQ0FBQyxDQUFDO1FBQ3hGLEVBQUUsQ0FBQyxhQUFhLENBQUMsVUFBSSxDQUFDLE9BQU8sRUFBRSxFQUFFLFVBQUksQ0FBQyxHQUFHLENBQUMsRUFBRSxDQUFDLEVBQUUseUJBQXlCLENBQUMsQ0FBQztRQUMxRSxFQUFFLENBQUMsY0FBYyxDQUFDLFVBQUksQ0FBQyxPQUFPLEVBQUUsRUFBRSxVQUFJLENBQUMsR0FBRyxDQUFDLEVBQUUsQ0FBQyxFQUFFLHlCQUF5QixDQUFDLENBQUM7UUFFM0UsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxZQUFHLENBQUMscUJBQVksQ0FBQyx5QkFBeUIsRUFBRTtZQUMzRCxtQkFBbUIsRUFBRTtnQkFDbkI7b0JBQ0UsTUFBTSxFQUFFLFdBQVc7b0JBQ25CLFdBQVcsRUFBRSx5QkFBeUI7b0JBQ3RDLFFBQVEsRUFBRSxFQUFFO29CQUNaLFVBQVUsRUFBRSxLQUFLO29CQUNqQixNQUFNLEVBQUUsRUFBRTtpQkFDWDthQUNGO1NBQ0YsQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUVMLGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMsWUFBRyxDQUFDLHFCQUFZLENBQUMseUJBQXlCLEVBQUU7WUFDM0Qsb0JBQW9CLEVBQUU7Z0JBQ3BCO29CQUNFLE1BQU0sRUFBRSxXQUFXO29CQUNuQixXQUFXLEVBQUUseUJBQXlCO29CQUN0QyxRQUFRLEVBQUUsRUFBRTtvQkFDWixVQUFVLEVBQUUsS0FBSztvQkFDakIsTUFBTSxFQUFFLEVBQUU7aUJBQ1g7YUFDRjtTQUNGLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFFTCxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsc0RBQXNELENBQUMsSUFBVTtRQUMvRCxRQUFRO1FBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLENBQUMsU0FBUyxFQUFFLFdBQVcsRUFBRSxFQUFFLEdBQUcsRUFBRSxFQUFFLE9BQU8sRUFBRSxVQUFVLEVBQUUsTUFBTSxFQUFFLE9BQU8sRUFBRSxFQUFDLENBQUMsQ0FBQztRQUNsRyxNQUFNLEdBQUcsR0FBRyxJQUFJLFNBQUcsQ0FBQyxLQUFLLEVBQUUsS0FBSyxDQUFDLENBQUM7UUFDbEMsTUFBTSxFQUFFLEdBQUcsSUFBSSxtQkFBYSxDQUFDLEtBQUssRUFBRSxJQUFJLEVBQUUsRUFBRSxHQUFHLEVBQUUsQ0FBQyxDQUFDO1FBRW5ELE1BQU0sS0FBSyxHQUFHO1lBQ1osSUFBSSxtQkFBYSxDQUFDLEtBQUssRUFBRSxXQUFXLEVBQUUsRUFBRSxHQUFHLEVBQUUsQ0FBQztZQUM5QyxVQUFJLENBQUMsT0FBTyxFQUFFO1lBQ2QsVUFBSSxDQUFDLE9BQU8sRUFBRTtZQUNkLFVBQUksQ0FBQyxVQUFVLENBQUMsV0FBVyxDQUFDO1NBQzdCLENBQUM7UUFFRixNQUFNLEtBQUssR0FBRztZQUNaLFVBQUksQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDO1lBQ2QsVUFBSSxDQUFDLEdBQUcsQ0FBQyxXQUFJLENBQUMsV0FBVyxDQUFDLEVBQUUsT0FBTyxFQUFFLEdBQUcsRUFBRSxDQUFDLElBQUksRUFBRSxDQUFDLENBQUM7WUFDbkQsVUFBSSxDQUFDLE1BQU0sRUFBRTtZQUNiLFVBQUksQ0FBQyxRQUFRLENBQUMsRUFBRSxFQUFFLEVBQUUsQ0FBQztZQUNyQixVQUFJLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQztZQUNkLFVBQUksQ0FBQyxHQUFHLENBQUMsV0FBSSxDQUFDLFdBQVcsQ0FBQyxFQUFFLE9BQU8sRUFBRSxHQUFHLEVBQUUsQ0FBQyxJQUFJLEVBQUUsQ0FBQyxDQUFDO1lBQ25ELFVBQUksQ0FBQyxNQUFNLEVBQUU7WUFDYixVQUFJLENBQUMsUUFBUSxDQUFDLEVBQUUsRUFBRSxFQUFFLENBQUM7WUFDckIsVUFBSSxDQUFDLGVBQWUsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDO1lBQzFCLFVBQUksQ0FBQyxRQUFRLENBQUMsQ0FBQyxDQUFDO1lBQ2hCLFVBQUksQ0FBQyxPQUFPLEVBQUU7WUFDZCxVQUFJLENBQUMsUUFBUSxFQUFFO1lBQ2YsVUFBSSxDQUFDLFVBQVUsRUFBRTtTQUNsQixDQUFDO1FBRUYsT0FBTztRQUNQLEtBQUssTUFBTSxJQUFJLElBQUksS0FBSyxFQUFFO1lBQ3hCLEtBQUssTUFBTSxJQUFJLElBQUksS0FBSyxFQUFFO2dCQUN4QixFQUFFLENBQUMsV0FBVyxDQUFDLE9BQU8sQ0FBQyxJQUFJLEVBQUUsSUFBSSxDQUFDLENBQUM7Z0JBQ25DLEVBQUUsQ0FBQyxXQUFXLENBQUMsU0FBUyxDQUFDLElBQUksRUFBRSxJQUFJLENBQUMsQ0FBQzthQUN0QztTQUNGO1FBRUQsbUJBQW1CO1FBRW5CLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCw4RUFBOEUsQ0FBQyxJQUFVO1FBQ3ZGLFFBQVE7UUFDUixNQUFNLEVBQUUsR0FBRyxXQUFJLENBQUMsV0FBVyxDQUFDLEVBQUUsT0FBTyxFQUFFLEdBQUcsRUFBRSxDQUFDLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFDbkQsTUFBTSxFQUFFLEdBQUcsV0FBSSxDQUFDLFdBQVcsQ0FBQyxFQUFFLE9BQU8sRUFBRSxHQUFHLEVBQUUsQ0FBQyxJQUFJLEVBQUUsQ0FBQyxDQUFDO1FBRXJELE9BQU87UUFDUCxNQUFNLEtBQUssR0FBRztZQUNaLFVBQUksQ0FBQyxHQUFHLENBQUMsRUFBRSxDQUFDO1lBQ1osVUFBSSxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUM7WUFDWixVQUFJLENBQUMsUUFBUSxDQUFDLEVBQUUsRUFBRSxFQUFFLENBQUM7WUFDckIsVUFBSSxDQUFDLFFBQVEsQ0FBQyxFQUFFLEVBQUUsRUFBRSxDQUFDO1lBQ3JCLFVBQUksQ0FBQyxRQUFRLENBQUMsRUFBRSxFQUFFLEVBQUUsQ0FBQztZQUNyQixVQUFJLENBQUMsR0FBRyxDQUFDLEVBQUUsQ0FBQztZQUNaLFVBQUksQ0FBQyxRQUFRLENBQUMsRUFBRSxFQUFFLEVBQUUsQ0FBQztZQUNyQixVQUFJLENBQUMsUUFBUSxDQUFDLEVBQUUsRUFBRSxFQUFFLENBQUM7WUFDckIsVUFBSSxDQUFDLFFBQVEsQ0FBQyxFQUFFLEVBQUUsRUFBRSxDQUFDO1lBQ3JCLFVBQUksQ0FBQyxlQUFlLENBQUMsRUFBRSxFQUFFLENBQUMsQ0FBQztZQUMzQixVQUFJLENBQUMsZUFBZSxDQUFDLENBQUMsRUFBRSxFQUFFLENBQUM7WUFDM0IsVUFBSSxDQUFDLGVBQWUsQ0FBQyxFQUFFLEVBQUUsRUFBRSxDQUFDO1lBQzVCLFVBQUksQ0FBQyxRQUFRLENBQUMsRUFBRSxDQUFDO1NBQ2xCLENBQUM7UUFFRixPQUFPO1FBQ1AsS0FBSyxNQUFNLEtBQUssSUFBSSxLQUFLLEVBQUU7WUFDekIsSUFBSSxDQUFDLEtBQUssQ0FBQyxLQUFLLENBQUMsYUFBYSxFQUFFLEtBQUssRUFBRSxLQUFLLENBQUMsUUFBUSxFQUFFLENBQUMsQ0FBQztTQUMxRDtRQUVELElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCx5QkFBeUIsRUFBRTtRQUN6QixtQ0FBbUMsQ0FBQyxJQUFVO1lBQzVDLFFBQVE7WUFDUixNQUFNLE9BQU8sR0FBRyxDQUFDLFdBQVcsRUFBRSxvQkFBb0IsQ0FBQyxDQUFDO1lBRXBELE9BQU87WUFDUCxLQUFLLE1BQU0sTUFBTSxJQUFJLE9BQU8sRUFBRTtnQkFDNUIsSUFBSSxDQUFDLEtBQUssQ0FBQyxVQUFJLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxDQUFDLFFBQVEsRUFBRSxNQUFNLENBQUMsQ0FBQzthQUNoRDtZQUVELElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7UUFFRCxzQ0FBc0MsQ0FBQyxJQUFVO1lBQy9DLFFBQVE7WUFDUixZQUFLLENBQUMsUUFBUSxDQUFDLElBQUksZ0JBQVMsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDO1lBRXBDLG9CQUFvQjtZQUVwQixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO1FBRUQsbUNBQW1DLENBQUMsSUFBVTtZQUM1QyxPQUFPO1lBQ1AsSUFBSSxDQUFDLE1BQU0sQ0FBQyxHQUFHLEVBQUU7Z0JBQ2YsVUFBSSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsQ0FBQztZQUN2QixDQUFDLEVBQUUsbUJBQW1CLENBQUMsQ0FBQztZQUV4QixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO1FBRUQsMkNBQTJDLENBQUMsSUFBVTtZQUNwRCxJQUFJLENBQUMsTUFBTSxDQUFDLEdBQUcsRUFBRTtnQkFDZixVQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxDQUFDO1lBQ3ZCLENBQUMsRUFBRSw4QkFBOEIsQ0FBQyxDQUFDO1lBRW5DLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7UUFFRCxtQ0FBbUMsQ0FBQyxJQUFVO1lBQzVDLFFBQVE7WUFDUixNQUFNLE9BQU8sR0FBRztnQkFDZCxNQUFNO2dCQUNOLGVBQWU7Z0JBQ2YsNENBQTRDO2dCQUM1Qyw0QkFBNEI7YUFDN0IsQ0FBQztZQUVGLE9BQU87WUFDUCxLQUFLLE1BQU0sTUFBTSxJQUFJLE9BQU8sRUFBRTtnQkFDNUIsSUFBSSxDQUFDLEtBQUssQ0FBQyxVQUFJLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxDQUFDLFFBQVEsRUFBRSxNQUFNLENBQUMsQ0FBQzthQUNoRDtZQUVELElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7UUFFRCxtQ0FBbUMsQ0FBQyxJQUFVO1lBQzVDLE9BQU87WUFDUCxJQUFJLENBQUMsTUFBTSxDQUFDLEdBQUcsRUFBRTtnQkFDZixVQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxDQUFDO1lBQ3ZCLENBQUMsRUFBRSxtQkFBbUIsQ0FBQyxDQUFDO1lBRXhCLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7UUFFRCwyQ0FBMkMsQ0FBQyxJQUFVO1lBQ3BELElBQUksQ0FBQyxNQUFNLENBQUMsR0FBRyxFQUFFO2dCQUNmLFVBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7WUFDbEIsQ0FBQyxFQUFFLDZCQUE2QixDQUFDLENBQUM7WUFFbEMsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2QsQ0FBQztLQUNGO0NBQ0YsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IGV4cGVjdCwgaGF2ZVJlc291cmNlLCBub3QgfSBmcm9tICdAYXdzLWNkay9hc3NlcnQnO1xuaW1wb3J0IHsgSW50cmluc2ljLCBMYXp5LCBTdGFjaywgVG9rZW4gfSBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCB7IFRlc3QgfSBmcm9tICdub2RldW5pdCc7XG5pbXBvcnQgeyBQZWVyLCBQb3J0LCBTZWN1cml0eUdyb3VwLCBWcGMgfSBmcm9tIFwiLi4vbGliXCI7XG5cbmV4cG9ydCA9IHtcbiAgJ3NlY3VyaXR5IGdyb3VwIGNhbiBhbGxvd3MgYWxsIG91dGJvdW5kIHRyYWZmaWMgYnkgZGVmYXVsdCcodGVzdDogVGVzdCkge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgICBjb25zdCB2cGMgPSBuZXcgVnBjKHN0YWNrLCAnVlBDJyk7XG5cbiAgICAvLyBXSEVOXG4gICAgbmV3IFNlY3VyaXR5R3JvdXAoc3RhY2ssICdTRzEnLCB7IHZwYywgYWxsb3dBbGxPdXRib3VuZDogdHJ1ZSB9KTtcblxuICAgIC8vIFRIRU5cbiAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQVdTOjpFQzI6OlNlY3VyaXR5R3JvdXAnLCB7XG4gICAgICBTZWN1cml0eUdyb3VwRWdyZXNzOiBbXG4gICAgICAgIHtcbiAgICAgICAgICBDaWRySXA6IFwiMC4wLjAuMC8wXCIsXG4gICAgICAgICAgRGVzY3JpcHRpb246IFwiQWxsb3cgYWxsIG91dGJvdW5kIHRyYWZmaWMgYnkgZGVmYXVsdFwiLFxuICAgICAgICAgIElwUHJvdG9jb2w6IFwiLTFcIlxuICAgICAgICB9XG4gICAgICBdLFxuICAgIH0pKTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdubyBuZXcgb3V0Ym91bmQgcnVsZSBpcyBhZGRlZCBpZiB3ZSBhcmUgYWxsb3dpbmcgYWxsIHRyYWZmaWMgYW55d2F5Jyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuICAgIGNvbnN0IHZwYyA9IG5ldyBWcGMoc3RhY2ssICdWUEMnKTtcblxuICAgIC8vIFdIRU5cbiAgICBjb25zdCBzZyA9IG5ldyBTZWN1cml0eUdyb3VwKHN0YWNrLCAnU0cxJywgeyB2cGMsIGFsbG93QWxsT3V0Ym91bmQ6IHRydWUgfSk7XG4gICAgc2cuYWRkRWdyZXNzUnVsZShQZWVyLmFueUlwdjQoKSwgUG9ydC50Y3AoODYpLCAnVGhpcyBkb2VzIG5vdCBzaG93IHVwJyk7XG5cbiAgICAvLyBUSEVOXG4gICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2UoJ0FXUzo6RUMyOjpTZWN1cml0eUdyb3VwJywge1xuICAgICAgU2VjdXJpdHlHcm91cEVncmVzczogW1xuICAgICAgICB7XG4gICAgICAgICAgQ2lkcklwOiBcIjAuMC4wLjAvMFwiLFxuICAgICAgICAgIERlc2NyaXB0aW9uOiBcIkFsbG93IGFsbCBvdXRib3VuZCB0cmFmZmljIGJ5IGRlZmF1bHRcIixcbiAgICAgICAgICBJcFByb3RvY29sOiBcIi0xXCJcbiAgICAgICAgfSxcbiAgICAgIF0sXG4gICAgfSkpO1xuXG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ3NlY3VyaXR5IGdyb3VwIGRpc2FsbG93IG91dGJvdW5kIHRyYWZmaWMgYnkgZGVmYXVsdCcodGVzdDogVGVzdCkge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgICBjb25zdCB2cGMgPSBuZXcgVnBjKHN0YWNrLCAnVlBDJyk7XG5cbiAgICAvLyBXSEVOXG4gICAgbmV3IFNlY3VyaXR5R3JvdXAoc3RhY2ssICdTRzEnLCB7IHZwYywgYWxsb3dBbGxPdXRib3VuZDogZmFsc2UgfSk7XG5cbiAgICAvLyBUSEVOXG4gICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2UoJ0FXUzo6RUMyOjpTZWN1cml0eUdyb3VwJywge1xuICAgICAgU2VjdXJpdHlHcm91cEVncmVzczogW1xuICAgICAgICB7XG4gICAgICAgICAgQ2lkcklwOiBcIjI1NS4yNTUuMjU1LjI1NS8zMlwiLFxuICAgICAgICAgIERlc2NyaXB0aW9uOiBcIkRpc2FsbG93IGFsbCB0cmFmZmljXCIsXG4gICAgICAgICAgRnJvbVBvcnQ6IDI1MixcbiAgICAgICAgICBJcFByb3RvY29sOiBcImljbXBcIixcbiAgICAgICAgICBUb1BvcnQ6IDg2XG4gICAgICAgIH1cbiAgICAgIF0sXG4gICAgfSkpO1xuXG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ2JvZ3VzIG91dGJvdW5kIHJ1bGUgZGlzYXBwZWFycyBpZiBhbm90aGVyIHJ1bGUgaXMgYWRkZWQnKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG4gICAgY29uc3QgdnBjID0gbmV3IFZwYyhzdGFjaywgJ1ZQQycpO1xuXG4gICAgLy8gV0hFTlxuICAgIGNvbnN0IHNnID0gbmV3IFNlY3VyaXR5R3JvdXAoc3RhY2ssICdTRzEnLCB7IHZwYywgYWxsb3dBbGxPdXRib3VuZDogZmFsc2UgfSk7XG4gICAgc2cuYWRkRWdyZXNzUnVsZShQZWVyLmFueUlwdjQoKSwgUG9ydC50Y3AoODYpLCAnVGhpcyByZXBsYWNlcyB0aGUgb3RoZXIgb25lJyk7XG5cbiAgICAvLyBUSEVOXG4gICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2UoJ0FXUzo6RUMyOjpTZWN1cml0eUdyb3VwJywge1xuICAgICAgU2VjdXJpdHlHcm91cEVncmVzczogW1xuICAgICAgICB7XG4gICAgICAgICAgQ2lkcklwOiBcIjAuMC4wLjAvMFwiLFxuICAgICAgICAgIERlc2NyaXB0aW9uOiBcIlRoaXMgcmVwbGFjZXMgdGhlIG90aGVyIG9uZVwiLFxuICAgICAgICAgIEZyb21Qb3J0OiA4NixcbiAgICAgICAgICBJcFByb3RvY29sOiBcInRjcFwiLFxuICAgICAgICAgIFRvUG9ydDogODZcbiAgICAgICAgfVxuICAgICAgXSxcbiAgICB9KSk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnYWxsIG91dGJvdW5kIHJ1bGUgY2Fubm90IGJlIGFkZGVkIGFmdGVyIGNyZWF0aW9uJyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuICAgIGNvbnN0IHZwYyA9IG5ldyBWcGMoc3RhY2ssICdWUEMnKTtcblxuICAgIC8vIFdIRU5cbiAgICBjb25zdCBzZyA9IG5ldyBTZWN1cml0eUdyb3VwKHN0YWNrLCAnU0cxJywgeyB2cGMsIGFsbG93QWxsT3V0Ym91bmQ6IGZhbHNlIH0pO1xuICAgIHRlc3QudGhyb3dzKCgpID0+IHtcbiAgICAgIHNnLmFkZEVncmVzc1J1bGUoUGVlci5hbnlJcHY0KCksIFBvcnQuYWxsVHJhZmZpYygpLCAnQWxsIHRyYWZmaWMnKTtcbiAgICB9LCAvQ2Fubm90IGFkZC8pO1xuXG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ2ltbXV0YWJsZSBpbXBvcnRzIGRvIG5vdCBhZGQgcnVsZXMnKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG5cbiAgICAvLyBXSEVOXG4gICAgY29uc3Qgc2cgPSBTZWN1cml0eUdyb3VwLmZyb21TZWN1cml0eUdyb3VwSWQoc3RhY2ssICdTRzEnLCBcInRlc3QtaWRcIiwge211dGFibGU6IGZhbHNlfSk7XG4gICAgc2cuYWRkRWdyZXNzUnVsZShQZWVyLmFueUlwdjQoKSwgUG9ydC50Y3AoODYpLCAnVGhpcyBydWxlIHdhcyBub3QgYWRkZWQnKTtcbiAgICBzZy5hZGRJbmdyZXNzUnVsZShQZWVyLmFueUlwdjQoKSwgUG9ydC50Y3AoODYpLCAnVGhpcyBydWxlIHdhcyBub3QgYWRkZWQnKTtcblxuICAgIGV4cGVjdChzdGFjaykudG8obm90KGhhdmVSZXNvdXJjZSgnQVdTOjpFQzI6OlNlY3VyaXR5R3JvdXAnLCB7XG4gICAgICBTZWN1cml0eUdyb3VwRWdyZXNzOiBbXG4gICAgICAgIHtcbiAgICAgICAgICBDaWRySXA6IFwiMC4wLjAuMC8wXCIsXG4gICAgICAgICAgRGVzY3JpcHRpb246IFwiVGhpcyBydWxlIHdhcyBub3QgYWRkZWRcIixcbiAgICAgICAgICBGcm9tUG9ydDogODYsXG4gICAgICAgICAgSXBQcm90b2NvbDogXCJ0Y3BcIixcbiAgICAgICAgICBUb1BvcnQ6IDg2XG4gICAgICAgIH1cbiAgICAgIF0sXG4gICAgfSkpKTtcblxuICAgIGV4cGVjdChzdGFjaykudG8obm90KGhhdmVSZXNvdXJjZSgnQVdTOjpFQzI6OlNlY3VyaXR5R3JvdXAnLCB7XG4gICAgICBTZWN1cml0eUdyb3VwSW5ncmVzczogW1xuICAgICAgICB7XG4gICAgICAgICAgQ2lkcklwOiBcIjAuMC4wLjAvMFwiLFxuICAgICAgICAgIERlc2NyaXB0aW9uOiBcIlRoaXMgcnVsZSB3YXMgbm90IGFkZGVkXCIsXG4gICAgICAgICAgRnJvbVBvcnQ6IDg2LFxuICAgICAgICAgIElwUHJvdG9jb2w6IFwidGNwXCIsXG4gICAgICAgICAgVG9Qb3J0OiA4NlxuICAgICAgICB9XG4gICAgICBdLFxuICAgIH0pKSk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAncGVlciBiZXR3ZWVuIGFsbCB0eXBlcyBvZiBwZWVycyBhbmQgcG9ydCByYW5nZSB0eXBlcycodGVzdDogVGVzdCkge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2sodW5kZWZpbmVkLCAnVGVzdFN0YWNrJywgeyBlbnY6IHsgYWNjb3VudDogJzEyMzQ1Njc4JywgcmVnaW9uOiAnZHVtbXknIH19KTtcbiAgICBjb25zdCB2cGMgPSBuZXcgVnBjKHN0YWNrLCAnVlBDJyk7XG4gICAgY29uc3Qgc2cgPSBuZXcgU2VjdXJpdHlHcm91cChzdGFjaywgJ1NHJywgeyB2cGMgfSk7XG5cbiAgICBjb25zdCBwZWVycyA9IFtcbiAgICAgIG5ldyBTZWN1cml0eUdyb3VwKHN0YWNrLCAnUGVlckdyb3VwJywgeyB2cGMgfSksXG4gICAgICBQZWVyLmFueUlwdjQoKSxcbiAgICAgIFBlZXIuYW55SXB2NigpLFxuICAgICAgUGVlci5wcmVmaXhMaXN0KCdwbC0wMTIzNDUnKSxcbiAgICBdO1xuXG4gICAgY29uc3QgcG9ydHMgPSBbXG4gICAgICBQb3J0LnRjcCgxMjM0KSxcbiAgICAgIFBvcnQudGNwKExhenkubnVtYmVyVmFsdWUoeyBwcm9kdWNlOiAoKSA9PiA1MDAwIH0pKSxcbiAgICAgIFBvcnQuYWxsVGNwKCksXG4gICAgICBQb3J0LnRjcFJhbmdlKDgwLCA5MCksXG4gICAgICBQb3J0LnVkcCgyMzQ1KSxcbiAgICAgIFBvcnQudWRwKExhenkubnVtYmVyVmFsdWUoeyBwcm9kdWNlOiAoKSA9PiA3Nzc3IH0pKSxcbiAgICAgIFBvcnQuYWxsVWRwKCksXG4gICAgICBQb3J0LnVkcFJhbmdlKDg1LCA5NSksXG4gICAgICBQb3J0LmljbXBUeXBlQW5kQ29kZSg1LCAxKSxcbiAgICAgIFBvcnQuaWNtcFR5cGUoOCksXG4gICAgICBQb3J0LmFsbEljbXAoKSxcbiAgICAgIFBvcnQuaWNtcFBpbmcoKSxcbiAgICAgIFBvcnQuYWxsVHJhZmZpYygpXG4gICAgXTtcblxuICAgIC8vIFdIRU5cbiAgICBmb3IgKGNvbnN0IHBlZXIgb2YgcGVlcnMpIHtcbiAgICAgIGZvciAoY29uc3QgcG9ydCBvZiBwb3J0cykge1xuICAgICAgICBzZy5jb25uZWN0aW9ucy5hbGxvd1RvKHBlZXIsIHBvcnQpO1xuICAgICAgICBzZy5jb25uZWN0aW9ucy5hbGxvd0Zyb20ocGVlciwgcG9ydCk7XG4gICAgICB9XG4gICAgfVxuXG4gICAgLy8gVEhFTiAtLSBubyBjcmFzaFxuXG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ2lmIHRva2VucyBhcmUgdXNlZCBpbiBwb3J0cywgYGNhbklubGluZVJ1bGVgIHNob3VsZCBiZSBmYWxzZSB0byBhdm9pZCBjeWNsZXMnKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IHAxID0gTGF6eS5udW1iZXJWYWx1ZSh7IHByb2R1Y2U6ICgpID0+IDgwIH0pO1xuICAgIGNvbnN0IHAyID0gTGF6eS5udW1iZXJWYWx1ZSh7IHByb2R1Y2U6ICgpID0+IDUwMDAgfSk7XG5cbiAgICAvLyBXSEVOXG4gICAgY29uc3QgcG9ydHMgPSBbXG4gICAgICBQb3J0LnRjcChwMSksXG4gICAgICBQb3J0LnRjcChwMiksXG4gICAgICBQb3J0LnRjcFJhbmdlKHAxLCA5MCksXG4gICAgICBQb3J0LnRjcFJhbmdlKDgwLCBwMiksXG4gICAgICBQb3J0LnRjcFJhbmdlKHAxLCBwMiksXG4gICAgICBQb3J0LnVkcChwMSksXG4gICAgICBQb3J0LnVkcFJhbmdlKHAxLCA5NSksXG4gICAgICBQb3J0LnVkcFJhbmdlKDg1LCBwMiksXG4gICAgICBQb3J0LnVkcFJhbmdlKHAxLCBwMiksXG4gICAgICBQb3J0LmljbXBUeXBlQW5kQ29kZShwMSwgMSksXG4gICAgICBQb3J0LmljbXBUeXBlQW5kQ29kZSg1LCBwMSksXG4gICAgICBQb3J0LmljbXBUeXBlQW5kQ29kZShwMSwgcDIpLFxuICAgICAgUG9ydC5pY21wVHlwZShwMSksXG4gICAgXTtcblxuICAgIC8vIFRIRU5cbiAgICBmb3IgKGNvbnN0IHJhbmdlIG9mIHBvcnRzKSB7XG4gICAgICB0ZXN0LmVxdWFsKHJhbmdlLmNhbklubGluZVJ1bGUsIGZhbHNlLCByYW5nZS50b1N0cmluZygpKTtcbiAgICB9XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnUGVlciBJUCBDSURSIHZhbGlkYXRpb24nOiB7XG4gICAgJ3Bhc3NlcyB3aXRoIHZhbGlkIElQdjQgQ0lEUiBibG9jaycodGVzdDogVGVzdCkge1xuICAgICAgLy8gR0lWRU5cbiAgICAgIGNvbnN0IGNpZHJJcHMgPSBbJzAuMC4wLjAvMCcsICcxOTIuMTY4LjI1NS4yNTUvMjQnXTtcblxuICAgICAgLy8gVEhFTlxuICAgICAgZm9yIChjb25zdCBjaWRySXAgb2YgY2lkcklwcykge1xuICAgICAgICB0ZXN0LmVxdWFsKFBlZXIuaXB2NChjaWRySXApLnVuaXF1ZUlkLCBjaWRySXApO1xuICAgICAgfVxuXG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9LFxuXG4gICAgJ3Bhc3NlcyB3aXRoIHVucmVzb2x2ZWQgSVAgQ0lEUiB0b2tlbicodGVzdDogVGVzdCkge1xuICAgICAgLy8gR0lWRU5cbiAgICAgIFRva2VuLmFzU3RyaW5nKG5ldyBJbnRyaW5zaWMoJ2lwJykpO1xuXG4gICAgICAvLyBUSEVOOiBkb24ndCB0aHJvd1xuXG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9LFxuXG4gICAgJ3Rocm93cyBpZiBpbnZhbGlkIElQdjQgQ0lEUiBibG9jaycodGVzdDogVGVzdCkge1xuICAgICAgLy8gVEhFTlxuICAgICAgdGVzdC50aHJvd3MoKCkgPT4ge1xuICAgICAgICBQZWVyLmlwdjQoJ2ludmFsaWQnKTtcbiAgICAgIH0sIC9JbnZhbGlkIElQdjQgQ0lEUi8pO1xuXG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9LFxuXG4gICAgJ3Rocm93cyBpZiBtaXNzaW5nIG1hc2sgaW4gSVB2NCBDSURSIGJsb2NrJyh0ZXN0OiBUZXN0KSB7XG4gICAgICB0ZXN0LnRocm93cygoKSA9PiB7XG4gICAgICAgIFBlZXIuaXB2NCgnMC4wLjAuMCcpO1xuICAgICAgfSwgL0NJRFIgbWFzayBpcyBtaXNzaW5nIGluIElQdjQvKTtcblxuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfSxcblxuICAgICdwYXNzZXMgd2l0aCB2YWxpZCBJUHY2IENJRFIgYmxvY2snKHRlc3Q6IFRlc3QpIHtcbiAgICAgIC8vIEdJVkVOXG4gICAgICBjb25zdCBjaWRySXBzID0gW1xuICAgICAgICAnOjovMCcsXG4gICAgICAgICcyMDAxOmRiODo6LzMyJyxcbiAgICAgICAgJzIwMDE6MGRiODowMDAwOjAwMDA6MDAwMDo4YTJlOjAzNzA6NzMzNC8zMicsXG4gICAgICAgICcyMDAxOmRiODo6OGEyZTozNzA6NzMzNC8zMicsXG4gICAgICBdO1xuXG4gICAgICAvLyBUSEVOXG4gICAgICBmb3IgKGNvbnN0IGNpZHJJcCBvZiBjaWRySXBzKSB7XG4gICAgICAgIHRlc3QuZXF1YWwoUGVlci5pcHY2KGNpZHJJcCkudW5pcXVlSWQsIGNpZHJJcCk7XG4gICAgICB9XG5cbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH0sXG5cbiAgICAndGhyb3dzIGlmIGludmFsaWQgSVB2NiBDSURSIGJsb2NrJyh0ZXN0OiBUZXN0KSB7XG4gICAgICAvLyBUSEVOXG4gICAgICB0ZXN0LnRocm93cygoKSA9PiB7XG4gICAgICAgIFBlZXIuaXB2NignaW52YWxpZCcpO1xuICAgICAgfSwgL0ludmFsaWQgSVB2NiBDSURSLyk7XG5cbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH0sXG5cbiAgICAndGhyb3dzIGlmIG1pc3NpbmcgbWFzayBpbiBJUHY2IENJRFIgYmxvY2snKHRlc3Q6IFRlc3QpIHtcbiAgICAgIHRlc3QudGhyb3dzKCgpID0+IHtcbiAgICAgICAgUGVlci5pcHY2KCc6OicpO1xuICAgICAgfSwgL0lEUiBtYXNrIGlzIG1pc3NpbmcgaW4gSVB2Ni8pO1xuXG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9XG4gIH1cbn07XG4iXX0=