"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const vpc_1 = require("./vpc");
/**
 * Turn an arbitrary string into one that can be used as a CloudFormation identifier by stripping special characters
 *
 * (At the moment, no efforts are taken to prevent collissions, but we can add that later when it becomes necessary).
 */
function slugify(x) {
    return x.replace(/[^a-zA-Z0-9]/g, '');
}
exports.slugify = slugify;
/**
 * The default names for every subnet type
 */
function defaultSubnetName(type) {
    switch (type) {
        case vpc_1.SubnetType.PUBLIC: return 'Public';
        case vpc_1.SubnetType.PRIVATE: return 'Private';
        case vpc_1.SubnetType.ISOLATED: return 'Isolated';
    }
}
exports.defaultSubnetName = defaultSubnetName;
/**
 * Return a subnet name from its construct ID
 *
 * All subnet names look like NAME <> "Subnet" <> INDEX
 */
function subnetGroupNameFromConstructId(subnet) {
    return subnet.node.id.replace(/Subnet\d+$/, '');
}
exports.subnetGroupNameFromConstructId = subnetGroupNameFromConstructId;
/**
 * Make the subnet construct ID from a name and number
 */
function subnetId(name, i) {
    return `${name}Subnet${i + 1}`;
}
exports.subnetId = subnetId;
class ImportSubnetGroup {
    constructor(subnetIds, names, routeTableIds, type, availabilityZones, idField, nameField, routeTableIdField) {
        this.availabilityZones = availabilityZones;
        this.subnetIds = subnetIds || [];
        this.routeTableIds = routeTableIds || [];
        this.groups = this.subnetIds.length / this.availabilityZones.length;
        if (Math.floor(this.groups) !== this.groups) {
            // tslint:disable-next-line:max-line-length
            throw new Error(`Number of ${idField} (${this.subnetIds.length}) must be a multiple of availability zones (${this.availabilityZones.length}).`);
        }
        if (this.routeTableIds.length !== this.subnetIds.length && routeTableIds != null) {
            // We don't err if no routeTableIds were provided to maintain backwards-compatibility. See https://github.com/aws/aws-cdk/pull/3171
            // tslint:disable-next-line: max-line-length
            throw new Error(`Number of ${routeTableIdField} (${this.routeTableIds.length}) must be equal to the amount of ${idField} (${this.subnetIds.length}).`);
        }
        this.names = this.normalizeNames(names, defaultSubnetName(type), nameField);
    }
    import(scope) {
        return range(this.subnetIds.length).map(i => {
            const k = Math.floor(i / this.availabilityZones.length);
            return vpc_1.Subnet.fromSubnetAttributes(scope, subnetId(this.names[k], i), {
                availabilityZone: this.pickAZ(i),
                subnetId: this.subnetIds[i],
                routeTableId: this.routeTableIds[i],
            });
        });
    }
    /**
     * Return a list with a name for every subnet
     */
    normalizeNames(names, defaultName, fieldName) {
        // If not given, return default
        if (names === undefined || names.length === 0) {
            return [defaultName];
        }
        // If given, must match given subnets
        if (names.length !== this.groups) {
            throw new Error(`${fieldName} must have an entry for every corresponding subnet group, got: ${JSON.stringify(names)}`);
        }
        return names;
    }
    /**
     * Return the i'th AZ
     */
    pickAZ(i) {
        return this.availabilityZones[i % this.availabilityZones.length];
    }
}
exports.ImportSubnetGroup = ImportSubnetGroup;
/**
 * Generate the list of numbers of [0..n)
 */
function range(n) {
    const ret = [];
    for (let i = 0; i < n; i++) {
        ret.push(i);
    }
    return ret;
}
exports.range = range;
/**
 * Return the union of table IDs from all selected subnets
 */
function allRouteTableIds(...ssns) {
    const ret = new Set();
    for (const ssn of ssns) {
        for (const subnet of ssn.subnets) {
            if (subnet.routeTable && subnet.routeTable.routeTableId) {
                ret.add(subnet.routeTable.routeTableId);
            }
        }
    }
    return Array.from(ret);
}
exports.allRouteTableIds = allRouteTableIds;
//# sourceMappingURL=data:application/json;base64,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