"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const ec2_generated_1 = require("./ec2.generated");
/**
 * A NetworkAclBase that is not created in this template
 *
 * @experimental
 */
class NetworkAclBase extends core_1.Resource {
    /**
     * Add a new entry to the ACL
     */
    addEntry(id, options) {
        return new NetworkAclEntry(this, id, {
            networkAcl: this,
            ...options
        });
    }
}
/**
 * Define a new custom network ACL
 *
 * By default, will deny all inbound and outbound traffic unless entries are
 * added explicitly allowing it.
 *
 * @experimental
 */
class NetworkAcl extends NetworkAclBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.networkAclName
        });
        this.vpc = props.vpc;
        this.networkAcl = new ec2_generated_1.CfnNetworkAcl(this, 'Resource', {
            vpcId: props.vpc.vpcId,
        });
        this.networkAclId = this.networkAcl.ref;
        this.networkAclVpcId = this.networkAcl.vpcId;
        if (props.subnetSelection !== undefined) {
            this.associateWithSubnet('DefaultAssociation', props.subnetSelection);
        }
    }
    /**
     * Import an existing NetworkAcl into this app.
     */
    static fromNetworkAclId(scope, id, networkAclId) {
        class Import extends NetworkAclBase {
            constructor() {
                super(...arguments);
                this.networkAclId = networkAclId;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Associate the ACL with a given set of subnets
     */
    associateWithSubnet(id, selection) {
        const subnets = this.vpc.selectSubnets(selection);
        for (const subnet of subnets.subnets) {
            subnet.associateNetworkAcl(id, this);
        }
    }
}
exports.NetworkAcl = NetworkAcl;
/**
 * What action to apply to traffic matching the ACL
 *
 * @experimental
 */
var Action;
(function (Action) {
    /**
     * Allow the traffic
     */
    Action["ALLOW"] = "allow";
    /**
     * Deny the traffic
     */
    Action["DENY"] = "deny";
})(Action = exports.Action || (exports.Action = {}));
/**
 * Base class for NetworkAclEntries
 *
 * @experimental
 */
class NetworkAclEntryBase extends core_1.Resource {
}
/**
 * Direction of traffic the AclEntry applies to
 *
 * @experimental
 */
var TrafficDirection;
(function (TrafficDirection) {
    /**
     * Traffic leaving the subnet
     */
    TrafficDirection[TrafficDirection["EGRESS"] = 0] = "EGRESS";
    /**
     * Traffic entering the subnet
     */
    TrafficDirection[TrafficDirection["INGRESS"] = 1] = "INGRESS";
})(TrafficDirection = exports.TrafficDirection || (exports.TrafficDirection = {}));
/**
 * Define an entry in a Network ACL table
 *
 * @experimental
 */
class NetworkAclEntry extends NetworkAclEntryBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.networkAclEntryName
        });
        this.networkAcl = props.networkAcl;
        new ec2_generated_1.CfnNetworkAclEntry(this, 'Resource', {
            networkAclId: this.networkAcl.networkAclId,
            ruleNumber: props.ruleNumber,
            ruleAction: props.ruleAction !== undefined ? props.ruleAction : Action.ALLOW,
            egress: props.direction !== undefined ? props.direction === TrafficDirection.EGRESS : undefined,
            ...props.traffic.toTrafficConfig(),
            ...props.cidr.toCidrConfig(),
        });
    }
}
exports.NetworkAclEntry = NetworkAclEntry;
/**
 * Associate a network ACL with a subnet
 *
 * @experimental
 */
class SubnetNetworkAclAssociationBase extends core_1.Resource {
}
class SubnetNetworkAclAssociation extends SubnetNetworkAclAssociationBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.subnetNetworkAclAssociationName
        });
        this.association = new ec2_generated_1.CfnSubnetNetworkAclAssociation(this, 'Resource', {
            networkAclId: props.networkAcl.networkAclId,
            subnetId: props.subnet.subnetId
        });
        this.networkAcl = props.networkAcl;
        this.subnet = props.subnet;
        this.subnetNetworkAclAssociationAssociationId = this.association.attrAssociationId;
    }
    static fromSubnetNetworkAclAssociationAssociationId(scope, id, subnetNetworkAclAssociationAssociationId) {
        class Import extends SubnetNetworkAclAssociationBase {
            constructor() {
                super(...arguments);
                this.subnetNetworkAclAssociationAssociationId = subnetNetworkAclAssociationAssociationId;
            }
        }
        return new Import(scope, id);
    }
}
exports.SubnetNetworkAclAssociation = SubnetNetworkAclAssociation;
//# sourceMappingURL=data:application/json;base64,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