"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * Either an IPv4 or an IPv6 CIDR
 *
 * @experimental
 */
class AclCidr {
    /**
     * An IP network range in CIDR notation (for example, 172.16.0.0/24).
     */
    static ipv4(ipv4Cidr) {
        return new AclCidrImpl({
            cidrBlock: ipv4Cidr
        });
    }
    /**
     * The CIDR containing all IPv4 addresses (i.e., 0.0.0.0/0)
     */
    static anyIpv4() {
        return AclCidr.ipv4('0.0.0.0/0');
    }
    /**
     * An IPv6 network range in CIDR notation (for example, 2001:db8::/48)
     */
    static ipv6(ipv6Cidr) {
        return new AclCidrImpl({
            ipv6CidrBlock: ipv6Cidr
        });
    }
    /**
     * The CIDR containing all IPv6 addresses (i.e., ::/0)
     */
    static anyIpv6() {
        return AclCidr.ipv6('::/0');
    }
}
exports.AclCidr = AclCidr;
class AclCidrImpl extends AclCidr {
    constructor(config) {
        super();
        this.config = config;
    }
    toCidrConfig() {
        return this.config;
    }
}
/**
 * The traffic that is configured using a Network ACL entry
 *
 * @experimental
 */
class AclTraffic {
    /**
     * Apply the ACL entry to all traffic
     */
    static allTraffic() {
        return new AclTrafficImpl({
            protocol: -1
        });
    }
    /**
     * Apply the ACL entry to ICMP traffic of given type and code
     */
    static icmp(props) {
        return new AclTrafficImpl({
            protocol: 1,
            icmp: props
        });
    }
    /**
     * Apply the ACL entry to ICMPv6 traffic of given type and code
     *
     * Requires an IPv6 CIDR block.
     */
    static icmpv6(props) {
        return new AclTrafficImpl({
            protocol: 58,
            icmp: props
        });
    }
    /**
     * Apply the ACL entry to TCP traffic on a given port
     */
    static tcpPort(port) {
        return new AclTrafficImpl({
            protocol: 6,
            portRange: {
                from: port,
                to: port
            }
        });
    }
    /**
     * Apply the ACL entry to TCP traffic on a given port range
     */
    static tcpPortRange(startPort, endPort) {
        return new AclTrafficImpl({
            protocol: 6,
            portRange: {
                from: startPort,
                to: endPort,
            }
        });
    }
    /**
     * Apply the ACL entry to UDP traffic on a given port
     */
    static udpPort(port) {
        return new AclTrafficImpl({
            protocol: 17,
            portRange: {
                from: port,
                to: port
            }
        });
    }
    /**
     * Apply the ACL entry to UDP traffic on a given port range
     */
    static udpPortRange(startPort, endPort) {
        return new AclTrafficImpl({
            protocol: 17,
            portRange: {
                from: startPort,
                to: endPort,
            }
        });
    }
}
exports.AclTraffic = AclTraffic;
class AclTrafficImpl extends AclTraffic {
    constructor(config) {
        super();
        this.config = config;
    }
    toTrafficConfig() {
        return this.config;
    }
}
//# sourceMappingURL=data:application/json;base64,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