import { InstanceType } from './instance-types';
import { IMachineImage, LookupMachineImage } from "./machine-image";
import { PrivateSubnet, PublicSubnet, Vpc } from './vpc';
/**
 * Pair represents a gateway created by NAT Provider
 */
export interface GatewayConfig {
    /**
     * Availability Zone
     */
    readonly az: string;
    /**
     * Identity of gateway spawned by the provider
     */
    readonly gatewayId: string;
}
/**
 * NAT providers
 *
 * Determines what type of NAT provider to create, either NAT gateways or NAT
 * instance.
 *
 * @experimental
 */
export declare abstract class NatProvider {
    /**
     * Use NAT Gateways to provide NAT services for your VPC
     *
     * NAT gateways are managed by AWS.
     *
     * @see https://docs.aws.amazon.com/vpc/latest/userguide/vpc-nat-gateway.html
     */
    static gateway(): NatProvider;
    /**
     * Use NAT instances to provide NAT services for your VPC
     *
     * NAT instances are managed by you, but in return allow more configuration.
     *
     * Be aware that instances created using this provider will not be
     * automatically replaced if they are stopped for any reason. You should implement
     * your own NatProvider based on AutoScaling groups if you need that.
     *
     * @see https://docs.aws.amazon.com/vpc/latest/userguide/VPC_NAT_Instance.html
     */
    static instance(props: NatInstanceProps): NatProvider;
    /**
     * Return list of gateways spawned by the provider
     */
    abstract readonly configuredGateways: GatewayConfig[];
    /**
     * Called by the VPC to configure NAT
     */
    abstract configureNat(options: ConfigureNatOptions): void;
    /**
     * Configures subnet with the gateway
     */
    abstract configureSubnet(subnet: PrivateSubnet): void;
}
/**
 * Options passed by the VPC when NAT needs to be configured
 *
 * @experimental
 */
export interface ConfigureNatOptions {
    /**
     * The VPC we're configuring NAT for
     */
    readonly vpc: Vpc;
    /**
     * The public subnets where the NAT providers need to be placed
     */
    readonly natSubnets: PublicSubnet[];
    /**
     * The private subnets that need to route through the NAT providers.
     *
     * There may be more private subnets than public subnets with NAT providers.
     */
    readonly privateSubnets: PrivateSubnet[];
}
/**
 * Properties for a NAT instance
 *
 * @experimental
 */
export interface NatInstanceProps {
    /**
     * The machine image (AMI) to use
     *
     * By default, will do an AMI lookup for the latest NAT instance image.
     *
     * If you have a specific AMI ID you want to use, pass a `GenericLinuxImage`. For example:
     *
     * ```ts
     * ec2.NatProvider.instance({
     *   instanceType: new ec2.InstanceType('t3.micro'),
     *   machineImage: new ec2.GenericLinuxImage({
     *     'us-east-2': 'ami-0f9c61b5a562a16af'
     *   })
     * })
     * ```
     *
     * @default - Latest NAT instance image
     */
    readonly machineImage?: IMachineImage;
    /**
     * Instance type of the NAT instance
     */
    readonly instanceType: InstanceType;
    /**
     * Name of SSH keypair to grant access to instance
     *
     * @default - No SSH access will be possible.
     */
    readonly keyName?: string;
}
/**
 * Machine image representing the latest NAT instance image
 *
 * @experimental
 */
export declare class NatInstanceImage extends LookupMachineImage {
    constructor();
}
