"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const _1 = require(".");
const instance_1 = require("./instance");
const machine_image_1 = require("./machine-image");
const port_1 = require("./port");
/**
 * This creates a linux bastion host you can use to connect to other instances or services in your VPC.
 * The recommended way to connect to the bastion host is by using AWS Systems Manager Session Manager.
 *
 * The operating system is Amazon Linux 2 with the latest SSM agent installed
 *
 * You can also configure this bastion host to allow connections via SSH
 *
 * @experimental
 */
class BastionHostLinux extends core_1.Construct {
    constructor(scope, id, props) {
        var _a, _b, _c, _d;
        super(scope, id);
        this.stack = core_1.Stack.of(scope);
        this.instance = new instance_1.Instance(this, 'Resource', {
            vpc: props.vpc,
            availabilityZone: props.availabilityZone,
            securityGroup: props.securityGroup,
            instanceName: (_a = props.instanceName) !== null && _a !== void 0 ? _a : 'BastionHost',
            instanceType: (_b = props.instanceType) !== null && _b !== void 0 ? _b : _1.InstanceType.of(_1.InstanceClass.T3, _1.InstanceSize.NANO),
            machineImage: (_c = props.machineImage) !== null && _c !== void 0 ? _c : machine_image_1.MachineImage.latestAmazonLinux({ generation: _1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
            vpcSubnets: (_d = props.subnetSelection) !== null && _d !== void 0 ? _d : {},
        });
        this.instance.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: [
                'ssmmessages:*',
                'ssm:UpdateInstanceInformation',
                'ec2messages:*'
            ],
            resources: ['*'],
        }));
        this.instance.addUserData('yum install -y https://s3.amazonaws.com/ec2-downloads-windows/SSMAgent/latest/linux_amd64/amazon-ssm-agent.rpm');
        this.connections = this.instance.connections;
        this.role = this.instance.role;
        this.grantPrincipal = this.instance.role;
        this.instanceId = this.instance.instanceId;
        this.instancePrivateIp = this.instance.instancePrivateIp;
        this.instanceAvailabilityZone = this.instance.instanceAvailabilityZone;
        this.instancePrivateDnsName = this.instance.instancePrivateDnsName;
        this.instancePublicIp = this.instance.instancePublicIp;
        this.instancePublicDnsName = this.instance.instancePublicDnsName;
        new core_1.CfnOutput(this, 'BastionHostId', {
            description: 'Instance ID of the bastion host. Use this to connect via SSM Session Manager',
            value: this.instanceId,
        });
    }
    /**
     * Allow SSH access from the given peer or peers
     *
     * Necessary if you want to connect to the instance using ssh. If not
     * called, you should use SSM Session Manager to connect to the instance.
     */
    allowSshAccessFrom(...peer) {
        peer.forEach(p => {
            this.connections.allowFrom(p, port_1.Port.tcp(22), 'SSH access');
        });
    }
}
exports.BastionHostLinux = BastionHostLinux;
//# sourceMappingURL=data:application/json;base64,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