"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Schema = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const yaml = require("js-yaml");
const lodash_1 = require("lodash");
const asset_1 = require("./asset");
const version_1 = require("./version");
// eslint-disable-next-line @typescript-eslint/no-require-imports
const omitDeep = require('omit-deep-lodash');
/**
 * Represents an OpenApi v3 Schema which can be deserialized from YAML-file, modified
 * and then serialized back to YAML.
 */
class Schema {
    /** Construct a new Schema instance from OpenApi v3 JSON.  */
    constructor(props) {
        this.document = props;
        this.openApiVersion = version_1.getValidVersion(Schema.openApiSupportedVersions, this.document.openapi);
    }
    /** Parse OpenApi v3 schema from inline YAML content. */
    static fromInline(content) {
        const schemaJson = yaml.load(content);
        return new Schema(schemaJson);
    }
    /** Parse OpenApi v3 schema by loading a YAML file from given path. */
    static fromAsset(path) {
        const schema = fs.readFileSync(path, 'utf-8');
        return Schema.fromInline(schema);
    }
    /** Serialize to YAML string. */
    toYaml() {
        return yaml.dump(this.document);
    }
    /** Serialize to JSON string. */
    toJson() {
        return JSON.stringify(this.document);
    }
    /** Return the actual OpenApi v3 document. */
    toDocument() {
        return this.document;
    }
    /** Return the OpenApi v3 document as an S3 Asset. */
    toAsset(scope, id) {
        return new asset_1.SchemaAsset(scope, id, this.document);
    }
    /** Check if definition has a value in given object path. */
    has(path) {
        return lodash_1.has(this.document, path);
    }
    /** Get a value from given object path. */
    get(path) {
        return lodash_1.get(this.document, path);
    }
    /** Set a value to given object path. */
    set(path, value) {
        lodash_1.set(this.document, path, value);
    }
    /** Inject multiple values to given paths. */
    inject(records = {}) {
        Object.keys(records).forEach(path => lodash_1.set(this.document, path, records[path]));
    }
    /** Reject – i.e. remove values – from given object paths. */
    reject(paths = []) {
        paths.forEach(path => lodash_1.unset(this.document, path));
    }
    /** Reject deep within object – i.e. remove all nested object paths. */
    rejectDeep(paths = []) {
        this.document = omitDeep(this.document, ...paths);
    }
}
exports.Schema = Schema;
_a = JSII_RTTI_SYMBOL_1;
Schema[_a] = { fqn: "@alma-cdk/openapix.Schema", version: "0.0.49" };
/**
 * A string representing supported SemVer range.
 * @see https://github.com/npm/node-semver
 */
Schema.openApiSupportedVersions = '3.0.x';
//# sourceMappingURL=data:application/json;base64,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