"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const aws_cdk_lib_1 = require("aws-cdk-lib");
const cognito = require("aws-cdk-lib/aws-cognito");
const dynamodb = require("aws-cdk-lib/aws-dynamodb");
const iam = require("aws-cdk-lib/aws-iam");
const lambda = require("aws-cdk-lib/aws-lambda");
const _1 = require(".");
const __1 = require("../..");
const api_1 = require("../../api");
describe('SchemaAsset', () => {
    test('Token Resolving', () => {
        const stack = new aws_cdk_lib_1.Stack();
        const userPool = new cognito.UserPool(stack, 'MyUserPool');
        const handler = new lambda.Function(stack, 'MyFunction', {
            runtime: lambda.Runtime.NODEJS_12_X,
            handler: 'index.handler',
            code: lambda.Code.fromInline(`module.exports = {
          handler: async (event) => {
            console.log(event);
            return {
              statusCode: 200,
              body: JSON.stringify({
                message: 'Hello',
              }),
            };
          },
        }`),
        });
        const pkName = 'item';
        const table = new dynamodb.Table(stack, 'MyTable', {
            partitionKey: {
                type: dynamodb.AttributeType.STRING,
                name: pkName,
            },
        });
        const role = new iam.Role(stack, 'MyRole', {
            assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com'),
        });
        table.grantReadData(role);
        const schema = new __1.Schema({
            openapi: '3.0.1',
            info: {
                title: 'Test Api',
                version: '0.0.0',
            },
            paths: {
                '/item': {
                    post: {
                        operationId: 'postItem',
                        responses: {},
                        security: [
                            {
                                MyCognitoAuthorizer: ['test/write'],
                            },
                        ],
                    },
                    get: {
                        operationId: 'getItem',
                        responses: {},
                    },
                },
            },
            components: {
                securitySchemes: {
                    MyCognitoAuthorizer: {
                        type: 'apiKey',
                        name: 'Authorization',
                        in: 'header',
                    },
                },
            },
        });
        const { document } = new api_1.Api(stack, 'Api', {
            source: schema,
            authorizers: [
                new __1.CognitoUserPoolsAuthorizer(stack, 'MyCognitoAuthorizer', {
                    cognitoUserPools: [userPool],
                    resultsCacheTtl: aws_cdk_lib_1.Duration.minutes(5),
                }),
            ],
            paths: {
                '/item': {
                    post: new __1.LambdaIntegration(stack, handler),
                    get: new __1.AwsIntegration(stack, {
                        validator: 'params-only',
                        service: 'dynamodb',
                        action: 'GetItem',
                        options: {
                            credentialsRole: role,
                            requestTemplates: {
                                'application/json': JSON.stringify({
                                    TableName: table.tableName,
                                    Key: {
                                        [pkName]: {
                                            S: "$input.params('item')",
                                        },
                                    },
                                }),
                            },
                        },
                    }),
                },
            },
        });
        const asset = new _1.SchemaAsset(stack, 'asset', document);
        expect(asset.cloudFormationDocument).toMatchObject({
            info: { title: 'Test Api', version: '0.0.0' },
            openapi: '3.0.1',
            paths: {
                '/item': {
                    post: {
                        'operationId': 'postItem',
                        'responses': {},
                        'security': [{ MyCognitoAuthorizer: ['test/write'] }],
                        'x-amazon-apigateway-integration': {
                            httpMethod: 'POST',
                            responses: {},
                            type: 'AWS_PROXY',
                            uri: {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        { Ref: 'AWS::Partition' },
                                        ':apigateway:',
                                        { Ref: 'AWS::Region' },
                                        ':lambda:path/2015-03-31/functions/',
                                        { 'Fn::GetAtt': [expect.stringMatching(/^MyFunction.*/), 'Arn'] },
                                        '/invocations',
                                    ],
                                ],
                            },
                        },
                    },
                    get: {
                        'operationId': 'getItem',
                        'responses': {},
                        'x-amazon-apigateway-integration': {
                            httpMethod: 'POST',
                            responses: {},
                            type: 'AWS',
                            uri: {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:aws:apigateway:',
                                        {
                                            Ref: 'AWS::Region',
                                        },
                                        ':dynamodb:action/GetItem',
                                    ],
                                ],
                            },
                            credentials: {
                                'Fn::GetAtt': [
                                    expect.stringMatching(/^MyRole.*/),
                                    'Arn',
                                ],
                            },
                            requestTemplates: {
                                'application/json': {
                                    'Fn::Join': [
                                        '',
                                        [
                                            '{"TableName":"',
                                            {
                                                Ref: expect.stringMatching(/^MyTable.*/),
                                            },
                                            "\",\"Key\":{\"item\":{\"S\":\"$input.params('item')\"}}}",
                                        ],
                                    ],
                                },
                            },
                        },
                    },
                },
            },
            components: {
                securitySchemes: {
                    MyCognitoAuthorizer: {
                        'type': 'apiKey',
                        'name': 'Authorization',
                        'in': 'header',
                        'x-amazon-apigateway-authtype': 'cognito_user_pools',
                        'x-amazon-apigateway-authorizer': {
                            type: 'cognito_user_pools',
                            providerARNs: [
                                {
                                    'Fn::GetAtt': [
                                        expect.stringMatching(/^MyUserPool.*/),
                                        'Arn',
                                    ],
                                },
                            ],
                            authorizerResultTtlInSeconds: 300,
                        },
                    },
                },
            },
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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