import copy
import numpy as np

from qiskit import QuantumRegister, AncillaRegister, ClassicalRegister, QuantumCircuit
from qiskit.circuit import Bit, Measure
from qiskit.circuit.classical import expr
from qiskit.quantum_info import Statevector, DensityMatrix, Pauli
from qiskit_addon_utils.slicing import slice_by_depth
from qiskit import _numpy_compat
from qiskit.transpiler import PassManager

from .Transpilation.ClearQEC import ClearQEC
from .Transpilation.UnBox import UnBox

from typing import Iterable

class LogicalCircuitGeneral(QuantumCircuit):
    """
    Core LogicalQ representation of a logical quantum circuit.

    Prototyping class for implementing and testing generalized functionality.
    """

    def __init__(
        self,
        n_logical_qubits,
        label,
        stabilizer_tableau,
        name=None,
    ):
        # Quantum error correcting code preparation
        self.n_logical_qubits = n_logical_qubits

        self.stabilizer_tableau = stabilizer_tableau
        self.n_stabilizers = len(self.stabilizer_tableau)

        self.label = label
        self.n, self.k, self.d = label
        if any([len(stabilizer) != self.n for stabilizer in self.stabilizer_tableau]):
            raise ValueError(f"Code label n ({self.n}) does not match individual stabilizer length ({self.n_physical_qubits})")

        self.n_physical_qubits = self.n
        # @TODO - obtain an exact estimate for the number of ancilla qubits
        self.n_ancilla_qubits = self.n_stabilizers
        self.n_measure_qubits = self.n_ancilla_qubits

        self.stabilizers = []

        # Generate the code, including its stabilizer groups
        self.generate_code()

        self.logical_qregs = []
        self.ancilla_qregs = []
        self.logical_op_qregs = []
        self.enc_verif_cregs = []
        self.curr_syndrome_cregs = []
        self.prev_syndrome_cregs = []
        self.unflagged_syndrome_diff_cregs = []
        self.pauli_frame_cregs = []
        self.logical_op_meas_cregs = []
        self.final_measurement_cregs = []

        self.qreg_lists = [
            self.logical_qregs,
            self.ancilla_qregs,
            self.logical_op_qregs,
        ]
        self.creg_lists = [
            self.enc_verif_cregs,
            self.curr_syndrome_cregs,
            self.prev_syndrome_cregs,
            self.unflagged_syndrome_diff_cregs,
            self.pauli_frame_cregs,
            self.logical_op_meas_cregs,
            self.final_measurement_cregs,
        ]

        # The underlying (empty) QuantumCircuit is generated by first calling super()...
        super().__init__(name=name)
        # ...then adding the logical qubits
        self.add_logical_qubits(self.n_logical_qubits)

        # Also add a classical measurement output register at the end
        self.output_creg = ClassicalRegister(self.n_logical_qubits, name="output")
        super().add_register(self.output_creg)

        # @TODO - find alternative, possibly by implementing upstream
        # Create setter qreg for purpose of setting classical bits dynamically
        self.cbit_setter_qreg = QuantumRegister(2, name="qsetter")
        self.add_register(self.cbit_setter_qreg)
        super().x(self.cbit_setter_qreg[1])


    def add_logical_qubits(self, logical_qubit_count):
        current_logical_qubit_count = len(self.logical_qregs)

        for i in range(current_logical_qubit_count, current_logical_qubit_count + logical_qubit_count):
            # Physical qubits for logical qubit
            logical_qreg_i = QuantumRegister(self.n_physical_qubits, name=f"qlog{i}")
            # Ancilla qubits needed for measurements
            ancilla_qreg_i = AncillaRegister(self.n_ancilla_qubits, name=f"qanc{i}")
            # Ancilla qubits needed for logical operations
            logical_op_qreg_i = AncillaRegister(2, name=f"qlogical_op{i}")
            # Classical bits needed for encoding verification
            enc_verif_creg_i = ClassicalRegister(1, name=f"cenc_verif{i}")
            # Classical bits needed for measurements
            curr_syndrome_creg_i = ClassicalRegister(self.n_measure_qubits, name=f"ccurr_syndrome{i}")
            # Classical bits needed for previous syndrome measurements
            prev_syndrome_creg_i = ClassicalRegister(self.n_stabilizers, name=f"cprev_syndrome{i}")
            # Classical bits needed for flagged syndrome difference measurements
            unflagged_syndrome_diff_creg_i = ClassicalRegister(self.n_stabilizers, name=f"cunflagged_syndrome_diff{i}")
            # Classical bits needed to track the Pauli Frame
            pauli_frame_creg_i = ClassicalRegister(2, name=f"cpauli_frame{i}")
            # Classical bits needed to take measurements of logical operation qubits
            logical_op_meas_creg_i = ClassicalRegister(2, name=f"clogical_op_meas{i}")
            # Classical bits needed to take measurements of logical operation qubits
            final_measurement_creg_i = ClassicalRegister(self.n_physical_qubits, name=f"cfinal_meas{i}")

            # Add new registers to storage lists
            self.logical_qregs.append(logical_qreg_i)
            self.ancilla_qregs.append(ancilla_qreg_i)
            self.logical_op_qregs.append(logical_op_qreg_i)
            self.enc_verif_cregs.append(enc_verif_creg_i)
            self.curr_syndrome_cregs.append(curr_syndrome_creg_i)
            self.prev_syndrome_cregs.append(prev_syndrome_creg_i)
            self.unflagged_syndrome_diff_cregs.append(unflagged_syndrome_diff_creg_i)
            self.pauli_frame_cregs.append(pauli_frame_creg_i)
            self.logical_op_meas_cregs.append(logical_op_meas_creg_i)
            self.final_measurement_cregs.append(final_measurement_creg_i)

            # Add new registers to quantum circuit
            super().add_register(logical_qreg_i)
            super().add_register(ancilla_qreg_i)
            super().add_register(logical_op_qreg_i)
            super().add_register(enc_verif_creg_i)
            super().add_register(curr_syndrome_creg_i)
            super().add_register(prev_syndrome_creg_i)
            super().add_register(unflagged_syndrome_diff_creg_i)
            super().add_register(pauli_frame_creg_i)
            super().add_register(logical_op_meas_creg_i)
            super().add_register(final_measurement_creg_i)

    ####################################
    ##### Quantum error correction #####
    ####################################


    # Function which generates encoding circuit and logical operators for a given tableau
    def generate_code(self):
        m = len(self.stabilizer_tableau)

        # Step 1: Assemble generator matrix
        G = np.zeros((2, m, self.n))
        for i, stabilizer in enumerate(self.stabilizer_tableau):
            for j, pauli_j in enumerate(stabilizer):
                if pauli_j == "X":
                    G[0, i, j] = 1
                elif pauli_j == "Z":
                    G[1, i, j] = 1
                elif pauli_j == "Y":
                    G[:, i, j] = 1

        self.G_non_standard = G.copy()

        # Step 2: Perform Gaussian reduction in base 2
        row = 0
        for col in range(self.n):
            pivot_row = None
            for i in range(row, m):
                if G[0, i, col] == 1:
                    pivot_row = i
                    break

            if pivot_row is None:
                continue

            G[:, [row, pivot_row]] = G[:, [pivot_row, row]]

            # Flip any other rows with a "1" in the same column
            for i in range(m):
                if i != row and G[0, i, col] == 1:
                    G[:, i] = G[:, i].astype(int) ^ G[:, row].astype(int)
                    # G[0, i] = G[0, i].astype(int) ^ G[0, row].astype(int)
                    # G[1, i] = G[1, i].astype(int) ^ G[1, row].astype(int)

            # Move to the next row, if we haven't reached the end of the matrix
            row += 1
            if row >= m:
                break

        r = np.linalg.matrix_rank(G[0])

        E = np.copy(G[:, r:, r:])
        row = 0
        for col in range(self.n-r):
            pivot_row = None
            for i in range(row, m-r):
                if E[1, i, col] == 1:
                    pivot_row = i
                    break

            if pivot_row is None:
                continue

            E[:, [row, pivot_row]] = E[:, [pivot_row, row]]
            G[:, [r+row, r+pivot_row]] = G[:, [r+pivot_row, r+row]]

            # Flip any other rows with a "1" in the same column
            for i in range(m-r):
                if i != row and E[1, i, col] == 1:
                    E[:, i] = E[:, i].astype(int) ^ E[:, row].astype(int)
                    G[:, r+i] = G[:, r+i].astype(int) ^ G[:, r+row].astype(int)

            # Move to the next row, if we haven't reached the end of the matrix
            row += 1
            if row >= m:
                break

        # Since G is in RREF, a pivot row is also a pivot column, so find the pivot columns and move them forward
        pivot_indices = []
        for row in G[0]:
            if 1 in row:
                pivot_indices.append(int(np.where(row == 1)[0][0]))
        
        for diagonal_index, pivot_index in enumerate(pivot_indices):
            if pivot_index > -1:
                G[:, :, [diagonal_index, pivot_index]] = G[:, :, [pivot_index, diagonal_index]]

        self.G = G

        # Step 3: Construct logical operators using Pauli vector representations due to Gottesmann (1997)
        r = np.linalg.matrix_rank(self.G[0])
        A_2 = self.G[0, 0:r, m:self.n] # r x k
        C_1 = self.G[1, 0:r, r:m] # r x m-r
        C_2 = self.G[1, 0:r, m:self.n] # r x k
        E_2 = self.G[1, r:m, m:self.n] # m-r x k

        self.LogicalXVector = np.block([
            [[np.zeros((self.k, r)), E_2.T,                        np.eye(self.k, self.k)    ]],
            [[E_2.T @ C_1.T + C_2.T,      np.zeros((self.k, m-r)), np.zeros((self.k, self.k))]]
        ])

        # Create Logical X circuit corresponding to X's and Z's at 1's in Pauli vector
        self.LogicalXCircuit = QuantumCircuit(self.n)
        for i in range(self.k):
            # X part
            for q, bit in enumerate(self.LogicalXVector[0][i]):
                if bit == 1:
                    self.LogicalXCircuit.x(q)
            # Z part
            for q, bit in enumerate(self.LogicalXVector[1][i]):
                if bit == 1:
                    self.LogicalXCircuit.z(q)
        self.LogicalXGate = self.LogicalXCircuit.to_gate(label="$X_L$")

        self.LogicalZVector = np.block([
            [[np.zeros((self.k, r)), np.zeros((self.k, m-r)), np.zeros((self.k, self.k))]],
            [[A_2.T,                 np.zeros((self.k, m-r)), np.eye(self.k, self.k)    ]]
        ])

        # Create Logical Z circuit corresponding to X's and Z's at 1's in Pauli vector
        self.LogicalZCircuit = QuantumCircuit(self.n)
        for i in range(self.k):
            # X part
            for q, bit in enumerate(self.LogicalZVector[0][i]):
                if bit == 1:
                    self.LogicalZCircuit.x(q)
            # Z part
            for q, bit in enumerate(self.LogicalZVector[1][i]):
                if bit == 1:
                    self.LogicalZCircuit.z(q)
        self.LogicalZGate = self.LogicalZCircuit.to_gate(label="$Z_L$")

        self.LogicalYCircuit = self.LogicalXCircuit.compose(self.LogicalZCircuit)
        self.LogicalYGate = self.LogicalYCircuit.to_gate(label="$Y_L$")

        # Create Logical H circuit using Childs and Wiebe's linear combination of unitaries method
        self.LogicalHCircuit_LCU = QuantumCircuit(self.n + 1)
        self.LogicalHCircuit_LCU.h(self.n)
        self.LogicalHCircuit_LCU.compose(self.LogicalXCircuit.control(1), [self.n, *list(range(self.n))], inplace=True)
        self.LogicalHCircuit_LCU.x(self.n)
        self.LogicalHCircuit_LCU.compose(self.LogicalZCircuit.control(1), [self.n, *list(range(self.n))], inplace=True)
        self.LogicalHCircuit_LCU.x(self.n)
        self.LogicalHCircuit_LCU.h(self.n)
        self.LogicalHGate_LCU = self.LogicalHCircuit_LCU.to_gate(label="$H_L$")

        # Creates Logical H circuit using coherent feedback
        self.LogicalHCircuit_CF = QuantumCircuit(self.n + 1)
        self.LogicalHCircuit_CF.h(self.n)
        self.LogicalHCircuit_CF.compose(self.LogicalXCircuit.control(1), [self.LogicalHCircuit_CF.qubits[self.n]] + self.LogicalHCircuit_CF.qubits[:self.n], inplace=True)
        self.LogicalHCircuit_CF.compose(self.LogicalZCircuit.control(1), [self.LogicalHCircuit_CF.qubits[self.n]] + self.LogicalHCircuit_CF.qubits[:self.n], inplace=True)
        self.LogicalHCircuit_CF.h(self.n)
        self.LogicalHCircuit_CF.compose(self.LogicalXCircuit.control(1), [self.LogicalHCircuit_CF.qubits[self.n]] + self.LogicalHCircuit_CF.qubits[:self.n], inplace=True)
        self.LogicalHCircuit_CF.x(self.n)
        self.LogicalHCircuit_CF.compose(self.LogicalZCircuit.control(1), [self.LogicalHCircuit_CF.qubits[self.n]] + self.LogicalHCircuit_CF.qubits[:self.n], inplace=True)
        self.LogicalHCircuit_CF.h(self.n)
        self.LogicalHGate_CF = self.LogicalHCircuit_CF.to_gate(label="$H_{CF}$")

        # @TODO - Logical S
        # Creates Logical S circuit using coherent feedback
        self.LogicalSCircuit_CF = QuantumCircuit(self.n + 1)
        self.LogicalSCircuit_CF.h(self.n)
        self.LogicalSCircuit_CF.s(self.n)
        self.LogicalSCircuit_CF.h(self.n)
        self.LogicalSCircuit_CF.compose(self.LogicalZCircuit.control(1), [self.LogicalSCircuit_CF.qubits[self.n]] + self.LogicalSCircuit_CF.qubits[:self.n], inplace=True)
        self.LogicalSCircuit_CF.h(self.n)
        self.LogicalSCircuit_CF.compose(self.LogicalZCircuit.control(1), [self.LogicalSCircuit_CF.qubits[self.n]] + self.LogicalSCircuit_CF.qubits[:self.n], inplace=True)
        self.LogicalSCircuit_CF.sdg(self.n)
        self.LogicalSCircuit_CF.h(self.n)
        self.LogicalSGate_CF = self.LogicalSCircuit_CF.to_gate(label="$S_{CF}$")

        # @TODO - Logical S†
        # Creates Logical S† circuit using coherent feedback
        self.LogicalSdgCircuit_CF = QuantumCircuit(self.n + 1)
        self.LogicalSdgCircuit_CF.h(self.n)
        self.LogicalSdgCircuit_CF.sdg(self.n)
        self.LogicalSdgCircuit_CF.h(self.n)
        self.LogicalSdgCircuit_CF.compose(self.LogicalZCircuit.control(1), [self.LogicalSdgCircuit_CF.qubits[self.n]] + self.LogicalSdgCircuit_CF.qubits[:self.n], inplace=True)
        self.LogicalSdgCircuit_CF.h(self.n)
        self.LogicalSdgCircuit_CF.compose(self.LogicalZCircuit.control(1), [self.LogicalSdgCircuit_CF.qubits[self.n]] + self.LogicalSdgCircuit_CF.qubits[:self.n], inplace=True)
        self.LogicalSdgCircuit_CF.s(self.n)
        self.LogicalSdgCircuit_CF.h(self.n)
        self.LogicalSdgGate_CF = self.LogicalSdgCircuit_CF.to_gate(label="$S†_{CF}$")

        # @TODO - Logical T
        # Creates Logical T circuit using coherent feedback
        self.LogicalTCircuit_CF = QuantumCircuit(self.n + 2)
        self.LogicalTCircuit_CF.h(self.n)
        self.LogicalTCircuit_CF.h(self.n + 1)
        self.LogicalTCircuit_CF.t(self.n)
        self.LogicalTCircuit_CF.s(self.n + 1)
        self.LogicalTCircuit_CF.h(self.n)
        self.LogicalTCircuit_CF.compose(self.LogicalZCircuit.control(1), [self.LogicalTCircuit_CF.qubits[self.n]] + self.LogicalTCircuit_CF.qubits[:self.n], inplace=True)
        self.LogicalTCircuit_CF.h(self.n)
        self.LogicalTCircuit_CF.h(self.n + 1)
        self.LogicalTCircuit_CF.compose(self.LogicalZCircuit.control(2), self.LogicalTCircuit_CF.qubits[self.n:self.n+2] + self.LogicalTCircuit_CF.qubits[:self.n], inplace=True)
        self.LogicalTCircuit_CF.h(self.n + 1)
        self.LogicalTCircuit_CF.compose(self.LogicalZCircuit.control(2), self.LogicalTCircuit_CF.qubits[self.n:self.n+2] + self.LogicalTCircuit_CF.qubits[:self.n], inplace=True)
        self.LogicalTCircuit_CF.tdg(self.n)
        self.LogicalTCircuit_CF.sdg(self.n + 1)
        self.LogicalTCircuit_CF.h(self.n)
        self.LogicalTCircuit_CF.h(self.n + 1)
        self.LogicalTGate_CF = self.LogicalTCircuit_CF.to_gate(label="$T_{CF}$")

        # @TODO - Logical T†
        # Creates Logical T† circuit using coherent feedback
        self.LogicalTdgCircuit_CF = QuantumCircuit(self.n + 2)
        self.LogicalTdgCircuit_CF.h(self.n)
        self.LogicalTdgCircuit_CF.h(self.n + 1)
        self.LogicalTdgCircuit_CF.tdg(self.n)
        self.LogicalTdgCircuit_CF.sdg(self.n + 1)
        self.LogicalTdgCircuit_CF.h(self.n)
        self.LogicalTdgCircuit_CF.compose(self.LogicalZCircuit.control(1), [self.LogicalTdgCircuit_CF.qubits[self.n]] + self.LogicalTdgCircuit_CF.qubits[:self.n], inplace=True)
        self.LogicalTdgCircuit_CF.h(self.n)
        self.LogicalTdgCircuit_CF.h(self.n + 1)
        self.LogicalTdgCircuit_CF.compose(self.LogicalZCircuit.control(2), self.LogicalTdgCircuit_CF.qubits[self.n:self.n+2] + self.LogicalTdgCircuit_CF.qubits[:self.n], inplace=True)
        self.LogicalTdgCircuit_CF.h(self.n + 1)
        self.LogicalTdgCircuit_CF.compose(self.LogicalZCircuit.control(2), self.LogicalTdgCircuit_CF.qubits[self.n:self.n+2] + self.LogicalTdgCircuit_CF.qubits[:self.n], inplace=True)
        self.LogicalTdgCircuit_CF.t(self.n)
        self.LogicalTdgCircuit_CF.s(self.n + 1)
        self.LogicalTdgCircuit_CF.h(self.n)
        self.LogicalTdgCircuit_CF.h(self.n + 1)
        self.LogicalTdgGate_CF = self.LogicalTdgCircuit_CF.to_gate(label="$T†_{CF}$")

        # @TODO - Logical CX

        # Step 4: Apply the respective stabilizers
        self.encoding_circuit = QuantumCircuit(self.n)
        for i in range(self.k):
            for j in range(r, self.n-self.k):
                if self.LogicalXVector[0, i, j]:
                    self.encoding_circuit.cx(self.n-self.k+i, j)

        for i in range(r):
            self.encoding_circuit.h(i)
            for j in range(self.n):
                if i != j:
                    if self.G[0, i, j] and self.G[1, i, j]:
                        self.encoding_circuit.cx(i, j)
                        self.encoding_circuit.cz(i, j)
                    elif self.G[0, i, j]:
                        self.encoding_circuit.cx(i, j)
                    elif self.G[1, i, j]:
                        self.encoding_circuit.cz(i, j)
                    

        self.encoding_gate = self.encoding_circuit.to_gate(label="$U_{enc}$")

        #Collect syndromes
        self.pauli_frame_x_syndromes = []
        self.pauli_frame_z_syndromes = []

        for i in range(self.n_physical_qubits):
            if self.LogicalXVector[0][0][i] == 1:
                self.pauli_frame_x_syndromes.append(self.G_non_standard[0].T[i])
            if self.LogicalXVector[1][0][i] == 1:
                self.pauli_frame_x_syndromes.append(self.G_non_standard[1].T[i])

            if self.LogicalZVector[0][0][i] == 1:
                self.pauli_frame_z_syndromes.append(self.G_non_standard[0].T[i])
            if self.LogicalZVector[1][0][i] == 1:
                self.pauli_frame_z_syndromes.append(self.G_non_standard[1].T[i])

    # Encodes logical qubits for a given number of iterations
    def encode(self, *qubits, max_iterations=1, initial_states=None):
        """
        Prepare logical qubit(s) in the specified initial state
        """

        if self.encoding_circuit is None:
            raise RuntimeError("LogicalCircuit code has not been properly constructed (missing encoding circuit)")

        if qubits is None:
            qubits = list(range(self.n_logical_qubits))
        elif (hasattr(qubits, "__iter__") and len(qubits) == 0):
            raise ValueError("No qubits specified for logical state encoding")
        else:
            if len(qubits) > 0 and hasattr(qubits[0], "__iter__"):
                # Double unwrapping in case qubits is actually a list of lists
                qubits = [qj for qi in qubits for qj in qi]
            else:
                # Simple list conversion to guarantee type
                qubits = list(qubits)

        if initial_states is None:
            initial_states = [0] * len(qubits)

        if initial_states is None or len(qubits) != len(initial_states):
            raise ValueError("Number of qubits should equal number of initial states if initial states are provided")

        for q, init_state in zip(qubits, initial_states):

            # Preliminary physical qubit reset
            super().reset(self.logical_qregs[q])

            # Initial encoding
            super().compose(self.encoding_circuit, self.logical_qregs[q], inplace=True)

            #Encoding verification

            #Measure Z_L on ancilla
            super().barrier()
            super().h(self.ancilla_qregs[q][0])
            # X part
            for i, bit in enumerate(self.LogicalZVector[0][0]):
                if bit == 1:
                    super().cx(self.ancilla_qregs[q][0], self.logical_qregs[q][i])
            # Z part
            for i, bit in enumerate(self.LogicalZVector[1][0]):
                if bit == 1:
                    super().cz(self.ancilla_qregs[q][0], self.logical_qregs[q][i])
            super().h(self.ancilla_qregs[q][0])
            super().barrier()

            super().append(Measure(), [self.ancilla_qregs[q][0]], [self.enc_verif_cregs[q][0]], copy=False)

            for _ in range(max_iterations - 1):
                # If the ancilla stores a 1, reset the entire logical qubit and redo
                with super().if_test((self.enc_verif_cregs[q][0], 1)):
                    super().reset(self.logical_qregs[q])

                    # Initial encoding
                    super().compose(self.encoding_circuit, self.logical_qregs[q], inplace=True)

                    #Measure Z_L on ancilla
                    super().h(self.ancilla_qregs[q][0])
                    # X part
                    for i, bit in enumerate(self.LogicalZVector[0][0]):
                        if bit == 1:
                            super().cx(self.ancilla_qregs[q][0], self.logical_qregs[q][i])
                    # Z part
                    for i, bit in enumerate(self.LogicalZVector[1][0]):
                        if bit == 1:
                            super().cz(self.ancilla_qregs[q][0], self.logical_qregs[q][i])
                    super().h(self.ancilla_qregs[q][0])

                    # Measure ancilla
                    # super().measure(self.ancilla_qregs[q][0], self.enc_verif_cregs[q][0])
                    super().append(Measure(), [self.ancilla_qregs[q][0]], [self.enc_verif_cregs[q][0]], copy=False)

            # Reset ancilla qubit
            super().reset(self.ancilla_qregs[q][0])

            # Flip qubits if necessary
            if init_state == 1:
                self.x(q)
            elif init_state != 0:
                raise ValueError("Initial state should be either 0 or 1 (arbitrary statevectors not yet supported)!")

        return True

    # Reset all ancillas associated with specified logical qubits
    def reset_ancillas(self, logical_qubit_indices=None):
        if logical_qubit_indices is None or len(logical_qubit_indices) == 0:
            logical_qubit_indices = list(range(self.n_logical_qubits))

        for q in logical_qubit_indices:
            self.reset(self.ancilla_qregs[q])


    # Measure specified specifiers to the circuit as controlled Pauli operators
    def measure_stabilizers(self, logical_qubit_indices=None, stabilizer_indices=None):
        if logical_qubit_indices is None or len(logical_qubit_indices) == 0:
            logical_qubit_indices = list(range(self.n_logical_qubits))

        if stabilizer_indices is None or len(logical_qubit_indices) == 0:
            stabilizer_indices = list(range(self.n_stabilizers))

        for q in logical_qubit_indices:
            for s, stabilizer_index in enumerate(stabilizer_indices):

                stabilizer = self.stabilizer_tableau[stabilizer_index]
                super().h(self.ancilla_qregs[q][s])
                for p in range(self.n_physical_qubits):
                    stabilizer_pauli = Pauli(stabilizer[p])
                    if stabilizer[p] != 'I':
                        CPauliInstruction = stabilizer_pauli.to_instruction().control(1)
                        super().append(CPauliInstruction, [self.ancilla_qregs[q][s], self.logical_qregs[q][p]])
                super().h(self.ancilla_qregs[q][s])
                super().barrier()

    # Measure flagged or unflagged syndrome differences for specified logical qubits and stabilizers
    def measure_syndrome_diff(self, logical_qubit_indices=None):
        if logical_qubit_indices is None or len(logical_qubit_indices) == 0:
            logical_qubit_indices = list(range(self.n_logical_qubits))

        for q in logical_qubit_indices:

            self.measure_stabilizers(logical_qubit_indices=[q])

            for n in range(self.n_ancilla_qubits):
                super().append(Measure(), [self.ancilla_qregs[q][n]], [self.curr_syndrome_cregs[q][n]], copy=False)

            # Determine the syndrome difference
            for n in range(self.n_ancilla_qubits):
                with self.if_test(self.cbit_xor([self.curr_syndrome_cregs[q][n], self.prev_syndrome_cregs[q][n]])) as _else:
                    self.set_cbit(self.unflagged_syndrome_diff_cregs[q][n], 1)
                with _else:
                    self.set_cbit(self.unflagged_syndrome_diff_cregs[q][n], 0)

        self.reset_ancillas(logical_qubit_indices=logical_qubit_indices)


    # Append a QEC cycle to the end of the circuit
    def append_qec_cycle(self, logical_qubit_indices=None):

        if logical_qubit_indices is None or len(logical_qubit_indices) == 0:
            logical_qubit_indices = list(range(self.n_logical_qubits))

        for q in logical_qubit_indices:

            super().reset(self.ancilla_qregs[q])

            # If change in syndrome, perform unflagged syndrome measurement, decode, and correct
            self.measure_syndrome_diff(logical_qubit_indices=[q])
            self.apply_decoding(logical_qubit_indices=[q])

            # Update previous syndrome
            for n in range(self.n_stabilizers):
                with self.if_test(expr.lift(self.unflagged_syndrome_diff_cregs[q][n])):
                    self.cbit_not(self.prev_syndrome_cregs[q][n])



    # @TODO - determine appropriate syndrome decoding mappings dynamically
    def apply_decoding(self, logical_qubit_indices):
        for q in logical_qubit_indices:
            for x_syn in self.pauli_frame_x_syndromes:
                with super().if_test(self.cbit_and(self.unflagged_syndrome_diff_cregs[q], x_syn)):
                    self.cbit_not(self.pauli_frame_cregs[q][0])
            for z_syn in self.pauli_frame_z_syndromes:
                with super().if_test(self.cbit_and(self.unflagged_syndrome_diff_cregs[q], z_syn)):
                    self.cbit_not(self.pauli_frame_cregs[q][1])


    def measure(self, logical_qubit_indices, cbit_indices, with_error_correction=True, meas_basis='Z'):
        if not hasattr(logical_qubit_indices, "__iter__"):
            raise ValueError("Logical qubit indices must be an iterable!")

        if not hasattr(cbit_indices, "__iter__"):
            raise ValueError("Classical bit indices must be an iterable!")

        if len(logical_qubit_indices) != len(cbit_indices):
            raise ValueError("Number of qubits should equal number of classical bits")

        for q, c in zip(logical_qubit_indices, cbit_indices):

            meas_inds = []

            #Prepares logical qreg for measurement and collects indices that will determine the logical state
            if meas_basis == 'X':
                for i in range(self.n_physical_qubits):
                    if self.LogicalXVector[0][0][i] == 1.:
                        super().h(self.logical_qregs[q][i])
                        meas_inds.append(i)
                    if self.LogicalXVector[1][0][i] == 1.:
                        meas_inds.append(i)
            
            elif meas_basis == 'Z':
                for i in range(self.n_physical_qubits):
                    if self.LogicalZVector[1][0][i] == 1.:
                        meas_inds.append(i)

            elif meas_basis == 'Y':
                for i in range(self.n_physical_qubits):
                    if self.LogicalXVector[0][0][i] == 1. and self.LogicalZVector[1][0][i] == 1.:
                        super().sdg(self.logical_qregs[q][i])
                        super().h(self.logical_qregs[q][i])
                        meas_inds.append(i)
                    elif self.LogicalXVector[0][0][i] == 1.:
                        super().h(self.logical_qregs[q][i])
                        meas_inds.append(i)
                    elif int(self.LogicalXVector[1][0][i]) ^ int(self.LogicalZVector[1][0][i]):
                        meas_inds.append(i)


            # Measurement of state
            for n in range(self.n_physical_qubits):
                super().append(Measure(), [self.logical_qregs[q][n]], [self.final_measurement_cregs[q][n]], copy=False)

            #Determining logical state
            with super().if_test(self.cbit_xor([self.final_measurement_cregs[q][m] for m in meas_inds])):
                self.set_cbit(self.output_creg[c], 1)

            #Apply pauli frame correction
            if with_error_correction:
                if meas_basis == 'X':
                    with super().if_test(expr.lift(self.pauli_frame_cregs[q][0])):
                        self.cbit_not(self.output_creg[c])
                elif meas_basis == 'Z':
                    with super().if_test(expr.lift(self.pauli_frame_cregs[q][1])):
                        self.cbit_not(self.output_creg[c])
                elif meas_basis == 'Y':
                    with super().if_test(expr.bit_xor(self.pauli_frame_cregs[q][0], self.pauli_frame_cregs[q][1])):
                        self.cbit_not(self.output_creg[c])

    def measure_all(self, with_error_correction=True, meas_basis='Z'):
        self.measure(range(self.n_logical_qubits), range(self.n_logical_qubits), with_error_correction=with_error_correction, meas_basis=meas_basis)

    def remove_final_measurements(self, inplace=False):
        if inplace:
            raise NotImplementedError("Inplace measurement removal is not supported")

        lqc_no_meas = LogicalCircuitGeneral(self.n_logical_qubits, self.label, self.stabilizer_tableau, self.name + "_no_meas")

        for circuit_instruction in self.data:
            if circuit_instruction.name != "measure":
                lqc_no_meas._append(circuit_instruction)

        return lqc_no_meas

    def get_logical_counts(
            self,
            physical_counts: Iterable[int],
            logical_qubit_indices: Iterable[int] = None
    ) -> dict[str, int]:
        """Get logical counts from physical counts.

        Args:
            physical_counts: Physical counts to convert to logical counts.
            logical_qubit_indices: Logical qubits to get counts for. If `None`, then get counts for all.

        Returns:
            Logical qubit counts.
        """
        if logical_qubit_indices is None:
            logical_qubit_indices = range(self.n_logical_qubits)

        logical_counts = {}
        for physical_outcome, physical_outcome_counts in physical_counts.items():
            logical_outcome = "".join([physical_outcome[self.n_logical_qubits-1-l] for l in logical_qubit_indices])

            logical_counts[logical_outcome] = logical_counts.get(logical_outcome, 0) + physical_outcome_counts

        return logical_counts

    ######################################
    ##### Logical quantum operations #####
    ######################################

    # @TODO - generalize logical quantum operations using stabilizers

    def h(self, *targets, method="Coherent_Feedback"):
        """
        Logical Hadamard gate
        """

        if len(targets) == 1 and hasattr(targets[0], "__iter__"):
            targets = targets[0]

        if method == "LCU":
            for t in targets:
                
                super().compose(self.LogicalHCircuit_LCU, [self.logical_op_qregs[t][0]] + self.logical_qregs[t][:], inplace=True)

            # @TODO - perform resets after main operation is complete to allow for faster(?) parallel operation
            # for t in targets:
                # @TODO - determine whether extra reset is necessary at the end
                # with self.box(label="logical.logicalop.lcu"):
                    # super().reset(self.logical_op_qregs[t])
        elif method == "LCU_Corrected": 
            for t in targets:

                # Construct circuit for implementing a Hadamard gate through the use of an ancilla
                super().h(self.logical_op_qregs[t][0])
                super().compose(self.LogicalXCircuit.control(1), [self.logical_op_qregs[t][0]] + self.logical_qregs[t][:], inplace=True)
                super().compose(self.LogicalZCircuit.control(1), [self.logical_op_qregs[t][0]] + self.logical_qregs[t][:], inplace=True)
                super().h(self.logical_op_qregs[t][0])
                super().append(Measure(), [self.logical_op_qregs[t][0]], [self.logical_op_meas_cregs[t][0]], copy=False)
                super().reset(self.logical_op_qregs[t][0])

                # Corrections to apply based on ancilla measurement
                with super().if_test((self.logical_op_meas_cregs[t][0], 1)) as else_:
                    self.x(t)
                with else_:
                    self.z(t)

        elif method == "Coherent_Feedback":
            for t in targets:

                super().compose(self.LogicalHCircuit_CF, self.logical_qregs[t][:] + [self.logical_op_qregs[t][0]], inplace=True)

        elif method == "Transversal_Uniform":
            for t in targets:

                super().h(self.logical_qregs[t][:])
        else:
            raise ValueError(f"'{method}' is not a valid method for the logical Hadamard gate")
        

        #Pauli frame update
        for t in targets:
            with super().if_test(expr.bit_xor(self.pauli_frame_cregs[t][0], self.pauli_frame_cregs[t][1])) as _else:
                self.set_cbit(self.pauli_frame_cregs[t][0], 1)
            with _else:
                self.set_cbit(self.pauli_frame_cregs[t][0], 0)
            with super().if_test(expr.bit_xor(self.pauli_frame_cregs[t][0], self.pauli_frame_cregs[t][1])) as _else:
                self.set_cbit(self.pauli_frame_cregs[t][1], 1)
            with _else:
                self.set_cbit(self.pauli_frame_cregs[t][1], 0)
            with super().if_test(expr.bit_xor(self.pauli_frame_cregs[t][0], self.pauli_frame_cregs[t][1])) as _else:
                self.set_cbit(self.pauli_frame_cregs[t][0], 1)
            with _else:
                self.set_cbit(self.pauli_frame_cregs[t][0], 0)

    def x(self, *targets):
        """
        Logical PauliX gate
        """

        if len(targets) == 1 and hasattr(targets[0], "__iter__"):
            targets = targets[0]

        for t in targets:
            
            super().compose(self.LogicalXCircuit, self.logical_qregs[t], inplace=True)

    def y(self, *targets):
        """
        Logical PauliY gate
        """

        if len(targets) == 1 and hasattr(targets[0], "__iter__"):
            targets = targets[0]


        self.z(targets)
        self.x(targets)

    def z(self, *targets):
        """
        Logical PauliZ gate
        """

        if len(targets) == 1 and hasattr(targets[0], "__iter__"):
            targets = targets[0]

        for t in targets:

            super().compose(self.LogicalZCircuit, self.logical_qregs[t], inplace=True)

    def s(self, *targets, method="Coherent_Feedback"):
        """
        Logical S gate

        Definition:
        [1   0]
        [0   i]
        """

        if len(targets) == 1 and hasattr(targets[0], "__iter__"):
            targets = targets[0]

        if method == "LCU_Corrected" or method == "T_Utility":
            for t in targets:

                super().h(self.logical_op_qregs[t][0])
                super().s(self.logical_op_qregs[t][0])
                super().h(self.logical_op_qregs[t][0])
                super().compose(self.LogicalZCircuit.control(1), [self.logical_op_qregs[t][0]] + self.logical_qregs[t][:], inplace=True)
                super().h(self.logical_op_qregs[t][0])
                super().append(Measure(), [self.logical_op_qregs[t][0]], [self.logical_op_meas_cregs[t][0]], copy=False)

                with super().if_test((self.logical_op_meas_cregs[t][0], 1)):
                    self.z(t)

                super().reset(self.logical_op_qregs[t][0])
        elif method == "Coherent_Feedback":
            for t in targets:

                super().compose(self.LogicalSCircuit_CF, self.logical_qregs[t][:] + [self.logical_op_qregs[t][0]], inplace=True)
        elif method == "Transversal_Uniform":
            for t in targets:

                super().sdg(self.logical_qregs[t][:])

        else:
            raise ValueError(f"'{method}' is not a valid method for the logical S gate")
        
        #Pauli frame update
        if method != "T_Utility":
            for t in targets:
                with super().if_test(expr.lift(self.pauli_frame_cregs[t][1])):
                    self.cbit_not(self.pauli_frame_cregs[t][0])

    def sdg(self, *targets, method="Coherent_Feedback"):
        """
        Logical S^dagger gate

        Definition:
        [1    0]
        [0   -i]
        """
        
        if len(targets) == 1 and hasattr(targets[0], "__iter__"):
            targets = targets[0]

        if method == "LCU_Corrected" or method == "T_Utility":
            for t in targets:

                super().h(self.logical_op_qregs[t][0])
                super().sdg(self.logical_op_qregs[t][0])
                super().h(self.logical_op_qregs[t][0])
                super().compose(self.LogicalZCircuit.control(1), [self.logical_op_qregs[t][0]] + self.logical_qregs[t][:], inplace=True)
                super().h(self.logical_op_qregs[t][0])
                super().append(Measure(), [self.logical_op_qregs[t][0]], [self.logical_op_meas_cregs[t][0]], copy=False)

                with super().if_test((self.logical_op_meas_cregs[t][0], 1)):
                    self.z(t)

                super().reset(self.logical_op_qregs[t][0])
        elif method == "Coherent_Feedback":
            for t in targets:

                super().compose(self.LogicalSdgCircuit_CF, self.logical_qregs[t][:] + [self.logical_op_qregs[t][0]], inplace=True)

        elif method == "Transversal_Uniform":
            for t in targets:

                super().s(self.logical_qregs[t][:])

        else:
            raise ValueError(f"'{method}' is not a valid method for the logical S^dagger gate")
        
        #Pauli frame update
        if method != "T_Utility":
            for t in targets:
                with super().if_test(expr.lift(self.pauli_frame_cregs[t][1])):
                    self.cbit_not(self.pauli_frame_cregs[t][0])

    def t(self, *targets, method="Coherent_Feedback"):
        """
        Logical T gate

        Definition:
        [1    0        ]
        [0    e^(ipi/4)]
        """

        if len(targets) == 1 and hasattr(targets[0], "__iter__"):
            targets = targets[0]

        if method == "LCU_Corrected":
            for t in targets:

                super().h(self.logical_op_qregs[t][0])

                with super().if_test((self.pauli_frame_cregs[t][1], 1)) as _else:
                    super().tdg(self.logical_op_qregs[t][0])
                with _else:
                    super().t(self.logical_op_qregs[t][0])
                
                super().h(self.logical_op_qregs[t][0])
                super().compose(self.LogicalZCircuit.control(1), [self.logical_op_qregs[t][0]] + self.logical_qregs[t][:], inplace=True)
                super().h(self.logical_op_qregs[t][0])

                super().append(Measure(), [self.logical_op_qregs[t][0]], [self.logical_op_meas_cregs[t][0]], copy=False)
                super().reset(self.logical_op_qregs[t][0])

                with super().if_test((self.logical_op_meas_cregs[t][0], 1)):
                    #T Utility method performs the gate without Pauli frame updates in order to match the T gate protocol
                    with super().if_test((self.pauli_frame_cregs[t][1], 1)) as _else:
                        self.sdg(t, method='T_Utility')
                    with _else:
                        self.s(t, method='T_Utility')
                    

        elif method == "Coherent_Feedback":
            for t in targets:

                #If the Z Pauli frame bit is 1, perform T^dagger instead of T
                with super().if_test(expr.lift(self.pauli_frame_cregs[t][1])) as _else:
                    super().compose(self.LogicalTdgCircuit_CF, self.logical_qregs[t][:] + self.logical_op_qregs[t][:], inplace=True)
                with _else:
                    super().compose(self.LogicalTCircuit_CF, self.logical_qregs[t][:] + self.logical_op_qregs[t][:], inplace=True)

        else:
            raise ValueError(f"'{method}' is not a valid method for the logical T gate")

    def tdg(self, *targets, method="Coherent_Feedback"):
            """
            Logical T^dagger gate

            Definition:
            [1    0         ]
            [0    e^(-ipi/4)]
            """

            if len(targets) == 1 and hasattr(targets[0], "__iter__"):
                targets = targets[0]

            if method == "LCU_Corrected":
                for t in targets:

                    super().h(self.logical_op_qregs[t][0])

                    with super().if_test((self.pauli_frame_cregs[t][1], 1)) as _else:
                        super().t(self.logical_op_qregs[t][0])
                    with _else:
                        super().tdg(self.logical_op_qregs[t][0])
                    
                    super().h(self.logical_op_qregs[t][0])
                    super().compose(self.LogicalZCircuit.control(1), [self.logical_op_qregs[t][0]] + self.logical_qregs[t][:], inplace=True)
                    super().h(self.logical_op_qregs[t][0])

                    super().append(Measure(), [self.logical_op_qregs[t][0]], [self.logical_op_meas_cregs[t][0]], copy=False)
                    super().reset(self.logical_op_qregs[t][0])

                    with super().if_test((self.logical_op_meas_cregs[t][0], 1)):
                        #T Utility method performs the gate without Pauli frame updates in order to match the T^dagger gate protocol
                        with super().if_test((self.pauli_frame_cregs[t][1], 1)) as _else:
                            self.s(t, method='T_Utility')
                        with _else:
                            self.sdg(t, method='T_Utility')

            elif method == "Coherent_Feedback":
                for t in targets:

                    #If the Z Pauli frame bit is 1, perform T instead of T^dagger
                    with super().if_test(expr.lift(self.pauli_frame_cregs[t][1])) as _else:
                        super().compose(self.LogicalTCircuit_CF, self.logical_qregs[t][:] + self.logical_op_qregs[t][:], inplace=True)
                    with _else:
                        super().compose(self.LogicalTdgCircuit_CF, self.logical_qregs[t][:] + self.logical_op_qregs[t][:], inplace=True)


            else:
                raise ValueError(f"'{method}' is not a valid method for the logical T^dagger gate")

    def cx(self, control, *_targets, method="Ancilla_Assisted"):
        """
        Logical Controlled-PauliX gate
        """

        if hasattr(_targets, "__iter__"):
            targets = _targets
        else:
            targets = [_targets]

        # @TODO - implement a better, more generalized CNOT gate
        if method == "Ancilla_Assisted":
            for t in targets:

                super().h(self.logical_op_qregs[t][0])
                super().compose(self.LogicalZCircuit.control(1), [self.logical_op_qregs[t][0]] + self.logical_qregs[control][:], inplace=True)
                super().h(self.logical_op_qregs[t][0])
                super().compose(self.LogicalXCircuit.control(1), [self.logical_op_qregs[t][0]] + self.logical_qregs[t][:], inplace=True)
                super().h(self.logical_op_qregs[t][0])
                super().compose(self.LogicalZCircuit.control(1), [self.logical_op_qregs[t][0]] + self.logical_qregs[control][:], inplace=True)
                super().h(self.logical_op_qregs[t][0])
        elif method == "Transversal_Uniform":
            for t in targets:

                super().cx(self.logical_qregs[control][:], self.logical_qregs[t][:])
        else:
            raise ValueError(f"'{method}' is not a valid method for the logical CX gate")
        
        #Pauli frame update
        for t in targets:
            with super().if_test(expr.lift(self.pauli_frame_cregs[t][0])):
                self.cbit_not(self.pauli_frame_cregs[control][0])
            with super().if_test(expr.lift(self.pauli_frame_cregs[control][1])):
                self.cbit_not(self.pauli_frame_cregs[t][1])

    def cz(self, control, *_targets, method="Ancilla_Assisted"):
        """
        Logical Controlled-PauliZ gate
        """

        if hasattr(_targets, "__iter__"):
            targets = _targets
        else:
            targets = [_targets]

        # @TODO - implement a better, more generalized CZ gate
        if method == "Ancilla_Assisted":
            for t in targets:

                super().h(self.logical_op_qregs[t][0])
                super().compose(self.LogicalZCircuit.control(1), [self.logical_op_qregs[t][0]] + self.logical_qregs[control][:], inplace=True)
                super().h(self.logical_op_qregs[t][0])
                super().compose(self.LogicalZCircuit.control(1), [self.logical_op_qregs[t][0]] + self.logical_qregs[t][:], inplace=True)
                super().h(self.logical_op_qregs[t][0])
                super().compose(self.LogicalZCircuit.control(1), [self.logical_op_qregs[t][0]] + self.logical_qregs[control][:], inplace=True)
                super().h(self.logical_op_qregs[t][0])
        elif method == "Transversal_Uniform":
            for t in targets:

                super().cz(self.logical_qregs[control][:], self.logical_qregs[t][:])
        else:
            raise ValueError(f"'{method}' is not a valid method for the logical CZ gate")
        
        #Pauli frame update
        for t in targets:
            with super().if_test(expr.lift(self.pauli_frame_cregs[t][1])):
                self.cbit_not(self.pauli_frame_cregs[control][0])
            with super().if_test(expr.lift(self.pauli_frame_cregs[control][1])):
                self.cbit_not(self.pauli_frame_cregs[t][0])
        
    def cy(self, control, *_targets, method="Ancilla_Assisted"):
        """
        Logical Controlled-PauliY gate
        """

        if hasattr(_targets, "__iter__"):
            targets = _targets
        else:
            targets = [_targets]

        # @TODO - implement a better, more generalized CY gate
        if method == "Ancilla_Assisted":
            for t in targets:

                super().h(self.logical_op_qregs[t][0])
                super().compose(self.LogicalZCircuit.control(1), [self.logical_op_qregs[t][0]] + self.logical_qregs[control][:], inplace=True)
                super().h(self.logical_op_qregs[t][0])
                super().s(self.logical_op_qregs[t][0])
                super().compose(self.LogicalZCircuit.control(1), [self.logical_op_qregs[t][0]] + self.logical_qregs[t][:], inplace=True)
                super().compose(self.LogicalXCircuit.control(1), [self.logical_op_qregs[t][0]] + self.logical_qregs[t][:], inplace=True)
                super().h(self.logical_op_qregs[t][0])
                super().compose(self.LogicalZCircuit.control(1), [self.logical_op_qregs[t][0]] + self.logical_qregs[control][:], inplace=True)
                super().h(self.logical_op_qregs[t][0])

        elif method == "Transversal_Uniform":
            for t in targets:

                super().cy(self.logical_qregs[control][:], self.logical_qregs[t][:])
        else:
            raise ValueError(f"'{method}' is not a valid method for the logical CY gate")

        #Pauli frame update
        for t in targets:
            with super().if_test(expr.lift(self.pauli_frame_cregs[control][1])):
                self.cbit_not(self.pauli_frame_cregs[t][0])
                self.cbit_not(self.pauli_frame_cregs[t][1])
            with super().if_test(self.cbit_xor(self.pauli_frame_cregs[control][:] + self.pauli_frame_cregs[t][:])) as _else:
                self.set_cbit(self.pauli_frame_cregs[control][0], 1)
            with _else:
                self.set_cbit(self.pauli_frame_cregs[control][0], 0)

    def mcmt(self, gate, controls, targets):
        """
        Logical Multi-Control Multi-Target gate
        """

        if len(controls) == 1 and hasattr(controls[0], "__iter__"):
            controls = controls[0]

        if len(targets) == 1 and hasattr(targets[0], "__iter__"):
            targets = targets[0]

        control_qubits = [self.logical_qregs[c][:] for c in controls]
        target_qubits = [self.logical_qregs[t][:] for t in targets]

        if not set(control_qubits).isdisjoint(target_qubits):
            raise ValueError("Qubit(s) specified as both control and target")


        super().append(gate.control(len(controls)), control_qubits + target_qubits)


    ###########################
    ##### Utility methods #####
    ###########################

    # Adds a desired error for testing
    def add_error(self, l_ind, p_ind, error_type):
        if error_type == 'X':
            super().x(self.logical_qregs[l_ind][p_ind])
        if error_type == 'Z':
            super().z(self.logical_qregs[l_ind][p_ind])

    # @TODO - find alternative to classical methods, possibly by implementing upstream

    # Set values of classical bits
    def set_cbit(self, cbit, value):
        if value == 0:
            # super().measure(self.cbit_setter_qreg[0], cbit)
            super().append(Measure(), [self.cbit_setter_qreg[0]], [cbit], copy=False)
        else:
            # super().measure(self.cbit_setter_qreg[1], cbit)
            super().append(Measure(), [self.cbit_setter_qreg[1]], [cbit], copy=False)

    # Performs a NOT statement on a classical bit
    def cbit_not(self, cbit):
        with self.if_test(expr.lift(cbit)) as _else:
            self.set_cbit(cbit, 0)
        with _else:
            self.set_cbit(cbit, 1)

    # Performs AND and NOT statements on multiple classical bits, e.g. (~c[0] & ~c[1] & c[2])
    def cbit_and(self, cbits, values):
        result = expr.bit_not(cbits[0]) if values[0] == 0 else expr.lift(cbits[0])
        for n in range(len(cbits)-1):
            result = expr.bit_and(result, expr.bit_not(cbits[n+1])) if values[n+1] == 0 else expr.bit_and(result, cbits[n+1])
        return result

    # XOR multiple classical bits
    def cbit_xor(self, cbits):
        result = expr.lift(cbits[0])
        for n in range(len(cbits)-1):
            result = expr.bit_xor(result, cbits[n+1])
        return result

    ######################################
    ##### Visualization and analysis #####
    ######################################

    # def draw(
    #     self,
    #     output=None,
    #     scale=None,
    #     filename=None,
    #     style=None,
    #     interactive=False,
    #     plot_barriers=True,
    #     reverse_bits=None,
    #     justify=None,
    #     vertical_compression="medium",
    #     idle_wires=None,
    #     with_layout=True,
    #     fold=None,
    #     # The type of ax is matplotlib.axes.Axes, but this is not a fixed dependency, so cannot be
    #     # safely forward-referenced.
    #     ax=None,
    #     initial_state=False,
    #     cregbundle=None,
    #     wire_order=None,
    #     expr_len=30,
    #     fold_qec=True,
    #     fold_logicalop=True,
    # ):
    #     """
    #     LogicalCircuit drawer based on Qiskit circuit drawer
    #     """

    #     from .Visualization.LogicalCircuitVisualization import logical_circuit_drawer

    #     return logical_circuit_drawer(
    #         self,
    #         scale=scale,
    #         filename=filename,
    #         style=style,
    #         output=output,
    #         interactive=interactive,
    #         plot_barriers=plot_barriers,
    #         reverse_bits=reverse_bits,
    #         justify=justify,
    #         vertical_compression=vertical_compression,
    #         idle_wires=idle_wires,
    #         with_layout=with_layout,
    #         fold=fold,
    #         ax=ax,
    #         initial_state=initial_state,
    #         cregbundle=cregbundle,
    #         wire_order=wire_order,
    #         expr_len=expr_len,
    #         fold_qec=fold_qec,
    #         fold_logicalop=fold_logicalop,
    #     )

