"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const cdk = require("@aws-cdk/core");
const events = require("@aws-cdk/aws-events");
const defaults = require("@aws-solutions-constructs/core");
require("@aws-cdk/assert/jest");
const lib_1 = require("../lib");
function deployNewStack(stack) {
    const props = {
        eventRuleProps: {
            schedule: events.Schedule.rate(cdk.Duration.minutes(5))
        }
    };
    return new lib_1.EventbridgeToSns(stack, 'test', props);
}
test('snapshot test EventbridgeToSns default params', () => {
    const stack = new cdk.Stack();
    deployNewStack(stack);
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
test('check if the event rule has permission/policy in place in sns for it to be able to publish to the topic', () => {
    const stack = new cdk.Stack();
    deployNewStack(stack);
    expect(stack).toHaveResource('AWS::SNS::TopicPolicy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        "SNS:Publish",
                        "SNS:RemovePermission",
                        "SNS:SetTopicAttributes",
                        "SNS:DeleteTopic",
                        "SNS:ListSubscriptionsByTopic",
                        "SNS:GetTopicAttributes",
                        "SNS:Receive",
                        "SNS:AddPermission",
                        "SNS:Subscribe"
                    ],
                    Condition: {
                        StringEquals: {
                            "AWS:SourceOwner": {
                                Ref: "AWS::AccountId"
                            }
                        }
                    },
                    Effect: "Allow",
                    Principal: {
                        AWS: {
                            "Fn::Join": [
                                "",
                                [
                                    "arn:",
                                    {
                                        Ref: "AWS::Partition"
                                    },
                                    ":iam::",
                                    {
                                        Ref: "AWS::AccountId"
                                    },
                                    ":root"
                                ]
                            ]
                        }
                    },
                    Resource: {
                        Ref: "testSnsTopic42942701"
                    },
                    Sid: "TopicOwnerOnlyAccess"
                },
                {
                    Action: [
                        "SNS:Publish",
                        "SNS:RemovePermission",
                        "SNS:SetTopicAttributes",
                        "SNS:DeleteTopic",
                        "SNS:ListSubscriptionsByTopic",
                        "SNS:GetTopicAttributes",
                        "SNS:Receive",
                        "SNS:AddPermission",
                        "SNS:Subscribe"
                    ],
                    Condition: {
                        Bool: {
                            "aws:SecureTransport": "false"
                        }
                    },
                    Effect: "Deny",
                    Principal: {
                        AWS: "*"
                    },
                    Resource: {
                        Ref: "testSnsTopic42942701"
                    },
                    Sid: "HttpsOnly"
                },
                {
                    Action: "sns:Publish",
                    Effect: "Allow",
                    Principal: {
                        Service: "events.amazonaws.com"
                    },
                    Resource: {
                        Ref: "testSnsTopic42942701"
                    },
                    Sid: "2"
                }
            ],
            Version: "2012-10-17"
        },
        Topics: [
            {
                Ref: "testSnsTopic42942701"
            }
        ]
    });
});
test('check events rule properties', () => {
    const stack = new cdk.Stack();
    deployNewStack(stack);
    expect(stack).toHaveResource('AWS::Events::Rule', {
        ScheduleExpression: "rate(5 minutes)",
        State: "ENABLED",
        Targets: [
            {
                Arn: {
                    Ref: "testSnsTopic42942701"
                },
                Id: {
                    "Fn::GetAtt": [
                        "testSnsTopic42942701",
                        "TopicName"
                    ]
                }
            }
        ]
    });
});
test('check properties', () => {
    const stack = new cdk.Stack();
    const construct = deployNewStack(stack);
    expect(construct.eventsRule !== null);
    expect(construct.snsTopic !== null);
    expect(construct.encryptionKey !== null);
});
test('check the sns topic properties', () => {
    const stack = new cdk.Stack();
    deployNewStack(stack);
    expect(stack).toHaveResource('AWS::SNS::Topic', {
        KmsMasterKeyId: {
            "Fn::GetAtt": [
                "testEncryptionKeyB55BFDBC",
                "Arn"
            ]
        }
    });
});
test('check the sns topic properties with existing KMS key', () => {
    const stack = new cdk.Stack();
    const key = defaults.buildEncryptionKey(stack, {
        description: 'my-key'
    });
    const props = {
        eventRuleProps: {
            schedule: events.Schedule.rate(cdk.Duration.minutes(5))
        },
        encryptionKey: key
    };
    new lib_1.EventbridgeToSns(stack, 'test-events-rule-sqs', props);
    expect(stack).toHaveResource('AWS::SNS::Topic', {
        KmsMasterKeyId: {
            "Fn::GetAtt": [
                "EncryptionKey1B843E66",
                "Arn"
            ]
        }
    });
    expect(stack).toHaveResource('AWS::KMS::Key', {
        KeyPolicy: {
            Statement: [
                {
                    Action: [
                        "kms:Create*",
                        "kms:Describe*",
                        "kms:Enable*",
                        "kms:List*",
                        "kms:Put*",
                        "kms:Update*",
                        "kms:Revoke*",
                        "kms:Disable*",
                        "kms:Get*",
                        "kms:Delete*",
                        "kms:ScheduleKeyDeletion",
                        "kms:CancelKeyDeletion",
                        "kms:GenerateDataKey",
                        "kms:TagResource",
                        "kms:UntagResource"
                    ],
                    Effect: "Allow",
                    Principal: {
                        AWS: {
                            "Fn::Join": [
                                "",
                                [
                                    "arn:",
                                    {
                                        Ref: "AWS::Partition"
                                    },
                                    ":iam::",
                                    {
                                        Ref: "AWS::AccountId"
                                    },
                                    ":root"
                                ]
                            ]
                        }
                    },
                    Resource: "*"
                },
                {
                    Action: [
                        "kms:Decrypt",
                        "kms:Encrypt",
                        "kms:ReEncrypt*",
                        "kms:GenerateDataKey*"
                    ],
                    Effect: "Allow",
                    Principal: {
                        Service: "events.amazonaws.com"
                    },
                    Resource: "*"
                }
            ],
            Version: "2012-10-17"
        },
        Description: "my-key",
        EnableKeyRotation: true
    });
});
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZXZlbnRicmlkZ2Utc25zLXRvcGljLnRlc3QuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJldmVudGJyaWRnZS1zbnMtdG9waWMudGVzdC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiO0FBQUE7Ozs7Ozs7Ozs7O0dBV0c7O0FBRUgsNENBQTZDO0FBQzdDLHFDQUFxQztBQUNyQyw4Q0FBOEM7QUFDOUMsMkRBQTJEO0FBQzNELGdDQUE4QjtBQUM5QixnQ0FBaUU7QUFFakUsU0FBUyxjQUFjLENBQUMsS0FBZ0I7SUFDdEMsTUFBTSxLQUFLLEdBQTBCO1FBQ25DLGNBQWMsRUFBRTtZQUNkLFFBQVEsRUFBRSxNQUFNLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsUUFBUSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsQ0FBQztTQUN4RDtLQUNGLENBQUM7SUFDRixPQUFPLElBQUksc0JBQWdCLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRSxLQUFLLENBQUMsQ0FBQztBQUNwRCxDQUFDO0FBRUQsSUFBSSxDQUFDLCtDQUErQyxFQUFFLEdBQUcsRUFBRTtJQUN6RCxNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztJQUM5QixjQUFjLENBQUMsS0FBSyxDQUFDLENBQUM7SUFDdEIsTUFBTSxDQUFDLG1CQUFVLENBQUMsZ0JBQWdCLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxlQUFlLEVBQUUsQ0FBQztBQUMvRCxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyx5R0FBeUcsRUFBRSxHQUFHLEVBQUU7SUFDbkgsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7SUFDOUIsY0FBYyxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBQ3RCLE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxjQUFjLENBQUMsdUJBQXVCLEVBQUU7UUFDcEQsY0FBYyxFQUFFO1lBQ2QsU0FBUyxFQUFFO2dCQUNUO29CQUNFLE1BQU0sRUFBRTt3QkFDTixhQUFhO3dCQUNiLHNCQUFzQjt3QkFDdEIsd0JBQXdCO3dCQUN4QixpQkFBaUI7d0JBQ2pCLDhCQUE4Qjt3QkFDOUIsd0JBQXdCO3dCQUN4QixhQUFhO3dCQUNiLG1CQUFtQjt3QkFDbkIsZUFBZTtxQkFDaEI7b0JBQ0QsU0FBUyxFQUFFO3dCQUNULFlBQVksRUFBRTs0QkFDWixpQkFBaUIsRUFBRTtnQ0FDakIsR0FBRyxFQUFFLGdCQUFnQjs2QkFDdEI7eUJBQ0Y7cUJBQ0Y7b0JBQ0QsTUFBTSxFQUFFLE9BQU87b0JBQ2YsU0FBUyxFQUFFO3dCQUNULEdBQUcsRUFBRTs0QkFDSCxVQUFVLEVBQUU7Z0NBQ1YsRUFBRTtnQ0FDRjtvQ0FDRSxNQUFNO29DQUNOO3dDQUNFLEdBQUcsRUFBRSxnQkFBZ0I7cUNBQ3RCO29DQUNELFFBQVE7b0NBQ1I7d0NBQ0UsR0FBRyxFQUFFLGdCQUFnQjtxQ0FDdEI7b0NBQ0QsT0FBTztpQ0FDUjs2QkFDRjt5QkFDRjtxQkFDRjtvQkFDRCxRQUFRLEVBQUU7d0JBQ1IsR0FBRyxFQUFFLHNCQUFzQjtxQkFDNUI7b0JBQ0QsR0FBRyxFQUFFLHNCQUFzQjtpQkFDNUI7Z0JBQ0Q7b0JBQ0UsTUFBTSxFQUFFO3dCQUNOLGFBQWE7d0JBQ2Isc0JBQXNCO3dCQUN0Qix3QkFBd0I7d0JBQ3hCLGlCQUFpQjt3QkFDakIsOEJBQThCO3dCQUM5Qix3QkFBd0I7d0JBQ3hCLGFBQWE7d0JBQ2IsbUJBQW1CO3dCQUNuQixlQUFlO3FCQUNoQjtvQkFDRCxTQUFTLEVBQUU7d0JBQ1QsSUFBSSxFQUFFOzRCQUNKLHFCQUFxQixFQUFFLE9BQU87eUJBQy9CO3FCQUNGO29CQUNELE1BQU0sRUFBRSxNQUFNO29CQUNkLFNBQVMsRUFBRTt3QkFDVCxHQUFHLEVBQUUsR0FBRztxQkFDVDtvQkFDRCxRQUFRLEVBQUU7d0JBQ1IsR0FBRyxFQUFFLHNCQUFzQjtxQkFDNUI7b0JBQ0QsR0FBRyxFQUFFLFdBQVc7aUJBQ2pCO2dCQUNEO29CQUNFLE1BQU0sRUFBRSxhQUFhO29CQUNyQixNQUFNLEVBQUUsT0FBTztvQkFDZixTQUFTLEVBQUU7d0JBQ1QsT0FBTyxFQUFFLHNCQUFzQjtxQkFDaEM7b0JBQ0QsUUFBUSxFQUFFO3dCQUNSLEdBQUcsRUFBRSxzQkFBc0I7cUJBQzVCO29CQUNELEdBQUcsRUFBRSxHQUFHO2lCQUNUO2FBQ0Y7WUFDRCxPQUFPLEVBQUUsWUFBWTtTQUN0QjtRQUNELE1BQU0sRUFBRTtZQUNOO2dCQUNFLEdBQUcsRUFBRSxzQkFBc0I7YUFDNUI7U0FDRjtLQUNGLENBQ0EsQ0FBQztBQUNKLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLDhCQUE4QixFQUFFLEdBQUcsRUFBRTtJQUN4QyxNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztJQUM5QixjQUFjLENBQUMsS0FBSyxDQUFDLENBQUM7SUFFdEIsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLGNBQWMsQ0FBQyxtQkFBbUIsRUFBRTtRQUNoRCxrQkFBa0IsRUFBRSxpQkFBaUI7UUFDckMsS0FBSyxFQUFFLFNBQVM7UUFDaEIsT0FBTyxFQUFFO1lBQ1A7Z0JBQ0UsR0FBRyxFQUFFO29CQUNILEdBQUcsRUFBRSxzQkFBc0I7aUJBQzVCO2dCQUNELEVBQUUsRUFBRTtvQkFDRixZQUFZLEVBQUU7d0JBQ1osc0JBQXNCO3dCQUN0QixXQUFXO3FCQUNaO2lCQUNGO2FBQ0Y7U0FDRjtLQUNGLENBQUMsQ0FBQztBQUNMLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLGtCQUFrQixFQUFFLEdBQUcsRUFBRTtJQUM1QixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztJQUM5QixNQUFNLFNBQVMsR0FBcUIsY0FBYyxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBRTFELE1BQU0sQ0FBQyxTQUFTLENBQUMsVUFBVSxLQUFLLElBQUksQ0FBQyxDQUFDO0lBQ3RDLE1BQU0sQ0FBQyxTQUFTLENBQUMsUUFBUSxLQUFLLElBQUksQ0FBQyxDQUFDO0lBQ3BDLE1BQU0sQ0FBQyxTQUFTLENBQUMsYUFBYSxLQUFLLElBQUksQ0FBQyxDQUFDO0FBQzNDLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLGdDQUFnQyxFQUFFLEdBQUcsRUFBRTtJQUMxQyxNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztJQUM5QixjQUFjLENBQUMsS0FBSyxDQUFDLENBQUM7SUFDdEIsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLGNBQWMsQ0FBQyxpQkFBaUIsRUFBRTtRQUM5QyxjQUFjLEVBQUU7WUFDZCxZQUFZLEVBQUU7Z0JBQ1osMkJBQTJCO2dCQUMzQixLQUFLO2FBQ047U0FDRjtLQUNGLENBQUMsQ0FBQztBQUNMLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLHNEQUFzRCxFQUFFLEdBQUcsRUFBRTtJQUNoRSxNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztJQUM5QixNQUFNLEdBQUcsR0FBRyxRQUFRLENBQUMsa0JBQWtCLENBQUMsS0FBSyxFQUFFO1FBQzdDLFdBQVcsRUFBRSxRQUFRO0tBQ3RCLENBQUMsQ0FBQztJQUVILE1BQU0sS0FBSyxHQUEwQjtRQUNuQyxjQUFjLEVBQUU7WUFDZCxRQUFRLEVBQUUsTUFBTSxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLFFBQVEsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLENBQUM7U0FDeEQ7UUFDRCxhQUFhLEVBQUUsR0FBRztLQUNuQixDQUFDO0lBRUYsSUFBSSxzQkFBZ0IsQ0FBQyxLQUFLLEVBQUUsc0JBQXNCLEVBQUUsS0FBSyxDQUFDLENBQUM7SUFFM0QsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLGNBQWMsQ0FBQyxpQkFBaUIsRUFBRTtRQUM5QyxjQUFjLEVBQUU7WUFDZCxZQUFZLEVBQUU7Z0JBQ1osdUJBQXVCO2dCQUN2QixLQUFLO2FBQ047U0FDRjtLQUNGLENBQUMsQ0FBQztJQUVILE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxjQUFjLENBQUMsZUFBZSxFQUFFO1FBQzVDLFNBQVMsRUFBRTtZQUNULFNBQVMsRUFBRTtnQkFDVDtvQkFDRSxNQUFNLEVBQUU7d0JBQ04sYUFBYTt3QkFDYixlQUFlO3dCQUNmLGFBQWE7d0JBQ2IsV0FBVzt3QkFDWCxVQUFVO3dCQUNWLGFBQWE7d0JBQ2IsYUFBYTt3QkFDYixjQUFjO3dCQUNkLFVBQVU7d0JBQ1YsYUFBYTt3QkFDYix5QkFBeUI7d0JBQ3pCLHVCQUF1Qjt3QkFDdkIscUJBQXFCO3dCQUNyQixpQkFBaUI7d0JBQ2pCLG1CQUFtQjtxQkFDcEI7b0JBQ0QsTUFBTSxFQUFFLE9BQU87b0JBQ2YsU0FBUyxFQUFFO3dCQUNULEdBQUcsRUFBRTs0QkFDSCxVQUFVLEVBQUU7Z0NBQ1YsRUFBRTtnQ0FDRjtvQ0FDRSxNQUFNO29DQUNOO3dDQUNFLEdBQUcsRUFBRSxnQkFBZ0I7cUNBQ3RCO29DQUNELFFBQVE7b0NBQ1I7d0NBQ0UsR0FBRyxFQUFFLGdCQUFnQjtxQ0FDdEI7b0NBQ0QsT0FBTztpQ0FDUjs2QkFDRjt5QkFDRjtxQkFDRjtvQkFDRCxRQUFRLEVBQUUsR0FBRztpQkFDZDtnQkFDRDtvQkFDRSxNQUFNLEVBQUU7d0JBQ04sYUFBYTt3QkFDYixhQUFhO3dCQUNiLGdCQUFnQjt3QkFDaEIsc0JBQXNCO3FCQUN2QjtvQkFDRCxNQUFNLEVBQUUsT0FBTztvQkFDZixTQUFTLEVBQUU7d0JBQ1QsT0FBTyxFQUFFLHNCQUFzQjtxQkFDaEM7b0JBQ0QsUUFBUSxFQUFFLEdBQUc7aUJBQ2Q7YUFDRjtZQUNELE9BQU8sRUFBRSxZQUFZO1NBQ3RCO1FBQ0QsV0FBVyxFQUFFLFFBQVE7UUFDckIsaUJBQWlCLEVBQUUsSUFBSTtLQUN4QixDQUFDLENBQUM7QUFDTCxDQUFDLENBQUMsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbIi8qKlxuICogIENvcHlyaWdodCAyMDIxIEFtYXpvbi5jb20sIEluYy4gb3IgaXRzIGFmZmlsaWF0ZXMuIEFsbCBSaWdodHMgUmVzZXJ2ZWQuXG4gKlxuICogIExpY2Vuc2VkIHVuZGVyIHRoZSBBcGFjaGUgTGljZW5zZSwgVmVyc2lvbiAyLjAgKHRoZSBcIkxpY2Vuc2VcIikuIFlvdSBtYXkgbm90IHVzZSB0aGlzIGZpbGUgZXhjZXB0IGluIGNvbXBsaWFuY2VcbiAqICB3aXRoIHRoZSBMaWNlbnNlLiBBIGNvcHkgb2YgdGhlIExpY2Vuc2UgaXMgbG9jYXRlZCBhdFxuICpcbiAqICAgICAgaHR0cDovL3d3dy5hcGFjaGUub3JnL2xpY2Vuc2VzL0xJQ0VOU0UtMi4wXG4gKlxuICogIG9yIGluIHRoZSAnbGljZW5zZScgZmlsZSBhY2NvbXBhbnlpbmcgdGhpcyBmaWxlLiBUaGlzIGZpbGUgaXMgZGlzdHJpYnV0ZWQgb24gYW4gJ0FTIElTJyBCQVNJUywgV0lUSE9VVCBXQVJSQU5USUVTXG4gKiAgT1IgQ09ORElUSU9OUyBPRiBBTlkgS0lORCwgZXhwcmVzcyBvciBpbXBsaWVkLiBTZWUgdGhlIExpY2Vuc2UgZm9yIHRoZSBzcGVjaWZpYyBsYW5ndWFnZSBnb3Zlcm5pbmcgcGVybWlzc2lvbnNcbiAqICBhbmQgbGltaXRhdGlvbnMgdW5kZXIgdGhlIExpY2Vuc2UuXG4gKi9cblxuaW1wb3J0IHsgU3ludGhVdGlscyB9IGZyb20gJ0Bhd3MtY2RrL2Fzc2VydCc7XG5pbXBvcnQgKiBhcyBjZGsgZnJvbSBcIkBhd3MtY2RrL2NvcmVcIjtcbmltcG9ydCAqIGFzIGV2ZW50cyBmcm9tIFwiQGF3cy1jZGsvYXdzLWV2ZW50c1wiO1xuaW1wb3J0ICogYXMgZGVmYXVsdHMgZnJvbSAnQGF3cy1zb2x1dGlvbnMtY29uc3RydWN0cy9jb3JlJztcbmltcG9ydCAnQGF3cy1jZGsvYXNzZXJ0L2plc3QnO1xuaW1wb3J0IHsgRXZlbnRicmlkZ2VUb1NucywgRXZlbnRicmlkZ2VUb1Nuc1Byb3BzIH0gZnJvbSBcIi4uL2xpYlwiO1xuXG5mdW5jdGlvbiBkZXBsb3lOZXdTdGFjayhzdGFjazogY2RrLlN0YWNrKSB7XG4gIGNvbnN0IHByb3BzOiBFdmVudGJyaWRnZVRvU25zUHJvcHMgPSB7XG4gICAgZXZlbnRSdWxlUHJvcHM6IHtcbiAgICAgIHNjaGVkdWxlOiBldmVudHMuU2NoZWR1bGUucmF0ZShjZGsuRHVyYXRpb24ubWludXRlcyg1KSlcbiAgICB9XG4gIH07XG4gIHJldHVybiBuZXcgRXZlbnRicmlkZ2VUb1NucyhzdGFjaywgJ3Rlc3QnLCBwcm9wcyk7XG59XG5cbnRlc3QoJ3NuYXBzaG90IHRlc3QgRXZlbnRicmlkZ2VUb1NucyBkZWZhdWx0IHBhcmFtcycsICgpID0+IHtcbiAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gIGRlcGxveU5ld1N0YWNrKHN0YWNrKTtcbiAgZXhwZWN0KFN5bnRoVXRpbHMudG9DbG91ZEZvcm1hdGlvbihzdGFjaykpLnRvTWF0Y2hTbmFwc2hvdCgpO1xufSk7XG5cbnRlc3QoJ2NoZWNrIGlmIHRoZSBldmVudCBydWxlIGhhcyBwZXJtaXNzaW9uL3BvbGljeSBpbiBwbGFjZSBpbiBzbnMgZm9yIGl0IHRvIGJlIGFibGUgdG8gcHVibGlzaCB0byB0aGUgdG9waWMnLCAoKSA9PiB7XG4gIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICBkZXBsb3lOZXdTdGFjayhzdGFjayk7XG4gIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2UoJ0FXUzo6U05TOjpUb3BpY1BvbGljeScsIHtcbiAgICBQb2xpY3lEb2N1bWVudDoge1xuICAgICAgU3RhdGVtZW50OiBbXG4gICAgICAgIHtcbiAgICAgICAgICBBY3Rpb246IFtcbiAgICAgICAgICAgIFwiU05TOlB1Ymxpc2hcIixcbiAgICAgICAgICAgIFwiU05TOlJlbW92ZVBlcm1pc3Npb25cIixcbiAgICAgICAgICAgIFwiU05TOlNldFRvcGljQXR0cmlidXRlc1wiLFxuICAgICAgICAgICAgXCJTTlM6RGVsZXRlVG9waWNcIixcbiAgICAgICAgICAgIFwiU05TOkxpc3RTdWJzY3JpcHRpb25zQnlUb3BpY1wiLFxuICAgICAgICAgICAgXCJTTlM6R2V0VG9waWNBdHRyaWJ1dGVzXCIsXG4gICAgICAgICAgICBcIlNOUzpSZWNlaXZlXCIsXG4gICAgICAgICAgICBcIlNOUzpBZGRQZXJtaXNzaW9uXCIsXG4gICAgICAgICAgICBcIlNOUzpTdWJzY3JpYmVcIlxuICAgICAgICAgIF0sXG4gICAgICAgICAgQ29uZGl0aW9uOiB7XG4gICAgICAgICAgICBTdHJpbmdFcXVhbHM6IHtcbiAgICAgICAgICAgICAgXCJBV1M6U291cmNlT3duZXJcIjoge1xuICAgICAgICAgICAgICAgIFJlZjogXCJBV1M6OkFjY291bnRJZFwiXG4gICAgICAgICAgICAgIH1cbiAgICAgICAgICAgIH1cbiAgICAgICAgICB9LFxuICAgICAgICAgIEVmZmVjdDogXCJBbGxvd1wiLFxuICAgICAgICAgIFByaW5jaXBhbDoge1xuICAgICAgICAgICAgQVdTOiB7XG4gICAgICAgICAgICAgIFwiRm46OkpvaW5cIjogW1xuICAgICAgICAgICAgICAgIFwiXCIsXG4gICAgICAgICAgICAgICAgW1xuICAgICAgICAgICAgICAgICAgXCJhcm46XCIsXG4gICAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAgIFJlZjogXCJBV1M6OlBhcnRpdGlvblwiXG4gICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgICAgXCI6aWFtOjpcIixcbiAgICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICAgUmVmOiBcIkFXUzo6QWNjb3VudElkXCJcbiAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgICBcIjpyb290XCJcbiAgICAgICAgICAgICAgICBdXG4gICAgICAgICAgICAgIF1cbiAgICAgICAgICAgIH1cbiAgICAgICAgICB9LFxuICAgICAgICAgIFJlc291cmNlOiB7XG4gICAgICAgICAgICBSZWY6IFwidGVzdFNuc1RvcGljNDI5NDI3MDFcIlxuICAgICAgICAgIH0sXG4gICAgICAgICAgU2lkOiBcIlRvcGljT3duZXJPbmx5QWNjZXNzXCJcbiAgICAgICAgfSxcbiAgICAgICAge1xuICAgICAgICAgIEFjdGlvbjogW1xuICAgICAgICAgICAgXCJTTlM6UHVibGlzaFwiLFxuICAgICAgICAgICAgXCJTTlM6UmVtb3ZlUGVybWlzc2lvblwiLFxuICAgICAgICAgICAgXCJTTlM6U2V0VG9waWNBdHRyaWJ1dGVzXCIsXG4gICAgICAgICAgICBcIlNOUzpEZWxldGVUb3BpY1wiLFxuICAgICAgICAgICAgXCJTTlM6TGlzdFN1YnNjcmlwdGlvbnNCeVRvcGljXCIsXG4gICAgICAgICAgICBcIlNOUzpHZXRUb3BpY0F0dHJpYnV0ZXNcIixcbiAgICAgICAgICAgIFwiU05TOlJlY2VpdmVcIixcbiAgICAgICAgICAgIFwiU05TOkFkZFBlcm1pc3Npb25cIixcbiAgICAgICAgICAgIFwiU05TOlN1YnNjcmliZVwiXG4gICAgICAgICAgXSxcbiAgICAgICAgICBDb25kaXRpb246IHtcbiAgICAgICAgICAgIEJvb2w6IHtcbiAgICAgICAgICAgICAgXCJhd3M6U2VjdXJlVHJhbnNwb3J0XCI6IFwiZmFsc2VcIlxuICAgICAgICAgICAgfVxuICAgICAgICAgIH0sXG4gICAgICAgICAgRWZmZWN0OiBcIkRlbnlcIixcbiAgICAgICAgICBQcmluY2lwYWw6IHtcbiAgICAgICAgICAgIEFXUzogXCIqXCJcbiAgICAgICAgICB9LFxuICAgICAgICAgIFJlc291cmNlOiB7XG4gICAgICAgICAgICBSZWY6IFwidGVzdFNuc1RvcGljNDI5NDI3MDFcIlxuICAgICAgICAgIH0sXG4gICAgICAgICAgU2lkOiBcIkh0dHBzT25seVwiXG4gICAgICAgIH0sXG4gICAgICAgIHtcbiAgICAgICAgICBBY3Rpb246IFwic25zOlB1Ymxpc2hcIixcbiAgICAgICAgICBFZmZlY3Q6IFwiQWxsb3dcIixcbiAgICAgICAgICBQcmluY2lwYWw6IHtcbiAgICAgICAgICAgIFNlcnZpY2U6IFwiZXZlbnRzLmFtYXpvbmF3cy5jb21cIlxuICAgICAgICAgIH0sXG4gICAgICAgICAgUmVzb3VyY2U6IHtcbiAgICAgICAgICAgIFJlZjogXCJ0ZXN0U25zVG9waWM0Mjk0MjcwMVwiXG4gICAgICAgICAgfSxcbiAgICAgICAgICBTaWQ6IFwiMlwiXG4gICAgICAgIH1cbiAgICAgIF0sXG4gICAgICBWZXJzaW9uOiBcIjIwMTItMTAtMTdcIlxuICAgIH0sXG4gICAgVG9waWNzOiBbXG4gICAgICB7XG4gICAgICAgIFJlZjogXCJ0ZXN0U25zVG9waWM0Mjk0MjcwMVwiXG4gICAgICB9XG4gICAgXVxuICB9XG4gICk7XG59KTtcblxudGVzdCgnY2hlY2sgZXZlbnRzIHJ1bGUgcHJvcGVydGllcycsICgpID0+IHtcbiAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gIGRlcGxveU5ld1N0YWNrKHN0YWNrKTtcblxuICBleHBlY3Qoc3RhY2spLnRvSGF2ZVJlc291cmNlKCdBV1M6OkV2ZW50czo6UnVsZScsIHtcbiAgICBTY2hlZHVsZUV4cHJlc3Npb246IFwicmF0ZSg1IG1pbnV0ZXMpXCIsXG4gICAgU3RhdGU6IFwiRU5BQkxFRFwiLFxuICAgIFRhcmdldHM6IFtcbiAgICAgIHtcbiAgICAgICAgQXJuOiB7XG4gICAgICAgICAgUmVmOiBcInRlc3RTbnNUb3BpYzQyOTQyNzAxXCJcbiAgICAgICAgfSxcbiAgICAgICAgSWQ6IHtcbiAgICAgICAgICBcIkZuOjpHZXRBdHRcIjogW1xuICAgICAgICAgICAgXCJ0ZXN0U25zVG9waWM0Mjk0MjcwMVwiLFxuICAgICAgICAgICAgXCJUb3BpY05hbWVcIlxuICAgICAgICAgIF1cbiAgICAgICAgfVxuICAgICAgfVxuICAgIF1cbiAgfSk7XG59KTtcblxudGVzdCgnY2hlY2sgcHJvcGVydGllcycsICgpID0+IHtcbiAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gIGNvbnN0IGNvbnN0cnVjdDogRXZlbnRicmlkZ2VUb1NucyA9IGRlcGxveU5ld1N0YWNrKHN0YWNrKTtcblxuICBleHBlY3QoY29uc3RydWN0LmV2ZW50c1J1bGUgIT09IG51bGwpO1xuICBleHBlY3QoY29uc3RydWN0LnNuc1RvcGljICE9PSBudWxsKTtcbiAgZXhwZWN0KGNvbnN0cnVjdC5lbmNyeXB0aW9uS2V5ICE9PSBudWxsKTtcbn0pO1xuXG50ZXN0KCdjaGVjayB0aGUgc25zIHRvcGljIHByb3BlcnRpZXMnLCAoKSA9PiB7XG4gIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICBkZXBsb3lOZXdTdGFjayhzdGFjayk7XG4gIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2UoJ0FXUzo6U05TOjpUb3BpYycsIHtcbiAgICBLbXNNYXN0ZXJLZXlJZDoge1xuICAgICAgXCJGbjo6R2V0QXR0XCI6IFtcbiAgICAgICAgXCJ0ZXN0RW5jcnlwdGlvbktleUI1NUJGREJDXCIsXG4gICAgICAgIFwiQXJuXCJcbiAgICAgIF1cbiAgICB9XG4gIH0pO1xufSk7XG5cbnRlc3QoJ2NoZWNrIHRoZSBzbnMgdG9waWMgcHJvcGVydGllcyB3aXRoIGV4aXN0aW5nIEtNUyBrZXknLCAoKSA9PiB7XG4gIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICBjb25zdCBrZXkgPSBkZWZhdWx0cy5idWlsZEVuY3J5cHRpb25LZXkoc3RhY2ssIHtcbiAgICBkZXNjcmlwdGlvbjogJ215LWtleSdcbiAgfSk7XG5cbiAgY29uc3QgcHJvcHM6IEV2ZW50YnJpZGdlVG9TbnNQcm9wcyA9IHtcbiAgICBldmVudFJ1bGVQcm9wczoge1xuICAgICAgc2NoZWR1bGU6IGV2ZW50cy5TY2hlZHVsZS5yYXRlKGNkay5EdXJhdGlvbi5taW51dGVzKDUpKVxuICAgIH0sXG4gICAgZW5jcnlwdGlvbktleToga2V5XG4gIH07XG5cbiAgbmV3IEV2ZW50YnJpZGdlVG9TbnMoc3RhY2ssICd0ZXN0LWV2ZW50cy1ydWxlLXNxcycsIHByb3BzKTtcblxuICBleHBlY3Qoc3RhY2spLnRvSGF2ZVJlc291cmNlKCdBV1M6OlNOUzo6VG9waWMnLCB7XG4gICAgS21zTWFzdGVyS2V5SWQ6IHtcbiAgICAgIFwiRm46OkdldEF0dFwiOiBbXG4gICAgICAgIFwiRW5jcnlwdGlvbktleTFCODQzRTY2XCIsXG4gICAgICAgIFwiQXJuXCJcbiAgICAgIF1cbiAgICB9XG4gIH0pO1xuXG4gIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2UoJ0FXUzo6S01TOjpLZXknLCB7XG4gICAgS2V5UG9saWN5OiB7XG4gICAgICBTdGF0ZW1lbnQ6IFtcbiAgICAgICAge1xuICAgICAgICAgIEFjdGlvbjogW1xuICAgICAgICAgICAgXCJrbXM6Q3JlYXRlKlwiLFxuICAgICAgICAgICAgXCJrbXM6RGVzY3JpYmUqXCIsXG4gICAgICAgICAgICBcImttczpFbmFibGUqXCIsXG4gICAgICAgICAgICBcImttczpMaXN0KlwiLFxuICAgICAgICAgICAgXCJrbXM6UHV0KlwiLFxuICAgICAgICAgICAgXCJrbXM6VXBkYXRlKlwiLFxuICAgICAgICAgICAgXCJrbXM6UmV2b2tlKlwiLFxuICAgICAgICAgICAgXCJrbXM6RGlzYWJsZSpcIixcbiAgICAgICAgICAgIFwia21zOkdldCpcIixcbiAgICAgICAgICAgIFwia21zOkRlbGV0ZSpcIixcbiAgICAgICAgICAgIFwia21zOlNjaGVkdWxlS2V5RGVsZXRpb25cIixcbiAgICAgICAgICAgIFwia21zOkNhbmNlbEtleURlbGV0aW9uXCIsXG4gICAgICAgICAgICBcImttczpHZW5lcmF0ZURhdGFLZXlcIixcbiAgICAgICAgICAgIFwia21zOlRhZ1Jlc291cmNlXCIsXG4gICAgICAgICAgICBcImttczpVbnRhZ1Jlc291cmNlXCJcbiAgICAgICAgICBdLFxuICAgICAgICAgIEVmZmVjdDogXCJBbGxvd1wiLFxuICAgICAgICAgIFByaW5jaXBhbDoge1xuICAgICAgICAgICAgQVdTOiB7XG4gICAgICAgICAgICAgIFwiRm46OkpvaW5cIjogW1xuICAgICAgICAgICAgICAgIFwiXCIsXG4gICAgICAgICAgICAgICAgW1xuICAgICAgICAgICAgICAgICAgXCJhcm46XCIsXG4gICAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAgIFJlZjogXCJBV1M6OlBhcnRpdGlvblwiXG4gICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgICAgXCI6aWFtOjpcIixcbiAgICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICAgUmVmOiBcIkFXUzo6QWNjb3VudElkXCJcbiAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgICBcIjpyb290XCJcbiAgICAgICAgICAgICAgICBdXG4gICAgICAgICAgICAgIF1cbiAgICAgICAgICAgIH1cbiAgICAgICAgICB9LFxuICAgICAgICAgIFJlc291cmNlOiBcIipcIlxuICAgICAgICB9LFxuICAgICAgICB7XG4gICAgICAgICAgQWN0aW9uOiBbXG4gICAgICAgICAgICBcImttczpEZWNyeXB0XCIsXG4gICAgICAgICAgICBcImttczpFbmNyeXB0XCIsXG4gICAgICAgICAgICBcImttczpSZUVuY3J5cHQqXCIsXG4gICAgICAgICAgICBcImttczpHZW5lcmF0ZURhdGFLZXkqXCJcbiAgICAgICAgICBdLFxuICAgICAgICAgIEVmZmVjdDogXCJBbGxvd1wiLFxuICAgICAgICAgIFByaW5jaXBhbDoge1xuICAgICAgICAgICAgU2VydmljZTogXCJldmVudHMuYW1hem9uYXdzLmNvbVwiXG4gICAgICAgICAgfSxcbiAgICAgICAgICBSZXNvdXJjZTogXCIqXCJcbiAgICAgICAgfVxuICAgICAgXSxcbiAgICAgIFZlcnNpb246IFwiMjAxMi0xMC0xN1wiXG4gICAgfSxcbiAgICBEZXNjcmlwdGlvbjogXCJteS1rZXlcIixcbiAgICBFbmFibGVLZXlSb3RhdGlvbjogdHJ1ZVxuICB9KTtcbn0pOyJdfQ==