/****************************************
 * MIT License
 *
 * Copyright (c) 2020 Miguel Ramos Pernas
 ****************************************/

/** Implementation of the Philox algorithm to generate random numbers.
 *
 * Information about the algorithm can be found at:
 *
 *  doi:10.1145/2063384.2063405
 *
 */
#define __philox4x32_a 0xD2511F53UL
#define __philox4x32_b 0xCD9E8D57UL
#define __philox2x64 0xD2B74407B1CE6E93UL
#define __philox4x64_a 0xD2E7470EE14C6C93UL
#define __philox4x64_b 0xCA5A826395121157UL
#define __golden_ratio 0x9e3779b97f4a7c15UL
#define __sqrt_3_minus_1 0xBB67AE8584CAA73BUL
#define __greatest_uint64_as_double                                            \
  18446744073709551616.                        // 0xffffffffffffffffUL
#define __greatest_uint32_as_float 4294967296. // 0xffffffffU

typedef unsigned int uint32;
typedef unsigned long uint64;

/// Keys for Philox
struct Key {
  uint64 _0, _1;
};

/// Hold the four random numbers generated by Philox
struct Counters {
  uint64 _0, _1, _2, _3;
};

/// Multiply the higher and lower parts of a 64-bit integer
WITHIN_KERNEL inline void mulhilo(uint64 a, uint64 b, uint64 *rl, uint64 *rh) {

  *rl = a * b; // low multiply

#ifdef CUDA
  *rh = __umul64hi(a, b);
#elif defined(OPENCL)
  *rh = mul_hi(a, b);
#else // an hypothetical CPU case (it is left for cross-checks)
  const uint32 mask = (((uint64)1) << 32) - 1;

  uint64 ahi = a >> 32;
  uint64 alo = a & mask;
  uint64 bhi = b >> 32;
  uint64 blo = b & mask;

  uint64 ahbl = ahi * blo;
  uint64 albh = alo * bhi;

  uint64 ahbl_albh = ((ahbl & mask) + (albh & mask));

  *rh = ahi * bhi + (ahbl >> 32) + (albh >> 32) + (ahbl_albh >> 32) +
        ((*rh >> 32) < (ahbl_albh & mask));
#endif
}

/// Modify the counters of the generator
WITHIN_KERNEL inline void compute_round(struct Counters *counter,
                                        struct Key *key) {

  uint64 h0, h1, l0, l1;

  mulhilo(__philox4x64_a, counter->_0, &l0, &h0);
  mulhilo(__philox4x64_b, counter->_2, &l1, &h1);

  counter->_0 = h1 ^ counter->_1 ^ key->_0;
  counter->_1 = l1;
  counter->_2 = h0 ^ counter->_3 ^ key->_1;
  counter->_3 = l0;
}

/// Modify the key using the constants
WITHIN_KERNEL inline void bump_key(struct Key *key) {
  key->_0 += __golden_ratio;
  key->_1 += __sqrt_3_minus_1;
}

/** Generate a random sample between 0 and 1 using the Philox RNG
 *
 * Each call fills four numbers of the output array. The number of rounds
 * must be smaller or equal to 16 (10 is the default). The function must
 * be included like:
 *
 * double *generate(uint64 len, uint64 seed) {
 *
 *  double *data = new double[len];
 *
 * for (uint64 i = 0; i < len; i += 4)
 *   philox(len, i, seed, data);
 *
 * return data;
 * }
 */
KERNEL void philox(int length, GLOBAL_MEM double *data, int seed, int ctr,
                   int rounds) {

  uint64 idx = (uint64)4 * get_global_id(0); // each call sets four numbers

  if (idx >= (uint64)length) // pad condition
    return;

  struct Key key; // must provide the first value (the seed)
  key._0 = (uint64)seed;
  key._1 = 0;

  struct Counters counter; // must provide the two first values
  counter._0 = ctr + idx;
  counter._1 = ctr + idx;
  counter._2 = 0;
  counter._3 = 0;

  // Compute the random number after 10 rounds
  for (int i = 0; i < rounds; ++i) {

    compute_round(&counter, &key);

    if (i != rounds - 1)
      bump_key(&key);
  }

  // Set numbers till the index reaches the maximum length
  data[idx] = ((double)counter._0) / __greatest_uint64_as_double;
  if (++idx < length) {
    data[idx] = ((double)counter._1) / __greatest_uint64_as_double;
    if (++idx < length) {
      data[idx] = ((double)counter._2) / __greatest_uint64_as_double;
      if (++idx < length) {
        data[idx] = ((double)counter._3) / __greatest_uint64_as_double;
      }
    }
  }
}
