"use strict";
const assert_1 = require("@aws-cdk/assert");
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const eks = require("../lib");
const CLUSTER_VERSION = eks.KubernetesVersion.V1_16;
module.exports = {
    'can be added to a cluster'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const cluster = new eks.Cluster(stack, 'MyCluster', { version: CLUSTER_VERSION });
        // WHEN
        cluster.addFargateProfile('MyProfile', {
            selectors: [{ namespace: 'default' }],
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Custom::AWSCDK-EKS-FargateProfile', {
            Config: {
                clusterName: { Ref: 'MyCluster8AD82BF8' },
                podExecutionRoleArn: { 'Fn::GetAtt': ['MyClusterfargateprofileMyProfilePodExecutionRole4795C054', 'Arn'] },
                selectors: [{ namespace: 'default' }],
            },
        }));
        test.done();
    },
    'supports specifying a profile name'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const cluster = new eks.Cluster(stack, 'MyCluster', { version: CLUSTER_VERSION });
        // WHEN
        cluster.addFargateProfile('MyProfile', {
            fargateProfileName: 'MyProfileName',
            selectors: [{ namespace: 'default' }],
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Custom::AWSCDK-EKS-FargateProfile', {
            Config: {
                clusterName: { Ref: 'MyCluster8AD82BF8' },
                podExecutionRoleArn: { 'Fn::GetAtt': ['MyClusterfargateprofileMyProfilePodExecutionRole4795C054', 'Arn'] },
                selectors: [{ namespace: 'default' }],
                fargateProfileName: 'MyProfileName',
            },
        }));
        test.done();
    },
    'supports custom execution role'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const cluster = new eks.Cluster(stack, 'MyCluster', { version: CLUSTER_VERSION });
        const myRole = new iam.Role(stack, 'MyRole', { assumedBy: new iam.AnyPrincipal() });
        // WHEN
        cluster.addFargateProfile('MyProfile', {
            podExecutionRole: myRole,
            selectors: [{ namespace: 'default' }],
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Custom::AWSCDK-EKS-FargateProfile', {
            Config: {
                clusterName: { Ref: 'MyCluster8AD82BF8' },
                podExecutionRoleArn: { 'Fn::GetAtt': ['MyRoleF48FFE04', 'Arn'] },
                selectors: [{ namespace: 'default' }],
            },
        }));
        test.done();
    },
    'supports tags through aspects'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const cluster = new eks.Cluster(stack, 'MyCluster', { version: CLUSTER_VERSION });
        // WHEN
        cluster.addFargateProfile('MyProfile', {
            selectors: [{ namespace: 'default' }],
        });
        core_1.Tag.add(stack, 'aspectTag', 'hello');
        core_1.Tag.add(cluster, 'propTag', '123');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Custom::AWSCDK-EKS-FargateProfile', {
            Config: {
                selectors: [{ namespace: 'default' }],
                clusterName: { Ref: 'MyCluster8AD82BF8' },
                podExecutionRoleArn: { 'Fn::GetAtt': ['MyClusterfargateprofileMyProfilePodExecutionRole4795C054', 'Arn'] },
                tags: {
                    propTag: '123',
                    aspectTag: 'hello',
                },
            },
        }));
        test.done();
    },
    'supports specifying vpc'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const cluster = new eks.Cluster(stack, 'MyCluster', { version: CLUSTER_VERSION });
        const vpc = ec2.Vpc.fromVpcAttributes(stack, 'MyVpc', {
            vpcId: 'vpc123',
            availabilityZones: ['az1'],
            privateSubnetIds: ['priv1'],
        });
        // WHEN
        cluster.addFargateProfile('MyProfile', {
            selectors: [{ namespace: 'default' }],
            vpc,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Custom::AWSCDK-EKS-FargateProfile', {
            Config: {
                clusterName: { Ref: 'MyCluster8AD82BF8' },
                podExecutionRoleArn: { 'Fn::GetAtt': ['MyClusterfargateprofileMyProfilePodExecutionRole4795C054', 'Arn'] },
                selectors: [{ namespace: 'default' }],
                subnets: ['priv1'],
            },
        }));
        test.done();
    },
    'fails if there are no selectors or if there are more than 5'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const cluster = new eks.Cluster(stack, 'MyCluster', { version: CLUSTER_VERSION });
        // THEN
        test.throws(() => cluster.addFargateProfile('MyProfile', { selectors: [] }));
        test.throws(() => cluster.addFargateProfile('MyProfile', {
            selectors: [
                { namespace: '1' },
                { namespace: '2' },
                { namespace: '3' },
                { namespace: '4' },
                { namespace: '5' },
                { namespace: '6' },
            ],
        }));
        test.done();
    },
    'FargateCluster creates an EKS cluster fully managed by Fargate'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new eks.FargateCluster(stack, 'FargateCluster', { version: CLUSTER_VERSION });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Custom::AWSCDK-EKS-KubernetesPatch', {
            ResourceName: 'deployment/coredns',
            ResourceNamespace: 'kube-system',
            ApplyPatchJson: '{"spec":{"template":{"metadata":{"annotations":{"eks.amazonaws.com/compute-type":"fargate"}}}}}',
            RestorePatchJson: '{"spec":{"template":{"metadata":{"annotations":{"eks.amazonaws.com/compute-type":"ec2"}}}}}',
            ClusterName: {
                Ref: 'FargateCluster019F03E8',
            },
        }));
        assert_1.expect(stack).to(assert_1.haveResource('Custom::AWSCDK-EKS-FargateProfile', {
            Config: {
                clusterName: {
                    Ref: 'FargateCluster019F03E8',
                },
                podExecutionRoleArn: {
                    'Fn::GetAtt': [
                        'FargateClusterfargateprofiledefaultPodExecutionRole66F2610E',
                        'Arn',
                    ],
                },
                selectors: [
                    { namespace: 'default' },
                    { namespace: 'kube-system' },
                ],
            },
        }));
        test.done();
    },
    'can create FargateCluster with a custom profile'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new eks.FargateCluster(stack, 'FargateCluster', {
            defaultProfile: {
                fargateProfileName: 'my-app', selectors: [{ namespace: 'foo' }, { namespace: 'bar' }],
            },
            version: CLUSTER_VERSION,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Custom::AWSCDK-EKS-FargateProfile', {
            Config: {
                clusterName: {
                    Ref: 'FargateCluster019F03E8',
                },
                fargateProfileName: 'my-app',
                podExecutionRoleArn: {
                    'Fn::GetAtt': [
                        'FargateClusterfargateprofilemyappPodExecutionRole875B4635',
                        'Arn',
                    ],
                },
                selectors: [
                    { namespace: 'foo' },
                    { namespace: 'bar' },
                ],
            },
        }));
        test.done();
    },
    'custom profile name is "custom" if no custom profile name is provided'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new eks.FargateCluster(stack, 'FargateCluster', {
            defaultProfile: {
                selectors: [{ namespace: 'foo' }, { namespace: 'bar' }],
            },
            version: CLUSTER_VERSION,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Custom::AWSCDK-EKS-FargateProfile', {
            Config: {
                clusterName: {
                    Ref: 'FargateCluster019F03E8',
                },
                podExecutionRoleArn: {
                    'Fn::GetAtt': [
                        'FargateClusterfargateprofilecustomPodExecutionRoleDB415F19',
                        'Arn',
                    ],
                },
                selectors: [
                    { namespace: 'foo' },
                    { namespace: 'bar' },
                ],
            },
        }));
        test.done();
    },
    'multiple Fargate profiles added to a cluster are processed sequentially'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const cluster = new eks.Cluster(stack, 'MyCluster', { version: CLUSTER_VERSION });
        // WHEN
        cluster.addFargateProfile('MyProfile1', {
            selectors: [{ namespace: 'namespace1' }],
        });
        cluster.addFargateProfile('MyProfile2', {
            selectors: [{ namespace: 'namespace2' }],
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Custom::AWSCDK-EKS-FargateProfile', {
            Config: {
                clusterName: { Ref: 'MyCluster8AD82BF8' },
                podExecutionRoleArn: { 'Fn::GetAtt': ['MyClusterfargateprofileMyProfile1PodExecutionRole794E9E37', 'Arn'] },
                selectors: [{ namespace: 'namespace1' }],
            },
        }));
        assert_1.expect(stack).to(assert_1.haveResource('Custom::AWSCDK-EKS-FargateProfile', {
            Properties: {
                ServiceToken: {
                    'Fn::GetAtt': [
                        'awscdkawseksClusterResourceProviderNestedStackawscdkawseksClusterResourceProviderNestedStackResource9827C454',
                        'Outputs.awscdkawseksClusterResourceProviderframeworkonEventEA97AA31Arn',
                    ],
                },
                AssumeRoleArn: { 'Fn::GetAtt': ['MyClusterCreationRoleB5FA4FF3', 'Arn'] },
                Config: {
                    clusterName: { Ref: 'MyCluster8AD82BF8' },
                    podExecutionRoleArn: { 'Fn::GetAtt': ['MyClusterfargateprofileMyProfile2PodExecutionRoleD1151CCF', 'Arn'] },
                    selectors: [{ namespace: 'namespace2' }],
                },
            },
            DependsOn: [
                'MyClusterfargateprofileMyProfile1PodExecutionRole794E9E37',
                'MyClusterfargateprofileMyProfile1879D501A',
            ],
        }, assert_1.ResourcePart.CompleteDefinition));
        test.done();
    },
    'fargate role is added to RBAC'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new eks.FargateCluster(stack, 'FargateCluster', { version: CLUSTER_VERSION });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Custom::AWSCDK-EKS-KubernetesResource', {
            Manifest: {
                'Fn::Join': [
                    '',
                    [
                        '[{"apiVersion":"v1","kind":"ConfigMap","metadata":{"name":"aws-auth","namespace":"kube-system"},"data":{"mapRoles":"[{\\"rolearn\\":\\"',
                        {
                            'Fn::GetAtt': [
                                'FargateClusterMastersRole50BAF9FD',
                                'Arn',
                            ],
                        },
                        '\\",\\"username\\":\\"',
                        {
                            'Fn::GetAtt': [
                                'FargateClusterMastersRole50BAF9FD',
                                'Arn',
                            ],
                        },
                        '\\",\\"groups\\":[\\"system:masters\\"]},{\\"rolearn\\":\\"',
                        {
                            'Fn::GetAtt': [
                                'FargateClusterfargateprofiledefaultPodExecutionRole66F2610E',
                                'Arn',
                            ],
                        },
                        '\\",\\"username\\":\\"system:node:{{SessionName}}\\",\\"groups\\":[\\"system:bootstrappers\\",\\"system:nodes\\",\\"system:node-proxier\\"]}]","mapUsers":"[]","mapAccounts":"[]"}}]',
                    ],
                ],
            },
        }));
        test.done();
    },
    'allow cluster creation role to iam:PassRole on fargate pod execution role'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new eks.FargateCluster(stack, 'FargateCluster', { version: CLUSTER_VERSION });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: 'iam:PassRole',
                        Effect: 'Allow',
                        Resource: {
                            'Fn::GetAtt': [
                                'FargateClusterRole8E36B33A',
                                'Arn',
                            ],
                        },
                    },
                    {
                        Action: [
                            'ec2:DescribeSubnets',
                            'ec2:DescribeRouteTables',
                        ],
                        Effect: 'Allow',
                        Resource: '*',
                    },
                    {
                        Action: [
                            'eks:CreateCluster',
                            'eks:DescribeCluster',
                            'eks:DescribeUpdate',
                            'eks:DeleteCluster',
                            'eks:UpdateClusterVersion',
                            'eks:UpdateClusterConfig',
                            'eks:CreateFargateProfile',
                            'eks:TagResource',
                            'eks:UntagResource',
                        ],
                        Effect: 'Allow',
                        Resource: [
                            '*',
                        ],
                    },
                    {
                        Action: [
                            'eks:DescribeFargateProfile',
                            'eks:DeleteFargateProfile',
                        ],
                        Effect: 'Allow',
                        Resource: '*',
                    },
                    {
                        Action: ['iam:GetRole', 'iam:listAttachedRolePolicies'],
                        Effect: 'Allow',
                        Resource: '*',
                    },
                    {
                        Action: 'iam:CreateServiceLinkedRole',
                        Effect: 'Allow',
                        Resource: '*',
                    },
                    {
                        Action: 'ec2:DescribeVpcs',
                        Effect: 'Allow',
                        Resource: {
                            'Fn::Join': [
                                '',
                                [
                                    'arn:',
                                    {
                                        Ref: 'AWS::Partition',
                                    },
                                    ':ec2:',
                                    {
                                        Ref: 'AWS::Region',
                                    },
                                    ':',
                                    {
                                        Ref: 'AWS::AccountId',
                                    },
                                    ':vpc/',
                                    {
                                        Ref: 'FargateClusterDefaultVpcE69D3A13',
                                    },
                                ],
                            ],
                        },
                    },
                    {
                        Action: 'iam:PassRole',
                        Effect: 'Allow',
                        Resource: {
                            'Fn::GetAtt': [
                                'FargateClusterfargateprofiledefaultPodExecutionRole66F2610E',
                                'Arn',
                            ],
                        },
                    },
                ],
            },
        }));
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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