"use strict";
const cluster_1 = require("../lib/cluster-resource-handler/cluster");
const mocks = require("./cluster-resource-handler-mocks");
module.exports = {
    setUp(callback) {
        mocks.reset();
        callback();
    },
    create: {
        async 'onCreate: minimal defaults (vpc + role)'(test) {
            const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Create', mocks.MOCK_PROPS));
            await handler.onEvent();
            test.deepEqual(mocks.actualRequest.configureAssumeRoleRequest, {
                RoleArn: mocks.MOCK_ASSUME_ROLE_ARN,
                RoleSessionName: 'AWSCDK.EKSCluster.Create.fake-request-id',
            });
            test.deepEqual(mocks.actualRequest.createClusterRequest, {
                roleArn: 'arn:of:role',
                resourcesVpcConfig: {
                    subnetIds: ['subnet1', 'subnet2'],
                    securityGroupIds: ['sg1', 'sg2', 'sg3'],
                },
                name: 'MyResourceId-fakerequestid',
            });
            test.done();
        },
        async 'generated cluster name does not exceed 100 characters'(test) {
            var _a, _b;
            // GIVEN
            const req = {
                StackId: 'fake-stack-id',
                RequestId: '602c078a-6181-4352-9676-4f00352445aa',
                ResourceType: 'Custom::EKSCluster',
                ServiceToken: 'boom',
                LogicalResourceId: 'hello'.repeat(30),
                ResponseURL: 'http://response-url',
                RequestType: 'Create',
                ResourceProperties: {
                    ServiceToken: 'boom',
                    Config: mocks.MOCK_PROPS,
                    AssumeRoleArn: mocks.MOCK_ASSUME_ROLE_ARN,
                },
            };
            // WHEN
            const handler = new cluster_1.ClusterResourceHandler(mocks.client, req);
            await handler.onEvent();
            // THEN
            test.equal((_a = mocks.actualRequest.createClusterRequest) === null || _a === void 0 ? void 0 : _a.name.length, 100);
            test.deepEqual((_b = mocks.actualRequest.createClusterRequest) === null || _b === void 0 ? void 0 : _b.name, 'hellohellohellohellohellohellohellohellohellohellohellohellohellohe-602c078a6181435296764f00352445aa');
            test.done();
        },
        async 'onCreate: explicit cluster name'(test) {
            const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Create', {
                ...mocks.MOCK_PROPS,
                name: 'ExplicitCustomName',
            }));
            await handler.onEvent();
            test.deepEqual(mocks.actualRequest.createClusterRequest.name, 'ExplicitCustomName');
            test.done();
        },
        async 'with no specific version'(test) {
            const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Create', {
                ...mocks.MOCK_PROPS,
                version: '12.34.56',
            }));
            await handler.onEvent();
            test.deepEqual(mocks.actualRequest.createClusterRequest.version, '12.34.56');
            test.done();
        },
        async 'isCreateComplete still not complete if cluster is not ACTIVE'(test) {
            const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Create'));
            mocks.simulateResponse.describeClusterResponseMockStatus = 'CREATING';
            const resp = await handler.isComplete();
            test.deepEqual(mocks.actualRequest.describeClusterRequest.name, 'physical-resource-id');
            test.deepEqual(resp, { IsComplete: false });
            test.done();
        },
        async 'isCreateComplete throws if cluster is FAILED'(test) {
            const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Create'));
            mocks.simulateResponse.describeClusterResponseMockStatus = 'FAILED';
            try {
                await handler.isComplete();
                test.ok(false, 'expected error to be thrown');
            }
            catch (err) {
                test.equal(err.message, 'Cluster is in a FAILED status');
            }
            test.done();
        },
        async 'isUpdateComplete throws if cluster is FAILED'(test) {
            const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Update'));
            mocks.simulateResponse.describeClusterResponseMockStatus = 'FAILED';
            try {
                await handler.isComplete();
                test.ok(false, 'expected error to be thrown');
            }
            catch (err) {
                test.equal(err.message, 'Cluster is in a FAILED status');
            }
            test.done();
        },
        async 'isCreateComplete is complete when cluster is ACTIVE'(test) {
            const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Create'));
            mocks.simulateResponse.describeClusterResponseMockStatus = 'ACTIVE';
            const resp = await handler.isComplete();
            test.deepEqual(resp, {
                IsComplete: true,
                Data: {
                    Name: 'physical-resource-id',
                    Endpoint: 'http://endpoint',
                    Arn: 'arn:cluster-arn',
                    CertificateAuthorityData: 'certificateAuthority-data',
                    ClusterSecurityGroupId: '',
                    EncryptionConfigKeyArn: '',
                    OpenIdConnectIssuerUrl: '',
                    OpenIdConnectIssuer: '',
                },
            });
            test.done();
        },
    },
    delete: {
        async 'returns correct physical name'(test) {
            const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Delete'));
            const resp = await handler.onEvent();
            test.deepEqual(mocks.actualRequest.deleteClusterRequest.name, 'physical-resource-id');
            test.deepEqual(resp, { PhysicalResourceId: 'physical-resource-id' });
            test.done();
        },
        async 'onDelete ignores ResourceNotFoundException'(test) {
            const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Delete'));
            mocks.simulateResponse.deleteClusterErrorCode = 'ResourceNotFoundException';
            await handler.onEvent();
            test.done();
        },
        async 'isDeleteComplete returns false as long as describeCluster succeeds'(test) {
            const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Delete'));
            const resp = await handler.isComplete();
            test.deepEqual(mocks.actualRequest.describeClusterRequest.name, 'physical-resource-id');
            test.ok(!resp.IsComplete);
            test.done();
        },
        async 'isDeleteComplete returns true when describeCluster throws a ResourceNotFound exception'(test) {
            const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Delete'));
            mocks.simulateResponse.describeClusterExceptionCode = 'ResourceNotFoundException';
            const resp = await handler.isComplete();
            test.ok(resp.IsComplete);
            test.done();
        },
        async 'isDeleteComplete propagates other errors'(test) {
            const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Delete'));
            mocks.simulateResponse.describeClusterExceptionCode = 'OtherException';
            let error;
            try {
                await handler.isComplete();
            }
            catch (e) {
                error = e;
            }
            test.equal(error.code, 'OtherException');
            test.done();
        },
    },
    update: {
        async 'no change'(test) {
            const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Update', mocks.MOCK_PROPS, mocks.MOCK_PROPS));
            const resp = await handler.onEvent();
            test.equal(resp, undefined);
            test.equal(mocks.actualRequest.createClusterRequest, undefined);
            test.equal(mocks.actualRequest.updateClusterConfigRequest, undefined);
            test.equal(mocks.actualRequest.updateClusterVersionRequest, undefined);
            test.done();
        },
        async 'isUpdateComplete is not complete when status is not ACTIVE'(test) {
            const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Update'));
            mocks.simulateResponse.describeClusterResponseMockStatus = 'UPDATING';
            const resp = await handler.isComplete();
            test.deepEqual(resp.IsComplete, false);
            test.done();
        },
        async 'isUpdateComplete waits for ACTIVE'(test) {
            const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Update'));
            mocks.simulateResponse.describeClusterResponseMockStatus = 'ACTIVE';
            const resp = await handler.isComplete();
            test.deepEqual(resp.IsComplete, true);
            test.done();
        },
        'requires replacement': {
            'name change': {
                async 'explicit name change'(test) {
                    // GIVEN
                    const req = mocks.newRequest('Update', {
                        ...mocks.MOCK_PROPS,
                        name: 'new-cluster-name-1234',
                    }, {
                        name: 'old-cluster-name',
                    });
                    const handler = new cluster_1.ClusterResourceHandler(mocks.client, req);
                    // WHEN
                    const resp = await handler.onEvent();
                    // THEN
                    test.deepEqual(mocks.actualRequest.createClusterRequest, {
                        name: 'new-cluster-name-1234',
                        roleArn: 'arn:of:role',
                        resourcesVpcConfig: {
                            subnetIds: ['subnet1', 'subnet2'],
                            securityGroupIds: ['sg1', 'sg2', 'sg3'],
                        },
                    });
                    test.deepEqual(resp, { PhysicalResourceId: 'new-cluster-name-1234' });
                    test.done();
                },
                async 'from auto-gen name to explicit name'(test) {
                    // GIVEN
                    const req = mocks.newRequest('Update', {
                        ...mocks.MOCK_PROPS,
                        name: undefined,
                    }, {
                        name: 'explicit',
                    });
                    const handler = new cluster_1.ClusterResourceHandler(mocks.client, req);
                    // WHEN
                    const resp = await handler.onEvent();
                    // THEN
                    test.deepEqual(mocks.actualRequest.createClusterRequest, {
                        name: 'MyResourceId-fakerequestid',
                        roleArn: 'arn:of:role',
                        resourcesVpcConfig: {
                            subnetIds: ['subnet1', 'subnet2'],
                            securityGroupIds: ['sg1', 'sg2', 'sg3'],
                        },
                    });
                    test.deepEqual(resp, { PhysicalResourceId: 'MyResourceId-fakerequestid' });
                    test.done();
                },
            },
            async 'subnets or security groups requires a replacement'(test) {
                const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Update', {
                    ...mocks.MOCK_PROPS,
                    resourcesVpcConfig: {
                        subnetIds: ['subnet1', 'subnet2'],
                        securityGroupIds: ['sg1'],
                    },
                }, {
                    ...mocks.MOCK_PROPS,
                    resourcesVpcConfig: {
                        subnetIds: ['subnet1'],
                        securityGroupIds: ['sg2'],
                    },
                }));
                const resp = await handler.onEvent();
                test.deepEqual(resp, { PhysicalResourceId: 'MyResourceId-fakerequestid' });
                test.deepEqual(mocks.actualRequest.createClusterRequest, {
                    name: 'MyResourceId-fakerequestid',
                    roleArn: 'arn:of:role',
                    resourcesVpcConfig: {
                        subnetIds: ['subnet1', 'subnet2'],
                        securityGroupIds: ['sg1'],
                    },
                });
                test.done();
            },
            async '"roleArn" requires a replacement'(test) {
                const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Update', {
                    roleArn: 'new-arn',
                }, {
                    roleArn: 'old-arn',
                }));
                const resp = await handler.onEvent();
                test.deepEqual(resp, { PhysicalResourceId: 'MyResourceId-fakerequestid' });
                test.deepEqual(mocks.actualRequest.createClusterRequest, {
                    name: 'MyResourceId-fakerequestid',
                    roleArn: 'new-arn',
                });
                test.done();
            },
            async 'fails if cluster has an explicit "name" that is the same as the old "name"'(test) {
                // GIVEN
                const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Update', {
                    roleArn: 'new-arn',
                    name: 'explicit-cluster-name',
                }, {
                    roleArn: 'old-arn',
                    name: 'explicit-cluster-name',
                }));
                // THEN
                let err;
                try {
                    await handler.onEvent();
                }
                catch (e) {
                    err = e;
                }
                test.equal(err === null || err === void 0 ? void 0 : err.message, 'Cannot replace cluster "explicit-cluster-name" since it has an explicit physical name. Either rename the cluster or remove the "name" configuration');
                test.done();
            },
            async 'succeeds if cluster had an explicit "name" and now it does not'(test) {
                // GIVEN
                const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Update', {
                    roleArn: 'new-arn',
                    name: undefined,
                }, {
                    roleArn: 'old-arn',
                    name: 'explicit-cluster-name',
                }));
                // WHEN
                const resp = await handler.onEvent();
                // THEN
                test.deepEqual(resp, { PhysicalResourceId: 'MyResourceId-fakerequestid' });
                test.deepEqual(mocks.actualRequest.createClusterRequest, {
                    name: 'MyResourceId-fakerequestid',
                    roleArn: 'new-arn',
                });
                test.done();
            },
            async 'succeeds if cluster had an explicit "name" and now it has a different name'(test) {
                // GIVEN
                const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Update', {
                    roleArn: 'new-arn',
                    name: 'new-explicit-cluster-name',
                }, {
                    roleArn: 'old-arn',
                    name: 'old-explicit-cluster-name',
                }));
                // WHEN
                const resp = await handler.onEvent();
                // THEN
                test.deepEqual(resp, { PhysicalResourceId: 'new-explicit-cluster-name' });
                test.deepEqual(mocks.actualRequest.createClusterRequest, {
                    name: 'new-explicit-cluster-name',
                    roleArn: 'new-arn',
                });
                test.done();
            },
        },
        'isUpdateComplete with EKS update ID': {
            async 'with "Failed" status'(test) {
                const event = mocks.newRequest('Update');
                const isCompleteHandler = new cluster_1.ClusterResourceHandler(mocks.client, {
                    ...event,
                    EksUpdateId: 'foobar',
                });
                mocks.simulateResponse.describeUpdateResponseMockStatus = 'Failed';
                mocks.simulateResponse.describeUpdateResponseMockErrors = [
                    {
                        errorMessage: 'errorMessageMock',
                        errorCode: 'errorCodeMock',
                        resourceIds: [
                            'foo', 'bar',
                        ],
                    },
                ];
                let error;
                try {
                    await isCompleteHandler.isComplete();
                }
                catch (e) {
                    error = e;
                }
                test.ok(error);
                test.deepEqual(mocks.actualRequest.describeUpdateRequest, { name: 'physical-resource-id', updateId: 'foobar' });
                test.equal(error.message, 'cluster update id "foobar" failed with errors: [{"errorMessage":"errorMessageMock","errorCode":"errorCodeMock","resourceIds":["foo","bar"]}]');
                test.done();
            },
            async 'with "InProgress" status, returns IsComplete=false'(test) {
                const event = mocks.newRequest('Update');
                const isCompleteHandler = new cluster_1.ClusterResourceHandler(mocks.client, {
                    ...event,
                    EksUpdateId: 'foobar',
                });
                mocks.simulateResponse.describeUpdateResponseMockStatus = 'InProgress';
                const response = await isCompleteHandler.isComplete();
                test.deepEqual(mocks.actualRequest.describeUpdateRequest, { name: 'physical-resource-id', updateId: 'foobar' });
                test.equal(response.IsComplete, false);
                test.done();
            },
            async 'with "Successful" status, returns IsComplete=true with "Data"'(test) {
                const event = mocks.newRequest('Update');
                const isCompleteHandler = new cluster_1.ClusterResourceHandler(mocks.client, {
                    ...event,
                    EksUpdateId: 'foobar',
                });
                mocks.simulateResponse.describeUpdateResponseMockStatus = 'Successful';
                const response = await isCompleteHandler.isComplete();
                test.deepEqual(mocks.actualRequest.describeUpdateRequest, { name: 'physical-resource-id', updateId: 'foobar' });
                test.deepEqual(response, {
                    IsComplete: true,
                    Data: {
                        Name: 'physical-resource-id',
                        Endpoint: 'http://endpoint',
                        Arn: 'arn:cluster-arn',
                        CertificateAuthorityData: 'certificateAuthority-data',
                        ClusterSecurityGroupId: '',
                        EncryptionConfigKeyArn: '',
                        OpenIdConnectIssuerUrl: '',
                        OpenIdConnectIssuer: '',
                    },
                });
                test.done();
            },
        },
        'in-place': {
            'version change': {
                async 'from undefined to a specific value'(test) {
                    const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Update', {
                        version: '12.34',
                    }, {
                        version: undefined,
                    }));
                    const resp = await handler.onEvent();
                    test.deepEqual(resp, { EksUpdateId: mocks.MOCK_UPDATE_STATUS_ID });
                    test.deepEqual(mocks.actualRequest.updateClusterVersionRequest, {
                        name: 'physical-resource-id',
                        version: '12.34',
                    });
                    test.equal(mocks.actualRequest.createClusterRequest, undefined);
                    test.done();
                },
                async 'from a specific value to another value'(test) {
                    const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Update', {
                        version: '2.0',
                    }, {
                        version: '1.1',
                    }));
                    const resp = await handler.onEvent();
                    test.deepEqual(resp, { EksUpdateId: mocks.MOCK_UPDATE_STATUS_ID });
                    test.deepEqual(mocks.actualRequest.updateClusterVersionRequest, {
                        name: 'physical-resource-id',
                        version: '2.0',
                    });
                    test.equal(mocks.actualRequest.createClusterRequest, undefined);
                    test.done();
                },
                async 'to a new value that is already the current version'(test) {
                    const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Update', {
                        version: '1.0',
                    }));
                    const resp = await handler.onEvent();
                    test.equal(resp, undefined);
                    test.deepEqual(mocks.actualRequest.describeClusterRequest, { name: 'physical-resource-id' });
                    test.equal(mocks.actualRequest.updateClusterVersionRequest, undefined);
                    test.equal(mocks.actualRequest.createClusterRequest, undefined);
                    test.done();
                },
                async 'fails from specific value to undefined'(test) {
                    const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Update', {
                        version: undefined,
                    }, {
                        version: '1.2',
                    }));
                    let error;
                    try {
                        await handler.onEvent();
                    }
                    catch (e) {
                        error = e;
                    }
                    test.equal(error.message, 'Cannot remove cluster version configuration. Current version is 1.2');
                    test.done();
                },
            },
            'logging or access change': {
                async 'from undefined to partial logging enabled'(test) {
                    const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Update', {
                        logging: {
                            clusterLogging: [
                                {
                                    types: ['api'],
                                    enabled: true,
                                },
                            ],
                        },
                    }, {
                        logging: undefined,
                    }));
                    const resp = await handler.onEvent();
                    test.deepEqual(resp, { EksUpdateId: mocks.MOCK_UPDATE_STATUS_ID });
                    test.deepEqual(mocks.actualRequest.updateClusterConfigRequest, {
                        name: 'physical-resource-id',
                        logging: {
                            clusterLogging: [
                                {
                                    types: ['api'],
                                    enabled: true,
                                },
                            ],
                        },
                    });
                    test.equal(mocks.actualRequest.createClusterRequest, undefined);
                    test.done();
                },
                async 'from partial vpc configuration to only private access enabled'(test) {
                    const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Update', {
                        resourcesVpcConfig: {
                            securityGroupIds: ['sg1', 'sg2', 'sg3'],
                            endpointPrivateAccess: true,
                        },
                    }, {
                        resourcesVpcConfig: {
                            securityGroupIds: ['sg1', 'sg2', 'sg3'],
                        },
                    }));
                    const resp = await handler.onEvent();
                    test.deepEqual(resp, { EksUpdateId: mocks.MOCK_UPDATE_STATUS_ID });
                    test.deepEqual(mocks.actualRequest.updateClusterConfigRequest, {
                        name: 'physical-resource-id',
                        logging: undefined,
                        resourcesVpcConfig: {
                            endpointPrivateAccess: true,
                            endpointPublicAccess: undefined,
                            publicAccessCidrs: undefined,
                        },
                    });
                    test.equal(mocks.actualRequest.createClusterRequest, undefined);
                    test.done();
                },
                async 'from undefined to both logging and access fully enabled'(test) {
                    const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Update', {
                        logging: {
                            clusterLogging: [
                                {
                                    types: ['api', 'audit', 'authenticator', 'controllerManager', 'scheduler'],
                                    enabled: true,
                                },
                            ],
                        },
                        resourcesVpcConfig: {
                            endpointPrivateAccess: true,
                            endpointPublicAccess: true,
                            publicAccessCidrs: ['0.0.0.0/0'],
                        },
                    }, {
                        logging: undefined,
                        resourcesVpcConfig: undefined,
                    }));
                    const resp = await handler.onEvent();
                    test.deepEqual(resp, { EksUpdateId: mocks.MOCK_UPDATE_STATUS_ID });
                    test.deepEqual(mocks.actualRequest.updateClusterConfigRequest, {
                        name: 'physical-resource-id',
                        logging: {
                            clusterLogging: [
                                {
                                    types: ['api', 'audit', 'authenticator', 'controllerManager', 'scheduler'],
                                    enabled: true,
                                },
                            ],
                        },
                        resourcesVpcConfig: {
                            endpointPrivateAccess: true,
                            endpointPublicAccess: true,
                            publicAccessCidrs: ['0.0.0.0/0'],
                        },
                    });
                    test.equal(mocks.actualRequest.createClusterRequest, undefined);
                    test.done();
                },
            },
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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