"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/// !cdk-integ pragma:ignore-assets
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const eks = require("../lib");
const hello = require("./hello-k8s");
const pinger_1 = require("./pinger/pinger");
const util_1 = require("./util");
class EksClusterStack extends util_1.TestStack {
    constructor(scope, id) {
        super(scope, id);
        // allow all account users to assume this role in order to admin the cluster
        const mastersRole = new iam.Role(this, 'AdminRole', {
            assumedBy: new iam.AccountRootPrincipal(),
        });
        // just need one nat gateway to simplify the test
        this.vpc = new ec2.Vpc(this, 'Vpc', { maxAzs: 3, natGateways: 1 });
        // create the cluster with a default nodegroup capacity
        this.cluster = new eks.Cluster(this, 'Cluster', {
            vpc: this.vpc,
            mastersRole,
            defaultCapacity: 2,
            version: eks.KubernetesVersion.V1_16,
        });
        this.assertFargateProfile();
        this.assertCapacity();
        this.assertBottlerocket();
        this.assertSpotCapacity();
        this.assertInferenceInstances();
        this.assertNodeGroup();
        this.assertSimpleManifest();
        this.assertSimpleHelmChart();
        this.assertCreateNamespace();
        this.assertServiceAccount();
        this.assertServiceLoadBalancerAddress();
        new core_1.CfnOutput(this, 'ClusterEndpoint', { value: this.cluster.clusterEndpoint });
        new core_1.CfnOutput(this, 'ClusterArn', { value: this.cluster.clusterArn });
        new core_1.CfnOutput(this, 'ClusterCertificateAuthorityData', { value: this.cluster.clusterCertificateAuthorityData });
        new core_1.CfnOutput(this, 'ClusterSecurityGroupId', { value: this.cluster.clusterSecurityGroupId });
        new core_1.CfnOutput(this, 'ClusterEncryptionConfigKeyArn', { value: this.cluster.clusterEncryptionConfigKeyArn });
        new core_1.CfnOutput(this, 'ClusterName', { value: this.cluster.clusterName });
    }
    assertServiceAccount() {
        // add a service account connected to a IAM role
        this.cluster.addServiceAccount('MyServiceAccount');
    }
    assertCreateNamespace() {
        // deploy an nginx ingress in a namespace
        const nginxNamespace = this.cluster.addManifest('nginx-namespace', {
            apiVersion: 'v1',
            kind: 'Namespace',
            metadata: {
                name: 'nginx',
            },
        });
        const nginxIngress = this.cluster.addChart('nginx-ingress', {
            chart: 'nginx-ingress',
            repository: 'https://helm.nginx.com/stable',
            namespace: 'nginx',
            wait: true,
            createNamespace: false,
            timeout: core_1.Duration.minutes(15),
        });
        // make sure namespace is deployed before the chart
        nginxIngress.node.addDependency(nginxNamespace);
    }
    assertSimpleHelmChart() {
        // deploy the Kubernetes dashboard through a helm chart
        this.cluster.addChart('dashboard', {
            chart: 'kubernetes-dashboard',
            repository: 'https://kubernetes.github.io/dashboard/',
        });
    }
    assertSimpleManifest() {
        // apply a kubernetes manifest
        this.cluster.addManifest('HelloApp', ...hello.resources);
    }
    assertNodeGroup() {
        // add a extra nodegroup
        this.cluster.addNodegroup('extra-ng', {
            instanceType: new ec2.InstanceType('t3.small'),
            minSize: 1,
            // reusing the default capacity nodegroup instance role when available
            nodeRole: this.cluster.defaultCapacity ? this.cluster.defaultCapacity.role : undefined,
        });
    }
    assertInferenceInstances() {
        // inference instances
        this.cluster.addCapacity('InferenceInstances', {
            instanceType: new ec2.InstanceType('inf1.2xlarge'),
            minCapacity: 1,
        });
    }
    assertSpotCapacity() {
        // spot instances (up to 10)
        this.cluster.addCapacity('spot', {
            spotPrice: '0.1094',
            instanceType: new ec2.InstanceType('t3.large'),
            maxCapacity: 10,
            bootstrapOptions: {
                kubeletExtraArgs: '--node-labels foo=bar,goo=far',
                awsApiRetryAttempts: 5,
            },
        });
    }
    assertBottlerocket() {
        // add bottlerocket nodes
        this.cluster.addCapacity('BottlerocketNodes', {
            instanceType: new ec2.InstanceType('t3.small'),
            minCapacity: 2,
            machineImageType: eks.MachineImageType.BOTTLEROCKET,
        });
    }
    assertCapacity() {
        // add some capacity to the cluster. The IAM instance role will
        // automatically be mapped via aws-auth to allow nodes to join the cluster.
        this.cluster.addCapacity('Nodes', {
            instanceType: new ec2.InstanceType('t2.medium'),
            minCapacity: 3,
        });
    }
    assertFargateProfile() {
        // fargate profile for resources in the "default" namespace
        this.cluster.addFargateProfile('default', {
            selectors: [{ namespace: 'default' }],
        });
    }
    assertServiceLoadBalancerAddress() {
        const serviceName = 'webservice';
        const labels = { app: 'simple-web' };
        const containerPort = 80;
        const servicePort = 9000;
        const pingerSecurityGroup = new ec2.SecurityGroup(this, 'WebServiceSecurityGroup', {
            vpc: this.vpc,
        });
        pingerSecurityGroup.addIngressRule(pingerSecurityGroup, ec2.Port.tcp(servicePort), `allow http ${servicePort} access from myself`);
        this.cluster.addManifest('simple-web-pod', {
            kind: 'Pod',
            apiVersion: 'v1',
            metadata: { name: 'webpod', labels: labels },
            spec: {
                containers: [{
                        name: 'simplewebcontainer',
                        image: 'nginx',
                        ports: [{ containerPort: containerPort }],
                    }],
            },
        });
        this.cluster.addManifest('simple-web-service', {
            kind: 'Service',
            apiVersion: 'v1',
            metadata: {
                name: serviceName,
                annotations: {
                    // this is furtile soil for cdk8s-plus! :)
                    'service.beta.kubernetes.io/aws-load-balancer-internal': 'true',
                    'service.beta.kubernetes.io/aws-load-balancer-extra-security-groups': pingerSecurityGroup.securityGroupId,
                },
            },
            spec: {
                type: 'LoadBalancer',
                ports: [{ port: servicePort, targetPort: containerPort }],
                selector: labels,
            },
        });
        const loadBalancerAddress = this.cluster.getServiceLoadBalancerAddress(serviceName);
        // create a resource that hits the load balancer to make sure
        // everything is wired properly.
        const pinger = new pinger_1.Pinger(this, 'ServicePinger', {
            url: `http://${loadBalancerAddress}:${servicePort}`,
            securityGroup: pingerSecurityGroup,
            vpc: this.vpc,
        });
        // this should display a proper nginx response
        // <title>Welcome to nginx!</title>...
        new core_1.CfnOutput(this, 'Response', {
            value: pinger.response,
        });
    }
}
// this test uses the bottlerocket image, which is only supported in these
// regions. see https://github.com/aws/aws-cdk/tree/master/packages/%40aws-cdk/aws-eks#bottlerocket
const supportedRegions = [
    'ap-northeast-1',
    'ap-south-1',
    'eu-central-1',
    'us-east-1',
    'us-west-2',
];
const app = new core_1.App();
// since the EKS optimized AMI is hard-coded here based on the region,
// we need to actually pass in a specific region.
const stack = new EksClusterStack(app, 'aws-cdk-eks-cluster-test');
if (process.env.CDK_INTEG_ACCOUNT !== '12345678') {
    // only validate if we are about to actually deploy.
    // TODO: better way to determine this, right now the 'CDK_INTEG_ACCOUNT' seems like the only way.
    if (core_1.Token.isUnresolved(stack.region)) {
        throw new Error(`region (${stack.region}) cannot be a token and must be configured to one of: ${supportedRegions}`);
    }
    if (!supportedRegions.includes(stack.region)) {
        throw new Error(`region (${stack.region}) must be configured to one of: ${supportedRegions}`);
    }
}
app.synth();
//# sourceMappingURL=data:application/json;base64,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