"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.clusterArnComponents = exports.ClusterResource = void 0;
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const consts_1 = require("./cluster-resource-handler/consts");
const cluster_resource_provider_1 = require("./cluster-resource-provider");
/**
 * A low-level CFN resource Amazon EKS cluster implemented through a custom
 * resource.
 *
 * Implements EKS create/update/delete through a CloudFormation custom resource
 * in order to allow us to control the IAM role which creates the cluster. This
 * is required in order to be able to allow CloudFormation to interact with the
 * cluster via `kubectl` to enable Kubernetes management capabilities like apply
 * manifest and IAM role/user RBAC mapping.
 */
class ClusterResource extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.trustedPrincipals = [];
        const stack = core_1.Stack.of(this);
        const provider = cluster_resource_provider_1.ClusterResourceProvider.getOrCreate(this);
        if (!props.roleArn) {
            throw new Error('"roleArn" is required');
        }
        // the role used to create the cluster. this becomes the administrator role
        // of the cluster.
        this.creationRole = new iam.Role(this, 'CreationRole', {
            assumedBy: new iam.CompositePrincipal(...provider.roles.map(x => new iam.ArnPrincipal(x.roleArn))),
        });
        // the CreateCluster API will allow the cluster to assume this role, so we
        // need to allow the lambda execution role to pass it.
        this.creationRole.addToPolicy(new iam.PolicyStatement({
            actions: ['iam:PassRole'],
            resources: [props.roleArn],
        }));
        // if we know the cluster name, restrict the policy to only allow
        // interacting with this specific cluster otherwise, we will have to grant
        // this role to manage all clusters in the account. this must be lazy since
        // `props.name` may contain a lazy value that conditionally resolves to a
        // physical name.
        const resourceArns = core_1.Lazy.listValue({
            produce: () => {
                const arn = stack.formatArn(clusterArnComponents(stack.resolve(props.name)));
                return stack.resolve(props.name)
                    ? [arn, `${arn}/*`] // see https://github.com/aws/aws-cdk/issues/6060
                    : ['*'];
            },
        });
        const fargateProfileResourceArn = core_1.Lazy.stringValue({
            produce: () => stack.resolve(props.name)
                ? stack.formatArn({ service: 'eks', resource: 'fargateprofile', resourceName: stack.resolve(props.name) + '/*' })
                : '*',
        });
        this.creationRole.addToPolicy(new iam.PolicyStatement({
            actions: [
                'ec2:DescribeSubnets',
                'ec2:DescribeRouteTables',
            ],
            resources: ['*'],
        }));
        this.creationRole.addToPolicy(new iam.PolicyStatement({
            actions: [
                'eks:CreateCluster',
                'eks:DescribeCluster',
                'eks:DescribeUpdate',
                'eks:DeleteCluster',
                'eks:UpdateClusterVersion',
                'eks:UpdateClusterConfig',
                'eks:CreateFargateProfile',
                'eks:TagResource',
                'eks:UntagResource',
            ],
            resources: resourceArns,
        }));
        this.creationRole.addToPolicy(new iam.PolicyStatement({
            actions: ['eks:DescribeFargateProfile', 'eks:DeleteFargateProfile'],
            resources: [fargateProfileResourceArn],
        }));
        this.creationRole.addToPolicy(new iam.PolicyStatement({
            actions: ['iam:GetRole', 'iam:listAttachedRolePolicies'],
            resources: ['*'],
        }));
        this.creationRole.addToPolicy(new iam.PolicyStatement({
            actions: ['iam:CreateServiceLinkedRole'],
            resources: ['*'],
        }));
        const resource = new core_1.CustomResource(this, 'Resource', {
            resourceType: consts_1.CLUSTER_RESOURCE_TYPE,
            serviceToken: provider.serviceToken,
            properties: {
                // the structure of config needs to be that of 'aws.EKS.CreateClusterRequest' since its passed as is
                // to the eks.createCluster sdk invocation.
                Config: {
                    name: props.name,
                    version: props.version,
                    roleArn: props.roleArn,
                    encryptionConfig: props.encryptionConfig,
                    resourcesVpcConfig: {
                        subnetIds: props.resourcesVpcConfig.subnetIds,
                        securityGroupIds: props.resourcesVpcConfig.securityGroupIds,
                        endpointPublicAccess: props.endpointPublicAccess,
                        endpointPrivateAccess: props.endpointPrivateAccess,
                        publicAccessCidrs: props.publicAccessCidrs,
                    },
                },
                AssumeRoleArn: this.creationRole.roleArn,
                // IMPORTANT: increment this number when you add new attributes to the
                // resource. Otherwise, CloudFormation will error with "Vendor response
                // doesn't contain XXX key in object" (see #8276) by incrementing this
                // number, you will effectively cause a "no-op update" to the cluster
                // which will return the new set of attribute.
                AttributesRevision: 2,
            },
        });
        resource.node.addDependency(this.creationRole);
        this.ref = resource.ref;
        this.attrEndpoint = core_1.Token.asString(resource.getAtt('Endpoint'));
        this.attrArn = core_1.Token.asString(resource.getAtt('Arn'));
        this.attrCertificateAuthorityData = core_1.Token.asString(resource.getAtt('CertificateAuthorityData'));
        this.attrClusterSecurityGroupId = core_1.Token.asString(resource.getAtt('ClusterSecurityGroupId'));
        this.attrEncryptionConfigKeyArn = core_1.Token.asString(resource.getAtt('EncryptionConfigKeyArn'));
        this.attrOpenIdConnectIssuerUrl = core_1.Token.asString(resource.getAtt('OpenIdConnectIssuerUrl'));
        this.attrOpenIdConnectIssuer = core_1.Token.asString(resource.getAtt('OpenIdConnectIssuer'));
    }
    /**
     * Grants `trustedRole` permissions to assume the creation role.
     */
    addTrustedRole(trustedRole) {
        if (this.trustedPrincipals.includes(trustedRole.roleArn)) {
            return;
        }
        if (!this.creationRole.assumeRolePolicy) {
            throw new Error('unexpected: cluster creation role must have trust policy');
        }
        this.creationRole.assumeRolePolicy.addStatements(new iam.PolicyStatement({
            actions: ['sts:AssumeRole'],
            principals: [new iam.ArnPrincipal(trustedRole.roleArn)],
        }));
        this.trustedPrincipals.push(trustedRole.roleArn);
    }
}
exports.ClusterResource = ClusterResource;
function clusterArnComponents(clusterName) {
    return {
        service: 'eks',
        resource: 'cluster',
        resourceName: clusterName,
    };
}
exports.clusterArnComponents = clusterArnComponents;
//# sourceMappingURL=data:application/json;base64,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