#
#	toyecc - A small Elliptic Curve Cryptography Demonstration.
#	Copyright (C) 2011-2022 Johannes Bauer
#
#	This file is part of toyecc.
#
#	toyecc is free software; you can redistribute it and/or modify
#	it under the terms of the GNU General Public License as published by
#	the Free Software Foundation; this program is ONLY licensed under
#	version 3 of the License, later versions are explicitly excluded.
#
#	toyecc is distributed in the hope that it will be useful,
#	but WITHOUT ANY WARRANTY; without even the implied warranty of
#	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#	GNU General Public License for more details.
#
#	You should have received a copy of the GNU General Public License
#	along with toyecc; if not, write to the Free Software
#	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#	Johannes Bauer <JohannesBauer@gmx.de>
#

import unittest
import random
from .. import getcurveentry
from ..AffineCurvePoint import AffineCurvePoint

class OpenSSLCurveTests(unittest.TestCase):
	def test_prime192v1(self):
		curve_entry = getcurveentry("prime192v1")
		self.assertEqual(curve_entry.oid, "1.2.840.10045.3.1.1")
		curve = curve_entry()
		self.assertEqual(0x68ecc592f815652ee1e44f457165d6b3d5e9bc3e6c1378e4 * curve.G, AffineCurvePoint(0xa132b7bd410e4f486d876cf33361bda874256585343c449d, 0x3b21cdf76be18259a9ab676f98830369d3d23234e158a54, curve))
		self.assertEqual(0x28f2edfa104ed6736b4d0bac34d73d8ab888cbcea443b55e * curve.G, AffineCurvePoint(0xfa074c264988204bd7c88fc7da8d879a2dfeb817ad76675, 0xda53825f6766032f78967572c25aff47727c6bc846d55b29, curve))
		self.assertEqual(0xa0a1cfa44672d01eecafbba38d34cd1faf14f7e6b9a37766 * curve.G, AffineCurvePoint(0x93f57517bb42871633ce2d6d64ce0bac4e0a16faa75f472c, 0xe42ba3623148efa5ab5392637b8068bded2d0b8aeed2e98e, curve))
		self.assertEqual(0xa21b4e3542f513f2838ab1cd87a264a033a134f37bae9c87 * curve.G, AffineCurvePoint(0xc4c30f01262c671fc38c1acde0163e667b2d09c275d59079, 0x7992a20bc1668a927bbed1fdd61204ba267bc847d9ec7e7c, curve))
		self.assertEqual(0x92b51022db37c8860b97db2107f148ffafd574a92988ae80 * curve.G, AffineCurvePoint(0x82c4df078b27dbdaef727fe1519861ccebf3faffd8ed945a, 0x3c16d4592b840a3f85dc55584c1f48706e55382ca4bc1d43, curve))
		self.assertEqual(0xc891e79a290e75c5f91db54bbee0befb89e64c3275f7c46a * curve.G, AffineCurvePoint(0xdba12e9db216fb2851334bcc4fd38f25095dfe8fffd1bf43, 0x4e501992d3763fb76247c35810b1462fb87dddac3021da66, curve))
		self.assertEqual(0x971929fb25b84ba50f3d4c226cc22abeab6d511f81045be0 * curve.G, AffineCurvePoint(0xf9c5bf8d2b53531d5e931dc1542f99d86c742e3462e0ebe4, 0x8520113d0e81e11efc95a46fdc0c662619c3ea7b3f5cb836, curve))
		self.assertEqual(0xba47562370c806f98d2b909ab7f113fd2d916693fb9f1b6e * curve.G, AffineCurvePoint(0x6110920df4eb187d21b69412241898db491885a47716d9bc, 0x9af1d912cf8060563fb2cb28aa2a679102c9a73625f9c779, curve))
		self.assertEqual(0xe504ef8d443b08980b4a8194898ce04cbd3d1c6762631ff9 * curve.G, AffineCurvePoint(0x8a393f8fd714c2203e906c36cb597035884e09ccacb95f26, 0x19c5539abb4f897149d72e393836eb9ee44217d3e3d40ce8, curve))
		self.assertEqual(0xe52bd359dcd16d45f22422bd0fa37fe08d7a2cfa46036571 * curve.G, AffineCurvePoint(0x3e46184aa9f27d5b5e12e942c8f5109f574a6a4282ddeaeb, 0xae0b32337647ab0affc9979a6edafba5d6b3f775a095e2f1, curve))

	def test_prime192v2(self):
		curve_entry = getcurveentry("prime192v2")
		self.assertEqual(curve_entry.oid, "1.2.840.10045.3.1.2")
		curve = curve_entry()
		self.assertEqual(0x8c579e28594b38b14bc73da345225fb738d4c95aab0369ed * curve.G, AffineCurvePoint(0x1002724b5774d7072246ec6558547b47b3f844ef09d93ea0, 0x4c19b1d2c670cb14db576ee87077b3c3d6b43c192fb95421, curve))
		self.assertEqual(0x78c0229ca893ae32d27791d29968c7253b1951e65f3bb346 * curve.G, AffineCurvePoint(0xb86fbdcae2c1d8aa93ffffd094c91b8ce3be2845c912df2c, 0x7a6ac91354a861d3f9ada17823c9188333957ff5519915aa, curve))
		self.assertEqual(0x4dd927d66c4ddcbd1a49e5ab945da8735e7972cf701e8b05 * curve.G, AffineCurvePoint(0xcb84979b94e890beeae1ad1c74ce5d10f46bd7d98be612e8, 0xbc685d6dfe83394fc7b0a6c040da8e5b05f31c0223e0da93, curve))
		self.assertEqual(0xd98712f9a2e46c629b29519b91ba171e5c609232d7e27e0b * curve.G, AffineCurvePoint(0x1498acaf5a5bb54b35054c91605fb223c19c1316b6c7fbc4, 0x6464307e2f8be5e192a606e2abbddbcc1b5795621ad33924, curve))
		self.assertEqual(0x131a0a391836c2eb088dd6065c65e47490ebd4d5ce222379 * curve.G, AffineCurvePoint(0x294318e570a6a1db589be089db6a89f0f861f5ccf602571e, 0x30563eacd3367665f85870a3747ac09d6b9c07eca7d669ac, curve))
		self.assertEqual(0x4ae8263318a6ca51f54a8dd8e2375310901fb1be6585120 * curve.G, AffineCurvePoint(0x25fee3d087419307acd3548932598af50c7e295bd9640978, 0x4eba9bf1394401907f26d7d0c8d91dbbc80fb87a59662eda, curve))
		self.assertEqual(0xf9525037770a619e7967b8536cd2a089449bbe3ac758b4b4 * curve.G, AffineCurvePoint(0x6f0fdaec2887bb72b69c743b4e774cc30226b930f912d643, 0x81e54e18fe9460533b7d22e390bc0fa489fb073c23d859a, curve))
		self.assertEqual(0x7dd20ad7772340b64f2929f8f1112efa00422503217c2526 * curve.G, AffineCurvePoint(0x5ea71315d5150c0354f13c04990d882e51bf3e9ebe32f672, 0x84268f96c1f4a598ebbae1f2ec9ae1c2fcb5fd926a81abba, curve))
		self.assertEqual(0x2f85ec592354e1e54849eb8998f9a6760001e8c547eabcb4 * curve.G, AffineCurvePoint(0x6f28b8d0648542b84cc8334e3410c25da373becb95be3c4a, 0x58e4d6270ec4f7dd22230ae81daf131a55fa21e11629a314, curve))
		self.assertEqual(0xe3b1a02a3eb392718d79bd503b0c4f8ce647925c212d897f * curve.G, AffineCurvePoint(0x2f8db177eadaa43f8719574fce19685340acc8ff98cf64e4, 0x228588e61ea63a7d795f234d1156ca6bba505344f0e8a170, curve))

	def test_prime192v3(self):
		curve_entry = getcurveentry("prime192v3")
		self.assertEqual(curve_entry.oid, "1.2.840.10045.3.1.3")
		curve = curve_entry()
		self.assertEqual(0xa5d99892dbaf961404de5b5262808f6966fe52022af2d71e * curve.G, AffineCurvePoint(0x9aede7d92d80b9c25b7380d829db70d685893909c66adf5b, 0xb73210096c6b0237e3bc8a3f949f4f5f2e0fbe3ba05ac80c, curve))
		self.assertEqual(0x110cd69f68920ad51175438ce5bedfc9b9121304d8d310ba * curve.G, AffineCurvePoint(0x4a3c8a8ff25f280624f09c50f6c877cbadfdcc049317f839, 0x22e4b88a91a7219f8830b8937d3a12d9cb969f00c37b0f6f, curve))
		self.assertEqual(0xbac28db86e01db35d3718291892f14965270480e009db57d * curve.G, AffineCurvePoint(0x8d3389ed64a9465a790ce9cf74669f96a07c45babcc0a060, 0xfbc4dfeff08a8a3eb284c73dccdce2689e426c79b192e5c4, curve))
		self.assertEqual(0x149ea68e806705efa67762a223180f23455793448818d701 * curve.G, AffineCurvePoint(0x110915082eeb941e83e8d134002f71a6ca06879654c75240, 0xb49ad8333d03d76c8db3089e008ac697c3a85c8578c434da, curve))
		self.assertEqual(0xc3466541e1377b14c54636f9c90ff341720ac652872791f6 * curve.G, AffineCurvePoint(0x1e674bf456aab530f7348c986cd9ec9f92c6e00bfec22960, 0xe9ceb47761358169eef57ebf4a675a74f73ded7b9dca5c77, curve))
		self.assertEqual(0x908872f9cf6e2aa10d9c82cfcedbccdbb7b4a9ededf2ac25 * curve.G, AffineCurvePoint(0x9cb277525a4b988c40409bc703ed0a772ec273921939ff04, 0x422545dfe959765b2786dc84899ebb68a48400304819a180, curve))
		self.assertEqual(0x5933d547edb3eab87c3b4ec856fd89c8bac32e3ce5442cfd * curve.G, AffineCurvePoint(0x24db4966f07c439d57296414e4367155d6fa08df1195d6b2, 0xe30c034247d14c940f9870cecea46e24351ad170b88574a0, curve))
		self.assertEqual(0x9cfbc68483c81c61112750f56d3e5f53665522e1c1af5026 * curve.G, AffineCurvePoint(0x45aa5a8235a096f7e9491a69cc49d6312e0c61ed508f07f2, 0x45a9c163ea8db04c11e8fdea4372283b08bf841da778fe49, curve))
		self.assertEqual(0x619ae83459af4da1fbe0679f94769c39a109ddbc99c90c64 * curve.G, AffineCurvePoint(0xba44777ceee41a87820fa5274d2c7c930809ba859daf0e55, 0xd1b4eccb0b2b689e4823674104365af1d8eecc3e900d4468, curve))
		self.assertEqual(0xae7984df24eecc2ba790bcaa375a9094d3c80349702440c3 * curve.G, AffineCurvePoint(0x20d01432f83658ec9cf22e6bdb6e167d6a32c1c407b36d27, 0x457b293193f35c6f4f1c89b5918036f3b29d4bb2d0538563, curve))

	def test_prime239v1(self):
		curve_entry = getcurveentry("prime239v1")
		self.assertEqual(curve_entry.oid, "1.2.840.10045.3.1.4")
		curve = curve_entry()
		self.assertEqual(0x17d4ad425f9a7cf39080bd557ce3d9e14e09817e42b3a2dd678e81551b2b * curve.G, AffineCurvePoint(0x1eb9f65d29121c6c01f364a758a3a44275a4f660a9e8a39cf47b70dd2eaf, 0x134e581f5e637fbcf3d00d7605422a797719ec43f4f9ec6b7564a8624279, curve))
		self.assertEqual(0x3a87f513020f202c5884219eb429e98561f33fdde47911df1ecb99e4b89e * curve.G, AffineCurvePoint(0x4802d6284ba61efee3da437174926a3aa8b8e5f7bc6c381ea196e7a262f5, 0x2b84b68d59f1bf0804d66b7477d6ba05f95ed56b94a61218ccaf456ce7d8, curve))
		self.assertEqual(0x448c9365f6a1c4a287fb6e9bed1ed829714b6853a8e7f033785d2963d7c * curve.G, AffineCurvePoint(0x7a50771085cebb66ac8d63dccfc46775a5c0dc437dd84472bacf405265f4, 0x263e0d20aab431121c58466ac742781fa5661d1a5090846a203f5c48e35, curve))
		self.assertEqual(0x4b65f1b2b8c204ca109cdb8ede88da9ce3a27cfaae348b3b6515fee7d7b0 * curve.G, AffineCurvePoint(0x6482eee61a48c259b13623fdc27ef96cdd6b40d8a3487c64aed760261a68, 0x5f36b4aa7be377a43eeaf671c4f104300fdaa1adc8b0c161b894f656190b, curve))
		self.assertEqual(0x6be60387feea012fc1d5d15fa47fcda58c174b6c3e297c3bf32efbc3f2f9 * curve.G, AffineCurvePoint(0x1ed85d1c8b838de61e989f0618ce4a71a52f61b7bfc7555060857f4df1b7, 0x1f52a339a9b4776f3345d5336a3709885e9ba8fa05408a5f1514856a0dab, curve))
		self.assertEqual(0x1e7cdf3c177b637d3345e5012e2f46d2a5cfb74a32260476a8b87b5e8617 * curve.G, AffineCurvePoint(0x8913f4d39bcd914c0caec6281c686f7becc82d392ac2aabfcdd4924ffb, 0x69ea34696941a410f8899f06084a7f00f4e71d8dec2a68f9628124fc2e96, curve))
		self.assertEqual(0x1782344f59c9069561a29fe33bcf06800b3d939af3a9af132f0412172bc0 * curve.G, AffineCurvePoint(0x745a35f61e88f3454154758c945f8e19a583df63357d453563c5a177a920, 0x39fa8833174af27161e9a402e69884cba624c4c8ee6c8d2706d2eb22f3e0, curve))
		self.assertEqual(0x6f67ce4e685042b52edf160e6c2f5a43304467e117a4c48a62c70e72883 * curve.G, AffineCurvePoint(0x1ccbaf7fae41149f35bc6debb4bab3bd18b893d865f8faa7f5827d1316b9, 0x5e496d2575a4e56faf7bdb927cf5b205e65b7eabc24e1c6940af5dcd8d27, curve))
		self.assertEqual(0x3f296a852a03805ff7b71af56e87f8573e0e1150d7d9372fea56bff5951a * curve.G, AffineCurvePoint(0x42bf7602d51bf95b3b7f3e59a878bb6cb37ad16992dfa77dfd9eafb6f255, 0x590855986b199874d6e26b66444fadc5776e18b833f0c415674892f1a15, curve))
		self.assertEqual(0xf00187501fbf11e0176009fe0cf89e22ec957569a06fe423465623ca2c7 * curve.G, AffineCurvePoint(0x3510bdc24d8b742ab196392be99c22ace9f2cc483e9c5ef1c89f0699ec25, 0x66f3788f6fc9b7cfe909a6e97d75a43749b50fbc469ca03111c860ff1844, curve))

	def test_prime239v2(self):
		curve_entry = getcurveentry("prime239v2")
		self.assertEqual(curve_entry.oid, "1.2.840.10045.3.1.5")
		curve = curve_entry()
		self.assertEqual(0x1b0194f224024f36e6ebf471af3b99786249a98fb9e6212d24e43d0c3d5f * curve.G, AffineCurvePoint(0x56906549b6ab11a8737e66c0408d0d5756a078b8d5b2d2aa39cf4ab47647, 0x63664821aad4e16f6d58ef5d45ab6ccb3b9b67047cb8e8e36515383c7f11, curve))
		self.assertEqual(0x363ee062eff935f21bd412fa49d34222372e1f303416fbfaddbdbbf2e7ed * curve.G, AffineCurvePoint(0x302c7633f76bce2131991a31400d7cdb2270bc5379e7708fd2feb82cb8e5, 0x9b74ef39553b728d61477e910076eb7430663ab8bac05e7f5a0b36ca396, curve))
		self.assertEqual(0x6d84b0421fef0f8045cc49a5f6f430f2a44821a29b59ab6af9d348bd5128 * curve.G, AffineCurvePoint(0x872fdc76c5db29c6c097d2409d8ca98586d1d6506132e3ba46d69953f78, 0x76bf1c921cbb2743e358c1ec4e5b3ec511655e39691cf297083148160cac, curve))
		self.assertEqual(0x5a56fbbf1deec0dad0869de40e1a3f5c72f6072beb061d199099ee583807 * curve.G, AffineCurvePoint(0x6e6c2d0f0752498bf5e7973fc0802a0b9c94711c18b0b3352ac1e7760872, 0x4eb03ff8030d163d2abb80a782333f7a453416522218c528c264d46de74c, curve))
		self.assertEqual(0x1ae994ecd99bb03011a34addb6180f0816aa8d297f554d610f48f3773330 * curve.G, AffineCurvePoint(0x33422dcafcfce09debe15ff6e53a031e903c7e3193d27bddd195315e9b3, 0x7c21de0321660ce59db0b0008352b6d8131dd668f4388125016cb24674e1, curve))
		self.assertEqual(0x1d2ebd2d98e94b219d686fdadeb56cb59714d08553c6fa07811d69301788 * curve.G, AffineCurvePoint(0x6952c94fb570514441d23e0c204f44bb58a39a783e336270df5a29c06149, 0x795cefd5efcf23c1803c3ead5819c916b807757656b2edeac8ba019e086, curve))
		self.assertEqual(0x4860204a6d5bd64f24f71da0fe737b38b4bc24681b8bc0431ecae0d0865e * curve.G, AffineCurvePoint(0x1853bac6aeac7ac905c3731f0b72821f8bbbd29e3c9d851411d88cd64e96, 0x7689b31e7adab1ea49e960a7e38f73c50143685f9cbf9f0e25b65d0d5abd, curve))
		self.assertEqual(0x7de37ba8c4c464419170a6db93885daa6c5fcbed8fb3703961f70d22b901 * curve.G, AffineCurvePoint(0x2a10125aed30ad6127d7ead859d5503de233403d97ca2378dd67b48e6982, 0xa8efce870d5d91bca910eca330d3b3569a7130cfdd2a8ff15af6bad0848, curve))
		self.assertEqual(0x177d42229bc24b3922aa8cf819b72065e175dd7eef2f4478a6108965e9d2 * curve.G, AffineCurvePoint(0x6f5f580934a7a345f4420f54ebbe517498ce463b7cddb74fb598e391cfb8, 0x5ace76e0dda3351b47186f288db4930283068bb83f7900e4b0144a1a0548, curve))
		self.assertEqual(0x1fa72c9e6532e82ed02930a1ace3765e366e75d324b6cbf037508a18eba5 * curve.G, AffineCurvePoint(0x12ade58c606685d021b1a7c442157e97a4674fadc5861af5cc43509c29ae, 0x78b34e1445bed96bf5b99d287021f77dfc5e9974c48394657fb46a03d6df, curve))

	def test_prime239v3(self):
		curve_entry = getcurveentry("prime239v3")
		self.assertEqual(curve_entry.oid, "1.2.840.10045.3.1.6")
		curve = curve_entry()
		self.assertEqual(0x1f0a291b18123600e3b9a7875f96d4a318afe2833a7f8dac7e46c72265f9 * curve.G, AffineCurvePoint(0x254836cbe936ee027b4ae651b98cc132f33e8558990ea344197219371dd2, 0x5a4c38b0c4ff023af57c3bce0bfa2902360f6c76dcf5b979ace999fa60a2, curve))
		self.assertEqual(0x10ff57fe6064ca91b33773a263def77dc2cc68260327415ed563e5c6ed5f * curve.G, AffineCurvePoint(0x3cf155aa933810fce4811b3bffbfadceb40618fcf39048207d33eff112a4, 0x7144e87f11210b68f46a6f79bab156a774209ec516206ec40d5266cd5f62, curve))
		self.assertEqual(0x5f8e8b7669ab1d488843d1e5a5a0a09afcc84306b396771c8dc7bad71d4a * curve.G, AffineCurvePoint(0x6dfbfe2acaecf53cc618b5360145403ec2e9f0ec429dd375e185576d1eea, 0x1983301c9b0df3ed27c6e9c62806ecafa57ee23ec00d6f8e19f4c2837a0c, curve))
		self.assertEqual(0xeb7906a8334f6da4e1b5ad1a351931c1658e5d70b55c3ecf115eaaef8fe * curve.G, AffineCurvePoint(0x7696153dd7c56d4c522ef14a578f6062caa623d7ba363aecbbb2801406e3, 0x4e169630c4ee2636886d1ee72ff294209a4de4ddadb86035f0348a724751, curve))
		self.assertEqual(0x3155bbc5bbc9e13d1ec8d31ed936b2238b1262d9c5abb1362eeb5217c598 * curve.G, AffineCurvePoint(0x195503f89c25cc163bdeffc8583eb61524b9b5d67003248514139d391129, 0x4515798cf8f7700050c3ee6097927d80a5e18b32adb866452bfc6e0f21dd, curve))
		self.assertEqual(0x451f7684922d7fa4e4d956792edf69ff15cf9be56664e78b5f4541d800c1 * curve.G, AffineCurvePoint(0x148198c9610abf87b7821c041e69960eb1d89e11a78f98bff8a1f32f7ade, 0x6c905845a77d40ed8d60a0fcb5aa2a5085f597ad6105097728c54710f361, curve))
		self.assertEqual(0x3a2b2ce80dcbf6765821a89d8d0b3550f0d9325dd820e3a7ef4d99d2a7e5 * curve.G, AffineCurvePoint(0x625cca609a25afa559e6cd826f36921008de89bf84aa699ac9859e5606d5, 0x3d958a3d90321a84b7e663a95f4d04584f9d00036042eee461a9e571aa11, curve))
		self.assertEqual(0x27165726a06d608e9843ed175f8e19e85d1f17f290c60c1a40728c539a1d * curve.G, AffineCurvePoint(0x627b1f6690fd92137c7a75809aadab208b40f3d5d3b9a2925d4ef09d4a09, 0x47bdc4d8e8c7502e53fdb361deabe2da0cef7135914735b52612969b59cb, curve))
		self.assertEqual(0x3c09c56e65743fd0d34e9579cee9ab6d2c7520e5679ead404a91d48889d3 * curve.G, AffineCurvePoint(0x63d2d60e277fd84c93f9dbcbfddcf8d658ed0a44f9c9fcf0b5c152cd340a, 0x7c5b1d8a63279805d27c63d7860f7dd401675f51838601284eed5c47591, curve))
		self.assertEqual(0x4dfaad3e350637a1c1ec4f58eca29cdcf8b3e3a95ea2eb8734f70e572edb * curve.G, AffineCurvePoint(0x4a0085930eb74cb699599446927c9cea8a291936f85455f2c23a67f0299e, 0x157dc6b5cc1eb1d972c5564a16aada2ab9d496282b5f55ad692e75f7d69a, curve))

	def test_prime256v1(self):
		curve_entry = getcurveentry("prime256v1")
		self.assertEqual(curve_entry.oid, "1.2.840.10045.3.1.7")
		curve = curve_entry()
		self.assertEqual(0x369d77e5a164da61e57f3a5d361638e028c65755a18151f10893d593ce281fc5 * curve.G, AffineCurvePoint(0x98662b0742eb04bf5f7581d7be10aaf40bbc4e50c2269604e9abfb314a7c1b62, 0x78fcd60a9345534d5386abb399a174a53acfaea62c5581bf9b8fe874487e12cc, curve))
		self.assertEqual(0x9491b59d2e4631cf556acb7b42e6e317943ab13d43b9bfe105663ce740792164 * curve.G, AffineCurvePoint(0x564a8658c27609c1eac57ac330e145902a596a1ef130944e3e7df7045ef4d39b, 0x98b23ae9b0570fdc8a4085307e39978aec7c6c5f4fb39ee60ef7ac63b8891fbd, curve))
		self.assertEqual(0x743064068bada5ede2c9ee3267318b70adc723b2631cf2356c0652e462f46e85 * curve.G, AffineCurvePoint(0xa5322cf0dd24793bf246ba20bad999bab2fa36753a1b613e947005d2fb159815, 0x4747054ed89c64ff90b7ef64b0fdd4519d99707e55b23957da1bcf0264fcce1a, curve))
		self.assertEqual(0xb990907a2953e2d489f0ac7d930c97d724b183feadb51288cf1e1d54f40a5a67 * curve.G, AffineCurvePoint(0x24d23cec8df46653b781bdedbc6788f8b712ddaa8a722a69c79d50144fe695c2, 0x18f1e451c9a5b2ba50e7d3b64fc00354883e0155162fec9f9a6096bbe24e6734, curve))
		self.assertEqual(0x3bcaaf87e34be68993bb8baf0ae8b001aaf61ffe296ec5a381e7dd0c16c5abe6 * curve.G, AffineCurvePoint(0x3677bbbd6e7889a1013bba039b0ad49c1dd249aca42a8d020280766edd009863, 0x1dcc1974c88f1cf0eae4b670e6b008182a34092438dac8d51c5470fed24765fd, curve))
		self.assertEqual(0x9bd6e8a16ef7ec9cd3f53a059a3151dca54bd8de444fcbca10dfe6d75af1878b * curve.G, AffineCurvePoint(0xe462a59e4ed2b3162db8b5ce6f0e2c90cf35ad444706758577f622e6bff37a2, 0xbd17db3d62fb568b38abce6aff2a15924f690b8b3d666c77dc47c3bdc7394a02, curve))
		self.assertEqual(0xb969d7f1e4f14cdc4e7f87e0cf3ce44755fb4645aa93f00197fd3410fc0e6107 * curve.G, AffineCurvePoint(0x118a2bef430b2c9910bc3af17aaabe38e823b2a86e59bce196b07c4c69565e15, 0xf892d3fc663c58fbd9b41935e26b6ed3171b1cddc17b7705d2c81a85244ca059, curve))
		self.assertEqual(0x3f50225b555acd018cac91814222c6a1ddd499e28f43a34109509bb122482a25 * curve.G, AffineCurvePoint(0xad3388bd2f2dff8a7231d7fccbc221bf1a4cde46b8f0c63ff83c7659dd2eb641, 0x19b3a166fbd54621c1aa1e93811ee7fed79304b8224ee240d80cea4cdb15818a, curve))
		self.assertEqual(0x257c3493df22886a379929093399f63220863065f34aa68c41fe9f7b8994332f * curve.G, AffineCurvePoint(0xd0ca0a25b0411a922e3222d9d036afec4f18edeb71d59744d5e0b2f8a4b31595, 0xdc892ba7b26b078fa56886d8e4f26cef3b78b8861c1d85ed573cbdf1341c6b4, curve))
		self.assertEqual(0xdf80874d46ed3d3cb0e8dbe9828f04e71742b5a4a00ec752e918a44cbec2a35e * curve.G, AffineCurvePoint(0xae4957a327da5a37dc72a0fcadac8982cc4d8be38bfadefd385cdf4e6f68d0fe, 0xd036b3dfa59e3f39d79332aa062395284acc56a9a89953a4eafa7c0bea28615c, curve))

	def test_secp112r1(self):
		curve_entry = getcurveentry("secp112r1")
		self.assertEqual(curve_entry.oid, "1.3.132.0.6")
		curve = curve_entry()
		self.assertEqual(0xcefead19bd4439327d428bb78fe * curve.G, AffineCurvePoint(0xd5431d0ef7139c462597d62bf3b0, 0x29de07500ad8c30c32f0c6260c33, curve))
		self.assertEqual(0xad680eda56d3b1f2ac2a08d2f6fc * curve.G, AffineCurvePoint(0x4fd4e943a349d66da2292d9a3ac4, 0x2a0040dec825589fb951b5d43832, curve))
		self.assertEqual(0x1982ecd4980bd2563612c9554883 * curve.G, AffineCurvePoint(0x45ab9b10ff56e8e90ca93abab6dd, 0xd4245a6a22100c9fe43a2423f365, curve))
		self.assertEqual(0xb0ed6ca95b070014e3a2b3022102 * curve.G, AffineCurvePoint(0xcd166d170094f8d3fb663550065f, 0xacfb725964d3a67763dd2cb4cc0b, curve))
		self.assertEqual(0x31733fd6bc1792aa6a141baff3ee * curve.G, AffineCurvePoint(0x460380395f2bef719a5be7c27a9a, 0xc6faec064dbecc0de81621959c0c, curve))
		self.assertEqual(0x2f3bad067152e1ad02a3a48525e3 * curve.G, AffineCurvePoint(0x49c22bff4f2d79546fc82e2ab5ee, 0x59b8475f8ed5b7d3f7cfc945d602, curve))
		self.assertEqual(0xd113563f175bab947c98a1856dfe * curve.G, AffineCurvePoint(0xb66f8eae7d762c4eb5c5459aeaca, 0x2099a3bc503fabb9d07663b5d650, curve))
		self.assertEqual(0xd1482ae5e1922e17b78ae266371 * curve.G, AffineCurvePoint(0x28d3841d66994118fb2b5b474c34, 0x511d540469118b1ab572af2057f8, curve))
		self.assertEqual(0x153083698caef31e3b605ff92825 * curve.G, AffineCurvePoint(0x21754bcae5cef1f59744fb1a93a5, 0x837e436f845c48833899757bd5af, curve))
		self.assertEqual(0x43706b99d34cf4f25fab40c63f * curve.G, AffineCurvePoint(0x69a4aa6e31f99a04f74535d76cb5, 0x97bf522c0342879431f6870372dc, curve))

	def test_secp112r2(self):
		curve_entry = getcurveentry("secp112r2")
		self.assertEqual(curve_entry.oid, "1.3.132.0.7")
		curve = curve_entry()
		self.assertEqual(0xa0e7a88dc36d86960ad7618f988 * curve.G, AffineCurvePoint(0x1a77ae9ffdaac7dd74440806293a, 0x74aa2bdd002b0833be64bc43cf41, curve))
		self.assertEqual(0xfcb6f8cf070c000e7ae9963ba6c * curve.G, AffineCurvePoint(0x809b214f0c11b7740bac1edf61c4, 0x671513a806dfacb92eebacd19e35, curve))
		self.assertEqual(0x34fcb4ffd924285813944ccef7a1 * curve.G, AffineCurvePoint(0xab2634caa945831ac791662acbe3, 0xc25cc8761240bb3a3d17c531f690, curve))
		self.assertEqual(0x3549f99804e0e5ca2a25219de468 * curve.G, AffineCurvePoint(0x7fcc3e9a706979a67c663c5a6ae6, 0x4c1ff6532a2eb69d9887b9fc3e1f, curve))
		self.assertEqual(0x1836630aef6c37c0ed9515bbc27e * curve.G, AffineCurvePoint(0xaeba9d42ed9e2b3efbead8f9f82a, 0x6f9fe416793efb66a6891a42c295, curve))
		self.assertEqual(0xa04a7d30b0e9eacb64f1b80cf2f * curve.G, AffineCurvePoint(0x30995ad7844680532dcf2e6a6713, 0x15ce84f955aa23c81fe3c2c7abe8, curve))
		self.assertEqual(0x2b8f402066b4f8d44498d823b96e * curve.G, AffineCurvePoint(0x61934f0fe3b7cbcd8feeda0baf9, 0x255c092fdab3d67468fb243ce373, curve))
		self.assertEqual(0x40aacba5a6e0dd73071efeeb15d * curve.G, AffineCurvePoint(0xce92868f92cd3dbdaba9f14f41b4, 0xd6418402a8eee44106368f81a2c5, curve))
		self.assertEqual(0x221893ddfd905f0c66b37f88fe1 * curve.G, AffineCurvePoint(0x97531094f30b5d369320dddb21c6, 0x579fe127e47beba79ecc6eec2ccf, curve))
		self.assertEqual(0x2b1081553a9cbe43b29da1004b7b * curve.G, AffineCurvePoint(0x90faf9f9d3bfcacfff4c55ae7e07, 0x22a56066ad2d8f60081c77ea3e6, curve))

	def test_secp128r1(self):
		curve_entry = getcurveentry("secp128r1")
		self.assertEqual(curve_entry.oid, "1.3.132.0.28")
		curve = curve_entry()
		self.assertEqual(0x619e93d14596b9696ea806ba964f35f7 * curve.G, AffineCurvePoint(0x55871e91341340fc759b469f23e342d, 0xa85feffe7a5a63364edbbb1645415122, curve))
		self.assertEqual(0x6b9a034c4dc2fd86e59636f913ba3af6 * curve.G, AffineCurvePoint(0xc7fc636d4a4d6d7dfd7907d3944f4afe, 0x62c13843b97ff1cf971c2eb1a6960aaa, curve))
		self.assertEqual(0x5851ce1f7d634e9d11b9c719f397b97 * curve.G, AffineCurvePoint(0x795948f4eac961cafbbde87e77eb6a38, 0x6064e05e94c947d89bddb1757b157ef, curve))
		self.assertEqual(0x618394d0afd6ce725db9571151dd6e52 * curve.G, AffineCurvePoint(0x46c1b1b5376225904c76261b3b5fdb03, 0x57dac485846cabe2c3bc5473c10b2240, curve))
		self.assertEqual(0xb4928c164d2f1f93c8582709fc4a3f9 * curve.G, AffineCurvePoint(0x39350bcc3c21f2790a5f58e073fbfcdc, 0x321b8825e6b79f53e5cb9a6007a3661b, curve))
		self.assertEqual(0x531329af183b4357a29886235b72d484 * curve.G, AffineCurvePoint(0x6f8fdc6bf0f49ee74a07410f034439ec, 0xf7299abf33e73f2445fde6b380b7a9b, curve))
		self.assertEqual(0x1f94e00af849f714e350776feb6af1f5 * curve.G, AffineCurvePoint(0x3cfc3bd286fb874697e3d1773ccef66, 0x55b53ccdbf86f0190e44cecc5fbb4054, curve))
		self.assertEqual(0x1a2595d2bcffbe5acf7a2e06cbf88d3d * curve.G, AffineCurvePoint(0x44bbc7e1badecbf33af43af6d16a2c52, 0xb56fe7a9b3bd65eec096bccfd4262a32, curve))
		self.assertEqual(0x8ff0486023b14ddcf59dac639c927897 * curve.G, AffineCurvePoint(0xb619d3e6de2042cff6296bccb9a62157, 0x26b46c4f6c9464aae2d2f7efb2f7532c, curve))
		self.assertEqual(0x2cf623f3a9fd4fd11561c4c168ff9a31 * curve.G, AffineCurvePoint(0xa69a2189c62f7aeb6b88e40e2136ddd9, 0xc4f3d3523a1c302d55f8edd013dfad78, curve))

	def test_secp128r2(self):
		curve_entry = getcurveentry("secp128r2")
		self.assertEqual(curve_entry.oid, "1.3.132.0.29")
		curve = curve_entry()
		self.assertEqual(0x6c2651923ee732a8ed0336b6c79fbaa * curve.G, AffineCurvePoint(0x8138e16dcafbede88ad98176d2f6a263, 0xde487acd65ec330a5d0d092db0bac367, curve))
		self.assertEqual(0x1d7a41d19d183887def56a34ca546fe5 * curve.G, AffineCurvePoint(0x77cda58df312ec725d555030ac446e6, 0x3255309c39b08fa920a4b27e68f28ddd, curve))
		self.assertEqual(0x372f1087f3ca5f27c6e45f761355b1a * curve.G, AffineCurvePoint(0xeaf10185e3a11425401ca6abb4bb9ace, 0xe4d64cbcd81dea232ef1b7392b129afb, curve))
		self.assertEqual(0x141a87d9ab55fc5d2ec5b29a07761daf * curve.G, AffineCurvePoint(0xb489288f680240508b8873a25c22eb3, 0x9c31470cb936a91519d63d0b6c4e55fa, curve))
		self.assertEqual(0x26c2cf109f92cf1b63a4b0b18edfc5ca * curve.G, AffineCurvePoint(0x66ba02e5319c01aaf6c87dea74a26fb0, 0x73e4084d7dcce30b0c0a4b22b308592, curve))
		self.assertEqual(0x2111a04683393ab218fdf5fc85d97d52 * curve.G, AffineCurvePoint(0x8a067ea91f2be9784a15751a8146e80c, 0xdcad8ed975a80f093b4915fc2097c7be, curve))
		self.assertEqual(0x2f0a2b5395d962d3d39bdbf2cc4df2a0 * curve.G, AffineCurvePoint(0xc3d8be446df191ad752c29186957165f, 0xa7e31c29713cf7245e3448816daef25f, curve))
		self.assertEqual(0x6dfe7a95ebded64dd23bad4ab4bd793 * curve.G, AffineCurvePoint(0x272071b09967da067dd32b1b2ee26528, 0xb8970270b48ad7dcaa9813059db2e158, curve))
		self.assertEqual(0x26c50bf19ad50465e32680b3fcdb3462 * curve.G, AffineCurvePoint(0x95803a04bb709cdbe5e258e6e38612d, 0xd9456386b824781e19936e2f6bd21578, curve))
		self.assertEqual(0x3498a2104921107a513f0fda0f275286 * curve.G, AffineCurvePoint(0x6f118a14a1d7454a9b9f201d8573a34a, 0x669888757f5e6c45e401a44747619175, curve))

	def test_secp160k1(self):
		curve_entry = getcurveentry("secp160k1")
		self.assertEqual(curve_entry.oid, "1.3.132.0.9")
		curve = curve_entry()
		self.assertEqual(0x44fac7e90db3ded76ba0a5efe7fff97685e76b4d * curve.G, AffineCurvePoint(0x87523b1083658f382e8c1c939db2ae77828d00f0, 0xcb7454c8b02ba392ece9333ffea93be1e12ad1ee, curve))
		self.assertEqual(0xc7e60f297d935c682df6e00d283c68d0f4ee7c50 * curve.G, AffineCurvePoint(0x874d1f74b8d6fd5f2324b0c39e34155dc5a8b1d7, 0x9a5dd2233f23085175bc61a53b94e0c960991804, curve))
		self.assertEqual(0x6b58531acdb8b2f1054b49ff84f396d6ed69f436 * curve.G, AffineCurvePoint(0xabd29cabb57564887f39f0ff9cf9972e19a3c54a, 0xaf1d0d5d0ae1a6a8ed4a836fbfda2fca1c5c2904, curve))
		self.assertEqual(0xc155aac5161a5b78315c83c21acb71357c335996 * curve.G, AffineCurvePoint(0xf097b88bd8d6b1efd7428e1dc5f549ab987a10a2, 0x4f1cb27178453b794a80d2d88a12afedf42405c7, curve))
		self.assertEqual(0xaa050cd7a11e4aea6e635c20f57feb3b2226d01b * curve.G, AffineCurvePoint(0xd84dd7c9063e065253138aedaa244e67e4b6a292, 0x9e6d074cd25528a04e4fe2580055c2b3203b7572, curve))
		self.assertEqual(0x7912fa61386ef13b184e43bca502996df93e77a3 * curve.G, AffineCurvePoint(0x760f97126103db3c50363d228f2ad215343f9b5f, 0xf62363c54443aed6af033d323c610bb5514ca69b, curve))
		self.assertEqual(0xb70819fb9a64f5daeabe49d403e01b50820c8ec8 * curve.G, AffineCurvePoint(0x2caef99e8decf7a1c8c92afee7dc52cc564b9230, 0xd9628fcf3437a832f471b1f925c5221c86048b6b, curve))
		self.assertEqual(0xe10826236988a460e4c43dbd060630e917dd6217 * curve.G, AffineCurvePoint(0x48f11a69b55fd50c3e98c71a3d69007643770ac4, 0xe758e8272e8c55f6c615b0f3084f709d4cdd57eb, curve))
		self.assertEqual(0xeae0d2e71cf872cf1002dd9b1b13824a091bda4b * curve.G, AffineCurvePoint(0x56134b78b6791b55aa79f5ee2c13fa2577db4fe8, 0xe68b9f85c5476e19a063fc9488c640ab9f7ecb7b, curve))
		self.assertEqual(0xf3285ab32dce7ee5f01823913ff9e65805f7cf81 * curve.G, AffineCurvePoint(0x5c028d0da2b84650e4796a9d499e264b49579cad, 0xde7936037704f591286a250ed4943290386d5fb0, curve))

	def test_secp160r1(self):
		curve_entry = getcurveentry("secp160r1")
		self.assertEqual(curve_entry.oid, "1.3.132.0.8")
		curve = curve_entry()
		self.assertEqual(0x14f515cd8917eb60e8912ce0c6be49c84449f16e * curve.G, AffineCurvePoint(0x280279c1f6e7e98c3107dae8e87287989a25d7ba, 0xfbced313e110fe76d9869b0111e3473f1038f886, curve))
		self.assertEqual(0x3dc339f9eb79f41b6778bc2df44285dfbcafbd1b * curve.G, AffineCurvePoint(0x9ec71e76a10a988fb92f29ec8c9f290959ffd743, 0x318ae166c1af0a6611d9c5ae8edb210f2d3afe4e, curve))
		self.assertEqual(0x385dd05e6e70b1159ad65e52f74dd7c8f465e0e * curve.G, AffineCurvePoint(0x671a7be9330f27563a2a0cb8b7b85c9a474279d3, 0xbcfd6e4f4e260947bc8c3a733a597e213c27fb10, curve))
		self.assertEqual(0x1e03ce350fbd6f4ad8b91e425362331e064b76d * curve.G, AffineCurvePoint(0x25d33221e923ae805d0d16d83ecc0de68ff03c3f, 0xedf137ce10b17596cdb67e3a7678f92960885d74, curve))
		self.assertEqual(0x1d4e8b66e42c94c06f00fadea573288073777432 * curve.G, AffineCurvePoint(0x86153bc2f1e1519f9554088c00a0640f3dcf68f6, 0xfb9b58da5381b5e293c16441df1e26b5fc13abc5, curve))
		self.assertEqual(0xcc56860fe7962450de9755984be19b04f1b9bc2f * curve.G, AffineCurvePoint(0xe9973dab55b5b6023d46ab76f309621ca105e4eb, 0xfdd3431c0c78135019847264debe42fcf862ba6c, curve))
		self.assertEqual(0xfd16097b1ae87ff1b895abcc7da094ac310cd4f8 * curve.G, AffineCurvePoint(0xb9abad3e1e00371bb7c1a88fefaedd3067c2c99b, 0xedeb852f81219688e223f279c9339189583f51f6, curve))
		self.assertEqual(0x799c022c547cc790c9aef93e43c56d7a2bab22a0 * curve.G, AffineCurvePoint(0x73262100f5a8b415f433bccd092111a28c7c7247, 0xb9d9ec961bf8e9db72e67fe61e14b6d836c32937, curve))
		self.assertEqual(0x1e9086b9616a7640c62f79b699a5ea7303340284 * curve.G, AffineCurvePoint(0xebd1c72fae4d97eba4d314df252457ff6fed22c8, 0x31f0c1cb526dbc344702b5adee8fd063cbec189, curve))
		self.assertEqual(0x9df05af60736e9cd3a061bd4b001f924f04d9037 * curve.G, AffineCurvePoint(0x3e8d31b42a06e140e2724d9960fbb2098b8e8d76, 0x4e8483bc4bf14ef451f2f1ff4ee194c94a93539a, curve))

	def test_secp160r2(self):
		curve_entry = getcurveentry("secp160r2")
		self.assertEqual(curve_entry.oid, "1.3.132.0.30")
		curve = curve_entry()
		self.assertEqual(0xfc75b57fbdfd25e3630637321f0040e1992408d2 * curve.G, AffineCurvePoint(0x2c62e7e5bd013a402ebeed6d892ef7654b4e83af, 0xdba02cb7c2b5cfcf4227bc76947cdfcf6dd78cf1, curve))
		self.assertEqual(0xe0191d86a66dd375a0d6ee42c6a96b8f47b98264 * curve.G, AffineCurvePoint(0x87d4dade3687c5009e9ebfcced0f9c034a580c17, 0x987eac2a89e00c878e3f8a66289da2456c8e290, curve))
		self.assertEqual(0xef7623ae55d13bfc96b603798dd9c089826e3f5c * curve.G, AffineCurvePoint(0x5316d2c34047a261d427184b75370888b00cd599, 0x3d8dd7f1935cee650084b0184ac53887e8a3ba07, curve))
		self.assertEqual(0x879501cdad6c30cc5ac5a2c4236ea2adcd135581 * curve.G, AffineCurvePoint(0xe082fd4b3f0b6ff48a16add48e2fa1c8041b76ae, 0xdd6effac79ca674aec7e8f6e0c47f666ea4e928b, curve))
		self.assertEqual(0xa7fd2fa073ea1812c702ce609069a0068af470a5 * curve.G, AffineCurvePoint(0x464d6a46b2ea4d62f5108d3f335f1099c438440a, 0x155e14cecb8ac4bb98f4dba927263efdcc09176e, curve))
		self.assertEqual(0xfc6229774014b43e4dfab68a09bd19085e8cb016 * curve.G, AffineCurvePoint(0xa36f07fd9b9e6cff1acc13b4559f9c09a8b492b7, 0x25bb0b04d4c32a6d2a654af7376b13706073df0c, curve))
		self.assertEqual(0x83fdeeaf9336599596155c25589c179fa15f677b * curve.G, AffineCurvePoint(0xccf6d7dcd788965aecf40a34260cf15711ce295d, 0x8aeca3e991cb41f9742244b44c83738643f6ef3e, curve))
		self.assertEqual(0xce68ab7571165fb078f209cab81fe8d3c0ba17a4 * curve.G, AffineCurvePoint(0x3e725837787186e83bffc7e9a44ea60c8006b09a, 0xace1835663c8259e83b27f921e86ecce160d3012, curve))
		self.assertEqual(0x9584407807ee5ac208de61951d7956f71078a8ab * curve.G, AffineCurvePoint(0x97d9103277273d41b1adb00384bbddef4c59cf4d, 0x1d9c517b4a42ac5d0f600cdc065b03382d928eab, curve))
		self.assertEqual(0xa539d7ac23f11e8e256e4f4714b9bc4eae371edf * curve.G, AffineCurvePoint(0x5005dd83ed1184115a3dad6707a31a54209e175a, 0xc6913d4d2dfb22aa5fe2dac5195934560756e942, curve))

	def test_secp192k1(self):
		curve_entry = getcurveentry("secp192k1")
		self.assertEqual(curve_entry.oid, "1.3.132.0.31")
		curve = curve_entry()
		self.assertEqual(0x865f992a10e492ad27dc85123b200c4ac461cf32149c0335 * curve.G, AffineCurvePoint(0x6fad73d5b5d093f3974323d42da65d1f42811c2ec4fcaf16, 0xd142d1a1f42b72cf8213313344725c1989c41f5d5f6bdd47, curve))
		self.assertEqual(0x98a2f8b0b49dfcfe01df477218f3b3083686a0e1995e3ef1 * curve.G, AffineCurvePoint(0xfd035a199ffa6dc32625d13844d3e661901b78a861914317, 0x56f141c234f9fa412d2be5fbb9a92b07c7718f34a44eb3d6, curve))
		self.assertEqual(0x5c294b6b63fce1a164eaa4874a3610bf18d6d81d9cc2dcf5 * curve.G, AffineCurvePoint(0x4f4acdc92174dc4aaa55d860c440e2f949c547b853513cff, 0x6a706c0bb49a22c7f71b3df98d7e90663da5cf904a9ee95c, curve))
		self.assertEqual(0xe485cc03e2ce1abb23adb7ea4ba10d26c51634e555f8e257 * curve.G, AffineCurvePoint(0x44183b306e4b2d40b47f6f55c11f8c697d157173a25b174c, 0x6d69bd42b4aa4eaafc190b8795aab18bf67b2740717cca0e, curve))
		self.assertEqual(0x6df3caf9dbd224e7f3d103679d31c229002d980514f3fec4 * curve.G, AffineCurvePoint(0x9383eb8b11d345dbeee8530228fcc32b088d946261c67616, 0xa5d23a34ad1414ac8b7c787ba5e835a6b9c40bc7bf591bdf, curve))
		self.assertEqual(0x6996fadd60e7e7c7fe7743bb338072fec51128358e9686db * curve.G, AffineCurvePoint(0xae0ff0b04b22f1628c070e7e853f43ab4c09b14c32c1930f, 0x8096846eca6ec254d534242c9cb3537521ff535de3c199f7, curve))
		self.assertEqual(0x5cc62664703701fc3d53c79aca98f37cdcd652581ddee81a * curve.G, AffineCurvePoint(0x17074a04340bf3e01b6a44998227454d63fb39fa54e28cef, 0x4efa5377dadefa9b0ccea913d1193f48b89e9448db12cfc0, curve))
		self.assertEqual(0x1e1c9a766eabffff4402a64e8a7e3b3bc43e271f0802dbc6 * curve.G, AffineCurvePoint(0x4c1ee141212aff8b1ee1baf03acb343b953c23129c0bf753, 0x1e9568941d4d41aa4ad5844641352e1521190cfe194a34a9, curve))
		self.assertEqual(0xad9734c864cb0646c40ddf70a07a1e0568343aa521e85ec4 * curve.G, AffineCurvePoint(0xd4809eda4a532baf45378b1f844055ea849b99914f646970, 0x2436b3b34ec733c68c63e3887754c472190a657fa18dcf96, curve))
		self.assertEqual(0x9be1ec58e8db22496d8ef0b48a2b278131eb9774ba6542ed * curve.G, AffineCurvePoint(0xdd18a2ecf8c13660cf323f225866cd9a99719a02a865a247, 0x96fa00d5c01b239b048421d84807f92ec7e02f242aee4a6a, curve))

	def test_secp224k1(self):
		curve_entry = getcurveentry("secp224k1")
		self.assertEqual(curve_entry.oid, "1.3.132.0.32")
		curve = curve_entry()
		self.assertEqual(0x3a6214baf9fff3a4556ff3686b0ccc2aad5cf143d84787f1fb755678 * curve.G, AffineCurvePoint(0x67d7d7d6da938c8c9eb768559462bd83ca1392317623422a5dd3b456, 0xdbedf7e0d91c16395a8c52e7da3b9c51480806400f9803f59a2d76e, curve))
		self.assertEqual(0x3afc25c72665795a8a7cf0a16952a049e5a2c13f8d7fcc93d05b7922 * curve.G, AffineCurvePoint(0xb904bdaff98a353f9dc21548a0ceb59d3ce2752ad8b24cf18bf6c761, 0xa811a8a7b59a404f5c451e344b7426f3abee20c7f0b3aeea13f5c398, curve))
		self.assertEqual(0x96ece3955bf110423d9d8f07844227880cf492ab0ef42b36906023f6 * curve.G, AffineCurvePoint(0x72083bc930a7d4e73dc93dbf3bd157c41fe13ee75b9bf62bd1122f79, 0xe07b6e59025055d3186a4780253d224371d0c7e4eb64c498d73a056c, curve))
		self.assertEqual(0x698c6e21552013b138da1ef5c4679e496fb8f3879315a8e27820607c * curve.G, AffineCurvePoint(0x80dc8f42de29b3a10c97ab595c324204569a303f0f544937d64ac0f1, 0xe50efb34f309c10ba3fce0356cbf73c3830b7f120643510bdf9a4d49, curve))
		self.assertEqual(0x2e50c0c42389a2d4155adf5b363de7965264c034f84e61a968475379 * curve.G, AffineCurvePoint(0xcb8aaf7e170d7b592645b6144d60bf66d1d7206301e01bf4ff7a8c6b, 0xddf0a84588d3d2d87e4562d2dbea67f37d34aedca9c565df6ef84215, curve))
		self.assertEqual(0x99d66b88b71bf2e55698de68c83f75dbebda0fd973cf54018662177f * curve.G, AffineCurvePoint(0x29e138c14b9de4cf2dfef963bc22056ecc3a1a962ba13fc6024a355b, 0xacd4cf271c6148b10901d33d12abd9af3f0d34fa41e560ab5d1ca49c, curve))
		self.assertEqual(0x372f6535a838b59666909d4e01e6ed8e53edfc7ed0f22ca1fa08449f * curve.G, AffineCurvePoint(0x68b0b6b2019e3dec0b39f5174bd7804eff70f0fb2b0afe298bd35ced, 0x93aecaa2d58e7e7fddd391b195e9b9cec1a21a9ec5067b0514b48ddd, curve))
		self.assertEqual(0x86d23c7f09e1871c5bd4917275112ba6ed863ad9e34db34c005728c4 * curve.G, AffineCurvePoint(0xc78acf2a5601e0fbf13ca730832ec4a7e45924d8e35c0745382ccc2f, 0x4acb37822a5e24196e7aa75b6b860bc37720e37c5975c105930522d6, curve))
		self.assertEqual(0x2346fa18d915e5482dfcbba51f24d145942a6d57b9255b9f9c4e29e4 * curve.G, AffineCurvePoint(0x421943448594fecbfc54125841dbeb8f62c88855256641a8c419f34a, 0xa16de2b076428f3faaa63b38794818186a9836c3cbc1a185411963a1, curve))
		self.assertEqual(0x525e0adf5c688aa6ad8b3726ce7a8d7c0257ee6d0601e54c95a7cc7e * curve.G, AffineCurvePoint(0xea6ced5ad4e4899e5052243f5ecb330f6b461982a242b738eecea604, 0xf598c9279f9dd3037eddfff77ed6cabae830576486b66129ebd785f, curve))

	def test_secp224r1(self):
		curve_entry = getcurveentry("secp224r1")
		self.assertEqual(curve_entry.oid, "1.3.132.0.33")
		curve = curve_entry()
		self.assertEqual(0xccead6cc335f811594ffc91f5e22ad57d9d0622d51d9ec4fcaaf9ab6 * curve.G, AffineCurvePoint(0xb5bcd231610d6e8f72a2457b023dc0388f125c62f11c7fb3c8def62d, 0x5d54bcd2c012987fc6403fce32a19aef651de314dbdc1d31bcf1dc7b, curve))
		self.assertEqual(0xeb0e873f0e5c65ca4c2d082c2824e4c72f7b3d62f8e34776c0ea865 * curve.G, AffineCurvePoint(0xc3ea9b4bb2664ef9c25fce05446fd49ffc9eb8242162d964deb9e7e1, 0xf17fc11b628b7dae1c87d5e2e7815f4fda4fc37cbe279a29a2e08b4b, curve))
		self.assertEqual(0x8ddd86f03770b86639c51212f31a0a47b3b6f642599fc92f6d6c6fc6 * curve.G, AffineCurvePoint(0x1bdf49f2a1cdffeda8c8525d34ad1ff0c288d2a60c4eab5fbab2b877, 0xae436982485b3520dd043e090338b25c3f9d9c996c833af8a476ee71, curve))
		self.assertEqual(0x908eff6367e8ba8f426abf4d72e47b6466eb1c3042a77b37d8122d14 * curve.G, AffineCurvePoint(0x6458b257135a34adf2694832281c758d14528ea116af5ca255e31202, 0x55d3ea50ca98e9c77f8265273ce4fbbd433b65096b39b091953a474a, curve))
		self.assertEqual(0x315a07cc08a91a926a0bbd88dec09ebd128c7faf62bf393380bc1c0f * curve.G, AffineCurvePoint(0x50e782df64d4662496948d4dfcaf66e0119c48754d1d3230efc8342b, 0x89f49b1ea46a5f13332d44fbcefb7aac3d71fa54af76e4aabb886e31, curve))
		self.assertEqual(0xed3cce4c245d96248192f2cd1071751567a554ee9bf02424c6ade71e * curve.G, AffineCurvePoint(0xab5e5e5a1164e7f7276f0031997e6ef43f24e9b3d4545f6f10f01208, 0x5c29ad3d841c1fd3ed73c14ce33c9d3f5b074b163c786d352e449159, curve))
		self.assertEqual(0x91cc85ba856b7e1bd910a0d4271e518572fda3c4a22278a58e8f86e2 * curve.G, AffineCurvePoint(0x921887dbc5cc7104bc9d64ba426268776b706ae745e3b829c0ce206c, 0xf1ff5f2c38bd146aa51cf3b20c6f95ad16c7b5f47412737dcaecaeeb, curve))
		self.assertEqual(0xd7d409b8a9febb9bea84ac7740163974bd624228965fb034faf5c509 * curve.G, AffineCurvePoint(0x31c23dcac9adf8c0782a6f715a928434995be299bf9d5f4510532c6f, 0x5d3cf32c786966319c538f0a6e6ed53fb14c16eff3c9d94b4c888d6d, curve))
		self.assertEqual(0x4f1b0fd13edae8f4e8dfa043e02ec8a50ad3872a8d8329850f7627b1 * curve.G, AffineCurvePoint(0xf360f94e646aee71d1f57f9fa10bbf2645a95bf616a43cbf32971f79, 0xa68b0fc6b4d96102adedf62e72d875dc55120d7a1ff54ede0991f49b, curve))
		self.assertEqual(0x2e79a854acbcaddbd916b7a89e412df087dc12f9912c3fa7e650dbbe * curve.G, AffineCurvePoint(0xb52f7c9746b311f42510d9664f55c800cde12a0d561cf9b406221817, 0x89f494828af9e5de4ebd013d3297974ebe883337fcc90d1db571c66, curve))

	def test_secp256k1(self):
		curve_entry = getcurveentry("secp256k1")
		self.assertEqual(curve_entry.oid, "1.3.132.0.10")
		curve = curve_entry()
		self.assertEqual(0xea7303441621550bd92448df54e4b800558764594515594e0ab40f770346e219 * curve.G, AffineCurvePoint(0xf8d42d8dc701b848de285a8bc5f2c1bb0b8ed3b68d0c6bd06b1bdf14e86f51a9, 0x2830cfe786f577bff7195dc8d4bb5919eecc7b451af39332ce605cffc0537188, curve))
		self.assertEqual(0x37274232162e35a5f01bce83be3156bc80d1ba9bfb1d1e78d1a3d42168d93bf1 * curve.G, AffineCurvePoint(0x131be11d064fe9e92382d5d0d0b8aa546d452ab6acea95eca3c71a24f8547695, 0x105ff7c850ffeb6f2078e42fb7d6025a3dc6b16d16527a89c8f0f5ab7f6ebc00, curve))
		self.assertEqual(0x153f37d5e230c856b07e1348684a113da5ed6ea9474f4cedd6b2eb0ab8e4bcd4 * curve.G, AffineCurvePoint(0x2a982fa43a6ac66186040c7a665e8920527ceaff672cd23fbacb0d6717bd9899, 0xbd6af9119699978e1047ff2380e201d3170fdb187b2ec160e80b27b1ea5a346e, curve))
		self.assertEqual(0x7537a0ede88f730fdeac4737123c82e0ad4d2f133102921790b0859662d7ead5 * curve.G, AffineCurvePoint(0x7679dfb9d13e67b64a59ccaab14ac68e3d23ca799aef3ebd74a92402033a0f69, 0xc66090d0ab222857c940e95d50964756a4928b412feee785f3eef80810bd044e, curve))
		self.assertEqual(0x26daec614c9aebaad877d3b75c6aa2aa652a8eb97eb6f0def31ace05fdad7dc7 * curve.G, AffineCurvePoint(0x18d2dc2b8a04c02aecc852d6db3a96e2992a9910113ed6cb1f26a5ec9701ab71, 0xd9a08199273efa2775a5b6f48c7388bfa7a0feca40781f794aa1f1148a9397d8, curve))
		self.assertEqual(0xcde2b817fd029c90899708a096bc1742eedd5316a7042119d367f02d4888265f * curve.G, AffineCurvePoint(0xdd84efd9090f10b416d5349f8b34b243a274ccf464cbb8a77649de9b611a70a3, 0x777a8a9cfbb322fc0393e0e9840dc2594eee0132caf3ad94ffcfa24733ece7a1, curve))
		self.assertEqual(0x602118eec3f38d8e91ea3751dc8836076a76635a222521e7d23a271f460e2fd5 * curve.G, AffineCurvePoint(0xb85300469b981013c364515329e589fd32a8d368cde74626d3168f9adecc31fd, 0x82e3d2cc93d586f1bced65c878adb7254125abc71592ee5be70eeef34fb76193, curve))
		self.assertEqual(0xbab499c1eeb5495f8d4df5e75553e2fe7e4249e39bbe6fd74182101b22505303 * curve.G, AffineCurvePoint(0xdc00c9098694432f64ccf352c23f57ffce50309e86484bbe08a29f66b2acaa22, 0xbfe2724bc4ad34027d328bf586bbecfd48be5fbf149b37e5580b31834cb1673c, curve))
		self.assertEqual(0xa2682d1daec8066c93a6cf3c2ae77488615f1bd1e0613277a93cba3ec6098c43 * curve.G, AffineCurvePoint(0xccd354651e9fa8e1ce240b3dc90fb9a7078d7713c4f54e2ce5b72e71f7c41ee3, 0x49b59edb23fbadaae652173c3eb83654ca11b89dee541a39e3f4a9cc8347791, curve))
		self.assertEqual(0xfdae61d504bda9708d6a19859e7df697b6e717bf67837a7893246aca9b437338 * curve.G, AffineCurvePoint(0x7d7c8536ea1885e930b6e22bfed8444d1a087a47e1de1b2c86ff33094483aeb9, 0xaeeb898c41d8262526098c920c8bd56df3b588f1f70e505300ed8fd49deb366d, curve))

	def test_secp384r1(self):
		curve_entry = getcurveentry("secp384r1")
		self.assertEqual(curve_entry.oid, "1.3.132.0.34")
		curve = curve_entry()
		self.assertEqual(0x1a96e5414df30d616efeacd6edaff56912fab628356f4dd43f39cfae682e6f1788fcb3d0583ecc5198f07e17c70ae25f * curve.G, AffineCurvePoint(0x3096ac8745c5eb616b9a1881b6e703fa1f10f44ae4448838489052b7ad9f7e0eb475cb45217437a36b3efdfeecf3ce2d, 0xe43e49d808098cc07af8fde96f4d25420e2b1a70c5153ed313a057bc9f36057f7bcd3b11eaab7832d7dbc021360835fe, curve))
		self.assertEqual(0x4d4ea0f6dd696b1a12f63caa50843de5a982d3d73805b70531c220230f095827499f693e783ac9dc0cd1fbaecbce21bd * curve.G, AffineCurvePoint(0x53763c435b6aa5588c7856504de38059a03476cd30dfb9b510c2daad2fdf12ee270b94599f895eb7cd7b5245be438bd8, 0xead0a6647591142396d4171b6d8a0c17828775da8dcbd455994ddc69a578c3426cd7ccc207866a014885695aac3f9603, curve))
		self.assertEqual(0x140ac0249a7398c9dcbd5e16dfab4b97df41c2acdf49dd36d54b107d16b20c9a5a97fc27e049f85bc973eb620bdb76fa * curve.G, AffineCurvePoint(0x1404aabf9122274204c1f92c7fb578c679bdd1d8024cf143a41bc07e88b6aae4ff474a9723c1ff4b9577a0f5b78d78ac, 0xddac6b423a72c5b73deed63dd3ac9957094a24b3abfe3713cae2465a45f13bb02b07029c00c10299105104c9152e8f30, curve))
		self.assertEqual(0x9a8d308bea101ebb2f5669cc0255f61f5a98d27c50e0a987c9ec840e779798eccc4ba4f8ace18018375ec176ec611e72 * curve.G, AffineCurvePoint(0x223d140b9c0877dbc256dafe3a16db754c4837f92988eec570e967d9260095b965e72ca4d9434868ce6bbe1bc233b76e, 0x2f40894875d810820b6aacbb6a7f4b41f5f4d9e3345cd83c05c0147b41f6f52d8bac415792efe7c89037cacb4b4d7d58, curve))
		self.assertEqual(0x687ab147d1c3d43355ff48d2c63d78ac30b411accba2284a4fccfe6d97690e1262b5f3c684ba73d83ceee83e6f0a90a * curve.G, AffineCurvePoint(0x1683e8ad9f211facc9c5f5a626a8cf40eb33c1b717ac187350184d436c5185764b14118fb9bf877825df8f228cf1b7a4, 0x1f47e6c1b673e9d798feec9fd92183f1f144751d99841a294b442f2dbff732ad6b8c99787cb96090fdc6933aefa5e0ee, curve))
		self.assertEqual(0x589c31b15ae2ae6f70076d7f25639c784ecefe65de2ecc6078fa78d9aeabe9af05e64e7405b0a0921ea7e909e32ec3a6 * curve.G, AffineCurvePoint(0xbd643ae54ffa6330005a8a5742e4bb0f3e06c199edab9052fa12ebe9c6731c121d101823cb8dc10289a1e92c1a126872, 0x575e6a4ef906dc249bb88131640565b8f4c5774d82a8d537d6346789dadd545c23b743d00ea6fbf483409f758f1e1327, curve))
		self.assertEqual(0xe173fde17d1fbb9794b6397a7f603945e7f4c92636eb9e153d0838e653b8c262210189359e7859555692e20ebc61b465 * curve.G, AffineCurvePoint(0xa63676708ac075c8bb144b1f0757fda21a66c5eac75d867c304cb386b9e8709f95a7524925b473f2424bdc9ccffce98e, 0x1925d23a08e6a376d61313e090a8403687ae4dba236819900ad51de45c8c3cf5fd0984262c3f52fe5c602970056efc9f, curve))
		self.assertEqual(0x1ef37b9bdf507d3b73dd8ddceef523d1759dde097270b9ee4ddc706fc8d1a789754f2fb5130eb3f7532be53bbf1d9046 * curve.G, AffineCurvePoint(0xa2366b441bdd23c7d201951fd64f6a368b0d8c441be576f0a46cf5588b0e413026e96e3057adf2d41093cda88ba76191, 0x420f62d52467d4cfdb03a1b903a5848e4e26e1c55018f29dd9b31c2b503e4f25d9ad8cf46f1ac605bbfaf83a1a04fd50, curve))
		self.assertEqual(0x4eb9c39c2da2cff9597935213d506858b030759539e70f81c538617bd5e09d890580f0aecf74e77060f770fe0918c4c * curve.G, AffineCurvePoint(0xdb835efc9452a877e8cfae02428cd155985029052f72f8d4da6c44866bf01acbb645f3499c2e071392d4f5d767ab4089, 0x1458c8a4b27554a5ca350d94e8b0c0a47292f22e2b0e500b1cb3e8c37b6e29c3923c42e5b3ea7e0a165c7ec8c0c98290, curve))
		self.assertEqual(0xd38d6a9e76ed6e189ee4fefc19c67031a2c093ce9f3f5cee6c5b21aef4b99fe12c96c0d728eedf05c18928b88b0d43b2 * curve.G, AffineCurvePoint(0x87c18408b5c3610dbcc7f411553cfa49dd43574373071ac204f996b02b8781ece792cb75c5a32d6a513ee23ba87a9e8d, 0xa6bcaea5fdbe0c48896ad4f733f5ae113339a856478db89fe9f67971c8758df73bc526eddef78e652114f0822792ae0b, curve))

	def test_secp521r1(self):
		curve_entry = getcurveentry("secp521r1")
		self.assertEqual(curve_entry.oid, "1.3.132.0.35")
		curve = curve_entry()
		self.assertEqual(0x2c471dd5fba4b0c78e1e8c07332dc634ce0644e1ba60db49da2353e56796db23d03397172641a5078fe268166ee998117f00208094828b18b1509ececc1948becc * curve.G, AffineCurvePoint(0x646a198886c09937a1b5fec1bd9aef91dea743b74e3f2688af9e7d3f4d7e663d70defeaddd5510368fa316c4277a4fa7bedfed932841a6aeafb326a80a2cdf4503, 0x2a0f4f8b4e50499f1deef497cc4f13d24c92c3b79bb6cde9cad382f59bbbc7e8d4562a63ff6328a271788bfa1504fdf28c1708291ca5bb72d0e83044d5a76afc19, curve))
		self.assertEqual(0x1b14a220431194f861b57e366f0196207e60ab4eea61416de7bcca644ef1c4cc25a1f3d07a41d3d9c787a7061a88b7b1f3e065147f75450082ae45594c5302f62df * curve.G, AffineCurvePoint(0x17bcb833478b1ef8165d60e41224ce46cbb2e2c4ae5c52902516b72ba2ea07fc76b7ff7753c6d705ad69df106a72f84de54ce4a3bb13eb76dc1b5a54e7893eb3208, 0x17288fbb69befb296a765774f384374fd5afb7411c99e9ba96c3682aef7c9f84134d269397a15dacbe08211ed7df767e8dd9423a35c5a33d9b26893adde126318c9, curve))
		self.assertEqual(0x4821630013b1868aaedd6456b46c1c7a8cd86b8be4af69e369f283c929e6ff55d2fef4cdcacd8e030f000e49ffdd5c5f68ede6b8244c7e975f23f4bf36e91be704 * curve.G, AffineCurvePoint(0x107e2d8f14b33674a979f1aebe1a8c45789672ebe03b776dd339aacde9e2fa9ad139646c7cded13453ab764f6cd6e45f44a6284ebd20693170b2963b0f521f4ca9c, 0x1d5eba87b87f83249363f0d75e127d6e8632bcec5b7d47cb9ed8e9d6d47fc148d6c388edd6d8c7cc2817d63b000af00287398872554b4875f2bfd288ec9a318cafb, curve))
		self.assertEqual(0x3e895ab9796deb4b07f67f0d9a9b7ab33d295ab7f00a9f34a588599a4e56ca57e5b0881d96752c7c14d66ec4a8985d29a086c2cb413e702e4458dd948bdaa797a8 * curve.G, AffineCurvePoint(0x95998cba29d261a95463c4b28bd89d3a44a685c7be2d7a32be3a68bab592baf1f72b5dc17db1eff73ac2727fd83b4d63e2057a508545a9c3d40bb178d43eeb0e4d, 0x10134382d81c7bb7da7a5bd722ff11509e21c4d4c187ab21fb35305ac2844fe611484f106950f050b789ce408dca6c4b5c3471241dfedb55cd8111b568093017d36, curve))
		self.assertEqual(0x1fb5024a653a0f6f248a9b9b1370e2763af9cb96f6bc5389d780e655f923fdf19a4373fbe844758251fcf538bf4204a38d0dd21a41bb744e8e37ea01d87598213cb * curve.G, AffineCurvePoint(0x19d9e919449c818bc74c4047042f3e76603662ca1fb4622a8108400642b49270fb694dcf957bb29e98b6d054f36eea5c58d2f994f39736f7771d2bc351060ee915e, 0x1ada021777093a5fb7fa2d55db045a5347a2f57524ad87d60115140d53b1c5f22d49583506c1bc3cd4599cf62ab9b5e37c05c104b11d3bb8ddca68f5930237a2e3f, curve))
		self.assertEqual(0x1b96498ca007e43d0bdde378ea1d4e8ee6e5f3bfd60993d32c0348da9c34f16ddc356fea68257a2913a529bb503022be760a84d1f66b6431fccecbd47e113303ab5 * curve.G, AffineCurvePoint(0x16a67a36d74c103711f578ef84d4fddb75f020bf3157ae2f514e649496129885e836e56a02a30f34d746f0d1d3dddf7f80d2b788f1c78434026ac28b142f8e76e03, 0x14b22300b9928d4341658c49639eb4d477afc1454eba2049b722c5cf5dd985e895d13522537076edf2b3f29fb35a7771b497e61f0d993f89fb7d8c2d3694919ad07, curve))
		self.assertEqual(0x983d05b585b35a6a4c59740fd0521b9adc244032852ecf0a35095f6f8283535d56c85b0a4937d5853271f23aee16af7976b95bdb0aedb93eb5ab97b9cadc668fe7 * curve.G, AffineCurvePoint(0x10d9b5aea3199eb3f705bf37322c2d4f2900346ebe775613d2c093298683c85dbe3e25eb08ebefb725308d31e1852ffd5347d1838f1f534cbdb492ee1e02ca16439, 0x8adef66af456a38f65cb2d6915dbdbd2715e67d81965771cbee3bc56c443f99841dcba3ad791586a2e71946d403d9d04175d300c13b01ecca721dd8baeeaf3ebf9, curve))
		self.assertEqual(0x5a4f053fbf9570c383aaecf2591099086f00a634bbb8bbab04244f56e286a121fd15ec6b31201d420a77c5b7e7c9f05da82acdb2b7d494d5a16b389f9f3b90f930 * curve.G, AffineCurvePoint(0x1564f3fb568ec2979baf1186a1989477c1f1b14493702438652d0b58e475b894805ce25a697507dd0fbe162087ad371cfc61d0db9889f403f6172e4cf2a41cb346e, 0x1308a51056d20d641995734b2ae3b145ae887e620919e8d97d6661a72877c107a895e4bc772d1ee26323c47082c086b502d823d0e0318def94e59b5475acd1e82a5, curve))
		self.assertEqual(0xf51a86c6bfab09b719cabb4af559e635d8f33113b621ebc07113dde5e59d3d68091e6a21273c509d32e0a0a55a3b1f6376423d4d30a4257fa0972b775316365a86 * curve.G, AffineCurvePoint(0x91d8e4d75a01e483533901dd7ab28299ac6538ce6269466a3b83d8978e1dced2e4f6e99ebd6553bf3c8ede483decaf920a2d7c346334b89ee628dfbae4ab6ee7aa, 0x1dc3aa1f41c2bcb31ce2416a0a92d8b84af316c7098c6e2ad5dd8b901c9213bebf1f8d69de51dab5b904bbc2fc68b9627352fc0a30b49123305ff23818e22c48cf7, curve))
		self.assertEqual(0x4b66d754bc1e75accc8e82cd9107fbcc188cf305f258f8e440afeac01a9b175c2fc508bbd527711dde1fe2356d590ef4cffca350b36a58f47b160baace331bab73 * curve.G, AffineCurvePoint(0x170b1e0983f7ae1be211e3843753ebfe046e2a0ae309d496ac3c38da7a02d7590f5b1887181a123bbac8c4689cf2199155a014b0d03b58650757d7885e9d6ef98db, 0x56b3b71709b4a361939aa3055a3785a6829337978c52d5d9fd19da41d76659abfd35afe58806ce66c78c846a37438f0a239f6e3f07a764df2e763706f793e1aa2f, curve))

	def test_wap_wsg_idm_ecid_wtls12(self):
		curve_entry = getcurveentry("wap-wsg-idm-ecid-wtls12")
		self.assertEqual(curve_entry.oid, "2.23.43.1.4.12")
		curve = curve_entry()
		self.assertEqual(0xf3c44eed2623273eb2c4ed7e3abbe7f422b1e8a654413f88bb405a1a * curve.G, AffineCurvePoint(0x2ab3c0c01dd55e102bc0a18bd2b13063612abce23ee60fb60f88f250, 0x16fc89a04103d352f12a6833b905066d233fa065f2b87318e5ed4ee7, curve))
		self.assertEqual(0x575adc255596fad4107c4b1077cf6abb61c7ef9d319725e7a5838623 * curve.G, AffineCurvePoint(0x6918a8ff2289de32da19a10154dd0e38d1cfa548adebac75b5ea86c4, 0xaade044e2e5859f25cdfacf4ebd30be1d30e2daf7c94d83cb7b059a4, curve))
		self.assertEqual(0x3c070403ae5bfae44ddea95b4759186b225337548d4062fbf1cbf13d * curve.G, AffineCurvePoint(0x69e04e59e6c58ec06bf0f515999a937e7f6270dece2db7f33d0a2dc7, 0xa37b9d1d20aeea8591480007407745c5f4d7dc3df9c931a479cfcb2c, curve))
		self.assertEqual(0x604325b8a0d7bd519273785bfa36490dbb6beecc0d41ab6aed639362 * curve.G, AffineCurvePoint(0x5308b375484d46f60e5346ff63a7d18aa7877adfe5c3e48c3482e6b7, 0xecaaeaf6cf253f90dc2a37216ac724e3567c8abca7bc18f2ff044111, curve))
		self.assertEqual(0xd012110ecda3244cb16b53bf644b82f68eb51ef898e602aa9dfa779b * curve.G, AffineCurvePoint(0x9b39f23e6febd402bddd9702737b4c9032ab849b42a428469e541b2f, 0x71ef3745e980b8bc1f447faf5c4e2f8b25d802d527d4ad2fba14c8d9, curve))
		self.assertEqual(0x60cf29b889813a7e89282f64707d3a70d3824b23d0a325d4d2f1d6ed * curve.G, AffineCurvePoint(0xee97eb04ef0e1a2af6f1e863ee583719cd0ddc90e368df031f6ee86d, 0x2527313cf8433d21ee6153d78c6290962e720cc8d39845604b04ac3e, curve))
		self.assertEqual(0x6b3b400f713d844d374d04531a551065ed8fb36d102ff54467500637 * curve.G, AffineCurvePoint(0xf5bd4a045fe19d8304219a0f221ee29253652602cf32f5f46f8f89b8, 0x4fa296acae79b6cc94f50dcabad9072785ff764b4a175819cf5791e4, curve))
		self.assertEqual(0x646702220308c467efff5777eefb8cbb37d350b3aa526a17b47d3c92 * curve.G, AffineCurvePoint(0x925ef39b3f756245409ba24d6a6114fc6e3a4091b3d6621cc643aaee, 0xb556325354722e983d2e90ed566a590ee24e59614cfa9813c11b17bb, curve))
		self.assertEqual(0x41563982819d63135fe7d9ebe509a2158d5d5db25745adfb9c42c0b3 * curve.G, AffineCurvePoint(0xd1631859e81e56880d96f3af8311272fc9e8a83127e2bc803045164f, 0x99bee2869104aee1d07df6e4cf20b7588aa8ae729e0d4cef60695307, curve))
		self.assertEqual(0xbad02d87577a245e5240a48dcdb8bf18b61aed11fad1a456fb419e61 * curve.G, AffineCurvePoint(0x91d9968fc7759d5db7f7fb99ea991c759616da0dcba0c3c6b8fb18e7, 0xe9bf25406109ddc80acf199292fe3fff84578a750b65e8e6212fd877, curve))

	def test_wap_wsg_idm_ecid_wtls6(self):
		curve_entry = getcurveentry("wap-wsg-idm-ecid-wtls6")
		self.assertEqual(curve_entry.oid, "2.23.43.1.4.6")
		curve = curve_entry()
		self.assertEqual(0x5939c9ddec01896eaf7a10642b94 * curve.G, AffineCurvePoint(0x1df13d7dba4a500e70f7ef7b9b3e, 0x92ff3718da1696a6f1a9b657c369, curve))
		self.assertEqual(0xa6d42e0fcc093ef9e0da09bb78f4 * curve.G, AffineCurvePoint(0xaf2eb4999d76f32644adbbe64472, 0x75a7668289df28ea95bbd4421484, curve))
		self.assertEqual(0x2457bdeb8950a1edd0842fa1775a * curve.G, AffineCurvePoint(0x40b92d3c01c1abeac725632a51d7, 0xd526a0677e98b81ab134391e35, curve))
		self.assertEqual(0x87f8f22fed231162e64b4fbe99e * curve.G, AffineCurvePoint(0x45bf0b43aab63076d8882f80590a, 0x56f2b519caefcc73f31fb1f47fe2, curve))
		self.assertEqual(0x778c34778c62e0ea93942dcceba3 * curve.G, AffineCurvePoint(0x68880285e63ff66a06cfbcd6f866, 0x1597ab64564ea2c5efe5a60c70fc, curve))
		self.assertEqual(0xb63f76fa49b431ded8a490b4de5d * curve.G, AffineCurvePoint(0xc57d3d5c8f2dcfedd031e8feabdf, 0xb723a9abfac9fdedd6e2e28f4af3, curve))
		self.assertEqual(0x83960d3464f5b32a28e3d4b6a447 * curve.G, AffineCurvePoint(0xb8fd8cac271950df4a39a7e32bc6, 0x3bd93960380d44a46b9017335d1b, curve))
		self.assertEqual(0x150971100e774ca18da084d6479c * curve.G, AffineCurvePoint(0x8f1d88e83e84c5e535f347b368c8, 0x78bdbce50f8924ba44785edfc865, curve))
		self.assertEqual(0xa0779d24223b74a3801c71f82976 * curve.G, AffineCurvePoint(0x31c951d4aeb56c144ea705783816, 0x8caa8b9a2676d34df1cce22bf917, curve))
		self.assertEqual(0xcb3e21b5fc4015679a111afc8b9f * curve.G, AffineCurvePoint(0xce9c7870eafd5fc899b367edcf34, 0xb6df7548c53357b5d72e2c96696b, curve))

	def test_wap_wsg_idm_ecid_wtls7(self):
		curve_entry = getcurveentry("wap-wsg-idm-ecid-wtls7")
		self.assertEqual(curve_entry.oid, "2.23.43.1.4.7")
		curve = curve_entry()
		self.assertEqual(0x3b9355e34c80e04f8e9b63ce35eddd986e538706 * curve.G, AffineCurvePoint(0xf1943be1a07c74a42763abf8940d735a03edca99, 0xe011022bf7cfe76f4747353127c20899e31e78aa, curve))
		self.assertEqual(0xf8bf013f87b5a405a6d5a48d26180cc657ec6844 * curve.G, AffineCurvePoint(0x1f2795f05a7043164783090957ca8288b4ed4c0f, 0xd2aceea2325e08620763a20405de166ab3ed6c0, curve))
		self.assertEqual(0x812a1b34047934e2cce48e4586fc096b72a9741c * curve.G, AffineCurvePoint(0xe53b78e92572932f1554c16064e7f3dd49ed439, 0xe0e9be8f617ce4677ab9b24c8b6e78c4023377ac, curve))
		self.assertEqual(0x3b7f35a86c34e605f50de032a57cdd2a96b9c536 * curve.G, AffineCurvePoint(0xdc23ee2d9294e4d33bfc2f01442503c973064cb8, 0x88e4d7e2fbbe851a82e52c1f512c4e24d52ff4c6, curve))
		self.assertEqual(0x83e66bad8fe30d3ed2f0502f2d520f8a9c5793b7 * curve.G, AffineCurvePoint(0xc2c26701e8776f669c6ce1d326c54e9721b310b6, 0x1e33b68d2a31576010924936e5d63b3a96aa011b, curve))
		self.assertEqual(0x55fa25a99837d77263e7ebeaadf597553960f74 * curve.G, AffineCurvePoint(0xedfeb07abd41999b8530cd7f3d70ce79fd856590, 0xa632557cf120e310ff26fbee98b482f0fce9e933, curve))
		self.assertEqual(0xfffe6cd6084f82dab37fa742a2797fb193b2d5a5 * curve.G, AffineCurvePoint(0x4c09b62404be973c1f8665f446d68d64ae362ed3, 0x818086d074fd89f23e0feb1906808a521b4576e5, curve))
		self.assertEqual(0xed6d51304fb92956c1fc82b2643121886e5d778b * curve.G, AffineCurvePoint(0x8d81c76c91875bf8cebebedb45e8253e42191380, 0xb376ef2dfbf5717b78438b812f2c05ddcb835912, curve))
		self.assertEqual(0xe69f9a6dd89d6a00aef994923d4cf1bbbfaf4e24 * curve.G, AffineCurvePoint(0x9bba1b734180a5a6dd51bc180f22f0f26b6169c4, 0x4a2f71d1e3c314caeb580bfd571545b251cc2bde, curve))
		self.assertEqual(0x5d426c0e575803171aaa6e898433e5aeccc01ed3 * curve.G, AffineCurvePoint(0xf1715e22eef694eef69d07e4b44ff29fa5132464, 0xda8a76535b451ed4312e34a2434de0f3c26c894b, curve))

	def test_wap_wsg_idm_ecid_wtls8(self):
		curve_entry = getcurveentry("wap-wsg-idm-ecid-wtls8")
		self.assertEqual(curve_entry.oid, "2.23.43.1.4.8")
		curve = curve_entry()
		self.assertEqual(0x9a7f5efdec8e0a31bce67e5167c * curve.G, AffineCurvePoint(0x4945318d66037ba2d8c8ec47d7fa, 0x324a26a17b4aa31e4d0e58bb1184, curve))
		self.assertEqual(0x28a50bdb919dc164037ee3e473ff * curve.G, AffineCurvePoint(0x375a1b0fc901ed050cb1fdc2233d, 0xa23012c1d4a401ed08beead0c054, curve))
		self.assertEqual(0x5025de096b1c9e6ede415fb3e2ee * curve.G, AffineCurvePoint(0xe610709dc98eef94d99c5b5b72d1, 0xb8b7aaeea01b1397e2db01766cf9, curve))
		self.assertEqual(0x93f352ec28573fab93913ec9dc3b * curve.G, AffineCurvePoint(0xfe042b27af83334b60d6cb98e47f, 0x406fcb03422c2c35ac5435ea4533, curve))
		self.assertEqual(0x984d2d3d5f612c13e4eec781dbb9 * curve.G, AffineCurvePoint(0x65515c1b6f606fb003c729a7d9d5, 0x777740e9c00502f3edf15a959a09, curve))
		self.assertEqual(0x382bef62e706ef35de889766db8 * curve.G, AffineCurvePoint(0x407135b2aefe31386ab60dedad21, 0x8e232c614288e4c443b5bffc5bbc, curve))
		self.assertEqual(0x5056fbcbfdb18264632e38cee684 * curve.G, AffineCurvePoint(0x87535268764b27c3a8164e980c71, 0x1571d2f1685eba4d1298ad60b311, curve))
		self.assertEqual(0x8cd4f76c81cd30c8f3e30b0a878a * curve.G, AffineCurvePoint(0x287c650c576afe2146f341b489a1, 0xea20cdc97c061f894a7c2989e035, curve))
		self.assertEqual(0x548d6d9990dbc70f05dc829d864 * curve.G, AffineCurvePoint(0x6a6b8a6c8e8d37658b522a3f9ee4, 0x5e7cec47e78936b46a74de5103ca, curve))
		self.assertEqual(0x64633629a2b5b23d9ca752566a72 * curve.G, AffineCurvePoint(0x74d642d74ff18006a69ac4bc3a1e, 0xed4833e9fffc6cbc82dd66fa47ae, curve))

	def test_wap_wsg_idm_ecid_wtls9(self):
		curve_entry = getcurveentry("wap-wsg-idm-ecid-wtls9")
		self.assertEqual(curve_entry.oid, "2.23.43.1.4.9")
		curve = curve_entry()
		self.assertEqual(0xf6d3eef597bdecaf0f3cc6b6588d32b2d8f8111e * curve.G, AffineCurvePoint(0x47e43d4510a49fadfa8f0f9033bc174be4114f7a, 0x912761eab4092039cfe999c44fff18c213becdfb, curve))
		self.assertEqual(0x9f853a93b470591575ef721706b0427a0f5bd0f5 * curve.G, AffineCurvePoint(0xee2cb883e004fffc2cc768b36857962094f93814, 0x536aaac2ee6b7d227760b6b124de7e840353bb63, curve))
		self.assertEqual(0x98c5c8a9048663c2e2d4735a60a07a7380475bd2 * curve.G, AffineCurvePoint(0x5f1072d280f99ad13690995f3a067c1597172aeb, 0x7fbc2b4b114750e09220744b9ec57b8af418d7ce, curve))
		self.assertEqual(0x522617e5d0582ed2b32a5c1b82e46e8ba55e348c * curve.G, AffineCurvePoint(0x4e76def4711c52ccc6f25d1fcf647f699ee583e8, 0x155e4cd544c0184450c3d94019a69bcae17b375, curve))
		self.assertEqual(0xe8738295c9200a084975424d308dbc57ae167911 * curve.G, AffineCurvePoint(0x2d7f25ea369b6a682a27afd24dbf5db13ddd15f5, 0xc44adbd869f2c9867b225db58715932723a451b5, curve))
		self.assertEqual(0x31fab08d02f82a432f60e188325dd089e23f4f34 * curve.G, AffineCurvePoint(0xb5c2ac58ceef4ee9dad5f121ea3566a3d84e249, 0x2e8c52d7492955cf10a86bbf9726e139b1836d72, curve))
		self.assertEqual(0x54d32bfe7ca7a94cda92cc5addfc21492681c94a * curve.G, AffineCurvePoint(0x720d105e22f51ff39d8be47971f868d4071216ac, 0xfe1f3b640f1f0f770f5a174708be908f3e96eb5, curve))
		self.assertEqual(0x9564d7488afc8e46f30fd6084670ef67f55eb929 * curve.G, AffineCurvePoint(0xb4fc39d622259ee2bb190c633705ea4c4c080ea7, 0x2b2b4b911a52ced0685d2c3f30e5beafeedc6b56, curve))
		self.assertEqual(0xebb777ea6017741a2002df69f47545d10dabee01 * curve.G, AffineCurvePoint(0x44dfecd9dc5bcfc831d8df4f1fd77ab5cf1e4e7d, 0x1026b370fe80a10b0de3c6b65182863285e32101, curve))
		self.assertEqual(0x84c47fe1efade1a3fa9d2a9f6ef0deb0867aa64a * curve.G, AffineCurvePoint(0xbed1d763a06cdd71992ca35fdf0dd9845382bcde, 0x36641f1f69ab96ab9fdc9e08c446b6b8ae3f082, curve))
