#
#	toyecc - A small Elliptic Curve Cryptography Demonstration.
#	Copyright (C) 2011-2022 Johannes Bauer
#
#	This file is part of toyecc.
#
#	toyecc is free software; you can redistribute it and/or modify
#	it under the terms of the GNU General Public License as published by
#	the Free Software Foundation; this program is ONLY licensed under
#	version 3 of the License, later versions are explicitly excluded.
#
#	toyecc is distributed in the hope that it will be useful,
#	but WITHOUT ANY WARRANTY; without even the implied warranty of
#	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#	GNU General Public License for more details.
#
#	You should have received a copy of the GNU General Public License
#	along with toyecc; if not, write to the Free Software
#	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#	Johannes Bauer <JohannesBauer@gmx.de>
#

import unittest
from ..DivisionPolynomial import DivisionPolynomial
from ..Polynomial import Polynomial
from .. import getcurvebyname

class DivPolyTests(unittest.TestCase):
	def test_init_poly_secp112r1(self):
		#0: 0
		#1: 1
		#2: 2
		#3: 3*x^4 - 18*x^2 - 1976690588858450197671635644628394*x - 9
		#4: 4*x^6 - 60*x^4 + 177118416224809665109557189789321*x^3 - 180*x^2 + 996608094753628753482653968583278*x + 361727871463520656904304672391508
		curve = getcurvebyname("secp112r1")
		divpoly = DivisionPolynomial(curve)

		terms = dict(divpoly[0])
		self.assertEqual(len(terms), 0)

		terms = dict(divpoly[1])
		self.assertEqual(len(terms), 1)
		self.assertEqual(terms[0], 1)

		terms = dict(divpoly[2])
		self.assertEqual(len(terms), 1)
		self.assertEqual(terms[0], 2)

		terms = dict(divpoly[3])
		self.assertEqual(len(terms), 4)
		self.assertEqual(terms[0], -9)
		self.assertEqual(terms[1], -1976690588858450197671635644628394)
		self.assertEqual(terms[2], -18)
		self.assertEqual(terms[4], 3)

		terms = dict(divpoly[4])
		self.assertEqual(len(terms), 6)
		self.assertEqual(terms[0], 361727871463520656904304672391508)
		self.assertEqual(terms[1], 996608094753628753482653968583278)
		self.assertEqual(terms[2], -180)
		self.assertEqual(terms[3], 177118416224809665109557189789321)
		self.assertEqual(terms[4], -60)
		self.assertEqual(terms[6], 4)

	def test_next_poly_secp112r1(self):
		curve = getcurvebyname("secp112r1")
		divpoly = DivisionPolynomial(curve)

		expect_poly = {
			5:	Polynomial.parse_poly("5*x^12 + 4451685225093714772084598273548241*x^10 + 4180076395888131988333845356660595*x^9 + 4451685225093714772084598273547482*x^8 + 2857619479070427786098583565444538*x^7 + 2712959035976404926782285042943600*x^6 + 3493200387462890833051605306335031*x^5 + 1664261512972003338494133072764280*x^4 + 3183863769631769346019475564892149*x^3 + 2158205198319070480617574018689515*x^2 + 1568917158483904691717779985650957*x + 2739612518230210911306565236772771", curve.p),
			6:	Polynomial.parse_poly("6*x^16 + 4451685225093714772084598273547995*x^14 + 1194915302539316465006721479041222*x^13 + 4451685225093714772084598273541875*x^12 + 2898374479653178322823407405695201*x^10 + 3719321063853246276336125227239347*x^9 + 1062574130317608316194067554664377*x^8 + 618046791624865520158047013326227*x^7 + 3826944191410031044987675283514674*x^6 + 3050271178929523578700004086723618*x^5 + 549767483272659484268496478222797*x^4 + 1547149110974161262945903528856829*x^3 + 2786230832016007637291640041113797*x^2 + 811553965571218381173706067634815*x + 1495046048390295323017551608142995", curve.p),
			7:	Polynomial.parse_poly("7*x^24 + 4451685225093714772084598273547503*x^22 + 273735992205058422940432736871514*x^21 + 4451685225093714772084598273521841*x^20 + 2524571438181686502293979506534519*x^19 + 4114207386728245195083799797977588*x^18 + 1620388122818144499065285293522715*x^17 + 2676152959582283362108276927514528*x^16 + 1672420714775905024583270968923946*x^15 + 193393698609250602304129888059251*x^14 + 1769860823362734514734199565115987*x^13 + 4169611734101507129267026710302771*x^12 + 959249953044793216954466495769189*x^11 + 3393484819899811558683962921578600*x^10 + 1010036780223634139137376029938124*x^9 + 578750277210731414086133307847090*x^8 + 3317080618724145066399096279794243*x^7 + 3310935971559099810831262193041169*x^6 + 1816923577284921415815450650228941*x^5 + 1744460954974479586785792913639308*x^4 + 2265584000578191728789884993178815*x^3 + 2812372518753609959493071964143079*x^2 + 2073900817445062675378190181306479*x + 2741492574487110057543512588298830", curve.p),
			8:	Polynomial.parse_poly("8*x^30 + 4451685225093714772084598273546675*x^28 + 1697473675915082120127408103300225*x^27 + 4451685225093714772084598273450579*x^26 + 4141627112286086060624993846295813*x^25 + 3636607100657965372792573721027335*x^24 + 464130930722125164287463664927639*x^23 + 3669165000555824423293476104103818*x^22 + 3516981484705596689659379470261675*x^21 + 844469384650074815059421145063992*x^20 + 955382669624209521074711644702366*x^19 + 3526654862150206090477618434855056*x^18 + 3061453562140697702046278878410541*x^17 + 3440262394299858578853820470036519*x^16 + 903949486842755381665929857639957*x^15 + 3568475161222598654356578637996349*x^14 + 4436605869391856976166047280385156*x^13 + 1642051871972302205124236727559798*x^12 + 2817715439948800254444615851603162*x^11 + 1966004243052164557304466592648727*x^10 + 4091422359341304476451503473145137*x^9 + 1503460424835474381566891686262528*x^8 + 1900235258371851607328147583833423*x^7 + 3402915185697381254748160117889147*x^6 + 1227833561642691423656483562391*x^5 + 4066920457418380485031985044988221*x^4 + 3788365871971905161969093080659792*x^3 + 867975263779872241140238621707318*x^2 + 1078570608239693210846522700720713*x + 2953741928199781644769684606744872", curve.p),
			9:	Polynomial.parse_poly("9*x^40 + 4451685225093714772084598273545295*x^38 + 2008515195416706440034545065215427*x^37 + 4451685225093714772084598273259473*x^36 + 2659809301888010147468243329995776*x^35 + 2504205766817988366790917241661410*x^34 + 2225897345892922907102287693323418*x^33 + 3819923871675967269582355921644944*x^32 + 3648663511200841643895723683258026*x^31 + 4084702527304611840276936136959848*x^30 + 2902489630875459875155468389786471*x^29 + 1235116993749109176152677289917334*x^28 + 420140021598999391924219947952388*x^27 + 1067578477391756238525172617453543*x^26 + 2243372214661388979817836330211619*x^25 + 1323005133744105389849983733447915*x^24 + 69300803065241374634211613289929*x^23 + 4087232836881371076727657467471430*x^22 + 1203003404910065227097132425257065*x^21 + 1063257055807141554126811430671050*x^20 + 3279729956639975345901301305293662*x^19 + 1810738865422606287439485878383229*x^18 + 719215740371081142768604404760877*x^17 + 2446345305392166539114922342542471*x^16 + 39785080648924288255846589168114*x^15 + 2781371407724723442196676703655667*x^14 + 2925005347597516171110864552590545*x^13 + 2394688382702939576941601956851678*x^12 + 3672637263605249557495845069298884*x^11 + 3120791660002436389643680880659259*x^10 + 1899203431588014381209119956122213*x^9 + 132685425727094259177020397650481*x^8 + 3684383558816680907431540795601354*x^7 + 2003467808655005764444879674438008*x^6 + 3676587746401267736492099239898679*x^5 + 2908016376585725042710893276535860*x^4 + 3672565089920940843517534927432248*x^3 + 2049909396814363495582679267380320*x^2 + 1174021582609270535762631939546323*x + 666211811872215342449014645435219", curve.p),
			10:	Polynomial.parse_poly("10*x^48 + 4451685225093714772084598273543195*x^46 + 3173815892831120647350324822821495*x^45 + 4451685225093714772084598272768019*x^44 + 4436086828986386069105975674894096*x^43 + 1354978649474439872061891158633883*x^42 + 2696843654991789411038489074870586*x^41 + 13938419464098804370808399337932*x^40 + 3667083037464978094310456280095073*x^39 + 1344304492400373501985367096019192*x^38 + 3504258735513643606020271362543691*x^37 + 1135058936337963357666029118157155*x^36 + 2765440454712573416184530390470212*x^35 + 3476802764213796947462921328660929*x^34 + 4020110081246684859212503592787421*x^33 + 1474309915217029770261817438088128*x^32 + 2619454296098976194445274588848334*x^31 + 3841987138443430634543001284509890*x^30 + 4354315125575898630267489886965017*x^29 + 799274516409838020750657582793763*x^28 + 3711129205029953850579861900944435*x^27 + 2561879895205508532226105321158755*x^26 + 902827535184642474990258644737528*x^25 + 3574333958284416099752222854361234*x^24 + 4245716201813065782558025237451049*x^23 + 3701318747332403409647709818034816*x^22 + 4259320722210790621577878945150949*x^21 + 3256362693715747872299617753143560*x^20 + 1851322680754309946679467086993558*x^19 + 65437373567394795218683385359437*x^18 + 2388835326466817906074226888872233*x^17 + 2095023183035485223878552241513151*x^16 + 1179295863426829977903472178611399*x^15 + 1694608227949363536062092152108692*x^14 + 326721048355090773505200484827627*x^13 + 314836577017681187658856441091074*x^12 + 2919803767732492825013462796210592*x^11 + 267167835924651806011008232539917*x^10 + 3670678897797665289854162308556021*x^9 + 4170436271685804753266449979076448*x^8 + 3571844293766501152269748033076942*x^7 + 3068952482569200853792058557641349*x^6 + 1021504386136898888032975567967103*x^5 + 958595267040048902390426612413364*x^4 + 2519495879455140768508773631758125*x^3 + 841718775256328949543606036096769*x^2 + 1103741113964767167633763881281609*x + 680202902003304833112564477034966", curve.p),
			11:	Polynomial.parse_poly("11*x^60 + 4451685225093714772084598273540045*x^58 + 198064019740324943747164880631241*x^57 + 4451685225093714772084598271679208*x^56 + 3154347797023076734621104177651468*x^55 + 3595722273686245327339607837940276*x^54 + 4422560917287987269362861376277236*x^53 + 917944121243943950840413727926845*x^52 + 1339290067873199068016390652326228*x^51 + 3342485808094824745953186697705516*x^50 + 3279732949555370760749841590226589*x^49 + 1669457729309284017846317278493463*x^48 + 3537631806176395171498369875224151*x^47 + 4084885518185053885165994473975540*x^46 + 3594796117269917235269239348035943*x^45 + 3743057776975365502185797612814861*x^44 + 4145875707122947361297768183218794*x^43 + 241264030221328505711275920664242*x^42 + 4037657961178471841513950916062898*x^41 + 676101669820477503649724683966846*x^40 + 737430012285684654082581323844183*x^39 + 1402753653077589988632380758213897*x^38 + 2993685088072856380105591073059102*x^37 + 343091427546048219008414750892405*x^36 + 3697987727755505248647037776379373*x^35 + 1109176302016554630487676546244243*x^34 + 1375114225512037115414278231267856*x^33 + 3727238924472220896911837282786503*x^32 + 3071488723944250682880352997251278*x^31 + 3109953785472219591819552163295458*x^30 + 2878798971392901575304646580696941*x^29 + 3104670662467655962925785941025829*x^28 + 190843467629876528947174292380075*x^27 + 2046567479911451420583527450455242*x^26 + 4327879927943396614410874592562859*x^25 + 1748442099144681665139759586840137*x^24 + 3729440683741079495752298790682364*x^23 + 860622396584707448736648949237432*x^22 + 3174376546280983946008457152362236*x^21 + 1371314108885949067301961643484141*x^20 + 937640528780632890836688818842337*x^19 + 3424522574047011897068059061646702*x^18 + 1775437951704120343214827174333943*x^17 + 327182151775961675477893723888538*x^16 + 4064431949983599560532375104893761*x^15 + 2865996732727448836144997475868593*x^14 + 3214378437838080449171784097372584*x^13 + 4204202347971537256136806343251725*x^12 + 4080392160105160901851674259647378*x^11 + 3437119612019602826433512288116653*x^10 + 2042386116434083272714952053086876*x^9 + 910257165176609560790239938968574*x^8 + 1451714050859189999317386675782910*x^7 + 22093237343622646951192253315302*x^6 + 2825972389049613590613957157595539*x^5 + 1385581851701026770835157944536435*x^4 + 3460681253620597964411260646577459*x^3 + 1501699001814114136084917391742794*x^2 + 3866600627229006155802550854769221*x + 4450681194569336241986395646304815", curve.p),
		}

		for degree in sorted(expect_poly.keys()):
			self.assertEqual(divpoly[degree], expect_poly[degree])
