"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const cloudfront = require("@aws-cdk/aws-cloudfront");
const s3 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
let app;
let stack;
beforeEach(() => {
    app = new core_1.App();
    stack = new core_1.Stack(app, 'Stack', {
        env: { account: '1234', region: 'testregion' },
    });
});
describe('With bucket', () => {
    test('renders minimal example', () => {
        const bucket = new s3.Bucket(stack, 'Bucket');
        const origin = new lib_1.S3Origin(bucket);
        const originBindConfig = origin.bind(stack, { originId: 'StackOrigin029E19582' });
        expect(originBindConfig.originProperty).toEqual({
            id: 'StackOrigin029E19582',
            domainName: bucket.bucketRegionalDomainName,
            s3OriginConfig: {
                originAccessIdentity: 'origin-access-identity/cloudfront/${Token[TOKEN.69]}',
            },
        });
    });
    test('can customize properties', () => {
        const bucket = new s3.Bucket(stack, 'Bucket');
        const origin = new lib_1.S3Origin(bucket, { originPath: '/assets' });
        const originBindConfig = origin.bind(stack, { originId: 'StackOrigin029E19582' });
        expect(originBindConfig.originProperty).toEqual({
            id: 'StackOrigin029E19582',
            domainName: bucket.bucketRegionalDomainName,
            originPath: '/assets',
            s3OriginConfig: {
                originAccessIdentity: 'origin-access-identity/cloudfront/${Token[TOKEN.89]}',
            },
        });
    });
    test('creates an OriginAccessIdentity and grants read permissions on the bucket', () => {
        const bucket = new s3.Bucket(stack, 'Bucket');
        const origin = new lib_1.S3Origin(bucket);
        new cloudfront.Distribution(stack, 'Dist', { defaultBehavior: { origin } });
        expect(stack).toHaveResourceLike('AWS::CloudFront::CloudFrontOriginAccessIdentity', {
            CloudFrontOriginAccessIdentityConfig: {
                Comment: 'Allows CloudFront to reach the bucket',
            },
        });
        expect(stack).toHaveResourceLike('AWS::S3::BucketPolicy', {
            PolicyDocument: {
                Statement: [{
                        Principal: {
                            CanonicalUser: { 'Fn::GetAtt': ['DistOrigin1S3Origin87D64058', 'S3CanonicalUserId'] },
                        },
                    }],
            },
        });
    });
});
describe('With website-configured bucket', () => {
    test('renders all required properties, including custom origin config', () => {
        const bucket = new s3.Bucket(stack, 'Bucket', {
            websiteIndexDocument: 'index.html',
        });
        const origin = new lib_1.S3Origin(bucket);
        const originBindConfig = origin.bind(stack, { originId: 'StackOrigin029E19582' });
        expect(originBindConfig.originProperty).toEqual({
            id: 'StackOrigin029E19582',
            domainName: bucket.bucketWebsiteDomainName,
            customOriginConfig: {
                originProtocolPolicy: 'http-only',
            },
        });
    });
    test('can customize properties', () => {
        const bucket = new s3.Bucket(stack, 'Bucket', {
            websiteIndexDocument: 'index.html',
        });
        const origin = new lib_1.S3Origin(bucket, { originPath: '/assets' });
        const originBindConfig = origin.bind(stack, { originId: 'StackOrigin029E19582' });
        expect(originBindConfig.originProperty).toEqual({
            id: 'StackOrigin029E19582',
            domainName: bucket.bucketWebsiteDomainName,
            originPath: '/assets',
            customOriginConfig: {
                originProtocolPolicy: 'http-only',
            },
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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