#############################################################################
# SRWLib for Python Utility module for Crystal Optics
# Author: A. Suvorov, O.C.
# v 0.01
#############################################################################

from math import *
import cmath

strMatDataNotDefined = 'The data for this crystal material is not defined (yet).'

#****************************************************************************
#def SiASF(s):   # calculates Si atomic scattering factor
def srwl_uti_cryst_ASF(_s, _mat='Si'):
    """Calculates Atomic Scattering Factor for some crystal materials
    :param _s: 
    :param _mat: crystal material
    """
    fa = None
    fa0 = 0
    if(_mat == 'Si'):
        fa = [6.2915,2.4386,3.0353,32.3337,1.9891,0.6785,1.541,81.6937,1.1407]
        fa0 = 13.985
    #to extend this for other crystal materials here
    else: raise Exception(strMatDataNotDefined)
        
    s2 = _s*_s ;
    if s2 != 0:
        f0 = fa[0]*exp(-fa[1]*s2) + fa[2]*exp(-fa[3]*s2) + fa[4]*exp(-fa[5]*s2) + fa[6]*exp(-fa[7]*s2) + fa[8]
    else:
        f0 = fa0
    return f0

#****************************************************************************
#def SiPlaneSpace(Hr):
def srwl_uti_cryst_pl_sp(_hr=None, _mat='Si'):
    """Calculates reflecting planes distance
    :param _hr: Miller's indices of reflection
    :param _mat: crystal material
    """

    d = None
    if(_mat == 'Si'): d = 5.43096890 # Si lattice constant (A)
    #to extend this for other crystal materials here
    else: raise Exception(strMatDataNotDefined)

    #return d/norm(Hr)
    if(_hr is None): return d
    else: return d/sqrt(sum(n*n for n in _hr))

#****************************************************************************
#def FourierPolarCompon(En,Hr):
def srwl_uti_cryst_pol_f(_en, _hr, _mat='Si'):
    """Calculates Fourier components of Si polarizability
    :param _en: energy [eV] of reflection
    :param _hr: Miller's indices of reflection
    :param _mat: crystal material
    :return [psi0r, psi0i, psiHr, psiHi]:
        psi0r, psi0i - real and imaginary parts of the 0-th Fourier components
        psiHr, psiHi - real and imaginary parts of the H-th Fourier components
    """

    # Linear interpolation (for anomalous scattering factors)
    # could be improved, taking into account that the table is monotonous
    def interp_f1f2(x, y1, y2, z): 
        for xn in x:
            if xn >= z:  break
        nn = x.index(xn)
        f1 = None; f2 = None
        if nn > 0 & nn < len(x):
            dE = x[nn] - x[nn-1]
            dF1 = y1[nn] - y1[nn-1]
            dF2 = y2[nn] - y2[nn-1]
            f1 = y1[nn-1] + (z-x[nn-1])*dF1/dE
            f2 = y2[nn-1] + (z-x[nn-1])*dF2/dE
        else: raise Exception('Argument is out of range of existing tabulated data')
        return [f1, f2]

    # Vector length
    #def norm(_a): return sqrt(sum(n*n for n in _a))
    
    # Constants
    re = 2.8179402894e-5  # classical radius of electron (A)
    eV2wA = 12398.4193009 # energy to wavelength conversion factor 12398.41930092394

    cDW = None
    rCell = None
    xE = None
    ff1 = None
    ff2 = None
    
    if(_mat == 'Si'):
        cDW = 0.5             # Debye-Waller constant for Si
        rCell = [[0.,0.,0.],[0.5,0.5,0.],[0.5,0.,0.5],[0.,0.5,0.5],[0.25,0.25,0.25],
                 [0.75,0.75,0.25],[0.75,0.25,0.75],[0.25,0.75,0.75]] # coordinates of atoms in Si cell
    
        # Tabulated parameters:
        # Photon Energy [eV] against which material-dependent parameters are tabulated (move to general, and use this mesh for all materials?)
        xE = [1000,1100,1200,1300,1400,1500,1600,1700,1741,1746,1751,1756,1761,1766,1771,1776,1781,1786,
              1791,1796,1800,1801,1806,1811,1816,1821,1826,1831,1836,1841,1846,1851,1856,1861,1866,1871,
              1876,1881,1886,1891,1896,1900,1901,1906,1911,1916,1921,1926,1931,1936,2000,2100,2200,2300,
              2400,2500,2600,2700,2800,2900,3000,3100,3200,3300,3400,3500,3600,3700,3800,3900,4000,4100,
              4200,4300,4400,4500,4600,4700,4800,4900,5000,5100,5200,5300,5400,5500,5600,5700,5800,5900,
              6000,6100,6200,6300,6400,6500,6600,6700,6800,6900,7000,7100,7200,7300,7400,7500,7600,7700,
              7800,7900,8000,8100,8200,8300,8400,8500,8600,8700,8800,8900,9000,9100,9200,9300,9400,9500,
              9600,9700,9800,9900,10000,10100,10200,10300,10400,10500,10600,10700,10800,10900,11000,11100,
              11200,11300,11400,11500,11600,11700,11800,11900,12000,12100,12200,12300,12400,12500,12600,
              12700,12800,12900,13000,13100,13200,13300,13400,13500,13600,13700,13800,13900,14000,14100,
              14200,14300,14400,14500,14600,14700,14800,14900,15000,15100,15200,15300,15400,15500,15600,
              15700,15800,15900,16000,16100,16200,16300,16400,16500,16600,16700,16800,16900,17000,17100,
              17200,17300,17400,17500,17600,17700,17800,17900,18000,18100,18200,18300,18400,18500,18600,
              18700,18800,18900,19000,19100,19200,19300,19400,19500,19600,19700,19800,19900,20000,20100,
              20200,20300,20400,20500,20600,20700,20800,20900,21000,21100,21200,21300,21400,21500,21600,
              21700,21800,21900,22000,22100,22200,22300,22400,22500,22600,22700,22800,22900,23000,23100,
              23200,23300,23400,23500,23600,23700,23800,23900,24000,24100,24200,24300,24400,24500,24600,
              24700,24800,24900,25000,25100,25200,25300,25400,25500,25600,25700,25800,25900,26000,26100,
              26200,26300,26400,26500,26600,26700,26800,26900,27000,27100,27200,27300,27400,27500,27600,
              27700,27800,27900,28000,28100,28200,28300,28400,28500,28600,28700,28800,28900,29000,29100,
              29200,29300,29400,29500,29600,29700,29800,29900,30000,30100,30200,30300,30400,30500,30600,
              30700,30800,30900,31000,31100,31200,31300,31400,31500,31600,31700,31800,31900,32000,32100,
              32200,32300,32400,32500,32600,32700,32800,32900,33000,33100,33200,33300,33400,33500,33600,
              33700,33800,33900,34000,34100,34200,34300,34400,34500,34600,34700,34800,34900,35000,35100,
              35200,35300,35400,35500,35600,35700,35800,35900,36000,36100,36200,36300,36400,36500,36600,
              36700,36800,36900,37000,37100,37200,37300,37400,37500,37600,37700,37800,37900,38000,38100,
              38200,38300,38400,38500,38600,38700,38800,38900,39000,39100,39200,39300,39400,39500,39600,
              39700,39800,39900,40000,40100,40200,40300,40400,40500,40600,40700,40800,40900,41000,41100,
              41200,41300,41400,41500,41600,41700,41800,41900,42000,42100,42200,42300,42400,42500,42600,
              42700,42800,42900,43000,43100,43200,43300,43400,43500,43600,43700,43800,43900,44000,44100,
              44200,44300,44400,44500,44600,44700,44800,44900,45000,45100,45200,45300,45400,45500,45600,
              45700,45800,45900,46000,46100,46200,46300,46400,46500,46600,46700,46800,46900,47000,47100,
              47200,47300,47400,47500,47600,47700,47800,47900,48000,48100,48200,48300,48400,48500,48600,
              48700,48800,48900,49000,49100,49200,49300,49400,49500,49600,49700,49800,49900,50000] 
        ff1 = [-1.1591,-1.2797,-1.4121,-1.5632,-1.7439,-1.9743,-2.2963,-2.8244,-3.1832,-3.2381,-3.2963,
               -3.3582,-3.4243,-3.4952,-3.5716,-3.6542,-3.7442,-3.843,-3.9522,-4.0743,-4.1833,-4.2124,
               -4.3712,-4.5574,-4.7822,-5.0647,-5.4435,-6.0156,-7.1968,-7.5667,-6.094,-5.4422,-5.0152,
               -4.6955,-4.441,-4.2263,-4.0413,-3.8785,-3.7329,-3.6011,-3.4805,-3.3909,-3.3693,-3.266,
               -3.1696,-3.079,-2.9937,-2.9129,-2.8362,-2.7632,-2.0432,-1.3246,-0.83998,-0.49318,-0.41208,
               -0.24945,-0.12251,-0.021709,0.057712,0.12383,0.17836,0.22264,0.25882,0.28838,0.31246,
               0.33201,0.34776,0.35877,0.36854,0.37608,0.38173,0.3858,0.3885,0.39006,0.39064,0.39039,
               0.38943,0.38786,0.38579,0.38328,0.3804,0.37722,0.37367,0.37001,0.36618,0.36219,0.35809,
               0.3539,0.34963,0.34532,0.34096,0.33659,0.3322,0.32781,0.32343,0.31906,0.31472,0.31041,
               0.30614,0.3019,0.2977,0.29355,0.28945,0.28539,0.28139,0.27744,0.27355,0.2697,0.26592,
               0.26219,0.25615,0.25254,0.249,0.24551,0.24208,0.23871,0.23539,0.23213,0.22893,0.22578,
               0.22266,0.21959,0.21658,0.21362,0.21071,0.20786,0.20505,0.20229,0.1995,0.19693,0.19432,
               0.19175,0.18923,0.18675,0.18432,0.18192,0.17957,0.17726,0.17499,0.17276,0.17057,0.16842,
               0.1663,0.16421,0.16217,0.16015,0.15817,0.15623,0.15431,0.15243,0.15058,0.14876,0.14696,
               0.1452,0.14347,0.14176,0.14008,0.13843,0.1368,0.1352,0.13363,0.13208,0.13055,0.12905,
               0.12757,0.12612,0.12468,0.12327,0.12188,0.12051,0.11916,0.11783,0.11652,0.11523,0.11396,
               0.11271,0.11148,0.11026,0.10906,0.10788,0.10672,0.10557,0.10444,0.10332,0.10222,0.10114,
               0.10007,0.099018,0.097978,0.096953,0.095942,0.094945,0.093961,0.092991,0.092034,0.091089,
               0.090157,0.089238,0.08833,0.087435,0.086551,0.085679,0.084818,0.083968,0.083129,0.082301,
               0.081484,0.080677,0.07988,0.079093,0.078316,0.077548,0.076791,0.076042,0.075303,0.074573,
               0.073851,0.073139,0.072435,0.07174,0.071053,0.070374,0.069703,0.06904,0.068386,0.067738,
               0.067099,0.066466,0.065842,0.065224,0.064613,0.06401,0.063413,0.062823,0.06224,0.061664,
               0.061093,0.06053,0.059972,0.059421,0.058876,0.058337,0.057803,0.057276,0.056754,0.056238,
               0.055728,0.055223,0.054723,0.054229,0.05374,0.053256,0.052777,0.052304,0.051835,0.051371,
               0.050912,0.050458,0.050008,0.049563,0.049123,0.048687,0.048255,0.047828,0.047405,0.046986,
               0.046572,0.046161,0.045755,0.045353,0.044954,0.04456,0.044169,0.043782,0.043399,0.04302,
               0.042644,0.042272,0.041903,0.041538,0.041185,0.040828,0.040475,0.040125,0.039778,0.039435,
               0.039095,0.038758,0.038424,0.038094,0.037766,0.037442,0.037121,0.036803,0.036488,0.036176,
               0.035866,0.03556,0.035256,0.034956,0.034658,0.034363,0.03407,0.033781,0.033494,0.033209,
               0.032927,0.032648,0.032371,0.032097,0.031826,0.031557,0.03129,0.031025,0.030764,0.030504,
               0.030247,0.029992,0.029739,0.029489,0.029241,0.028995,0.028751,0.02851,0.028271,0.028033,
               0.027798,0.027565,0.027334,0.027105,0.026878,0.026654,0.026431,0.02621,0.025991,0.025774,
               0.025558,0.025345,0.025134,0.024924,0.024716,0.024511,0.024306,0.024104,0.023903,0.023705,
               0.023507,0.023312,0.023118,0.022926,0.022736,0.022547,0.02236,0.022174,0.021991,0.021808,
               0.021627,0.021448,0.021271,0.021094,0.02092,0.020747,0.020575,0.020405,0.020236,0.020069,
               0.019903,0.019738,0.019575,0.019413,0.019253,0.019094,0.018937,0.01878,0.018625,0.018472,
               0.018319,0.018168,0.018019,0.01787,0.017723,0.017577,0.017432,0.017288,0.017146,0.017005,
               0.016865,0.016726,0.016589,0.016452,0.016317,0.016183,0.01605,0.015918,0.015787,0.015657,
               0.015528,0.015401,0.015274,0.015149,0.015024,0.014901,0.014779,0.014657,0.014537,0.014418,
               0.014299,0.014182,0.014066,0.01395,0.013836,0.013723,0.01361,0.013498,0.013388,0.013278,
               0.013169,0.013061,0.012954,0.012848,0.012743,0.012638,0.012535,0.012432,0.01233,0.012229,
               0.012129,0.01203,0.011932,0.011834,0.011737,0.011641,0.011546,0.011451,0.011357,0.011265,
               0.011172,0.011081,0.01099,0.0109,0.010811,0.010723,0.010635,0.010548,0.010462,0.010376,
               0.010292,0.010207,0.010124,0.010041,0.0099592,0.0098778,0.0097971,0.009717,0.0096376,
               0.0095589,0.0094808,0.0094033,0.0093265,0.0092503,0.0091747,0.0090998,0.0090254,0.0089517,
               0.0088786,0.0088061,0.0087342,0.0086628,0.0085921,0.0085219,0.0084524,0.0083833,0.0083149,
               0.008247,0.0081797,0.0081129,0.0080467,0.7981,0.0079158,0.0078512,0.0077871,0.0077236,
               0.0076606,0.007598,0.007536,0.0074745,0.0074135,0.007353,0.007293,0.0072335,0.0071745,
               0.007116,0.0070579,0.0070003,0.0069432,0.0068866,0.0068304,0.0067747,0.0067194,0.0066646,
               0.0066102,0.0065563,0.0065028,0.0064498,0.0063972,0.006345,0.0062932,0.0062419,0.006191,
               0.0061405,0.0060904,0.0060408,0.0059915,0.0059426,0.0058942,0.0058461,0.0057984,0.0057512,
               0.0057043,0.0056577,0.0056116,0.0055659,0.0055205,0.0054755,0.0054308,0.0053865,0.0053426,
               0.0052991,0.0052559,0.005213,0.0051705]
        ff2 = [0.97954,0.83919,0.72758,0.63724,0.56302,0.50123,0.44921,0.40497,0.38872,0.38681,0.38491,
               0.38302,0.38115,0.37929,0.37745,0.37562,0.3738,0.372,0.37021,0.36843,0.36701,0.36666,0.36491,
               0.36317,0.36144,0.35973,0.35803,0.35634,0.35466,4.1077,4.0995,4.0912,4.0827,4.0741,4.0654,
               4.0566,4.0476,4.0386,4.0294,4.0201,4.0107,4.0031,4.0012,3.9916,3.9819,3.972,3.9621,3.9521,
               3.942,3.9318,3.7934,3.5549,3.2999,3.0383,2.7847,2.6079,2.4484,2.3038,2.1723,2.0524,1.9427,
               1.84,1.7452,1.6573,1.5759,1.5002,1.4297,1.364,1.3029,1.2458,1.1923,1.1421,1.095,1.0507,
               1.009,0.96971,0.93263,0.8976,0.86448,0.83312,0.8034,0.77522,0.74846,0.72304,0.69886,
               0.67584,0.65392,0.63302,0.61308,0.59405,0.57586,0.55848,0.54184,0.52592,0.51067,0.49606,
               0.48204,0.46859,0.45568,0.44328,0.43136,0.4199,0.40888,0.39827,0.38805,0.37821,0.36872,
               0.35957,0.35075,0.34223,0.33404,0.32619,0.31861,0.31128,0.30419,0.29734,0.29071,0.28429,
               0.27808,0.27206,0.26625,0.26061,0.25515,0.24986,0.24473,0.23975,0.23491,0.23022,0.22567,
               0.22124,0.21694,0.21277,0.20871,0.20476,0.20092,0.19719,0.19355,0.19002,0.18657,0.18322,
               0.17995,0.17677,0.17367,0.17065,0.16771,0.16484,0.16204,0.15931,0.15664,0.15404,0.15151,
               0.14903,0.14661,0.14425,0.14195,0.13969,0.13749,0.13534,0.13324,0.13118,0.12917,0.12721,
               0.12529,0.12341,0.12157,0.11977,0.11801,0.11629,0.1146,0.11295,0.11133,0.10975,0.10819,
               0.10667,0.10519,0.10373,0.1023,0.1009,0.099522,0.098175,0.096854,0.095559,0.09429,0.093044,
               0.091822,0.090623,0.089447,0.088293,0.08716,0.086048,0.084957,0.083885,0.082833,0.0818,
               0.080786,0.079789,0.078811,0.077849,0.076905,0.075976,0.075064,0.074168,0.073287,0.072422,
               0.07157,0.070734,0.069911,0.069102,0.068307,0.067524,0.066755,0.065998,0.065253,0.064521,
               0.0638,0.063091,0.062393,0.061707,0.061031,0.060366,0.059711,0.059066,0.058431,0.057807,
               0.057191,0.056585,0.055989,0.055401,0.054822,0.054252,0.05369,0.053137,0.052592,0.052055,
               0.051525,0.051004,0.05049,0.049983,0.049484,0.048992,0.048507,0.048028,0.047557,0.047092,
               0.046634,0.046182,0.045736,0.045296,0.044863,0.044435,0.044013,0.043597,0.043187,0.042782,
               0.042383,0.041989,0.0416,0.041216,0.040837,0.040464,0.040095,0.039731,0.039371,0.039017,
               0.038667,0.038321,0.03798,0.037643,0.037311,0.036982,0.036658,0.036338,0.036022,0.035709,
               0.035401,0.035096,0.034796,0.034498,0.034205,0.033915,0.033602,0.033316,0.033033,0.032754,
               0.032477,0.032204,0.031933,0.031666,0.031402,0.03114,0.030882,0.030626,0.030373,0.030123,
               0.029875,0.02963,0.029388,0.029148,0.028911,0.028676,0.028444,0.028214,0.027987,0.027762,
               0.027539,0.027318,0.0271,0.026884,0.02667,0.026458,0.026248,0.02604,0.025835,0.025631,
               0.02543,0.02523,0.025033,0.024837,0.024643,0.024451,0.024261,0.024072,0.023886,0.023701,
               0.023518,0.023337,0.023157,0.022979,0.022803,0.022628,0.022455,0.022283,0.022113,0.021945,
               0.021778,0.021613,0.021449,0.021286,0.021125,0.020966,0.020808,0.020651,0.020495,0.020341,
               0.020189,0.020037,0.019887,0.019739,0.019591,0.019445,0.0193,0.019157,0.019014,0.018873,
               0.018733,0.018594,0.018457,0.01832,0.018185,0.01805,0.017917,0.017785,0.017654,0.017525,
               0.017396,0.017268,0.017142,0.017016,0.016891,0.016768,0.016645,0.016524,0.016403,0.016284,
               0.016165,0.016047,0.015931,0.015815,0.0157,0.015586,0.015473,0.015361,0.01525,0.015139,
               0.01503,0.014921,0.014813,0.014706,0.0146,0.014495,0.01439,0.014287,0.014184,0.014082,
               0.013981,0.01388,0.01378,0.013682,0.013583,0.013486,0.013389,0.013293,0.013198,0.013104,
               0.01301,0.012917,0.012825,0.012733,0.012642,0.012552,0.012462,0.012373,0.012285,0.012198,
               0.012111,0.012025,0.011939,0.011854,0.01177,0.011686,0.011603,0.011521,0.011439,0.011358,
               0.011277,0.011197,0.011118,0.011039,0.01096,0.010883,0.010806,0.010729,0.010653,0.010578,
               0.010503,0.010429,0.010355,0.010282,0.010209,0.010137,0.010065,0.0099943,0.0099238,
               0.0098537,0.0097842,0.0097151,0.0096466,0.0095786,0.0095111,0.009444,0.0093775,0.0093114,
               0.0092458,0.0091807,0.009116,0.0090518,0.0089881,0.0089249,0.0088621,0.0087997,0.0087378,
               0.0086764,0.0086154,0.0085548,0.0084947,0.008435,0.0083757,0.0083169,0.0082585,0.0082005,
               0.0081429,0.0080857,0.008029,0.0079726,0.0079167,0.0078611,0.007806,0.0077512,0.0076968,
               0.0076429,0.0075893,0.0075361,0.0074832,0.0074308,0.0073787,0.007327,0.0072756,0.0072247,
               0.007174,0.0071238,0.0070739,0.0070244,0.0069752,0.0069263,0.0068778,0.0068297,0.0067819,
               0.0067344,0.0066873,0.0066405,0.006594,0.0065479,0.0065021,0.0064566,0.0064114,0.0063666,
               0.0063221,0.0062779,0.006234,0.0061904,0.0061471,0.0061041,0.0060614,0.0060191,0.005977,
               0.0059352,0.0058938,0.0058526,0.0058117,0.0057711,0.0057307,0.0056907,0.005651,0.0056115,
               0.0055723,0.0055334,0.0054947,0.0054563,0.0054182,0.0053804,0.0053428]
    #to extend this for other crystal materials here
    else: raise Exception(strMatDataNotDefined)
    
    # Calculated parameters
    dLat = srwl_uti_cryst_pl_sp(None, _mat) # lattice constant (A)
    Vc = dLat*dLat*dLat                     # crystal cell volume
    
    #dr = dLat/norm(_hr)
    dr = srwl_uti_cryst_pl_sp(_hr, _mat)
    
    wA = eV2wA/_en
    #fE = interp(xE,ff1,ff2,En)
    fE = interp_f1f2(xE, ff1, ff2, _en)
    xh = 0.5/dr
    #f0 = SiASF(0.)
    f0 = srwl_uti_cryst_ASF(0., _mat)
    
    cF = wA*wA*re/(pi*Vc)
    psi0r = -8.*cF*(f0+fE[0])
    psi0i = 8.*cF*fE[1]
    #f0 = SiASF(xh)
    f0 = srwl_uti_cryst_ASF(xh, _mat)
    
    #fDW = math.exp(-cDW*xh*xh)
    fDW = exp(-cDW*xh*xh)
    
    cF = cF*fDW
    eF = complex(0., 0.)
    for x in rCell:
        eF = eF + cmath.exp(2j*pi*(x[0]*_hr[0]+x[1]*_hr[1]+x[2]*_hr[2]))
        aF = abs(eF)
    
    psiHr = -cF*aF*(f0+fE[0])
    psiHi = cF*aF*fE[1]
    return [psi0r, psi0i, psiHr, psiHi]
