# auto generated by update_py.py

import threading

from collections import defaultdict
from google.protobuf.json_format import MessageToDict, ParseDict

from ..pb_msg import message_pb
from ..pb_msg import Direction, OffsetFlag, OrderStatus, OrderType, SecurityType, Side


class OMSHelper:

    def __init__(self):
        self._cached_order = defaultdict(message_pb.RtnOrder)
        self._cached_position = defaultdict(message_pb.GatewayPosition)
        self._cached_capital = defaultdict(message_pb.RspAccount)

        self.order_lock = threading.Lock()
        self.position_lock = threading.Lock()
        self.capital_lock = threading.Lock()

    def to_direction(self, offset_flag: OffsetFlag, side: Side):
        # open buy, close sell -> long
        # open sell, close buy -> short
        direction = Direction.DIRECTION_UNSPECIFIED
        if offset_flag == OffsetFlag.OFFSET_FLAG_OPEN:
            if side == Side.SIDE_BUY:
                direction = Direction.DIRECTION_LONG
            elif side == Side.SIDE_SELL:
                direction = Direction.DIRECTION_SHORT
        elif offset_flag in (OffsetFlag.OFFSET_FLAG_CLOSE, OffsetFlag.OFFSET_FLAG_FORCE_CLOSE, OffsetFlag.OFFSET_FLAG_CLOSE_TODAY, OffsetFlag.OFFSET_FLAG_CLOSE_YESTERDAY):
            if side == Side.SIDE_BUY:
                direction = Direction.DIRECTION_SHORT
            elif side == Side.SIDE_SELL:
                direction = Direction.DIRECTION_LONG
        return direction

    ################
    # update stock #
    ################

    def _freeze_stock_position(self, account_id, sub_account, symbol, frz_pos, revert=False):
        with self.position_lock:
            pos = self._cached_position.get((account_id, sub_account))
            if pos is None:
                return

            for p in pos.positions:
                if p.symbol == symbol:
                    frz_pos = -frz_pos if revert else frz_pos
                    p.avail_pos -= frz_pos
                    p.frz_pos += frz_pos
                    return

    def _update_stock_position_from_trade(self, obj: message_pb.RtnTrade):
        with self.position_lock:
            pos = self._cached_position.get((obj.account_id, obj.sub_account))
            if pos is None:
                return

            # TODO: market_price 当前价格, from market
            market_price = obj.traded_price

            for p in pos.positions:
                if p.symbol == obj.symbol:
                    if obj.side == Side.SIDE_BUY:
                        p.pos_cost = (p.position * p.pos_cost + obj.traded_vol * obj.traded_price) / (p.position + obj.traded_vol)
                        p.position += obj.traded_vol
                    else:
                        p.position -= obj.traded_vol
                        p.frz_pos -= obj.traded_vol
                        p.realized_pnl += (obj.traded_price - p.pos_cost) * obj.traded_vol

                    p.mkt_value = market_price * p.position
                    p.unrealized_pnl = (market_price - p.pos_cost) * p.position
                    p.pnl = p.realized_pnl + p.unrealized_pnl
                    return

            if obj.side == Side.SIDE_BUY:
                p = ParseDict(MessageToDict(obj), message_pb.PositionInfo(), ignore_unknown_fields=True)
                p.direction = Direction.DIRECTION_NET
                p.position = obj.traded_vol
                p.pos_cost = obj.traded_price
                p.mkt_value = market_price * p.position
                p.unrealized_pnl = (market_price - p.pos_cost) * p.position
                p.pnl = p.realized_pnl + p.unrealized_pnl

                pos.positions.append(p)

    def _freeze_stock_capital(self, account_id, sub_account, frz_amt, revert=False):
        with self.capital_lock:
            cap = self._cached_capital.get((account_id, sub_account))
            if cap is None:
                return

            frz_amt = -frz_amt if revert else frz_amt
            cap.avail_amt -= frz_amt
            cap.balance -= frz_amt
            cap.frz_amt += frz_amt

    def _update_stock_capital_from_trade(self, obj: message_pb.RtnTrade):
        with self.capital_lock:
            cap = self._cached_capital.get((obj.account_id, obj.sub_account))
            if cap is None:
                return

            traded_amt = obj.traded_vol * obj.traded_price
            cap.avail_amt -= obj.commission
            cap.balance -= obj.commission

            if obj.side == Side.SIDE_SELL:
                cap.avail_amt += traded_amt

            elif obj.side == Side.SIDE_BUY:
                if obj.order_type == OrderType.ORDER_TYPE_LIMIT:
                    cap.frz_amt -= traded_amt
                elif obj.order_type == OrderType.ORDER_TYPE_MARKET:
                    cap.avail_amt -= traded_amt
                    cap.balance -= traded_amt

            # update from position
            pos = self.get_position_info(obj.account_id, obj.sub_account)
            if pos is None:
                return

            cap.mkt_value, cap.realized_pnl, cap.unrealized_pnl, _ = self._get_capital_from_position(pos)
            cap.pnl = cap.realized_pnl + cap.unrealized_pnl
            cap.equity = cap.avail_amt + cap.mkt_value + cap.frz_amt

    #################
    # update future #
    #################

    def _freeze_future_position(self, account_id, sub_account, symbol, frz_pos, direction, revert=False):
        with self.position_lock:
            pos = self._cached_position.get((account_id, sub_account))
            if pos is None:
                return

            for p in pos.positions:
                if p.symbol == symbol and p.direction == direction:
                    frz_pos = -frz_pos if revert else frz_pos
                    p.avail_pos -= frz_pos
                    p.frz_pos += frz_pos
                    return

    def _update_future_position_from_trade(self, obj: message_pb.RtnTrade):
        with self.position_lock:
            pos = self._cached_position.get((obj.account_id, obj.sub_account))
            if pos is None:
                return

            direction = self.to_direction(obj.offset_flag, obj.side)
            # TODO: market_price 当前价格, from market
            market_price = obj.traded_price
            # TODO: contract_constant 合约常数
            contract_constant = 1
            # TODO: margin_rate 保证金率
            margin_rate = 1

            for p in pos.positions:
                if p.symbol == obj.symbol and p.direction == direction:
                    if obj.offset_flag == OffsetFlag.OFFSET_FLAG_OPEN:
                        p.pos_cost = (p.position * p.pos_cost + obj.traded_vol * obj.traded_price) / (p.position + obj.traded_vol)
                        p.position += obj.traded_vol
                    else:
                        realized_pnl = (obj.traded_price - p.pos_cost) * obj.traded_vol

                        p.position -= obj.traded_vol
                        p.frz_pos -= obj.traded_vol
                        p.realized_pnl += realized_pnl if direction == Direction.DIRECTION_LONG else -realized_pnl

                    unrealized_pnl = (market_price - p.pos_cost) * p.position

                    p.mkt_value = market_price * p.position * contract_constant
                    p.margin = p.position * p.pos_cost * contract_constant * margin_rate
                    p.unrealized_pnl = unrealized_pnl if direction == Direction.DIRECTION_LONG else -unrealized_pnl
                    p.pnl = p.realized_pnl + p.unrealized_pnl
                    return

            if obj.offset_flag == OffsetFlag.OFFSET_FLAG_OPEN:
                unrealized_pnl = (market_price - obj.traded_price) * obj.traded_vol

                p = ParseDict(MessageToDict(obj), message_pb.PositionInfo(), ignore_unknown_fields=True)
                p.direction = direction
                p.position = obj.traded_vol
                p.pos_cost = obj.traded_price
                p.mkt_value = market_price * p.position * contract_constant
                p.margin = p.position * p.pos_cost * contract_constant * margin_rate
                p.unrealized_pnl = unrealized_pnl if direction == Direction.DIRECTION_LONG else -unrealized_pnl
                p.pnl = p.realized_pnl + p.unrealized_pnl

                pos.positions.append(p)

    def _freeze_future_capital(self, account_id, sub_account, frz_amt, revert=False):
        with self.capital_lock:
            cap = self._cached_capital.get((account_id, sub_account))
            if cap is None:
                return

            # TODO: contract_constant 合约常数
            contract_constant = 1
            # TODO: margin_rate 保证金率
            margin_rate = 1

            frz_amt *= contract_constant * margin_rate
            frz_amt = -frz_amt if revert else frz_amt
            cap.avail_amt -= frz_amt
            cap.frz_amt += frz_amt
            cap.balance = cap.avail_amt + cap.frz_amt

    def _update_future_capital_from_trade(self, obj: message_pb.RtnTrade):
        with self.capital_lock:
            cap = self._cached_capital.get((obj.account_id, obj.sub_account))
            if cap is None:
                return

            # TODO: contract_constant 合约常数
            contract_constant = 1
            # TODO: margin_rate 保证金率
            margin_rate = 1

            traded_amt = obj.traded_vol * obj.traded_price * contract_constant * margin_rate

            if obj.offset_flag == OffsetFlag.OFFSET_FLAG_OPEN:
                if obj.order_type == OrderType.ORDER_TYPE_LIMIT:
                    cap.frz_amt -= traded_amt
                elif obj.order_type == OrderType.ORDER_TYPE_MARKET:
                    cap.avail_amt -= traded_amt
            else:
                cap.avail_amt += traded_amt

            cap.avail_amt -= obj.commission
            cap.balance = cap.avail_amt + cap.frz_amt

            # update from position
            pos = self.get_position_info(obj.account_id, obj.sub_account)
            if pos is None:
                return

            cap.mkt_value, cap.realized_pnl, cap.unrealized_pnl, cap.frz_margin = self._get_capital_from_position(pos)
            cap.pnl = cap.realized_pnl + cap.unrealized_pnl
            cap.equity = cap.balance + cap.frz_margin + cap.unrealized_pnl

    ###########################
    # update position/capital #
    ###########################

    def _get_capital_from_position(self, obj: message_pb.GatewayPosition):
        mkt_value = realized_pnl = unrealized_pnl = frz_margin = 0

        for p in obj.positions:
            mkt_value += p.mkt_value
            realized_pnl += p.realized_pnl
            unrealized_pnl += p.unrealized_pnl
            frz_margin += p.margin

        return mkt_value, realized_pnl, unrealized_pnl, frz_margin

    def _freeze_position_or_capital(self, obj, revert=False):
        if hasattr(obj, 'traded_vol'):
            vol = obj.entrust_vol - obj.traded_vol
        else:
            vol = obj.entrust_vol

        if obj.security_type == SecurityType.SECURITY_TYPE_STOCK:
            if obj.side == Side.SIDE_SELL:
                self._freeze_stock_position(obj.account_id, obj.sub_account, obj.symbol, vol, revert=revert)
            elif obj.side == Side.SIDE_BUY and obj.order_type == OrderType.ORDER_TYPE_LIMIT:
                self._freeze_stock_capital(obj.account_id, obj.sub_account, obj.entrust_price * vol, revert=revert)

        elif obj.security_type == SecurityType.SECURITY_TYPE_FUTURE:
            if obj.offset_flag in (OffsetFlag.OFFSET_FLAG_CLOSE, OffsetFlag.OFFSET_FLAG_FORCE_CLOSE, OffsetFlag.OFFSET_FLAG_CLOSE_TODAY, OffsetFlag.OFFSET_FLAG_CLOSE_YESTERDAY):
                self._freeze_future_position(obj.account_id, obj.sub_account, obj.symbol, vol, self.to_direction(obj.offset_flag, obj.side), revert=revert)
            elif obj.offset_flag == OffsetFlag.OFFSET_FLAG_OPEN and obj.order_type == OrderType.ORDER_TYPE_LIMIT:
                self._freeze_future_capital(obj.account_id, obj.sub_account, obj.entrust_price * vol, revert=revert)

    ##############
    # on message #
    ##############

    def on_req_order_insert(self, obj: message_pb.ReqOrderInsert):
        self._freeze_position_or_capital(obj)

        with self.order_lock:
            return ParseDict(MessageToDict(obj), self._cached_order[obj.order_id])

    def on_rsp_order_insert(self, obj: message_pb.RspOrderInsert):
        with self.order_lock:
            order = self._cached_order[obj.order_id]

            if obj.err_id != 0:
                self._freeze_position_or_capital(order, revert=True)

            return ParseDict(MessageToDict(obj), order)

    def on_req_order_cancel(self, obj: message_pb.ReqOrderCancel):
        pass

    def on_rsp_order_cancel(self, obj: message_pb.RspOrderCancel):
        with self.order_lock:
            order = self._cached_order[obj.order_id]
            order.cancel_err_id = obj.err_id
            order.cancel_err_msg = obj.err_msg
            return order

    def on_rtn_order(self, obj: message_pb.RtnOrder):
        with self.order_lock:
            if obj.order_status in (OrderStatus.ORDER_STATUS_CANCELED, OrderStatus.ORDER_STATUS_REJECTED):
                self._freeze_position_or_capital(obj, revert=True)

            self._cached_order[obj.order_id].MergeFrom(obj)
            return self._cached_order[obj.order_id]

    def on_rtn_trade(self, obj: message_pb.RtnTrade):
        if obj.security_type == SecurityType.SECURITY_TYPE_STOCK:
            self._update_stock_position_from_trade(obj)
            self._update_stock_capital_from_trade(obj)

        elif obj.security_type == SecurityType.SECURITY_TYPE_FUTURE:
            self._update_future_position_from_trade(obj)
            self._update_future_capital_from_trade(obj)

    def on_rsp_position(self, obj: message_pb.GatewayPosition):
        with self.position_lock:
            self._cached_position[(obj.account_id, obj.sub_account)] = obj

    def on_rsp_account(self, obj: message_pb.RspAccount):
        with self.capital_lock:
            self._cached_capital[(obj.account_id, obj.sub_account)] = obj

    #####################
    # init data from db #
    #####################

    def init_cached_order(self, init_order):
        with self.order_lock:
            self._cached_order.update({o.order_id: o for o in init_order})

    def init_cached_position(self, init_position):
        with self.position_lock:
            self._cached_position.update({(p.account_id, p.sub_account): p for p in init_position})

    def init_cached_capital(self, init_capital):
        with self.capital_lock:
            self._cached_capital.update({(c.account_id, c.sub_account): c for c in init_capital})

    #####################
    # get data from oms #
    #####################

    def get_order_info(self, order_id):
        with self.order_lock:
            return self._cached_order.get(order_id)

    def get_position_info(self, account_id, sub_account=''):
        with self.position_lock:
            return self._cached_position.get((account_id, sub_account))

    def get_capital_info(self, account_id, sub_account=''):
        with self.capital_lock:
            return self._cached_capital.get((account_id, sub_account))

    def get_init_orders(self, account_id, sub_account=''):
        with self.order_lock:
            orders = [o for o in self._cached_order.values() if o.account_id == account_id and o.sub_account == sub_account]
            if not orders:
                return None

            obj = message_pb.GatewayOrders()
            obj.account_id = account_id
            obj.sub_account = sub_account
            obj.orders.extend(orders)
            return obj
