"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
exports.AUTOGEN_MARKER = '$$autogen$$';
/**
 * Renders an ARN for an SSM parameter given a parameter name.
 * @param scope definition scope
 * @param parameterName the parameter name to include in the ARN
 * @param physicalName optional physical name specified by the user (to auto-detect separator)
 */
function arnForParameterName(scope, parameterName, options = {}) {
    const physicalName = options.physicalName;
    const nameToValidate = physicalName || parameterName;
    if (!core_1.Token.isUnresolved(nameToValidate) && nameToValidate.includes('/') && !nameToValidate.startsWith('/')) {
        throw new Error(`Parameter names must be fully qualified (if they include "/" they must also begin with a "/"): ${nameToValidate}`);
    }
    return core_1.Stack.of(scope).formatArn({
        service: 'ssm',
        resource: 'parameter',
        sep: isSimpleName() ? '/' : '',
        resourceName: parameterName,
    });
    /**
     * Determines the ARN separator for this parameter: if we have a concrete
     * parameter name (or explicitly defined physical name), we will parse them
     * and decide whether a "/" is needed or not. Otherwise, users will have to
     * explicitly specify `simpleName` when they import the ARN.
     */
    function isSimpleName() {
        // look for a concrete name as a hint for determining the separator
        const concreteName = !core_1.Token.isUnresolved(parameterName) ? parameterName : physicalName;
        if (!concreteName || core_1.Token.isUnresolved(concreteName)) {
            if (options.simpleName === undefined) {
                throw new Error(`Unable to determine ARN separator for SSM parameter since the parameter name is an unresolved token. Use "fromAttributes" and specify "simpleName" explicitly`);
            }
            return options.simpleName;
        }
        const result = !concreteName.startsWith('/');
        // if users explicitly specify the separator and it conflicts with the one we need, it's an error.
        if (options.simpleName !== undefined && options.simpleName !== result) {
            if (concreteName === exports.AUTOGEN_MARKER) {
                throw new Error(`If "parameterName" is not explicitly defined, "simpleName" must be "true" or undefined since auto-generated parameter names always have simple names`);
            }
            throw new Error(`Parameter name "${concreteName}" is ${result ? 'a simple name' : 'not a simple name'}, but "simpleName" was explicitly set to ${options.simpleName}. Either omit it or set it to ${result}`);
        }
        return result;
    }
}
exports.arnForParameterName = arnForParameterName;
//# sourceMappingURL=data:application/json;base64,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