# PyPI Deployment Guide

Complete guide for deploying `parallel-llm` to PyPI.

## Prerequisites

### 1. PyPI Accounts

Create accounts if you haven't already:
- **Test PyPI**: https://test.pypi.org/account/register/
- **Production PyPI**: https://pypi.org/account/register/

### 2. API Tokens

Generate API tokens for authentication:

1. Go to your account settings on Test PyPI and PyPI
2. Navigate to "API tokens" section
3. Click "Add API token"
4. Name it (e.g., "parallel-llm-deployment")
5. Set scope to "Entire account" or specific to this project
6. **Save the token immediately** - it won't be shown again

### 3. Install Required Tools

```powershell
# Install build and upload tools
pip install --upgrade build twine
```

## Phase 1: Local Testing

### Step 1: Run Package Tests

```powershell
cd "c:\Users\furka\Desktop\NITSRI PHD\ParallelLLM"

# Install in editable mode for testing
pip install -e .

# Run test suite
python -m pytest tests/test_package.py -v
```

Expected output: All tests should pass ✓

### Step 2: Test Local Installation

```powershell
# Run the local installation test script
python scripts/test_local_install.py
```

This will:
- Create a temporary virtual environment
- Install the package in editable mode
- Test imports and basic functionality
- Clean up automatically

## Phase 2: Build Distributions

### Step 3: Build Package

```powershell
# Clean old builds (if any)
if (Test-Path dist) { Remove-Item -Recurse -Force dist }
if (Test-Path build) { Remove-Item -Recurse -Force build }
if (Test-Path parallel_llm.egg-info) { Remove-Item -Recurse -Force parallel_llm.egg-info }

# Build source distribution and wheel
python -m build
```

This creates:
- `dist/parallel-llm-0.1.1.tar.gz` (source distribution)
- `dist/parallel_llm-0.1.1-py3-none-any.whl` (wheel)

### Step 4: Verify Build

```powershell
# Run comprehensive build verification
python scripts/build_and_test.py
```

This will:
- Clean old artifacts
- Build fresh distributions
- Verify package contents
- Test wheel installation
- Report any issues

## Phase 3: Test PyPI Deployment

### Step 5: Upload to Test PyPI

```powershell
# Upload to Test PyPI
python -m twine upload --repository testpypi dist/*
```

When prompted:
- **Username**: `__token__`
- **Password**: Your Test PyPI API token (starts with `pypi-`)

### Step 6: Test Installation from Test PyPI

```powershell
# Create a new test environment
python -m venv test_testpypi
test_testpypi\Scripts\activate

# Install from Test PyPI
pip install --index-url https://test.pypi.org/simple/ --extra-index-url https://pypi.org/simple/ parallel-llm

# Test imports
python -c "import parallel_llm; print(f'Version: {parallel_llm.__version__}')"

# Test CLI
python -m parallel_llm.cli train --help

# Clean up
deactivate
Remove-Item -Recurse -Force test_testpypi
```

### Step 7: Verify Test PyPI Package Page

Visit: https://test.pypi.org/project/parallel-llm/

Check:
- [ ] Package description renders correctly
- [ ] Version number is correct (0.1.1)
- [ ] All metadata is accurate
- [ ] Links work (GitHub, documentation, etc.)
- [ ] Dependencies are listed correctly

## Phase 4: Production PyPI Deployment

⚠️ **Warning**: Once uploaded to PyPI, you **cannot** delete or re-upload the same version!

### Step 8: Final Pre-Deployment Checklist

- [ ] All tests pass on Test PyPI
- [ ] Package installs correctly from Test PyPI
- [ ] README renders correctly on Test PyPI
- [ ] Version 0.1.1 doesn't already exist on PyPI
- [ ] Git repository is committed and clean
- [ ] LICENSE file exists
- [ ] All documentation is up to date

### Step 9: Upload to Production PyPI

```powershell
# Upload to production PyPI
python -m twine upload dist/*
```

When prompted:
- **Username**: `__token__`
- **Password**: Your Production PyPI API token

### Step 10: Verify Production Deployment

```powershell
# Create fresh test environment
python -m venv test_pypi
test_pypi\Scripts\activate

# Install from PyPI (wait 1-2 minutes after upload for propagation)
pip install parallel-llm

# Verify installation
python -c "import parallel_llm; print(f'✓ Version: {parallel_llm.__version__}')"
python -c "from parallel_llm import ModelConfig, DiffusionTransformer, ParallelGenerator; print('✓ Core imports successful')"

# Test CLI
parallel-llm-train --help
parallel-llm-infer --help

# Clean up
deactivate
Remove-Item -Recurse -Force test_pypi
```

### Step 11: Check PyPI Package Page

Visit: https://pypi.org/project/parallel-llm/

Verify:
- Package page looks correct
- Installation instructions work
- All metadata is accurate

## Phase 5: Post-Deployment

### Step 12: Tag Release in Git

```powershell
# Tag the release
git tag v0.1.1
git push origin v0.1.1

# Optional: Create annotated tag with message
git tag -a v0.1.1 -m "Release version 0.1.1"
git push origin v0.1.1
```

### Step 13: Update Documentation

Update README.md badge (if desired):

```markdown
[![PyPI version](https://badge.fury.io/py/parallel-llm.svg)](https://badge.fury.io/py/parallel-llm)
```

### Step 14: Monitor

- Check download statistics on PyPI
- Monitor for any user issues or bug reports
- Watch for dependency compatibility issues

## Troubleshooting

### Issue: "Username/Password incorrect"

**Solution**: Make sure you're using `__token__` as username (with two underscores) and your API token as password.

### Issue: "File already exists"

**Solution**: You're trying to upload a version that already exists. You need to bump the version number in `pyproject.toml`.

### Issue: "Invalid distribution"

**Solution**: Run `python -m twine check dist/*` to validate your distributions before uploading.

### Issue: Package won't import after installation

**Solution**: 
1. Check that all dependencies are properly installed
2. Verify the package structure in the wheel
3. Ensure `__init__.py` correctly imports all modules

## Next Steps for Future Releases

When releasing version 0.1.2 or later:

1. Update version in `pyproject.toml`
2. Update version in `parallel_llm/__init__.py`
3. Update CHANGELOG (if exists)
4. Run all tests
5. Follow this deployment guide from Phase 1

## Configuration File (.pypirc)

For easier uploads, create `C:\Users\furka\.pypirc`:

```ini
[distutils]
index-servers =
    pypi
    testpypi

[pypi]
username = __token__
password = pypi-YOUR_PRODUCTION_TOKEN_HERE

[testpypi]
repository = https://test.pypi.org/legacy/
username = __token__
password = pypi-YOUR_TEST_TOKEN_HERE
```

⚠️ **Security**: Never commit this file to Git! Add `.pypirc` to `.gitignore`.

Then you can upload without entering credentials:

```powershell
# Upload to Test PyPI
python -m twine upload --repository testpypi dist/*

# Upload to Production PyPI
python -m twine upload --repository pypi dist/*
```

## Resources

- **PyPI**: https://pypi.org/
- **Test PyPI**: https://test.pypi.org/
- **Twine Documentation**: https://twine.readthedocs.io/
- **Packaging Guide**: https://packaging.python.org/
- **Package on PyPI**: https://pypi.org/project/parallel-llm/
