"use strict";
const assert_1 = require("@aws-cdk/assert");
const codebuild = require("@aws-cdk/aws-codebuild");
const codecommit = require("@aws-cdk/aws-codecommit");
const codepipeline = require("@aws-cdk/aws-codepipeline");
const targets = require("@aws-cdk/aws-events-targets");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const s3 = require("@aws-cdk/aws-s3");
const sns = require("@aws-cdk/aws-sns");
const core_1 = require("@aws-cdk/core");
const cpactions = require("../lib");
module.exports = {
    'basic pipeline'(test) {
        const stack = new core_1.Stack();
        const repository = new codecommit.Repository(stack, 'MyRepo', {
            repositoryName: 'my-repo',
        });
        const pipeline = new codepipeline.Pipeline(stack, 'Pipeline');
        const sourceOutput = new codepipeline.Artifact('SourceArtifact');
        const source = new cpactions.CodeCommitSourceAction({
            actionName: 'source',
            output: sourceOutput,
            repository,
        });
        pipeline.addStage({
            stageName: 'source',
            actions: [source],
        });
        const project = new codebuild.PipelineProject(stack, 'MyBuildProject');
        pipeline.addStage({
            stageName: 'build',
            actions: [
                new cpactions.CodeBuildAction({
                    actionName: 'build',
                    input: sourceOutput,
                    project,
                }),
            ],
        });
        test.notDeepEqual(assert_1.SynthUtils.toCloudFormation(stack), {});
        test.deepEqual([], core_1.ConstructNode.validate(pipeline.node));
        test.done();
    },
    'Tokens can be used as physical names of the Pipeline'(test) {
        const stack = new core_1.Stack(undefined, 'StackName');
        new codepipeline.Pipeline(stack, 'Pipeline', {
            pipelineName: core_1.Aws.STACK_NAME,
        });
        assert_1.expect(stack, true).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
            'Name': {
                'Ref': 'AWS::StackName',
            },
        }));
        test.done();
    },
    'pipeline with GitHub source with poll trigger'(test) {
        const stack = new core_1.Stack();
        const secret = new core_1.CfnParameter(stack, 'GitHubToken', { type: 'String', default: 'my-token' });
        const p = new codepipeline.Pipeline(stack, 'P');
        p.addStage({
            stageName: 'Source',
            actions: [
                new cpactions.GitHubSourceAction({
                    actionName: 'GH',
                    runOrder: 8,
                    output: new codepipeline.Artifact('A'),
                    branch: 'branch',
                    oauthToken: core_1.SecretValue.plainText(secret.valueAsString),
                    owner: 'foo',
                    repo: 'bar',
                    trigger: cpactions.GitHubTrigger.POLL,
                }),
            ],
        });
        p.addStage({
            stageName: 'Two',
            actions: [
                new cpactions.ManualApprovalAction({ actionName: 'Boo' }),
            ],
        });
        assert_1.expect(stack).to(assert_1.not(assert_1.haveResourceLike('AWS::CodePipeline::Webhook')));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
            'Stages': [
                {
                    'Actions': [
                        {
                            'Configuration': {
                                'PollForSourceChanges': true,
                            },
                            'Name': 'GH',
                        },
                    ],
                    'Name': 'Source',
                },
                {
                    'Actions': [
                        {
                            'Name': 'Boo',
                        },
                    ],
                    'Name': 'Two',
                },
            ],
        }));
        test.done();
    },
    'pipeline with GitHub source without triggers'(test) {
        const stack = new core_1.Stack();
        const secret = new core_1.CfnParameter(stack, 'GitHubToken', { type: 'String', default: 'my-token' });
        const p = new codepipeline.Pipeline(stack, 'P');
        p.addStage({
            stageName: 'Source',
            actions: [
                new cpactions.GitHubSourceAction({
                    actionName: 'GH',
                    runOrder: 8,
                    output: new codepipeline.Artifact('A'),
                    branch: 'branch',
                    oauthToken: core_1.SecretValue.plainText(secret.valueAsString),
                    owner: 'foo',
                    repo: 'bar',
                    trigger: cpactions.GitHubTrigger.NONE,
                }),
            ],
        });
        p.addStage({
            stageName: 'Two',
            actions: [
                new cpactions.ManualApprovalAction({ actionName: 'Boo' }),
            ],
        });
        assert_1.expect(stack).to(assert_1.not(assert_1.haveResourceLike('AWS::CodePipeline::Webhook')));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
            'Stages': [
                {
                    'Actions': [
                        {
                            'Configuration': {
                                'PollForSourceChanges': false,
                            },
                            'Name': 'GH',
                        },
                    ],
                    'Name': 'Source',
                },
                {
                    'Actions': [
                        {
                            'Name': 'Boo',
                        },
                    ],
                    'Name': 'Two',
                },
            ],
        }));
        test.done();
    },
    'github action uses ThirdParty owner'(test) {
        const stack = new core_1.Stack();
        const secret = new core_1.CfnParameter(stack, 'GitHubToken', { type: 'String', default: 'my-token' });
        const p = new codepipeline.Pipeline(stack, 'P');
        p.addStage({
            stageName: 'Source',
            actions: [
                new cpactions.GitHubSourceAction({
                    actionName: 'GH',
                    runOrder: 8,
                    output: new codepipeline.Artifact('A'),
                    branch: 'branch',
                    oauthToken: core_1.SecretValue.plainText(secret.valueAsString),
                    owner: 'foo',
                    repo: 'bar',
                }),
            ],
        });
        p.addStage({
            stageName: 'Two',
            actions: [
                new cpactions.ManualApprovalAction({ actionName: 'Boo' }),
            ],
        });
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Webhook'));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
            'ArtifactStore': {
                'Location': {
                    'Ref': 'PArtifactsBucket5E711C12',
                },
                'Type': 'S3',
            },
            'RoleArn': {
                'Fn::GetAtt': [
                    'PRole07BDC907',
                    'Arn',
                ],
            },
            'Stages': [
                {
                    'Actions': [
                        {
                            'ActionTypeId': {
                                'Category': 'Source',
                                'Owner': 'ThirdParty',
                                'Provider': 'GitHub',
                                'Version': '1',
                            },
                            'Configuration': {
                                'Owner': 'foo',
                                'Repo': 'bar',
                                'Branch': 'branch',
                                'OAuthToken': {
                                    'Ref': 'GitHubToken',
                                },
                                'PollForSourceChanges': false,
                            },
                            'Name': 'GH',
                            'OutputArtifacts': [
                                {
                                    'Name': 'A',
                                },
                            ],
                            'RunOrder': 8,
                        },
                    ],
                    'Name': 'Source',
                },
                {
                    'Actions': [
                        {
                            'ActionTypeId': {
                                'Category': 'Approval',
                                'Owner': 'AWS',
                                'Provider': 'Manual',
                                'Version': '1',
                            },
                            'Name': 'Boo',
                            'RunOrder': 1,
                        },
                    ],
                    'Name': 'Two',
                },
            ],
        }));
        test.deepEqual([], core_1.ConstructNode.validate(p.node));
        test.done();
    },
    'onStateChange'(test) {
        const stack = new core_1.Stack();
        const topic = new sns.Topic(stack, 'Topic');
        const pipeline = new codepipeline.Pipeline(stack, 'PL');
        pipeline.addStage({
            stageName: 'S1',
            actions: [
                new cpactions.S3SourceAction({
                    actionName: 'A1',
                    output: new codepipeline.Artifact('Artifact'),
                    bucket: new s3.Bucket(stack, 'Bucket'),
                    bucketKey: 'Key',
                }),
            ],
        });
        pipeline.addStage({
            stageName: 'S2',
            actions: [
                new cpactions.ManualApprovalAction({ actionName: 'A2' }),
            ],
        });
        pipeline.onStateChange('OnStateChange', {
            target: new targets.SnsTopic(topic),
            description: 'desc',
            eventPattern: {
                detail: {
                    state: ['FAILED'],
                },
            },
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Events::Rule', {
            'Description': 'desc',
            'EventPattern': {
                'detail': {
                    'state': [
                        'FAILED',
                    ],
                },
                'detail-type': [
                    'CodePipeline Pipeline Execution State Change',
                ],
                'source': [
                    'aws.codepipeline',
                ],
                'resources': [
                    {
                        'Fn::Join': [
                            '',
                            [
                                'arn:',
                                {
                                    'Ref': 'AWS::Partition',
                                },
                                ':codepipeline:',
                                {
                                    'Ref': 'AWS::Region',
                                },
                                ':',
                                {
                                    'Ref': 'AWS::AccountId',
                                },
                                ':',
                                {
                                    'Ref': 'PLD5425AEA',
                                },
                            ],
                        ],
                    },
                ],
            },
            'State': 'ENABLED',
            'Targets': [
                {
                    'Arn': {
                        'Ref': 'TopicBFC7AF6E',
                    },
                    'Id': 'Target0',
                },
            ],
        }));
        test.deepEqual([], core_1.ConstructNode.validate(pipeline.node));
        test.done();
    },
    'PipelineProject': {
        'with a custom Project Name': {
            'sets the source and artifacts to CodePipeline'(test) {
                const stack = new core_1.Stack();
                new codebuild.PipelineProject(stack, 'MyProject', {
                    projectName: 'MyProject',
                });
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodeBuild::Project', {
                    'Name': 'MyProject',
                    'Source': {
                        'Type': 'CODEPIPELINE',
                    },
                    'Artifacts': {
                        'Type': 'CODEPIPELINE',
                    },
                    'ServiceRole': {
                        'Fn::GetAtt': [
                            'MyProjectRole9BBE5233',
                            'Arn',
                        ],
                    },
                    'Environment': {
                        'Type': 'LINUX_CONTAINER',
                        'PrivilegedMode': false,
                        'Image': 'aws/codebuild/standard:1.0',
                        'ComputeType': 'BUILD_GENERAL1_SMALL',
                    },
                }));
                test.done();
            },
        },
    },
    'Lambda PipelineInvokeAction can be used to invoke Lambda functions from a CodePipeline'(test) {
        const stack = new core_1.Stack();
        const lambdaFun = new lambda.Function(stack, 'Function', {
            code: new lambda.InlineCode('bla'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        const pipeline = new codepipeline.Pipeline(stack, 'Pipeline');
        const bucket = new s3.Bucket(stack, 'Bucket');
        const source1Output = new codepipeline.Artifact('sourceArtifact1');
        const source1 = new cpactions.S3SourceAction({
            actionName: 'SourceAction1',
            bucketKey: 'some/key',
            output: source1Output,
            bucket,
        });
        const source2Output = new codepipeline.Artifact('sourceArtifact2');
        const source2 = new cpactions.S3SourceAction({
            actionName: 'SourceAction2',
            bucketKey: 'another/key',
            output: source2Output,
            bucket,
        });
        pipeline.addStage({
            stageName: 'Source',
            actions: [
                source1,
                source2,
            ],
        });
        const lambdaAction = new cpactions.LambdaInvokeAction({
            actionName: 'InvokeAction',
            lambda: lambdaFun,
            inputs: [
                source2Output,
                source1Output,
            ],
            outputs: [
                new codepipeline.Artifact('lambdaOutput1'),
                new codepipeline.Artifact('lambdaOutput2'),
                new codepipeline.Artifact('lambdaOutput3'),
            ],
        });
        pipeline.addStage({
            stageName: 'Stage',
            actions: [lambdaAction],
        });
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
            'ArtifactStore': {
                'Location': {
                    'Ref': 'PipelineArtifactsBucket22248F97',
                },
                'Type': 'S3',
            },
            'RoleArn': {
                'Fn::GetAtt': [
                    'PipelineRoleD68726F7',
                    'Arn',
                ],
            },
            'Stages': [
                {
                    'Name': 'Source',
                },
                {
                    'Actions': [
                        {
                            'ActionTypeId': {
                                'Category': 'Invoke',
                                'Owner': 'AWS',
                                'Provider': 'Lambda',
                                'Version': '1',
                            },
                            'Configuration': {
                                'FunctionName': {
                                    'Ref': 'Function76856677',
                                },
                            },
                            'InputArtifacts': [
                                { 'Name': 'sourceArtifact2' },
                                { 'Name': 'sourceArtifact1' },
                            ],
                            'Name': 'InvokeAction',
                            'OutputArtifacts': [
                                { 'Name': 'lambdaOutput1' },
                                { 'Name': 'lambdaOutput2' },
                                { 'Name': 'lambdaOutput3' },
                            ],
                            'RunOrder': 1,
                        },
                    ],
                    'Name': 'Stage',
                },
            ],
        }));
        test.equal((lambdaAction.actionProperties.outputs || []).length, 3);
        assert_1.expect(stack, /* skip validation */ true).to(assert_1.haveResource('AWS::IAM::Policy', {
            'PolicyDocument': {
                'Statement': [
                    {
                        'Action': [
                            'codepipeline:PutJobSuccessResult',
                            'codepipeline:PutJobFailureResult',
                        ],
                        'Effect': 'Allow',
                        'Resource': '*',
                    },
                ],
                'Version': '2012-10-17',
            },
            'PolicyName': 'FunctionServiceRoleDefaultPolicy2F49994A',
            'Roles': [
                {
                    'Ref': 'FunctionServiceRole675BB04A',
                },
            ],
        }));
        test.done();
    },
    'cross-region Pipeline': {
        'generates the required Action & ArtifactStores properties in the template'(test) {
            const pipelineRegion = 'us-west-2';
            const pipelineAccount = '123';
            const app = new core_1.App();
            const stack = new core_1.Stack(app, 'TestStack', {
                env: {
                    region: pipelineRegion,
                    account: pipelineAccount,
                },
            });
            const pipeline = new codepipeline.Pipeline(stack, 'MyPipeline', {
                crossRegionReplicationBuckets: {
                    'us-west-1': s3.Bucket.fromBucketName(stack, 'ImportedBucket', 'sfo-replication-bucket'),
                },
            });
            const sourceBucket = new s3.Bucket(stack, 'MyBucket');
            const sourceOutput = new codepipeline.Artifact('SourceOutput');
            const sourceAction = new cpactions.S3SourceAction({
                actionName: 'BucketSource',
                bucketKey: '/some/key',
                output: sourceOutput,
                bucket: sourceBucket,
            });
            pipeline.addStage({
                stageName: 'Stage1',
                actions: [sourceAction],
            });
            pipeline.addStage({
                stageName: 'Stage2',
                actions: [
                    new cpactions.CloudFormationCreateReplaceChangeSetAction({
                        actionName: 'Action1',
                        changeSetName: 'ChangeSet',
                        templatePath: sourceOutput.atPath('template.yaml'),
                        stackName: 'SomeStack',
                        region: pipelineRegion,
                        adminPermissions: false,
                    }),
                    new cpactions.CloudFormationCreateUpdateStackAction({
                        actionName: 'Action2',
                        templatePath: sourceOutput.atPath('template.yaml'),
                        stackName: 'OtherStack',
                        region: 'us-east-1',
                        adminPermissions: false,
                    }),
                    new cpactions.CloudFormationExecuteChangeSetAction({
                        actionName: 'Action3',
                        changeSetName: 'ChangeSet',
                        stackName: 'SomeStack',
                        region: 'us-west-1',
                    }),
                ],
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                'ArtifactStores': [
                    {
                        'Region': 'us-west-1',
                        'ArtifactStore': {
                            'Type': 'S3',
                            'Location': 'sfo-replication-bucket',
                        },
                    },
                    {
                        'Region': 'us-east-1',
                        'ArtifactStore': {
                            'Type': 'S3',
                            'Location': 'teststack-support-us-easteplicationbucket1a8063b3cdac6e7e0e73',
                        },
                    },
                    {
                        'Region': 'us-west-2',
                        'ArtifactStore': {
                            'Type': 'S3',
                            'EncryptionKey': {
                                'Type': 'KMS',
                                'Id': {},
                            },
                        },
                    },
                ],
                'Stages': [
                    {
                        'Name': 'Stage1',
                    },
                    {
                        'Name': 'Stage2',
                        'Actions': [
                            {
                                'Name': 'Action1',
                                'Region': 'us-west-2',
                            },
                            {
                                'Name': 'Action2',
                                'Region': 'us-east-1',
                            },
                            {
                                'Name': 'Action3',
                                'Region': 'us-west-1',
                            },
                        ],
                    },
                ],
            }));
            test.notEqual(pipeline.crossRegionSupport[pipelineRegion], undefined);
            test.notEqual(pipeline.crossRegionSupport['us-west-1'], undefined);
            const usEast1Support = pipeline.crossRegionSupport['us-east-1'];
            test.notEqual(usEast1Support, undefined);
            test.equal(usEast1Support.stack.region, 'us-east-1');
            test.equal(usEast1Support.stack.account, pipelineAccount);
            test.ok(usEast1Support.stack.node.id.indexOf('us-east-1') !== -1, `expected '${usEast1Support.stack.node.id}' to contain 'us-east-1'`);
            test.done();
        },
        'allows specifying only one of artifactBucket and crossRegionReplicationBuckets'(test) {
            const stack = new core_1.Stack();
            test.throws(() => {
                new codepipeline.Pipeline(stack, 'Pipeline', {
                    artifactBucket: new s3.Bucket(stack, 'Bucket'),
                    crossRegionReplicationBuckets: {
                    // even an empty map should trigger this validation...
                    },
                });
            }, /Only one of artifactBucket and crossRegionReplicationBuckets can be specified!/);
            test.done();
        },
        'does not create a new artifact Bucket if one was provided in the cross-region Buckets for the Pipeline region'(test) {
            const pipelineRegion = 'us-west-2';
            const stack = new core_1.Stack(undefined, undefined, {
                env: {
                    region: pipelineRegion,
                },
            });
            const sourceOutput = new codepipeline.Artifact();
            new codepipeline.Pipeline(stack, 'Pipeline', {
                crossRegionReplicationBuckets: {
                    [pipelineRegion]: new s3.Bucket(stack, 'Bucket', {
                        bucketName: 'my-pipeline-bucket',
                    }),
                },
                stages: [
                    {
                        stageName: 'Source',
                        actions: [
                            new cpactions.CodeCommitSourceAction({
                                actionName: 'Source',
                                output: sourceOutput,
                                repository: new codecommit.Repository(stack, 'Repo', { repositoryName: 'Repo' }),
                            }),
                        ],
                    },
                    {
                        stageName: 'Build',
                        actions: [
                            new cpactions.CodeBuildAction({
                                actionName: 'Build',
                                input: sourceOutput,
                                project: new codebuild.PipelineProject(stack, 'Project'),
                            }),
                        ],
                    },
                ],
            });
            assert_1.expect(stack).to(assert_1.countResources('AWS::S3::Bucket', 1));
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                'ArtifactStores': [
                    {
                        'Region': pipelineRegion,
                        'ArtifactStore': {
                            'Type': 'S3',
                            'Location': {
                                'Ref': 'Bucket83908E77',
                            },
                        },
                    },
                ],
            }));
            test.done();
        },
        'allows providing a resource-backed action from a different region directly'(test) {
            const account = '123456789012';
            const app = new core_1.App();
            const replicationRegion = 'us-west-1';
            const replicationStack = new core_1.Stack(app, 'ReplicationStack', { env: { region: replicationRegion, account } });
            const project = new codebuild.PipelineProject(replicationStack, 'CodeBuildProject', {
                projectName: 'MyCodeBuildProject',
            });
            const pipelineRegion = 'us-west-2';
            const pipelineStack = new core_1.Stack(app, 'TestStack', { env: { region: pipelineRegion, account } });
            const sourceOutput = new codepipeline.Artifact('SourceOutput');
            new codepipeline.Pipeline(pipelineStack, 'MyPipeline', {
                stages: [
                    {
                        stageName: 'Source',
                        actions: [new cpactions.CodeCommitSourceAction({
                                actionName: 'CodeCommitAction',
                                output: sourceOutput,
                                repository: codecommit.Repository.fromRepositoryName(pipelineStack, 'Repo', 'my-codecommit-repo'),
                            })],
                    },
                    {
                        stageName: 'Build',
                        actions: [new cpactions.CodeBuildAction({
                                actionName: 'CodeBuildAction',
                                input: sourceOutput,
                                project,
                            })],
                    },
                ],
            });
            assert_1.expect(pipelineStack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                'ArtifactStores': [
                    {
                        'Region': replicationRegion,
                        'ArtifactStore': {
                            'Type': 'S3',
                            'Location': 'replicationstackeplicationbucket2464cd5c33b386483b66',
                            'EncryptionKey': {
                                'Id': {
                                    'Fn::Join': [
                                        '',
                                        [
                                            'arn:',
                                            {
                                                'Ref': 'AWS::Partition',
                                            },
                                            ':kms:us-west-1:123456789012:alias/ionstacktencryptionalias043cb2f8ceac9da9c07c',
                                        ],
                                    ],
                                },
                                'Type': 'KMS',
                            },
                        },
                    },
                    {
                        'Region': pipelineRegion,
                    },
                ],
                'Stages': [
                    {
                        'Name': 'Source',
                    },
                    {
                        'Name': 'Build',
                        'Actions': [
                            {
                                'Name': 'CodeBuildAction',
                                'Region': replicationRegion,
                                'Configuration': {
                                    'ProjectName': 'MyCodeBuildProject',
                                },
                            },
                        ],
                    },
                ],
            }));
            assert_1.expect(replicationStack).to(assert_1.haveResourceLike('AWS::S3::Bucket', {
                'BucketName': 'replicationstackeplicationbucket2464cd5c33b386483b66',
            }));
            test.done();
        },
    },
    'cross-account Pipeline': {
        'with a CodeBuild Project in a different account works correctly'(test) {
            const app = new core_1.App();
            const buildAccount = '901234567890';
            const buildRegion = 'bermuda-triangle-1';
            const buildStack = new core_1.Stack(app, 'BuildStack', {
                env: { account: buildAccount, region: buildRegion },
            });
            const rolePhysicalName = 'ProjectRolePhysicalName';
            const projectRole = new iam.Role(buildStack, 'ProjectRole', {
                assumedBy: new iam.ServicePrincipal('codebuild.amazonaws.com'),
                roleName: rolePhysicalName,
            });
            const projectPhysicalName = 'ProjectPhysicalName';
            const project = new codebuild.PipelineProject(buildStack, 'Project', {
                projectName: projectPhysicalName,
                role: projectRole,
            });
            const pipelineStack = new core_1.Stack(app, 'PipelineStack', {
                env: { account: '123456789012', region: buildRegion },
            });
            const sourceBucket = new s3.Bucket(pipelineStack, 'ArtifactBucket', {
                bucketName: 'source-bucket',
                encryption: s3.BucketEncryption.KMS,
            });
            const sourceOutput = new codepipeline.Artifact();
            new codepipeline.Pipeline(pipelineStack, 'Pipeline', {
                stages: [
                    {
                        stageName: 'Source',
                        actions: [
                            new cpactions.S3SourceAction({
                                actionName: 'S3',
                                bucket: sourceBucket,
                                bucketKey: 'path/to/file.zip',
                                output: sourceOutput,
                            }),
                        ],
                    },
                    {
                        stageName: 'Build',
                        actions: [
                            new cpactions.CodeBuildAction({
                                actionName: 'CodeBuild',
                                project,
                                input: sourceOutput,
                            }),
                        ],
                    },
                ],
            });
            assert_1.expect(pipelineStack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                'Stages': [
                    {
                        'Name': 'Source',
                    },
                    {
                        'Name': 'Build',
                        'Actions': [
                            {
                                'Name': 'CodeBuild',
                                'Configuration': {
                                    'ProjectName': projectPhysicalName,
                                },
                                'RoleArn': {
                                    'Fn::Join': [
                                        '',
                                        [
                                            'arn:',
                                            {
                                                'Ref': 'AWS::Partition',
                                            },
                                            `:iam::${buildAccount}:role/buildstackebuildactionrole166c75d1d8be701b1ad8`,
                                        ],
                                    ],
                                },
                            },
                        ],
                    },
                ],
            }));
            assert_1.expect(buildStack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
                'PolicyDocument': {
                    'Statement': [
                        {
                        // log permissions from the CodeBuild Project Construct...
                        },
                        {
                        // report group permissions from the CodeBuild Project construct...
                        },
                        {
                            'Action': [
                                's3:GetObject*',
                                's3:GetBucket*',
                                's3:List*',
                            ],
                            'Effect': 'Allow',
                            'Resource': [
                                {
                                    'Fn::Join': [
                                        '',
                                        [
                                            'arn:',
                                            {
                                                'Ref': 'AWS::Partition',
                                            },
                                            ':s3:::pipelinestackeartifactsbucket5409dc84bb108027cb58',
                                        ],
                                    ],
                                },
                                {
                                    'Fn::Join': [
                                        '',
                                        [
                                            'arn:',
                                            {
                                                'Ref': 'AWS::Partition',
                                            },
                                            ':s3:::pipelinestackeartifactsbucket5409dc84bb108027cb58/*',
                                        ],
                                    ],
                                },
                            ],
                        },
                        {
                            'Action': [
                                'kms:Decrypt',
                                'kms:DescribeKey',
                            ],
                            'Effect': 'Allow',
                            'Resource': '*',
                        },
                    ],
                },
            }));
            test.done();
        },
        'adds a dependency on the Stack containing a new action Role'(test) {
            const region = 'us-west-2';
            const pipelineAccount = '123456789012';
            const buildAccount = '901234567890';
            const app = new core_1.App();
            const buildStack = new core_1.Stack(app, 'BuildStack', {
                env: { account: buildAccount, region },
            });
            const actionRolePhysicalName = 'ProjectRolePhysicalName';
            const actionRoleInOtherAccount = new iam.Role(buildStack, 'ProjectRole', {
                assumedBy: new iam.AccountPrincipal(pipelineAccount),
                roleName: actionRolePhysicalName,
            });
            const projectPhysicalName = 'ProjectPhysicalName';
            const project = codebuild.Project.fromProjectName(buildStack, 'Project', projectPhysicalName);
            const pipelineStack = new core_1.Stack(app, 'PipelineStack', {
                env: { account: pipelineAccount, region },
            });
            const bucket = new s3.Bucket(pipelineStack, 'ArtifactBucket', {
                bucketName: 'source-bucket',
                encryption: s3.BucketEncryption.KMS,
            });
            const sourceOutput = new codepipeline.Artifact();
            new codepipeline.Pipeline(pipelineStack, 'Pipeline', {
                artifactBucket: bucket,
                stages: [
                    {
                        stageName: 'Source',
                        actions: [
                            new cpactions.S3SourceAction({
                                actionName: 'S3',
                                bucket,
                                bucketKey: 'path/to/file.zip',
                                output: sourceOutput,
                            }),
                        ],
                    },
                    {
                        stageName: 'Build',
                        actions: [
                            new cpactions.CodeBuildAction({
                                actionName: 'CodeBuild',
                                project,
                                input: sourceOutput,
                                role: actionRoleInOtherAccount,
                            }),
                        ],
                    },
                ],
            });
            assert_1.expect(pipelineStack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                'Stages': [
                    {
                        'Name': 'Source',
                    },
                    {
                        'Name': 'Build',
                        'Actions': [
                            {
                                'Name': 'CodeBuild',
                                'Configuration': {
                                    'ProjectName': projectPhysicalName,
                                },
                                'RoleArn': {
                                    'Fn::Join': ['', [
                                            'arn:',
                                            { 'Ref': 'AWS::Partition' },
                                            `:iam::${buildAccount}:role/${actionRolePhysicalName}`,
                                        ]],
                                },
                            },
                        ],
                    },
                ],
            }));
            test.equal(pipelineStack.dependencies.length, 1);
            test.done();
        },
        'does not add a dependency on the Stack containing an imported action Role'(test) {
            const region = 'us-west-2';
            const pipelineAccount = '123456789012';
            const buildAccount = '901234567890';
            const app = new core_1.App();
            const buildStack = new core_1.Stack(app, 'BuildStack', {
                env: { account: buildAccount, region },
            });
            const actionRolePhysicalName = 'ProjectRolePhysicalName';
            const actionRoleInOtherAccount = iam.Role.fromRoleArn(buildStack, 'ProjectRole', `arn:aws:iam::${buildAccount}:role/${actionRolePhysicalName}`);
            const projectPhysicalName = 'ProjectPhysicalName';
            const project = new codebuild.PipelineProject(buildStack, 'Project', {
                projectName: projectPhysicalName,
            });
            const pipelineStack = new core_1.Stack(app, 'PipelineStack', {
                env: { account: pipelineAccount, region },
            });
            const bucket = new s3.Bucket(pipelineStack, 'ArtifactBucket', {
                bucketName: 'source-bucket',
                encryption: s3.BucketEncryption.KMS,
            });
            const sourceOutput = new codepipeline.Artifact();
            new codepipeline.Pipeline(pipelineStack, 'Pipeline', {
                artifactBucket: bucket,
                stages: [
                    {
                        stageName: 'Source',
                        actions: [
                            new cpactions.S3SourceAction({
                                actionName: 'S3',
                                bucket,
                                bucketKey: 'path/to/file.zip',
                                output: sourceOutput,
                            }),
                        ],
                    },
                    {
                        stageName: 'Build',
                        actions: [
                            new cpactions.CodeBuildAction({
                                actionName: 'CodeBuild',
                                project,
                                input: sourceOutput,
                                role: actionRoleInOtherAccount,
                            }),
                        ],
                    },
                ],
            });
            assert_1.expect(pipelineStack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                'Stages': [
                    {
                        'Name': 'Source',
                    },
                    {
                        'Name': 'Build',
                        'Actions': [
                            {
                                'Name': 'CodeBuild',
                                'Configuration': {
                                    'ProjectName': projectPhysicalName,
                                },
                                'RoleArn': `arn:aws:iam::${buildAccount}:role/${actionRolePhysicalName}`,
                            },
                        ],
                    },
                ],
            }));
            test.equal(pipelineStack.dependencies.length, 0);
            test.done();
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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