"""This module contains methods to featurize molecules."""
import numpy as np
from rdkit.Chem import rdmolops
from rdkit import Chem
from rdkit import DataStructs
from rdkit.Chem import AllChem
from rdkit.Chem.Fingerprints import FingerprintMols
from rdkit.Chem import MACCSkeys
from rdkit.Chem.AtomPairs import Pairs, Torsions
from rdkit.DataStructs import cDataStructs
from mordred import Calculator, descriptors
from aimsim.utils.ccbmlib_fingerprints import generate_fingerprints
from padelpy import from_smiles

from ..exceptions import (
    InvalidConfigurationError,
    NotInitializedError,
    MordredCalculatorError,
)


class Descriptor:
    """Class for descriptors.

    Attributes:
        label_ (str): Label used to denote the type of descriptor being used.
        numpy_ (np.ndarray): Value of the descriptor in the numpy format.
        rdkit_ (rdkit.DataStructs.cDataStructs.UIntSparseIntVec):  Value of
            the descriptor in the rdkit format.

    Methods:
        to_numpy: Get the numpy_ attribute. If it does not exist,
            it is created.
        to_rdkit: Get the rdkit_ attribute. If it does not exist,
            it is created.
        check_init: check if the Descriptor object is initialized. This is
            done by checking the existence of the numpy_ or rdkit_ attribute.


    """

    def __init__(self, value=None):
        if value is not None:
            self.set_manually(arbitrary_descriptor_val=value)

    def to_numpy(self):
        """Return numpy_ attribute of Descriptor.
        Attribute will be initialized if not present.

        Returns:
            (np.array): Numpy vector of descriptor.

        """
        if self.check_init() is False:
            raise NotInitializedError(
                "Descriptor value not generated. Use "
                "make_fingerprint() to initialize it."
            )
        if not hasattr(self, "numpy_"):
            self.numpy_ = np.zeros((0,), dtype=np.int8)
            DataStructs.ConvertToNumpyArray(self.rdkit_, self.numpy_)
        self.numpy_ = self.numpy_.ravel()
        return self.numpy_

    def to_rdkit(self):
        """Return rdkit_ attribute of Descriptor.

        Returns:
            (DataStructs.ExplicitBitVect): Fingerprint value as a bit vector.

        Raises:
            (NotInitializedError): If object not initialized with a fingerprint.
            (ValueError): If only arbitrary numpy descriptor is used to
                initialize the object. This cannot be converted to bit vectors.

        """
        if self.check_init() is False:
            raise NotInitializedError(
                "Descriptor value not generated. Use "
                "make_fingerprint() to initialize it."
            )
        if not hasattr(self, "rdkit_"):
            raise ValueError(
                "Attempting to convert arbitrary numpy array "
                "to rdkit bit vector is not supported"
            )
        return self.rdkit_

    def check_init(self):
        """Check initialization status of the Descriptor object.

        Returns:
            (bool): True if object is initialized.

        """
        return getattr(self, "numpy_", None) is not None \
            or getattr(self, "rdkit_", None) is not None

    def _set_morgan_fingerprint(self, molecule_graph, radius, n_bits, **kwargs):
        """Set the descriptor to a morgan fingerprint.

        Args:
            molecule_graph (RDKIT object): Graph of molecule to be
                fingerprinted.
            radius (int): Radius of fingerprint, 3 corresponds to ECFP6.
            n_bits (int):Number of bits to use if Morgan Fingerprint wanted as
                a bit vector. If set to None, Morgan fingerprint returned
                as count.

        """
        self.rdkit_ = AllChem.GetMorganFingerprintAsBitVect(
            molecule_graph, radius, nBits=n_bits, **kwargs
        )
        self.label_ = "morgan_fingerprint"
        self.params_ = {"radius": radius, "n_bits": n_bits}

    def _set_rdkit_topological_fingerprint(
        self, molecule_graph, min_path, max_path, **kwargs
    ):
        """Set the descriptor to a topological fingerprint.

        Args:
            molecule_graph (RDKIT object): Graph of molecule to be
                fingerprinted.
            min_path (int): Minimum path used to generate the
                topological fingerprint.
            max_path (int): Maximum path used to generate the topological
                fingerprint.

        """
        if molecule_graph.GetNumAtoms() <= min_path:
            raise InvalidConfigurationError(
                f"# atoms in molecule: "
                f"{molecule_graph.GetNumAtoms()}, "
                f"min_path: {min_path}. "
                f"For topological fingerprint, "
                f"the number of atoms has to be "
                f"greater than the minimum path "
                f"used for fingerprint."
            )
        self.rdkit_ = rdmolops.RDKFingerprint(
            molecule_graph, minPath=min_path, maxPath=max_path
        )
        self.label_ = "topological_fingerprint"
        self.params_ = {"min_path": min_path, "max_path": max_path}

    def _set_padelpy_descriptor(self, molecule_graph, descriptor, **kwargs):
        """Set the value of numpy_ to the descriptor as indicated by descriptor.

        Args:
            molecule_graph (RDKit object): Graph of the molecule of interest.
            descriptor (string): Name of descriptor, as implemented in Mordred.

        """
        try:
            local_smi = Chem.MolToSmiles(molecule_graph)
            all_desc = from_smiles(
                local_smi, timeout=kwargs.get("timeout", 60))
            self.numpy_ = np.array(all_desc[descriptor])
            self.label_ = descriptor
        except Exception as e:
            raise RuntimeError(
                f"Unable to retrieve PaDELPy Descriptor '{descriptor}'. Is it a valid descriptor?")

    def _set_mordred_descriptor(self, molecule_graph, descriptor, **kwargs):
        """Set the value of numpy_ to the descriptor as indicated by descriptor.

        Args:
            molecule_graph (RDKit object): Graph of the molecule of interest.
            descriptor (string): Name of descriptor, as implemented in Mordred.

        Raises:
            MordredCalculatorError: If Morded is unable to calculate a value
                for the indicated descriptor, this exception will be raised.
        """
        try:
            calc = Calculator(descriptors, ignore_3D=False)
            res = calc(molecule_graph)
            res.drop_missing()
            self.numpy_ = np.array(res[descriptor])
            self.label_ = descriptor
        except KeyError:
            raise MordredCalculatorError(
                """Mordred descriptor calculator unable to calculate descriptor \"{}\",
                ensure correct name is used (https://mordred-descriptor.github.io/documentation/master/descriptors.html).""".format(
                    descriptor
                )
            )

    def _set_daylight_fingerprint(self, molecule_graph, **kwargs):
        """Set the descriptor to a daylight fingerprint.

        Args:
            molecule_graph (RDKIT object): Graph of molecule to be
                fingerprinted.

        """
        # returns an ExplicitBitVect
        self.rdkit_ = FingerprintMols.FingerprintMol(molecule_graph)
        self.label_ = "daylight_fingerprint"
        self.params_ = {}

    def _set_maccs_keys(self, molecule_graph, **kwargs):
        """Set the descriptor to MACCS keys.

        Args:
            molecule_graph (RDKIT object): Graph of molecule to be
                fingerprinted.

        """
        # returns an ExplicitBitVect
        self.rdkit_ = MACCSkeys.GenMACCSKeys(molecule_graph)
        self.label_ = "maccs_keys"
        self.params_ = {}

    def _set_atom_pair_fingerprint(self, molecule_graph, **kwargs):
        """Set the descriptor to an atom-pair fingerprint.

        Args:
            molecule_graph (RDKIT object): Graph of molecule to be
                fingerprinted.

        """
        # returns a SparseBitVect
        fp = Pairs.GetAtomPairFingerprintAsBitVect(molecule_graph)
        fp2 = cDataStructs.ConvertToExplicit(fp)
        self.rdkit_ = fp2
        self.label_ = "atom-pair_fingerprint"
        self.params_ = {}

    def _set_torsion_fingerprint(self, molecule_graph, **kwargs):
        """Set the descriptor to a torsion fingerprint.

        Args:
            molecule_graph (RDKIT object): Graph of molecule to be
                fingerprinted.

        """
        # returns a long sparse int vector
        fp = Torsions.GetTopologicalTorsionFingerprintAsIntVect(molecule_graph)
        self.rdkit_ = fp
        self.label_ = "torsion_fingerprint"
        self.params_ = {}

    def _set_ccbmlib_fingerprint(self, molecule_graph, descriptor, **kwargs):
        """Set the descriptor to fingerprint from ccbmlib.

        Args:
        molecule_graph (RDKIT object): Graph of molecule to be fingerprinted.

        """
        # returns a list of ints which represent the on bits (features)
        fp = generate_fingerprints(molecule_graph, descriptor, **kwargs)
        self.numpy_ = fp
        self.label_ = descriptor
        self.params_ = {}

    def make_fingerprint(
        self, molecule_graph, fingerprint_type, fingerprint_params=None
    ):
        """Make fingerprint of a molecule based on a graph representation.
        Set the state of the descriptor to this fingerprint.

        Args:
            molecule_graph (RDKIT object): The graph object used to make a
                fingerprint.
            fingerprint_type (str): label for the type of fingerprint.
                Invokes get_supported_descriptors()['fingerprints']
                for list of supported fingerprints.
            fingerprint_params (dict): Keyword arguments used to modify
                parameters of fingerprint. Default is None.
        """
        if fingerprint_params is None:
            fingerprint_params = {}
        if fingerprint_type == "morgan_fingerprint":
            morgan_params = {"radius": 3, "n_bits": 1024}
            morgan_params.update(fingerprint_params)
            self._set_morgan_fingerprint(molecule_graph=molecule_graph,
                                         **morgan_params)
        elif fingerprint_type == "topological_fingerprint":
            topological_params = {"min_path": 1, "max_path": 7}
            topological_params.update(fingerprint_params)
            self._set_rdkit_topological_fingerprint(
                molecule_graph=molecule_graph, **topological_params
            )
        elif fingerprint_type == "daylight_fingerprint":
            daylight_params = {
                'minPath': 1,
                'maxPath': 7,
                'fpSize': 2048,
                'bitsPerHash': 2,
                'useHs': 0,
                'tgtDensity': 0.3,
                'minSize': 64,
            }
            daylight_params.update(fingerprint_params)
            self._set_daylight_fingerprint(
                molecule_graph=molecule_graph, **daylight_params
            )
        elif fingerprint_type == "maccs_keys":
            maccs_params = {}
            maccs_params.update(fingerprint_params)
            self._set_maccs_keys(molecule_graph=molecule_graph, **maccs_params)
        elif fingerprint_type == "atom-pair_fingerprint":
            atom_pair_fp_params = {}
            atom_pair_fp_params.update(fingerprint_params)
            self._set_atom_pair_fingerprint(
                molecule_graph=molecule_graph, **atom_pair_fp_params
            )
        elif fingerprint_type == "torsion_fingerprint":
            torsion_params = {}
            torsion_params.update(fingerprint_params)
            self._set_torsion_fingerprint(
                molecule_graph=molecule_graph, **torsion_params
            )
        elif fingerprint_type.split(":")[0] == "mordred":
            mordred_params = {}
            self._set_mordred_descriptor(
                molecule_graph=molecule_graph,
                descriptor=fingerprint_type.split(":")[1],
                **mordred_params,
            )
        elif fingerprint_type.split(":")[0] == "padelpy":
            padelpy_params = {}
            padelpy_params.update(fingerprint_params)
            self._set_padelpy_descriptor(
                molecule_graph=molecule_graph,
                descriptor=fingerprint_type.split(":")[1],
                **padelpy_params,
            )
        elif fingerprint_type.split(":")[0] == "ccbmlib":
            ccbmlib_params = {}
            self._set_ccbmlib_fingerprint(
                molecule_graph=molecule_graph,
                descriptor=fingerprint_type.split(":")[1],
                **ccbmlib_params,
            )
        else:
            raise ValueError(f"{fingerprint_type} not supported")

    def set_manually(self, arbitrary_descriptor_val):
        """Set the descriptor value manually based on user specified value.

        Args:
            arbitrary_descriptor_val (np.ndarray or list): Vectorized
                representation of descriptor values.
        """
        self.label_ = "arbitrary"
        self.numpy_ = np.array(arbitrary_descriptor_val)

    def get_label(self):
        if not self.check_init():
            raise NotInitializedError
        else:
            return self.label_

    def get_params(self):
        if not self.check_init():
            raise NotInitializedError
        else:
            return self.params_

    def is_fingerprint(self):
        return "fingerprint" in self.get_label()

    def get_folded_fprint(self, fold_to_length):
        """
        Get the folded value of a fingerprint to a specified length.
        Args:
            fold_to_length (int): Number of bits to fold to.

        Returns:
            (np.ndarray): Folded fingerprint.
        """
        if not self.is_fingerprint():
            raise ValueError("Can only fold fingerprints")
        fingerprint = self.to_numpy()
        if len(fingerprint) < fold_to_length:
            raise InvalidConfigurationError(
                f"Cannot fold fingerprint of "
                f"length {len(fingerprint)}to a "
                f"higher length {fold_to_length}"
            )
        n_folds = np.log2(len(fingerprint) / fold_to_length)
        if n_folds - int(n_folds) > 0.0:
            raise InvalidConfigurationError(
                f"Fingerprint length "
                f"{len(fingerprint)} not "
                f"a 2-multiple of required "
                f"folded length {fold_to_length}"
            )
        for _ in range(int(n_folds)):
            mid_point = int(len(fingerprint) / 2)
            assert mid_point - (len(fingerprint) / 2) == 0.0
            fingerprint = fingerprint[:mid_point] | fingerprint[mid_point:]
        assert len(fingerprint) == fold_to_length
        return fingerprint

    @staticmethod
    def shorten_label(label):
        """Shorten the label of a fingerprint. Useful for plotting purposes.

        Args:
            label (str): Label of fingerprint to shorten.

        Returns:
            (str): Shortened label.

        Raises:
            InvalidConfigurationError: if label not in
                get_supported_descriptors()

        Currently implemented shortening strategies:
            1. Fingerprints: remove '_fingerprint' from the label

        """
        if label not in Descriptor.get_all_supported_descriptors():
            raise InvalidConfigurationError(
                f"{label} not a " f"supported descriptor")
        if label in Descriptor.get_supported_fprints():
            return label.replace("_fingerprint", "")
        return label

    @staticmethod
    def fold_to_equal_length(fingerprint1, fingerprint2):
        """
        Get back two fingerprint arrays of equal length. The longer fingerprint
        is folded to the size of the smaller one.

        Args:
            fingerprint1 (Descriptor): Fingerprint one.
            fingerprint2 (Descriptor): Fingerprint two.

        Returns:
            (np.ndarray, np.ndarray)

        """
        fprint1_arr = fingerprint1.to_numpy()
        fprint2_arr = fingerprint2.to_numpy()
        if len(fprint1_arr) > len(fprint2_arr):
            return (
                fingerprint1.get_folded_fprint(
                    fold_to_length=len(fprint2_arr)),
                fprint2_arr,
            )
        else:
            return fprint1_arr, fingerprint2.get_folded_fprint(
                fold_to_length=len(fprint1_arr)
            )

    @staticmethod
    def get_supported_fprints():
        """Return a list of strings for the currently implemented
        molecular fingerprints.
        Returns:
            List: List of strings.
        """
        return [
            "morgan_fingerprint",
            "topological_fingerprint",
            "daylight_fingerprint",
        ]

    @staticmethod
    def get_all_supported_descriptors():
        """Returns a list of descriptors which _can_ be used with AIMSim
        but are considered experimental or are complex/rarely used and
        are therefore excluded from the UI.

        Returns:
            List: List of strings of all supported descriptors.
        """
        return [
            "morgan_fingerprint",
            "topological_fingerprint",
            "daylight_fingerprint",
            "maccs_keys",
            "atom-pair_fingerprint",
            "torsion_fingerprint",
            "ccbmlib:atom_pairs",
            "ccbmlib:hashed_atom_pairs",
            "ccbmlib:avalon",
            "ccbmlib:maccs_keys",
            "ccbmlib:morgan",
            "ccbmlib:hashed_morgan",
            "ccbmlib:rdkit_fingerprint",
            "ccbmlib:torsions",
            "ccbmlib:hashed_torsions",
            "mordred:ABC",
            "mordred:ABCGG",
            "mordred:nAcid",
            "mordred:nBase",
            "mordred:SpAbs_A",
            "mordred:SpMax_A",
            "mordred:SpDiam_A",
            "mordred:SpAD_A",
            "mordred:SpMAD_A",
            "mordred:LogEE_A",
            "mordred:VE1_A",
            "mordred:VE2_A",
            "mordred:VE3_A",
            "mordred:VR1_A",
            "mordred:VR2_A",
            "mordred:VR3_A",
            "mordred:nAromAtom",
            "mordred:nAromBond",
            "mordred:nAtom",
            "mordred:nHeavyAtom",
            "mordred:nSpiro",
            "mordred:nBridgehead",
            "mordred:nHetero",
            "mordred:nH",
            "mordred:nB",
            "mordred:nC",
            "mordred:nN",
            "mordred:nO",
            "mordred:nS",
            "mordred:nP",
            "mordred:nF",
            "mordred:nCl",
            "mordred:nBr",
            "mordred:nI",
            "mordred:nX",
            "mordred:ATS0dv",
            "mordred:ATS1dv",
            "mordred:ATS2dv",
            "mordred:ATS3dv",
            "mordred:ATS4dv",
            "mordred:ATS5dv",
            "mordred:ATS6dv",
            "mordred:ATS7dv",
            "mordred:ATS8dv",
            "mordred:ATS0d",
            "mordred:ATS1d",
            "mordred:ATS2d",
            "mordred:ATS3d",
            "mordred:ATS4d",
            "mordred:ATS5d",
            "mordred:ATS6d",
            "mordred:ATS7d",
            "mordred:ATS8d",
            "mordred:ATS0s",
            "mordred:ATS1s",
            "mordred:ATS2s",
            "mordred:ATS3s",
            "mordred:ATS4s",
            "mordred:ATS5s",
            "mordred:ATS6s",
            "mordred:ATS7s",
            "mordred:ATS8s",
            "mordred:ATS0Z",
            "mordred:ATS1Z",
            "mordred:ATS2Z",
            "mordred:ATS3Z",
            "mordred:ATS4Z",
            "mordred:ATS5Z",
            "mordred:ATS6Z",
            "mordred:ATS7Z",
            "mordred:ATS8Z",
            "mordred:ATS0m",
            "mordred:ATS1m",
            "mordred:ATS2m",
            "mordred:ATS3m",
            "mordred:ATS4m",
            "mordred:ATS5m",
            "mordred:ATS6m",
            "mordred:ATS7m",
            "mordred:ATS8m",
            "mordred:ATS0v",
            "mordred:ATS1v",
            "mordred:ATS2v",
            "mordred:ATS3v",
            "mordred:ATS4v",
            "mordred:ATS5v",
            "mordred:ATS6v",
            "mordred:ATS7v",
            "mordred:ATS8v",
            "mordred:ATS0se",
            "mordred:ATS1se",
            "mordred:ATS2se",
            "mordred:ATS3se",
            "mordred:ATS4se",
            "mordred:ATS5se",
            "mordred:ATS6se",
            "mordred:ATS7se",
            "mordred:ATS8se",
            "mordred:ATS0pe",
            "mordred:ATS1pe",
            "mordred:ATS2pe",
            "mordred:ATS3pe",
            "mordred:ATS4pe",
            "mordred:ATS5pe",
            "mordred:ATS6pe",
            "mordred:ATS7pe",
            "mordred:ATS8pe",
            "mordred:ATS0are",
            "mordred:ATS1are",
            "mordred:ATS2are",
            "mordred:ATS3are",
            "mordred:ATS4are",
            "mordred:ATS5are",
            "mordred:ATS6are",
            "mordred:ATS7are",
            "mordred:ATS8are",
            "mordred:ATS0p",
            "mordred:ATS1p",
            "mordred:ATS2p",
            "mordred:ATS3p",
            "mordred:ATS4p",
            "mordred:ATS5p",
            "mordred:ATS6p",
            "mordred:ATS7p",
            "mordred:ATS8p",
            "mordred:ATS0i",
            "mordred:ATS1i",
            "mordred:ATS2i",
            "mordred:ATS3i",
            "mordred:ATS4i",
            "mordred:ATS5i",
            "mordred:ATS6i",
            "mordred:ATS7i",
            "mordred:ATS8i",
            "mordred:AATS0dv",
            "mordred:AATS1dv",
            "mordred:AATS2dv",
            "mordred:AATS3dv",
            "mordred:AATS4dv",
            "mordred:AATS5dv",
            "mordred:AATS6dv",
            "mordred:AATS7dv",
            "mordred:AATS8dv",
            "mordred:AATS0d",
            "mordred:AATS1d",
            "mordred:AATS2d",
            "mordred:AATS3d",
            "mordred:AATS4d",
            "mordred:AATS5d",
            "mordred:AATS6d",
            "mordred:AATS7d",
            "mordred:AATS8d",
            "mordred:AATS0s",
            "mordred:AATS1s",
            "mordred:AATS2s",
            "mordred:AATS3s",
            "mordred:AATS4s",
            "mordred:AATS5s",
            "mordred:AATS6s",
            "mordred:AATS7s",
            "mordred:AATS8s",
            "mordred:AATS0Z",
            "mordred:AATS1Z",
            "mordred:AATS2Z",
            "mordred:AATS3Z",
            "mordred:AATS4Z",
            "mordred:AATS5Z",
            "mordred:AATS6Z",
            "mordred:AATS7Z",
            "mordred:AATS8Z",
            "mordred:AATS0m",
            "mordred:AATS1m",
            "mordred:AATS2m",
            "mordred:AATS3m",
            "mordred:AATS4m",
            "mordred:AATS5m",
            "mordred:AATS6m",
            "mordred:AATS7m",
            "mordred:AATS8m",
            "mordred:AATS0v",
            "mordred:AATS1v",
            "mordred:AATS2v",
            "mordred:AATS3v",
            "mordred:AATS4v",
            "mordred:AATS5v",
            "mordred:AATS6v",
            "mordred:AATS7v",
            "mordred:AATS8v",
            "mordred:AATS0se",
            "mordred:AATS1se",
            "mordred:AATS2se",
            "mordred:AATS3se",
            "mordred:AATS4se",
            "mordred:AATS5se",
            "mordred:AATS6se",
            "mordred:AATS7se",
            "mordred:AATS8se",
            "mordred:AATS0pe",
            "mordred:AATS1pe",
            "mordred:AATS2pe",
            "mordred:AATS3pe",
            "mordred:AATS4pe",
            "mordred:AATS5pe",
            "mordred:AATS6pe",
            "mordred:AATS7pe",
            "mordred:AATS8pe",
            "mordred:AATS0are",
            "mordred:AATS1are",
            "mordred:AATS2are",
            "mordred:AATS3are",
            "mordred:AATS4are",
            "mordred:AATS5are",
            "mordred:AATS6are",
            "mordred:AATS7are",
            "mordred:AATS8are",
            "mordred:AATS0p",
            "mordred:AATS1p",
            "mordred:AATS2p",
            "mordred:AATS3p",
            "mordred:AATS4p",
            "mordred:AATS5p",
            "mordred:AATS6p",
            "mordred:AATS7p",
            "mordred:AATS8p",
            "mordred:AATS0i",
            "mordred:AATS1i",
            "mordred:AATS2i",
            "mordred:AATS3i",
            "mordred:AATS4i",
            "mordred:AATS5i",
            "mordred:AATS6i",
            "mordred:AATS7i",
            "mordred:AATS8i",
            "mordred:ATSC0c",
            "mordred:ATSC1c",
            "mordred:ATSC2c",
            "mordred:ATSC3c",
            "mordred:ATSC4c",
            "mordred:ATSC5c",
            "mordred:ATSC6c",
            "mordred:ATSC7c",
            "mordred:ATSC8c",
            "mordred:ATSC0dv",
            "mordred:ATSC1dv",
            "mordred:ATSC2dv",
            "mordred:ATSC3dv",
            "mordred:ATSC4dv",
            "mordred:ATSC5dv",
            "mordred:ATSC6dv",
            "mordred:ATSC7dv",
            "mordred:ATSC8dv",
            "mordred:ATSC0d",
            "mordred:ATSC1d",
            "mordred:ATSC2d",
            "mordred:ATSC3d",
            "mordred:ATSC4d",
            "mordred:ATSC5d",
            "mordred:ATSC6d",
            "mordred:ATSC7d",
            "mordred:ATSC8d",
            "mordred:ATSC0s",
            "mordred:ATSC1s",
            "mordred:ATSC2s",
            "mordred:ATSC3s",
            "mordred:ATSC4s",
            "mordred:ATSC5s",
            "mordred:ATSC6s",
            "mordred:ATSC7s",
            "mordred:ATSC8s",
            "mordred:ATSC0Z",
            "mordred:ATSC1Z",
            "mordred:ATSC2Z",
            "mordred:ATSC3Z",
            "mordred:ATSC4Z",
            "mordred:ATSC5Z",
            "mordred:ATSC6Z",
            "mordred:ATSC7Z",
            "mordred:ATSC8Z",
            "mordred:ATSC0m",
            "mordred:ATSC1m",
            "mordred:ATSC2m",
            "mordred:ATSC3m",
            "mordred:ATSC4m",
            "mordred:ATSC5m",
            "mordred:ATSC6m",
            "mordred:ATSC7m",
            "mordred:ATSC8m",
            "mordred:ATSC0v",
            "mordred:ATSC1v",
            "mordred:ATSC2v",
            "mordred:ATSC3v",
            "mordred:ATSC4v",
            "mordred:ATSC5v",
            "mordred:ATSC6v",
            "mordred:ATSC7v",
            "mordred:ATSC8v",
            "mordred:ATSC0se",
            "mordred:ATSC1se",
            "mordred:ATSC2se",
            "mordred:ATSC3se",
            "mordred:ATSC4se",
            "mordred:ATSC5se",
            "mordred:ATSC6se",
            "mordred:ATSC7se",
            "mordred:ATSC8se",
            "mordred:ATSC0pe",
            "mordred:ATSC1pe",
            "mordred:ATSC2pe",
            "mordred:ATSC3pe",
            "mordred:ATSC4pe",
            "mordred:ATSC5pe",
            "mordred:ATSC6pe",
            "mordred:ATSC7pe",
            "mordred:ATSC8pe",
            "mordred:ATSC0are",
            "mordred:ATSC1are",
            "mordred:ATSC2are",
            "mordred:ATSC3are",
            "mordred:ATSC4are",
            "mordred:ATSC5are",
            "mordred:ATSC6are",
            "mordred:ATSC7are",
            "mordred:ATSC8are",
            "mordred:ATSC0p",
            "mordred:ATSC1p",
            "mordred:ATSC2p",
            "mordred:ATSC3p",
            "mordred:ATSC4p",
            "mordred:ATSC5p",
            "mordred:ATSC6p",
            "mordred:ATSC7p",
            "mordred:ATSC8p",
            "mordred:ATSC0i",
            "mordred:ATSC1i",
            "mordred:ATSC2i",
            "mordred:ATSC3i",
            "mordred:ATSC4i",
            "mordred:ATSC5i",
            "mordred:ATSC6i",
            "mordred:ATSC7i",
            "mordred:ATSC8i",
            "mordred:AATSC0c",
            "mordred:AATSC1c",
            "mordred:AATSC2c",
            "mordred:AATSC3c",
            "mordred:AATSC4c",
            "mordred:AATSC5c",
            "mordred:AATSC6c",
            "mordred:AATSC7c",
            "mordred:AATSC8c",
            "mordred:AATSC0dv",
            "mordred:AATSC1dv",
            "mordred:AATSC2dv",
            "mordred:AATSC3dv",
            "mordred:AATSC4dv",
            "mordred:AATSC5dv",
            "mordred:AATSC6dv",
            "mordred:AATSC7dv",
            "mordred:AATSC8dv",
            "mordred:AATSC0d",
            "mordred:AATSC1d",
            "mordred:AATSC2d",
            "mordred:AATSC3d",
            "mordred:AATSC4d",
            "mordred:AATSC5d",
            "mordred:AATSC6d",
            "mordred:AATSC7d",
            "mordred:AATSC8d",
            "mordred:AATSC0s",
            "mordred:AATSC1s",
            "mordred:AATSC2s",
            "mordred:AATSC3s",
            "mordred:AATSC4s",
            "mordred:AATSC5s",
            "mordred:AATSC6s",
            "mordred:AATSC7s",
            "mordred:AATSC8s",
            "mordred:AATSC0Z",
            "mordred:AATSC1Z",
            "mordred:AATSC2Z",
            "mordred:AATSC3Z",
            "mordred:AATSC4Z",
            "mordred:AATSC5Z",
            "mordred:AATSC6Z",
            "mordred:AATSC7Z",
            "mordred:AATSC8Z",
            "mordred:AATSC0m",
            "mordred:AATSC1m",
            "mordred:AATSC2m",
            "mordred:AATSC3m",
            "mordred:AATSC4m",
            "mordred:AATSC5m",
            "mordred:AATSC6m",
            "mordred:AATSC7m",
            "mordred:AATSC8m",
            "mordred:AATSC0v",
            "mordred:AATSC1v",
            "mordred:AATSC2v",
            "mordred:AATSC3v",
            "mordred:AATSC4v",
            "mordred:AATSC5v",
            "mordred:AATSC6v",
            "mordred:AATSC7v",
            "mordred:AATSC8v",
            "mordred:AATSC0se",
            "mordred:AATSC1se",
            "mordred:AATSC2se",
            "mordred:AATSC3se",
            "mordred:AATSC4se",
            "mordred:AATSC5se",
            "mordred:AATSC6se",
            "mordred:AATSC7se",
            "mordred:AATSC8se",
            "mordred:AATSC0pe",
            "mordred:AATSC1pe",
            "mordred:AATSC2pe",
            "mordred:AATSC3pe",
            "mordred:AATSC4pe",
            "mordred:AATSC5pe",
            "mordred:AATSC6pe",
            "mordred:AATSC7pe",
            "mordred:AATSC8pe",
            "mordred:AATSC0are",
            "mordred:AATSC1are",
            "mordred:AATSC2are",
            "mordred:AATSC3are",
            "mordred:AATSC4are",
            "mordred:AATSC5are",
            "mordred:AATSC6are",
            "mordred:AATSC7are",
            "mordred:AATSC8are",
            "mordred:AATSC0p",
            "mordred:AATSC1p",
            "mordred:AATSC2p",
            "mordred:AATSC3p",
            "mordred:AATSC4p",
            "mordred:AATSC5p",
            "mordred:AATSC6p",
            "mordred:AATSC7p",
            "mordred:AATSC8p",
            "mordred:AATSC0i",
            "mordred:AATSC1i",
            "mordred:AATSC2i",
            "mordred:AATSC3i",
            "mordred:AATSC4i",
            "mordred:AATSC5i",
            "mordred:AATSC6i",
            "mordred:AATSC7i",
            "mordred:AATSC8i",
            "mordred:MATS1c",
            "mordred:MATS2c",
            "mordred:MATS3c",
            "mordred:MATS4c",
            "mordred:MATS5c",
            "mordred:MATS6c",
            "mordred:MATS7c",
            "mordred:MATS8c",
            "mordred:MATS1dv",
            "mordred:MATS2dv",
            "mordred:MATS3dv",
            "mordred:MATS4dv",
            "mordred:MATS5dv",
            "mordred:MATS6dv",
            "mordred:MATS7dv",
            "mordred:MATS8dv",
            "mordred:MATS1d",
            "mordred:MATS2d",
            "mordred:MATS3d",
            "mordred:MATS4d",
            "mordred:MATS5d",
            "mordred:MATS6d",
            "mordred:MATS7d",
            "mordred:MATS8d",
            "mordred:MATS1s",
            "mordred:MATS2s",
            "mordred:MATS3s",
            "mordred:MATS4s",
            "mordred:MATS5s",
            "mordred:MATS6s",
            "mordred:MATS7s",
            "mordred:MATS8s",
            "mordred:MATS1Z",
            "mordred:MATS2Z",
            "mordred:MATS3Z",
            "mordred:MATS4Z",
            "mordred:MATS5Z",
            "mordred:MATS6Z",
            "mordred:MATS7Z",
            "mordred:MATS8Z",
            "mordred:MATS1m",
            "mordred:MATS2m",
            "mordred:MATS3m",
            "mordred:MATS4m",
            "mordred:MATS5m",
            "mordred:MATS6m",
            "mordred:MATS7m",
            "mordred:MATS8m",
            "mordred:MATS1v",
            "mordred:MATS2v",
            "mordred:MATS3v",
            "mordred:MATS4v",
            "mordred:MATS5v",
            "mordred:MATS6v",
            "mordred:MATS7v",
            "mordred:MATS8v",
            "mordred:MATS1se",
            "mordred:MATS2se",
            "mordred:MATS3se",
            "mordred:MATS4se",
            "mordred:MATS5se",
            "mordred:MATS6se",
            "mordred:MATS7se",
            "mordred:MATS8se",
            "mordred:MATS1pe",
            "mordred:MATS2pe",
            "mordred:MATS3pe",
            "mordred:MATS4pe",
            "mordred:MATS5pe",
            "mordred:MATS6pe",
            "mordred:MATS7pe",
            "mordred:MATS8pe",
            "mordred:MATS1are",
            "mordred:MATS2are",
            "mordred:MATS3are",
            "mordred:MATS4are",
            "mordred:MATS5are",
            "mordred:MATS6are",
            "mordred:MATS7are",
            "mordred:MATS8are",
            "mordred:MATS1p",
            "mordred:MATS2p",
            "mordred:MATS3p",
            "mordred:MATS4p",
            "mordred:MATS5p",
            "mordred:MATS6p",
            "mordred:MATS7p",
            "mordred:MATS8p",
            "mordred:MATS1i",
            "mordred:MATS2i",
            "mordred:MATS3i",
            "mordred:MATS4i",
            "mordred:MATS5i",
            "mordred:MATS6i",
            "mordred:MATS7i",
            "mordred:MATS8i",
            "mordred:GATS1c",
            "mordred:GATS2c",
            "mordred:GATS3c",
            "mordred:GATS4c",
            "mordred:GATS5c",
            "mordred:GATS6c",
            "mordred:GATS7c",
            "mordred:GATS8c",
            "mordred:GATS1dv",
            "mordred:GATS2dv",
            "mordred:GATS3dv",
            "mordred:GATS4dv",
            "mordred:GATS5dv",
            "mordred:GATS6dv",
            "mordred:GATS7dv",
            "mordred:GATS8dv",
            "mordred:GATS1d",
            "mordred:GATS2d",
            "mordred:GATS3d",
            "mordred:GATS4d",
            "mordred:GATS5d",
            "mordred:GATS6d",
            "mordred:GATS7d",
            "mordred:GATS8d",
            "mordred:GATS1s",
            "mordred:GATS2s",
            "mordred:GATS3s",
            "mordred:GATS4s",
            "mordred:GATS5s",
            "mordred:GATS6s",
            "mordred:GATS7s",
            "mordred:GATS8s",
            "mordred:GATS1Z",
            "mordred:GATS2Z",
            "mordred:GATS3Z",
            "mordred:GATS4Z",
            "mordred:GATS5Z",
            "mordred:GATS6Z",
            "mordred:GATS7Z",
            "mordred:GATS8Z",
            "mordred:GATS1m",
            "mordred:GATS2m",
            "mordred:GATS3m",
            "mordred:GATS4m",
            "mordred:GATS5m",
            "mordred:GATS6m",
            "mordred:GATS7m",
            "mordred:GATS8m",
            "mordred:GATS1v",
            "mordred:GATS2v",
            "mordred:GATS3v",
            "mordred:GATS4v",
            "mordred:GATS5v",
            "mordred:GATS6v",
            "mordred:GATS7v",
            "mordred:GATS8v",
            "mordred:GATS1se",
            "mordred:GATS2se",
            "mordred:GATS3se",
            "mordred:GATS4se",
            "mordred:GATS5se",
            "mordred:GATS6se",
            "mordred:GATS7se",
            "mordred:GATS8se",
            "mordred:GATS1pe",
            "mordred:GATS2pe",
            "mordred:GATS3pe",
            "mordred:GATS4pe",
            "mordred:GATS5pe",
            "mordred:GATS6pe",
            "mordred:GATS7pe",
            "mordred:GATS8pe",
            "mordred:GATS1are",
            "mordred:GATS2are",
            "mordred:GATS3are",
            "mordred:GATS4are",
            "mordred:GATS5are",
            "mordred:GATS6are",
            "mordred:GATS7are",
            "mordred:GATS8are",
            "mordred:GATS1p",
            "mordred:GATS2p",
            "mordred:GATS3p",
            "mordred:GATS4p",
            "mordred:GATS5p",
            "mordred:GATS6p",
            "mordred:GATS7p",
            "mordred:GATS8p",
            "mordred:GATS1i",
            "mordred:GATS2i",
            "mordred:GATS3i",
            "mordred:GATS4i",
            "mordred:GATS5i",
            "mordred:GATS6i",
            "mordred:GATS7i",
            "mordred:GATS8i",
            "mordred:BCUTc-1h",
            "mordred:BCUTc-1l",
            "mordred:BCUTdv-1h",
            "mordred:BCUTdv-1l",
            "mordred:BCUTd-1h",
            "mordred:BCUTd-1l",
            "mordred:BCUTs-1h",
            "mordred:BCUTs-1l",
            "mordred:BCUTZ-1h",
            "mordred:BCUTZ-1l",
            "mordred:BCUTm-1h",
            "mordred:BCUTm-1l",
            "mordred:BCUTv-1h",
            "mordred:BCUTv-1l",
            "mordred:BCUTse-1h",
            "mordred:BCUTse-1l",
            "mordred:BCUTpe-1h",
            "mordred:BCUTpe-1l",
            "mordred:BCUTare-1h",
            "mordred:BCUTare-1l",
            "mordred:BCUTp-1h",
            "mordred:BCUTp-1l",
            "mordred:BCUTi-1h",
            "mordred:BCUTi-1l",
            "mordred:BalabanJ",
            "mordred:SpAbs_DzZ",
            "mordred:SpMax_DzZ",
            "mordred:SpDiam_DzZ",
            "mordred:SpAD_DzZ",
            "mordred:SpMAD_DzZ",
            "mordred:LogEE_DzZ",
            "mordred:SM1_DzZ",
            "mordred:VE1_DzZ",
            "mordred:VE2_DzZ",
            "mordred:VE3_DzZ",
            "mordred:VR1_DzZ",
            "mordred:VR2_DzZ",
            "mordred:VR3_DzZ",
            "mordred:SpAbs_Dzm",
            "mordred:SpMax_Dzm",
            "mordred:SpDiam_Dzm",
            "mordred:SpAD_Dzm",
            "mordred:SpMAD_Dzm",
            "mordred:LogEE_Dzm",
            "mordred:SM1_Dzm",
            "mordred:VE1_Dzm",
            "mordred:VE2_Dzm",
            "mordred:VE3_Dzm",
            "mordred:VR1_Dzm",
            "mordred:VR2_Dzm",
            "mordred:VR3_Dzm",
            "mordred:SpAbs_Dzv",
            "mordred:SpMax_Dzv",
            "mordred:SpDiam_Dzv",
            "mordred:SpAD_Dzv",
            "mordred:SpMAD_Dzv",
            "mordred:LogEE_Dzv",
            "mordred:SM1_Dzv",
            "mordred:VE1_Dzv",
            "mordred:VE2_Dzv",
            "mordred:VE3_Dzv",
            "mordred:VR1_Dzv",
            "mordred:VR2_Dzv",
            "mordred:VR3_Dzv",
            "mordred:SpAbs_Dzse",
            "mordred:SpMax_Dzse",
            "mordred:SpDiam_Dzse",
            "mordred:SpAD_Dzse",
            "mordred:SpMAD_Dzse",
            "mordred:LogEE_Dzse",
            "mordred:SM1_Dzse",
            "mordred:VE1_Dzse",
            "mordred:VE2_Dzse",
            "mordred:VE3_Dzse",
            "mordred:VR1_Dzse",
            "mordred:VR2_Dzse",
            "mordred:VR3_Dzse",
            "mordred:SpAbs_Dzpe",
            "mordred:SpMax_Dzpe",
            "mordred:SpDiam_Dzpe",
            "mordred:SpAD_Dzpe",
            "mordred:SpMAD_Dzpe",
            "mordred:LogEE_Dzpe",
            "mordred:SM1_Dzpe",
            "mordred:VE1_Dzpe",
            "mordred:VE2_Dzpe",
            "mordred:VE3_Dzpe",
            "mordred:VR1_Dzpe",
            "mordred:VR2_Dzpe",
            "mordred:VR3_Dzpe",
            "mordred:SpAbs_Dzare",
            "mordred:SpMax_Dzare",
            "mordred:SpDiam_Dzare",
            "mordred:SpAD_Dzare",
            "mordred:SpMAD_Dzare",
            "mordred:LogEE_Dzare",
            "mordred:SM1_Dzare",
            "mordred:VE1_Dzare",
            "mordred:VE2_Dzare",
            "mordred:VE3_Dzare",
            "mordred:VR1_Dzare",
            "mordred:VR2_Dzare",
            "mordred:VR3_Dzare",
            "mordred:SpAbs_Dzp",
            "mordred:SpMax_Dzp",
            "mordred:SpDiam_Dzp",
            "mordred:SpAD_Dzp",
            "mordred:SpMAD_Dzp",
            "mordred:LogEE_Dzp",
            "mordred:SM1_Dzp",
            "mordred:VE1_Dzp",
            "mordred:VE2_Dzp",
            "mordred:VE3_Dzp",
            "mordred:VR1_Dzp",
            "mordred:VR2_Dzp",
            "mordred:VR3_Dzp",
            "mordred:SpAbs_Dzi",
            "mordred:SpMax_Dzi",
            "mordred:SpDiam_Dzi",
            "mordred:SpAD_Dzi",
            "mordred:SpMAD_Dzi",
            "mordred:LogEE_Dzi",
            "mordred:SM1_Dzi",
            "mordred:VE1_Dzi",
            "mordred:VE2_Dzi",
            "mordred:VE3_Dzi",
            "mordred:VR1_Dzi",
            "mordred:VR2_Dzi",
            "mordred:VR3_Dzi",
            "mordred:BertzCT",
            "mordred:nBonds",
            "mordred:nBondsO",
            "mordred:nBondsS",
            "mordred:nBondsD",
            "mordred:nBondsT",
            "mordred:nBondsA",
            "mordred:nBondsM",
            "mordred:nBondsKS",
            "mordred:nBondsKD",
            "mordred:PNSA1",
            "mordred:PNSA2",
            "mordred:PNSA3",
            "mordred:PNSA4",
            "mordred:PNSA5",
            "mordred:PPSA1",
            "mordred:PPSA2",
            "mordred:PPSA3",
            "mordred:PPSA4",
            "mordred:PPSA5",
            "mordred:DPSA1",
            "mordred:DPSA2",
            "mordred:DPSA3",
            "mordred:DPSA4",
            "mordred:DPSA5",
            "mordred:FNSA1",
            "mordred:FNSA2",
            "mordred:FNSA3",
            "mordred:FNSA4",
            "mordred:FNSA5",
            "mordred:FPSA1",
            "mordred:FPSA2",
            "mordred:FPSA3",
            "mordred:FPSA4",
            "mordred:FPSA5",
            "mordred:WNSA1",
            "mordred:WNSA2",
            "mordred:WNSA3",
            "mordred:WNSA4",
            "mordred:WNSA5",
            "mordred:WPSA1",
            "mordred:WPSA2",
            "mordred:WPSA3",
            "mordred:WPSA4",
            "mordred:WPSA5",
            "mordred:RNCG",
            "mordred:RPCG",
            "mordred:RNCS",
            "mordred:RPCS",
            "mordred:TASA",
            "mordred:TPSA",
            "mordred:RASA",
            "mordred:RPSA",
            "mordred:C1SP1",
            "mordred:C2SP1",
            "mordred:C1SP2",
            "mordred:C2SP2",
            "mordred:C3SP2",
            "mordred:C1SP3",
            "mordred:C2SP3",
            "mordred:C3SP3",
            "mordred:C4SP3",
            "mordred:HybRatio",
            "mordred:FCSP3",
            "mordred:Xch-3d",
            "mordred:Xch-4d",
            "mordred:Xch-5d",
            "mordred:Xch-6d",
            "mordred:Xch-7d",
            "mordred:Xch-3dv",
            "mordred:Xch-4dv",
            "mordred:Xch-5dv",
            "mordred:Xch-6dv",
            "mordred:Xch-7dv",
            "mordred:Xc-3d",
            "mordred:Xc-4d",
            "mordred:Xc-5d",
            "mordred:Xc-6d",
            "mordred:Xc-3dv",
            "mordred:Xc-4dv",
            "mordred:Xc-5dv",
            "mordred:Xc-6dv",
            "mordred:Xpc-4d",
            "mordred:Xpc-5d",
            "mordred:Xpc-6d",
            "mordred:Xpc-4dv",
            "mordred:Xpc-5dv",
            "mordred:Xpc-6dv",
            "mordred:Xp-0d",
            "mordred:Xp-1d",
            "mordred:Xp-2d",
            "mordred:Xp-3d",
            "mordred:Xp-4d",
            "mordred:Xp-5d",
            "mordred:Xp-6d",
            "mordred:Xp-7d",
            "mordred:AXp-0d",
            "mordred:AXp-1d",
            "mordred:AXp-2d",
            "mordred:AXp-3d",
            "mordred:AXp-4d",
            "mordred:AXp-5d",
            "mordred:AXp-6d",
            "mordred:AXp-7d",
            "mordred:Xp-0dv",
            "mordred:Xp-1dv",
            "mordred:Xp-2dv",
            "mordred:Xp-3dv",
            "mordred:Xp-4dv",
            "mordred:Xp-5dv",
            "mordred:Xp-6dv",
            "mordred:Xp-7dv",
            "mordred:AXp-0dv",
            "mordred:AXp-1dv",
            "mordred:AXp-2dv",
            "mordred:AXp-3dv",
            "mordred:AXp-4dv",
            "mordred:AXp-5dv",
            "mordred:AXp-6dv",
            "mordred:AXp-7dv",
            "mordred:SZ",
            "mordred:Sm",
            "mordred:Sv",
            "mordred:Sse",
            "mordred:Spe",
            "mordred:Sare",
            "mordred:Sp",
            "mordred:Si",
            "mordred:MZ",
            "mordred:Mm",
            "mordred:Mv",
            "mordred:Mse",
            "mordred:Mpe",
            "mordred:Mare",
            "mordred:Mp",
            "mordred:Mi",
            "mordred:SpAbs_Dt",
            "mordred:SpMax_Dt",
            "mordred:SpDiam_Dt",
            "mordred:SpAD_Dt",
            "mordred:SpMAD_Dt",
            "mordred:LogEE_Dt",
            "mordred:SM1_Dt",
            "mordred:VE1_Dt",
            "mordred:VE2_Dt",
            "mordred:VE3_Dt",
            "mordred:VR1_Dt",
            "mordred:VR2_Dt",
            "mordred:VR3_Dt",
            "mordred:DetourIndex",
            "mordred:SpAbs_D",
            "mordred:SpMax_D",
            "mordred:SpDiam_D",
            "mordred:SpAD_D",
            "mordred:SpMAD_D",
            "mordred:LogEE_D",
            "mordred:VE1_D",
            "mordred:VE2_D",
            "mordred:VE3_D",
            "mordred:VR1_D",
            "mordred:VR2_D",
            "mordred:VR3_D",
            "mordred:NsLi",
            "mordred:NssBe",
            "mordred:NssssBe",
            "mordred:NssBH",
            "mordred:NsssB",
            "mordred:NssssB",
            "mordred:NsCH3",
            "mordred:NdCH2",
            "mordred:NssCH2",
            "mordred:NtCH",
            "mordred:NdsCH",
            "mordred:NaaCH",
            "mordred:NsssCH",
            "mordred:NddC",
            "mordred:NtsC",
            "mordred:NdssC",
            "mordred:NaasC",
            "mordred:NaaaC",
            "mordred:NssssC",
            "mordred:NsNH3",
            "mordred:NsNH2",
            "mordred:NssNH2",
            "mordred:NdNH",
            "mordred:NssNH",
            "mordred:NaaNH",
            "mordred:NtN",
            "mordred:NsssNH",
            "mordred:NdsN",
            "mordred:NaaN",
            "mordred:NsssN",
            "mordred:NddsN",
            "mordred:NaasN",
            "mordred:NssssN",
            "mordred:NsOH",
            "mordred:NdO",
            "mordred:NssO",
            "mordred:NaaO",
            "mordred:NsF",
            "mordred:NsSiH3",
            "mordred:NssSiH2",
            "mordred:NsssSiH",
            "mordred:NssssSi",
            "mordred:NsPH2",
            "mordred:NssPH",
            "mordred:NsssP",
            "mordred:NdsssP",
            "mordred:NsssssP",
            "mordred:NsSH",
            "mordred:NdS",
            "mordred:NssS",
            "mordred:NaaS",
            "mordred:NdssS",
            "mordred:NddssS",
            "mordred:NsCl",
            "mordred:NsGeH3",
            "mordred:NssGeH2",
            "mordred:NsssGeH",
            "mordred:NssssGe",
            "mordred:NsAsH2",
            "mordred:NssAsH",
            "mordred:NsssAs",
            "mordred:NsssdAs",
            "mordred:NsssssAs",
            "mordred:NsSeH",
            "mordred:NdSe",
            "mordred:NssSe",
            "mordred:NaaSe",
            "mordred:NdssSe",
            "mordred:NddssSe",
            "mordred:NsBr",
            "mordred:NsSnH3",
            "mordred:NssSnH2",
            "mordred:NsssSnH",
            "mordred:NssssSn",
            "mordred:NsI",
            "mordred:NsPbH3",
            "mordred:NssPbH2",
            "mordred:NsssPbH",
            "mordred:NssssPb",
            "mordred:SsLi",
            "mordred:SssBe",
            "mordred:SssssBe",
            "mordred:SssBH",
            "mordred:SsssB",
            "mordred:SssssB",
            "mordred:SsCH3",
            "mordred:SdCH2",
            "mordred:SssCH2",
            "mordred:StCH",
            "mordred:SdsCH",
            "mordred:SaaCH",
            "mordred:SsssCH",
            "mordred:SddC",
            "mordred:StsC",
            "mordred:SdssC",
            "mordred:SaasC",
            "mordred:SaaaC",
            "mordred:SssssC",
            "mordred:SsNH3",
            "mordred:SsNH2",
            "mordred:SssNH2",
            "mordred:SdNH",
            "mordred:SssNH",
            "mordred:SaaNH",
            "mordred:StN",
            "mordred:SsssNH",
            "mordred:SdsN",
            "mordred:SaaN",
            "mordred:SsssN",
            "mordred:SddsN",
            "mordred:SaasN",
            "mordred:SssssN",
            "mordred:SsOH",
            "mordred:SdO",
            "mordred:SssO",
            "mordred:SaaO",
            "mordred:SsF",
            "mordred:SsSiH3",
            "mordred:SssSiH2",
            "mordred:SsssSiH",
            "mordred:SssssSi",
            "mordred:SsPH2",
            "mordred:SssPH",
            "mordred:SsssP",
            "mordred:SdsssP",
            "mordred:SsssssP",
            "mordred:SsSH",
            "mordred:SdS",
            "mordred:SssS",
            "mordred:SaaS",
            "mordred:SdssS",
            "mordred:SddssS",
            "mordred:SsCl",
            "mordred:SsGeH3",
            "mordred:SssGeH2",
            "mordred:SsssGeH",
            "mordred:SssssGe",
            "mordred:SsAsH2",
            "mordred:SssAsH",
            "mordred:SsssAs",
            "mordred:SsssdAs",
            "mordred:SsssssAs",
            "mordred:SsSeH",
            "mordred:SdSe",
            "mordred:SssSe",
            "mordred:SaaSe",
            "mordred:SdssSe",
            "mordred:SddssSe",
            "mordred:SsBr",
            "mordred:SsSnH3",
            "mordred:SssSnH2",
            "mordred:SsssSnH",
            "mordred:SssssSn",
            "mordred:SsI",
            "mordred:SsPbH3",
            "mordred:SssPbH2",
            "mordred:SsssPbH",
            "mordred:SssssPb",
            "mordred:MAXsLi",
            "mordred:MAXssBe",
            "mordred:MAXssssBe",
            "mordred:MAXssBH",
            "mordred:MAXsssB",
            "mordred:MAXssssB",
            "mordred:MAXsCH3",
            "mordred:MAXdCH2",
            "mordred:MAXssCH2",
            "mordred:MAXtCH",
            "mordred:MAXdsCH",
            "mordred:MAXaaCH",
            "mordred:MAXsssCH",
            "mordred:MAXddC",
            "mordred:MAXtsC",
            "mordred:MAXdssC",
            "mordred:MAXaasC",
            "mordred:MAXaaaC",
            "mordred:MAXssssC",
            "mordred:MAXsNH3",
            "mordred:MAXsNH2",
            "mordred:MAXssNH2",
            "mordred:MAXdNH",
            "mordred:MAXssNH",
            "mordred:MAXaaNH",
            "mordred:MAXtN",
            "mordred:MAXsssNH",
            "mordred:MAXdsN",
            "mordred:MAXaaN",
            "mordred:MAXsssN",
            "mordred:MAXddsN",
            "mordred:MAXaasN",
            "mordred:MAXssssN",
            "mordred:MAXsOH",
            "mordred:MAXdO",
            "mordred:MAXssO",
            "mordred:MAXaaO",
            "mordred:MAXsF",
            "mordred:MAXsSiH3",
            "mordred:MAXssSiH2",
            "mordred:MAXsssSiH",
            "mordred:MAXssssSi",
            "mordred:MAXsPH2",
            "mordred:MAXssPH",
            "mordred:MAXsssP",
            "mordred:MAXdsssP",
            "mordred:MAXsssssP",
            "mordred:MAXsSH",
            "mordred:MAXdS",
            "mordred:MAXssS",
            "mordred:MAXaaS",
            "mordred:MAXdssS",
            "mordred:MAXddssS",
            "mordred:MAXsCl",
            "mordred:MAXsGeH3",
            "mordred:MAXssGeH2",
            "mordred:MAXsssGeH",
            "mordred:MAXssssGe",
            "mordred:MAXsAsH2",
            "mordred:MAXssAsH",
            "mordred:MAXsssAs",
            "mordred:MAXsssdAs",
            "mordred:MAXsssssAs",
            "mordred:MAXsSeH",
            "mordred:MAXdSe",
            "mordred:MAXssSe",
            "mordred:MAXaaSe",
            "mordred:MAXdssSe",
            "mordred:MAXddssSe",
            "mordred:MAXsBr",
            "mordred:MAXsSnH3",
            "mordred:MAXssSnH2",
            "mordred:MAXsssSnH",
            "mordred:MAXssssSn",
            "mordred:MAXsI",
            "mordred:MAXsPbH3",
            "mordred:MAXssPbH2",
            "mordred:MAXsssPbH",
            "mordred:MAXssssPb",
            "mordred:MINsLi",
            "mordred:MINssBe",
            "mordred:MINssssBe",
            "mordred:MINssBH",
            "mordred:MINsssB",
            "mordred:MINssssB",
            "mordred:MINsCH3",
            "mordred:MINdCH2",
            "mordred:MINssCH2",
            "mordred:MINtCH",
            "mordred:MINdsCH",
            "mordred:MINaaCH",
            "mordred:MINsssCH",
            "mordred:MINddC",
            "mordred:MINtsC",
            "mordred:MINdssC",
            "mordred:MINaasC",
            "mordred:MINaaaC",
            "mordred:MINssssC",
            "mordred:MINsNH3",
            "mordred:MINsNH2",
            "mordred:MINssNH2",
            "mordred:MINdNH",
            "mordred:MINssNH",
            "mordred:MINaaNH",
            "mordred:MINtN",
            "mordred:MINsssNH",
            "mordred:MINdsN",
            "mordred:MINaaN",
            "mordred:MINsssN",
            "mordred:MINddsN",
            "mordred:MINaasN",
            "mordred:MINssssN",
            "mordred:MINsOH",
            "mordred:MINdO",
            "mordred:MINssO",
            "mordred:MINaaO",
            "mordred:MINsF",
            "mordred:MINsSiH3",
            "mordred:MINssSiH2",
            "mordred:MINsssSiH",
            "mordred:MINssssSi",
            "mordred:MINsPH2",
            "mordred:MINssPH",
            "mordred:MINsssP",
            "mordred:MINdsssP",
            "mordred:MINsssssP",
            "mordred:MINsSH",
            "mordred:MINdS",
            "mordred:MINssS",
            "mordred:MINaaS",
            "mordred:MINdssS",
            "mordred:MINddssS",
            "mordred:MINsCl",
            "mordred:MINsGeH3",
            "mordred:MINssGeH2",
            "mordred:MINsssGeH",
            "mordred:MINssssGe",
            "mordred:MINsAsH2",
            "mordred:MINssAsH",
            "mordred:MINsssAs",
            "mordred:MINsssdAs",
            "mordred:MINsssssAs",
            "mordred:MINsSeH",
            "mordred:MINdSe",
            "mordred:MINssSe",
            "mordred:MINaaSe",
            "mordred:MINdssSe",
            "mordred:MINddssSe",
            "mordred:MINsBr",
            "mordred:MINsSnH3",
            "mordred:MINssSnH2",
            "mordred:MINsssSnH",
            "mordred:MINssssSn",
            "mordred:MINsI",
            "mordred:MINsPbH3",
            "mordred:MINssPbH2",
            "mordred:MINsssPbH",
            "mordred:MINssssPb",
            "mordred:ECIndex",
            "mordred:ETA_alpha",
            "mordred:AETA_alpha",
            "mordred:ETA_shape_p",
            "mordred:ETA_shape_y",
            "mordred:ETA_shape_x",
            "mordred:ETA_beta",
            "mordred:AETA_beta",
            "mordred:ETA_beta_s",
            "mordred:AETA_beta_s",
            "mordred:ETA_beta_ns",
            "mordred:AETA_beta_ns",
            "mordred:ETA_beta_ns_d",
            "mordred:AETA_beta_ns_d",
            "mordred:ETA_eta",
            "mordred:AETA_eta",
            "mordred:ETA_eta_L",
            "mordred:AETA_eta_L",
            "mordred:ETA_eta_R",
            "mordred:AETA_eta_R",
            "mordred:ETA_eta_RL",
            "mordred:AETA_eta_RL",
            "mordred:ETA_eta_F",
            "mordred:AETA_eta_F",
            "mordred:ETA_eta_FL",
            "mordred:AETA_eta_FL",
            "mordred:ETA_eta_B",
            "mordred:AETA_eta_B",
            "mordred:ETA_eta_BR",
            "mordred:AETA_eta_BR",
            "mordred:ETA_dAlpha_A",
            "mordred:ETA_dAlpha_B",
            "mordred:ETA_epsilon_1",
            "mordred:ETA_epsilon_2",
            "mordred:ETA_epsilon_3",
            "mordred:ETA_epsilon_4",
            "mordred:ETA_epsilon_5",
            "mordred:ETA_dEpsilon_A",
            "mordred:ETA_dEpsilon_B",
            "mordred:ETA_dEpsilon_C",
            "mordred:ETA_dEpsilon_D",
            "mordred:ETA_dBeta",
            "mordred:AETA_dBeta",
            "mordred:ETA_psi_1",
            "mordred:ETA_dPsi_A",
            "mordred:ETA_dPsi_B",
            "mordred:fragCpx",
            "mordred:fMF",
            "mordred:GeomDiameter",
            "mordred:GeomRadius",
            "mordred:GeomShapeIndex",
            "mordred:GeomPetitjeanIndex",
            "mordred:GRAV",
            "mordred:GRAVH",
            "mordred:GRAVp",
            "mordred:GRAVHp",
            "mordred:nHBAcc",
            "mordred:nHBDon",
            "mordred:IC0",
            "mordred:IC1",
            "mordred:IC2",
            "mordred:IC3",
            "mordred:IC4",
            "mordred:IC5",
            "mordred:TIC0",
            "mordred:TIC1",
            "mordred:TIC2",
            "mordred:TIC3",
            "mordred:TIC4",
            "mordred:TIC5",
            "mordred:SIC0",
            "mordred:SIC1",
            "mordred:SIC2",
            "mordred:SIC3",
            "mordred:SIC4",
            "mordred:SIC5",
            "mordred:BIC0",
            "mordred:BIC1",
            "mordred:BIC2",
            "mordred:BIC3",
            "mordred:BIC4",
            "mordred:BIC5",
            "mordred:CIC0",
            "mordred:CIC1",
            "mordred:CIC2",
            "mordred:CIC3",
            "mordred:CIC4",
            "mordred:CIC5",
            "mordred:MIC0",
            "mordred:MIC1",
            "mordred:MIC2",
            "mordred:MIC3",
            "mordred:MIC4",
            "mordred:MIC5",
            "mordred:ZMIC0",
            "mordred:ZMIC1",
            "mordred:ZMIC2",
            "mordred:ZMIC3",
            "mordred:ZMIC4",
            "mordred:ZMIC5",
            "mordred:Kier1",
            "mordred:Kier2",
            "mordred:Kier3",
            "mordred:Lipinski",
            "mordred:GhoseFilter",
            "mordred:FilterItLogS",
            "mordred:VMcGowan",
            "mordred:Mor01",
            "mordred:Mor02",
            "mordred:Mor03",
            "mordred:Mor04",
            "mordred:Mor05",
            "mordred:Mor06",
            "mordred:Mor07",
            "mordred:Mor08",
            "mordred:Mor09",
            "mordred:Mor10",
            "mordred:Mor11",
            "mordred:Mor12",
            "mordred:Mor13",
            "mordred:Mor14",
            "mordred:Mor15",
            "mordred:Mor16",
            "mordred:Mor17",
            "mordred:Mor18",
            "mordred:Mor19",
            "mordred:Mor20",
            "mordred:Mor21",
            "mordred:Mor22",
            "mordred:Mor23",
            "mordred:Mor24",
            "mordred:Mor25",
            "mordred:Mor26",
            "mordred:Mor27",
            "mordred:Mor28",
            "mordred:Mor29",
            "mordred:Mor30",
            "mordred:Mor31",
            "mordred:Mor32",
            "mordred:Mor01m",
            "mordred:Mor02m",
            "mordred:Mor03m",
            "mordred:Mor04m",
            "mordred:Mor05m",
            "mordred:Mor06m",
            "mordred:Mor07m",
            "mordred:Mor08m",
            "mordred:Mor09m",
            "mordred:Mor10m",
            "mordred:Mor11m",
            "mordred:Mor12m",
            "mordred:Mor13m",
            "mordred:Mor14m",
            "mordred:Mor15m",
            "mordred:Mor16m",
            "mordred:Mor17m",
            "mordred:Mor18m",
            "mordred:Mor19m",
            "mordred:Mor20m",
            "mordred:Mor21m",
            "mordred:Mor22m",
            "mordred:Mor23m",
            "mordred:Mor24m",
            "mordred:Mor25m",
            "mordred:Mor26m",
            "mordred:Mor27m",
            "mordred:Mor28m",
            "mordred:Mor29m",
            "mordred:Mor30m",
            "mordred:Mor31m",
            "mordred:Mor32m",
            "mordred:Mor01v",
            "mordred:Mor02v",
            "mordred:Mor03v",
            "mordred:Mor04v",
            "mordred:Mor05v",
            "mordred:Mor06v",
            "mordred:Mor07v",
            "mordred:Mor08v",
            "mordred:Mor09v",
            "mordred:Mor10v",
            "mordred:Mor11v",
            "mordred:Mor12v",
            "mordred:Mor13v",
            "mordred:Mor14v",
            "mordred:Mor15v",
            "mordred:Mor16v",
            "mordred:Mor17v",
            "mordred:Mor18v",
            "mordred:Mor19v",
            "mordred:Mor20v",
            "mordred:Mor21v",
            "mordred:Mor22v",
            "mordred:Mor23v",
            "mordred:Mor24v",
            "mordred:Mor25v",
            "mordred:Mor26v",
            "mordred:Mor27v",
            "mordred:Mor28v",
            "mordred:Mor29v",
            "mordred:Mor30v",
            "mordred:Mor31v",
            "mordred:Mor32v",
            "mordred:Mor01se",
            "mordred:Mor02se",
            "mordred:Mor03se",
            "mordred:Mor04se",
            "mordred:Mor05se",
            "mordred:Mor06se",
            "mordred:Mor07se",
            "mordred:Mor08se",
            "mordred:Mor09se",
            "mordred:Mor10se",
            "mordred:Mor11se",
            "mordred:Mor12se",
            "mordred:Mor13se",
            "mordred:Mor14se",
            "mordred:Mor15se",
            "mordred:Mor16se",
            "mordred:Mor17se",
            "mordred:Mor18se",
            "mordred:Mor19se",
            "mordred:Mor20se",
            "mordred:Mor21se",
            "mordred:Mor22se",
            "mordred:Mor23se",
            "mordred:Mor24se",
            "mordred:Mor25se",
            "mordred:Mor26se",
            "mordred:Mor27se",
            "mordred:Mor28se",
            "mordred:Mor29se",
            "mordred:Mor30se",
            "mordred:Mor31se",
            "mordred:Mor32se",
            "mordred:Mor01p",
            "mordred:Mor02p",
            "mordred:Mor03p",
            "mordred:Mor04p",
            "mordred:Mor05p",
            "mordred:Mor06p",
            "mordred:Mor07p",
            "mordred:Mor08p",
            "mordred:Mor09p",
            "mordred:Mor10p",
            "mordred:Mor11p",
            "mordred:Mor12p",
            "mordred:Mor13p",
            "mordred:Mor14p",
            "mordred:Mor15p",
            "mordred:Mor16p",
            "mordred:Mor17p",
            "mordred:Mor18p",
            "mordred:Mor19p",
            "mordred:Mor20p",
            "mordred:Mor21p",
            "mordred:Mor22p",
            "mordred:Mor23p",
            "mordred:Mor24p",
            "mordred:Mor25p",
            "mordred:Mor26p",
            "mordred:Mor27p",
            "mordred:Mor28p",
            "mordred:Mor29p",
            "mordred:Mor30p",
            "mordred:Mor31p",
            "mordred:Mor32p",
            "mordred:LabuteASA",
            "mordred:PEOE_VSA1",
            "mordred:PEOE_VSA2",
            "mordred:PEOE_VSA3",
            "mordred:PEOE_VSA4",
            "mordred:PEOE_VSA5",
            "mordred:PEOE_VSA6",
            "mordred:PEOE_VSA7",
            "mordred:PEOE_VSA8",
            "mordred:PEOE_VSA9",
            "mordred:PEOE_VSA10",
            "mordred:PEOE_VSA11",
            "mordred:PEOE_VSA12",
            "mordred:PEOE_VSA13",
            "mordred:SMR_VSA1",
            "mordred:SMR_VSA2",
            "mordred:SMR_VSA3",
            "mordred:SMR_VSA4",
            "mordred:SMR_VSA5",
            "mordred:SMR_VSA6",
            "mordred:SMR_VSA7",
            "mordred:SMR_VSA8",
            "mordred:SMR_VSA9",
            "mordred:SlogP_VSA1",
            "mordred:SlogP_VSA2",
            "mordred:SlogP_VSA3",
            "mordred:SlogP_VSA4",
            "mordred:SlogP_VSA5",
            "mordred:SlogP_VSA6",
            "mordred:SlogP_VSA7",
            "mordred:SlogP_VSA8",
            "mordred:SlogP_VSA9",
            "mordred:SlogP_VSA10",
            "mordred:SlogP_VSA11",
            "mordred:EState_VSA1",
            "mordred:EState_VSA2",
            "mordred:EState_VSA3",
            "mordred:EState_VSA4",
            "mordred:EState_VSA5",
            "mordred:EState_VSA6",
            "mordred:EState_VSA7",
            "mordred:EState_VSA8",
            "mordred:EState_VSA9",
            "mordred:EState_VSA10",
            "mordred:VSA_EState1",
            "mordred:VSA_EState2",
            "mordred:VSA_EState3",
            "mordred:VSA_EState4",
            "mordred:VSA_EState5",
            "mordred:VSA_EState6",
            "mordred:VSA_EState7",
            "mordred:VSA_EState8",
            "mordred:VSA_EState9",
            "mordred:MDEC-11",
            "mordred:MDEC-12",
            "mordred:MDEC-13",
            "mordred:MDEC-14",
            "mordred:MDEC-22",
            "mordred:MDEC-23",
            "mordred:MDEC-24",
            "mordred:MDEC-33",
            "mordred:MDEC-34",
            "mordred:MDEC-44",
            "mordred:MDEO-11",
            "mordred:MDEO-12",
            "mordred:MDEO-22",
            "mordred:MDEN-11",
            "mordred:MDEN-12",
            "mordred:MDEN-13",
            "mordred:MDEN-22",
            "mordred:MDEN-23",
            "mordred:MDEN-33",
            "mordred:MID",
            "mordred:AMID",
            "mordred:MID_h",
            "mordred:AMID_h",
            "mordred:MID_C",
            "mordred:AMID_C",
            "mordred:MID_N",
            "mordred:AMID_N",
            "mordred:MID_O",
            "mordred:AMID_O",
            "mordred:MID_X",
            "mordred:AMID_X",
            "mordred:MOMI-X",
            "mordred:MOMI-Y",
            "mordred:MOMI-Z",
            "mordred:PBF",
            "mordred:MPC2",
            "mordred:MPC3",
            "mordred:MPC4",
            "mordred:MPC5",
            "mordred:MPC6",
            "mordred:MPC7",
            "mordred:MPC8",
            "mordred:MPC9",
            "mordred:MPC10",
            "mordred:TMPC10",
            "mordred:piPC1",
            "mordred:piPC2",
            "mordred:piPC3",
            "mordred:piPC4",
            "mordred:piPC5",
            "mordred:piPC6",
            "mordred:piPC7",
            "mordred:piPC8",
            "mordred:piPC9",
            "mordred:piPC10",
            "mordred:TpiPC10",
            "mordred:apol",
            "mordred:bpol",
            "mordred:nRing",
            "mordred:n3Ring",
            "mordred:n4Ring",
            "mordred:n5Ring",
            "mordred:n6Ring",
            "mordred:n7Ring",
            "mordred:n8Ring",
            "mordred:n9Ring",
            "mordred:n10Ring",
            "mordred:n11Ring",
            "mordred:n12Ring",
            "mordred:nG12Ring",
            "mordred:nHRing",
            "mordred:n3HRing",
            "mordred:n4HRing",
            "mordred:n5HRing",
            "mordred:n6HRing",
            "mordred:n7HRing",
            "mordred:n8HRing",
            "mordred:n9HRing",
            "mordred:n10HRing",
            "mordred:n11HRing",
            "mordred:n12HRing",
            "mordred:nG12HRing",
            "mordred:naRing",
            "mordred:n3aRing",
            "mordred:n4aRing",
            "mordred:n5aRing",
            "mordred:n6aRing",
            "mordred:n7aRing",
            "mordred:n8aRing",
            "mordred:n9aRing",
            "mordred:n10aRing",
            "mordred:n11aRing",
            "mordred:n12aRing",
            "mordred:nG12aRing",
            "mordred:naHRing",
            "mordred:n3aHRing",
            "mordred:n4aHRing",
            "mordred:n5aHRing",
            "mordred:n6aHRing",
            "mordred:n7aHRing",
            "mordred:n8aHRing",
            "mordred:n9aHRing",
            "mordred:n10aHRing",
            "mordred:n11aHRing",
            "mordred:n12aHRing",
            "mordred:nG12aHRing",
            "mordred:nARing",
            "mordred:n3ARing",
            "mordred:n4ARing",
            "mordred:n5ARing",
            "mordred:n6ARing",
            "mordred:n7ARing",
            "mordred:n8ARing",
            "mordred:n9ARing",
            "mordred:n10ARing",
            "mordred:n11ARing",
            "mordred:n12ARing",
            "mordred:nG12ARing",
            "mordred:nAHRing",
            "mordred:n3AHRing",
            "mordred:n4AHRing",
            "mordred:n5AHRing",
            "mordred:n6AHRing",
            "mordred:n7AHRing",
            "mordred:n8AHRing",
            "mordred:n9AHRing",
            "mordred:n10AHRing",
            "mordred:n11AHRing",
            "mordred:n12AHRing",
            "mordred:nG12AHRing",
            "mordred:nFRing",
            "mordred:n4FRing",
            "mordred:n5FRing",
            "mordred:n6FRing",
            "mordred:n7FRing",
            "mordred:n8FRing",
            "mordred:n9FRing",
            "mordred:n10FRing",
            "mordred:n11FRing",
            "mordred:n12FRing",
            "mordred:nG12FRing",
            "mordred:nFHRing",
            "mordred:n4FHRing",
            "mordred:n5FHRing",
            "mordred:n6FHRing",
            "mordred:n7FHRing",
            "mordred:n8FHRing",
            "mordred:n9FHRing",
            "mordred:n10FHRing",
            "mordred:n11FHRing",
            "mordred:n12FHRing",
            "mordred:nG12FHRing",
            "mordred:nFaRing",
            "mordred:n4FaRing",
            "mordred:n5FaRing",
            "mordred:n6FaRing",
            "mordred:n7FaRing",
            "mordred:n8FaRing",
            "mordred:n9FaRing",
            "mordred:n10FaRing",
            "mordred:n11FaRing",
            "mordred:n12FaRing",
            "mordred:nG12FaRing",
            "mordred:nFaHRing",
            "mordred:n4FaHRing",
            "mordred:n5FaHRing",
            "mordred:n6FaHRing",
            "mordred:n7FaHRing",
            "mordred:n8FaHRing",
            "mordred:n9FaHRing",
            "mordred:n10FaHRing",
            "mordred:n11FaHRing",
            "mordred:n12FaHRing",
            "mordred:nG12FaHRing",
            "mordred:nFARing",
            "mordred:n4FARing",
            "mordred:n5FARing",
            "mordred:n6FARing",
            "mordred:n7FARing",
            "mordred:n8FARing",
            "mordred:n9FARing",
            "mordred:n10FARing",
            "mordred:n11FARing",
            "mordred:n12FARing",
            "mordred:nG12FARing",
            "mordred:nFAHRing",
            "mordred:n4FAHRing",
            "mordred:n5FAHRing",
            "mordred:n6FAHRing",
            "mordred:n7FAHRing",
            "mordred:n8FAHRing",
            "mordred:n9FAHRing",
            "mordred:n10FAHRing",
            "mordred:n11FAHRing",
            "mordred:n12FAHRing",
            "mordred:nG12FAHRing",
            "mordred:nRot",
            "mordred:RotRatio",
            "mordred:SLogP",
            "mordred:SMR",
            "mordred:TopoPSA(NO)",
            "mordred:TopoPSA",
            "mordred:GGI1",
            "mordred:GGI2",
            "mordred:GGI3",
            "mordred:GGI4",
            "mordred:GGI5",
            "mordred:GGI6",
            "mordred:GGI7",
            "mordred:GGI8",
            "mordred:GGI9",
            "mordred:GGI10",
            "mordred:JGI1",
            "mordred:JGI2",
            "mordred:JGI3",
            "mordred:JGI4",
            "mordred:JGI5",
            "mordred:JGI6",
            "mordred:JGI7",
            "mordred:JGI8",
            "mordred:JGI9",
            "mordred:JGI10",
            "mordred:JGT10",
            "mordred:Diameter",
            "mordred:Radius",
            "mordred:TopoShapeIndex",
            "mordred:PetitjeanIndex",
            "mordred:Vabc",
            "mordred:VAdjMat",
            "mordred:MWC01",
            "mordred:MWC02",
            "mordred:MWC03",
            "mordred:MWC04",
            "mordred:MWC05",
            "mordred:MWC06",
            "mordred:MWC07",
            "mordred:MWC08",
            "mordred:MWC09",
            "mordred:MWC10",
            "mordred:TMWC10",
            "mordred:SRW02",
            "mordred:SRW03",
            "mordred:SRW04",
            "mordred:SRW05",
            "mordred:SRW06",
            "mordred:SRW07",
            "mordred:SRW08",
            "mordred:SRW09",
            "mordred:SRW10",
            "mordred:TSRW10",
            "mordred:MW",
            "mordred:AMW",
            "mordred:WPath",
            "mordred:WPol",
            "mordred:Zagreb1",
            "mordred:Zagreb2",
            "mordred:mZagreb1",
            "mordred:mZagreb2",
            'padelpy:nAcid',
            'padelpy:ALogP',
            'padelpy:ALogp2',
            'padelpy:AMR',
            'padelpy:apol',
            'padelpy:naAromAtom',
            'padelpy:nAromBond',
            'padelpy:nAtom',
            'padelpy:nHeavyAtom',
            'padelpy:nH',
            'padelpy:nB',
            'padelpy:nC',
            'padelpy:nN',
            'padelpy:nO',
            'padelpy:nS',
            'padelpy:nP',
            'padelpy:nF',
            'padelpy:nCl',
            'padelpy:nBr',
            'padelpy:nI',
            'padelpy:nX',
            'padelpy:ATS0m',
            'padelpy:ATS1m',
            'padelpy:ATS2m',
            'padelpy:ATS3m',
            'padelpy:ATS4m',
            'padelpy:ATS5m',
            'padelpy:ATS6m',
            'padelpy:ATS7m',
            'padelpy:ATS8m',
            'padelpy:ATS0v',
            'padelpy:ATS1v',
            'padelpy:ATS2v',
            'padelpy:ATS3v',
            'padelpy:ATS4v',
            'padelpy:ATS5v',
            'padelpy:ATS6v',
            'padelpy:ATS7v',
            'padelpy:ATS8v',
            'padelpy:ATS0e',
            'padelpy:ATS1e',
            'padelpy:ATS2e',
            'padelpy:ATS3e',
            'padelpy:ATS4e',
            'padelpy:ATS5e',
            'padelpy:ATS6e',
            'padelpy:ATS7e',
            'padelpy:ATS8e',
            'padelpy:ATS0p',
            'padelpy:ATS1p',
            'padelpy:ATS2p',
            'padelpy:ATS3p',
            'padelpy:ATS4p',
            'padelpy:ATS5p',
            'padelpy:ATS6p',
            'padelpy:ATS7p',
            'padelpy:ATS8p',
            'padelpy:ATS0i',
            'padelpy:ATS1i',
            'padelpy:ATS2i',
            'padelpy:ATS3i',
            'padelpy:ATS4i',
            'padelpy:ATS5i',
            'padelpy:ATS6i',
            'padelpy:ATS7i',
            'padelpy:ATS8i',
            'padelpy:ATS0s',
            'padelpy:ATS1s',
            'padelpy:ATS2s',
            'padelpy:ATS3s',
            'padelpy:ATS4s',
            'padelpy:ATS5s',
            'padelpy:ATS6s',
            'padelpy:ATS7s',
            'padelpy:ATS8s',
            'padelpy:AATS0m',
            'padelpy:AATS1m',
            'padelpy:AATS2m',
            'padelpy:AATS3m',
            'padelpy:AATS4m',
            'padelpy:AATS5m',
            'padelpy:AATS6m',
            'padelpy:AATS7m',
            'padelpy:AATS8m',
            'padelpy:AATS0v',
            'padelpy:AATS1v',
            'padelpy:AATS2v',
            'padelpy:AATS3v',
            'padelpy:AATS4v',
            'padelpy:AATS5v',
            'padelpy:AATS6v',
            'padelpy:AATS7v',
            'padelpy:AATS8v',
            'padelpy:AATS0e',
            'padelpy:AATS1e',
            'padelpy:AATS2e',
            'padelpy:AATS3e',
            'padelpy:AATS4e',
            'padelpy:AATS5e',
            'padelpy:AATS6e',
            'padelpy:AATS7e',
            'padelpy:AATS8e',
            'padelpy:AATS0p',
            'padelpy:AATS1p',
            'padelpy:AATS2p',
            'padelpy:AATS3p',
            'padelpy:AATS4p',
            'padelpy:AATS5p',
            'padelpy:AATS6p',
            'padelpy:AATS7p',
            'padelpy:AATS8p',
            'padelpy:AATS0i',
            'padelpy:AATS1i',
            'padelpy:AATS2i',
            'padelpy:AATS3i',
            'padelpy:AATS4i',
            'padelpy:AATS5i',
            'padelpy:AATS6i',
            'padelpy:AATS7i',
            'padelpy:AATS8i',
            'padelpy:AATS0s',
            'padelpy:AATS1s',
            'padelpy:AATS2s',
            'padelpy:AATS3s',
            'padelpy:AATS4s',
            'padelpy:AATS5s',
            'padelpy:AATS6s',
            'padelpy:AATS7s',
            'padelpy:AATS8s',
            'padelpy:ATSC0c',
            'padelpy:ATSC1c',
            'padelpy:ATSC2c',
            'padelpy:ATSC3c',
            'padelpy:ATSC4c',
            'padelpy:ATSC5c',
            'padelpy:ATSC6c',
            'padelpy:ATSC7c',
            'padelpy:ATSC8c',
            'padelpy:ATSC0m',
            'padelpy:ATSC1m',
            'padelpy:ATSC2m',
            'padelpy:ATSC3m',
            'padelpy:ATSC4m',
            'padelpy:ATSC5m',
            'padelpy:ATSC6m',
            'padelpy:ATSC7m',
            'padelpy:ATSC8m',
            'padelpy:ATSC0v',
            'padelpy:ATSC1v',
            'padelpy:ATSC2v',
            'padelpy:ATSC3v',
            'padelpy:ATSC4v',
            'padelpy:ATSC5v',
            'padelpy:ATSC6v',
            'padelpy:ATSC7v',
            'padelpy:ATSC8v',
            'padelpy:ATSC0e',
            'padelpy:ATSC1e',
            'padelpy:ATSC2e',
            'padelpy:ATSC3e',
            'padelpy:ATSC4e',
            'padelpy:ATSC5e',
            'padelpy:ATSC6e',
            'padelpy:ATSC7e',
            'padelpy:ATSC8e',
            'padelpy:ATSC0p',
            'padelpy:ATSC1p',
            'padelpy:ATSC2p',
            'padelpy:ATSC3p',
            'padelpy:ATSC4p',
            'padelpy:ATSC5p',
            'padelpy:ATSC6p',
            'padelpy:ATSC7p',
            'padelpy:ATSC8p',
            'padelpy:ATSC0i',
            'padelpy:ATSC1i',
            'padelpy:ATSC2i',
            'padelpy:ATSC3i',
            'padelpy:ATSC4i',
            'padelpy:ATSC5i',
            'padelpy:ATSC6i',
            'padelpy:ATSC7i',
            'padelpy:ATSC8i',
            'padelpy:ATSC0s',
            'padelpy:ATSC1s',
            'padelpy:ATSC2s',
            'padelpy:ATSC3s',
            'padelpy:ATSC4s',
            'padelpy:ATSC5s',
            'padelpy:ATSC6s',
            'padelpy:ATSC7s',
            'padelpy:ATSC8s',
            'padelpy:AATSC0c',
            'padelpy:AATSC1c',
            'padelpy:AATSC2c',
            'padelpy:AATSC3c',
            'padelpy:AATSC4c',
            'padelpy:AATSC5c',
            'padelpy:AATSC6c',
            'padelpy:AATSC7c',
            'padelpy:AATSC8c',
            'padelpy:AATSC0m',
            'padelpy:AATSC1m',
            'padelpy:AATSC2m',
            'padelpy:AATSC3m',
            'padelpy:AATSC4m',
            'padelpy:AATSC5m',
            'padelpy:AATSC6m',
            'padelpy:AATSC7m',
            'padelpy:AATSC8m',
            'padelpy:AATSC0v',
            'padelpy:AATSC1v',
            'padelpy:AATSC2v',
            'padelpy:AATSC3v',
            'padelpy:AATSC4v',
            'padelpy:AATSC5v',
            'padelpy:AATSC6v',
            'padelpy:AATSC7v',
            'padelpy:AATSC8v',
            'padelpy:AATSC0e',
            'padelpy:AATSC1e',
            'padelpy:AATSC2e',
            'padelpy:AATSC3e',
            'padelpy:AATSC4e',
            'padelpy:AATSC5e',
            'padelpy:AATSC6e',
            'padelpy:AATSC7e',
            'padelpy:AATSC8e',
            'padelpy:AATSC0p',
            'padelpy:AATSC1p',
            'padelpy:AATSC2p',
            'padelpy:AATSC3p',
            'padelpy:AATSC4p',
            'padelpy:AATSC5p',
            'padelpy:AATSC6p',
            'padelpy:AATSC7p',
            'padelpy:AATSC8p',
            'padelpy:AATSC0i',
            'padelpy:AATSC1i',
            'padelpy:AATSC2i',
            'padelpy:AATSC3i',
            'padelpy:AATSC4i',
            'padelpy:AATSC5i',
            'padelpy:AATSC6i',
            'padelpy:AATSC7i',
            'padelpy:AATSC8i',
            'padelpy:AATSC0s',
            'padelpy:AATSC1s',
            'padelpy:AATSC2s',
            'padelpy:AATSC3s',
            'padelpy:AATSC4s',
            'padelpy:AATSC5s',
            'padelpy:AATSC6s',
            'padelpy:AATSC7s',
            'padelpy:AATSC8s',
            'padelpy:MATS1c',
            'padelpy:MATS2c',
            'padelpy:MATS3c',
            'padelpy:MATS4c',
            'padelpy:MATS5c',
            'padelpy:MATS6c',
            'padelpy:MATS7c',
            'padelpy:MATS8c',
            'padelpy:MATS1m',
            'padelpy:MATS2m',
            'padelpy:MATS3m',
            'padelpy:MATS4m',
            'padelpy:MATS5m',
            'padelpy:MATS6m',
            'padelpy:MATS7m',
            'padelpy:MATS8m',
            'padelpy:MATS1v',
            'padelpy:MATS2v',
            'padelpy:MATS3v',
            'padelpy:MATS4v',
            'padelpy:MATS5v',
            'padelpy:MATS6v',
            'padelpy:MATS7v',
            'padelpy:MATS8v',
            'padelpy:MATS1e',
            'padelpy:MATS2e',
            'padelpy:MATS3e',
            'padelpy:MATS4e',
            'padelpy:MATS5e',
            'padelpy:MATS6e',
            'padelpy:MATS7e',
            'padelpy:MATS8e',
            'padelpy:MATS1p',
            'padelpy:MATS2p',
            'padelpy:MATS3p',
            'padelpy:MATS4p',
            'padelpy:MATS5p',
            'padelpy:MATS6p',
            'padelpy:MATS7p',
            'padelpy:MATS8p',
            'padelpy:MATS1i',
            'padelpy:MATS2i',
            'padelpy:MATS3i',
            'padelpy:MATS4i',
            'padelpy:MATS5i',
            'padelpy:MATS6i',
            'padelpy:MATS7i',
            'padelpy:MATS8i',
            'padelpy:MATS1s',
            'padelpy:MATS2s',
            'padelpy:MATS3s',
            'padelpy:MATS4s',
            'padelpy:MATS5s',
            'padelpy:MATS6s',
            'padelpy:MATS7s',
            'padelpy:MATS8s',
            'padelpy:GATS1c',
            'padelpy:GATS2c',
            'padelpy:GATS3c',
            'padelpy:GATS4c',
            'padelpy:GATS5c',
            'padelpy:GATS6c',
            'padelpy:GATS7c',
            'padelpy:GATS8c',
            'padelpy:GATS1m',
            'padelpy:GATS2m',
            'padelpy:GATS3m',
            'padelpy:GATS4m',
            'padelpy:GATS5m',
            'padelpy:GATS6m',
            'padelpy:GATS7m',
            'padelpy:GATS8m',
            'padelpy:GATS1v',
            'padelpy:GATS2v',
            'padelpy:GATS3v',
            'padelpy:GATS4v',
            'padelpy:GATS5v',
            'padelpy:GATS6v',
            'padelpy:GATS7v',
            'padelpy:GATS8v',
            'padelpy:GATS1e',
            'padelpy:GATS2e',
            'padelpy:GATS3e',
            'padelpy:GATS4e',
            'padelpy:GATS5e',
            'padelpy:GATS6e',
            'padelpy:GATS7e',
            'padelpy:GATS8e',
            'padelpy:GATS1p',
            'padelpy:GATS2p',
            'padelpy:GATS3p',
            'padelpy:GATS4p',
            'padelpy:GATS5p',
            'padelpy:GATS6p',
            'padelpy:GATS7p',
            'padelpy:GATS8p',
            'padelpy:GATS1i',
            'padelpy:GATS2i',
            'padelpy:GATS3i',
            'padelpy:GATS4i',
            'padelpy:GATS5i',
            'padelpy:GATS6i',
            'padelpy:GATS7i',
            'padelpy:GATS8i',
            'padelpy:GATS1s',
            'padelpy:GATS2s',
            'padelpy:GATS3s',
            'padelpy:GATS4s',
            'padelpy:GATS5s',
            'padelpy:GATS6s',
            'padelpy:GATS7s',
            'padelpy:GATS8s',
            'padelpy:SpAbs_DzZ',
            'padelpy:SpMax_DzZ',
            'padelpy:SpDiam_DzZ',
            'padelpy:SpAD_DzZ',
            'padelpy:SpMAD_DzZ',
            'padelpy:EE_DzZ',
            'padelpy:SM1_DzZ',
            'padelpy:VE1_DzZ',
            'padelpy:VE2_DzZ',
            'padelpy:VE3_DzZ',
            'padelpy:VR1_DzZ',
            'padelpy:VR2_DzZ',
            'padelpy:VR3_DzZ',
            'padelpy:SpAbs_Dzm',
            'padelpy:SpMax_Dzm',
            'padelpy:SpDiam_Dzm',
            'padelpy:SpAD_Dzm',
            'padelpy:SpMAD_Dzm',
            'padelpy:EE_Dzm',
            'padelpy:SM1_Dzm',
            'padelpy:VE1_Dzm',
            'padelpy:VE2_Dzm',
            'padelpy:VE3_Dzm',
            'padelpy:VR1_Dzm',
            'padelpy:VR2_Dzm',
            'padelpy:VR3_Dzm',
            'padelpy:SpAbs_Dzv',
            'padelpy:SpMax_Dzv',
            'padelpy:SpDiam_Dzv',
            'padelpy:SpAD_Dzv',
            'padelpy:SpMAD_Dzv',
            'padelpy:EE_Dzv',
            'padelpy:SM1_Dzv',
            'padelpy:VE1_Dzv',
            'padelpy:VE2_Dzv',
            'padelpy:VE3_Dzv',
            'padelpy:VR1_Dzv',
            'padelpy:VR2_Dzv',
            'padelpy:VR3_Dzv',
            'padelpy:SpAbs_Dze',
            'padelpy:SpMax_Dze',
            'padelpy:SpDiam_Dze',
            'padelpy:SpAD_Dze',
            'padelpy:SpMAD_Dze',
            'padelpy:EE_Dze',
            'padelpy:SM1_Dze',
            'padelpy:VE1_Dze',
            'padelpy:VE2_Dze',
            'padelpy:VE3_Dze',
            'padelpy:VR1_Dze',
            'padelpy:VR2_Dze',
            'padelpy:VR3_Dze',
            'padelpy:SpAbs_Dzp',
            'padelpy:SpMax_Dzp',
            'padelpy:SpDiam_Dzp',
            'padelpy:SpAD_Dzp',
            'padelpy:SpMAD_Dzp',
            'padelpy:EE_Dzp',
            'padelpy:SM1_Dzp',
            'padelpy:VE1_Dzp',
            'padelpy:VE2_Dzp',
            'padelpy:VE3_Dzp',
            'padelpy:VR1_Dzp',
            'padelpy:VR2_Dzp',
            'padelpy:VR3_Dzp',
            'padelpy:SpAbs_Dzi',
            'padelpy:SpMax_Dzi',
            'padelpy:SpDiam_Dzi',
            'padelpy:SpAD_Dzi',
            'padelpy:SpMAD_Dzi',
            'padelpy:EE_Dzi',
            'padelpy:SM1_Dzi',
            'padelpy:VE1_Dzi',
            'padelpy:VE2_Dzi',
            'padelpy:VE3_Dzi',
            'padelpy:VR1_Dzi',
            'padelpy:VR2_Dzi',
            'padelpy:VR3_Dzi',
            'padelpy:SpAbs_Dzs',
            'padelpy:SpMax_Dzs',
            'padelpy:SpDiam_Dzs',
            'padelpy:SpAD_Dzs',
            'padelpy:SpMAD_Dzs',
            'padelpy:EE_Dzs',
            'padelpy:SM1_Dzs',
            'padelpy:VE1_Dzs',
            'padelpy:VE2_Dzs',
            'padelpy:VE3_Dzs',
            'padelpy:VR1_Dzs',
            'padelpy:VR2_Dzs',
            'padelpy:VR3_Dzs',
            'padelpy:nBase',
            'padelpy:BCUTw-1l',
            'padelpy:BCUTw-1h',
            'padelpy:BCUTc-1l',
            'padelpy:BCUTc-1h',
            'padelpy:BCUTp-1l',
            'padelpy:BCUTp-1h',
            'padelpy:nBonds',
            'padelpy:nBonds2',
            'padelpy:nBondsS',
            'padelpy:nBondsS2',
            'padelpy:nBondsS3',
            'padelpy:nBondsD',
            'padelpy:nBondsD2',
            'padelpy:nBondsT',
            'padelpy:nBondsQ',
            'padelpy:nBondsM',
            'padelpy:bpol',
            'padelpy:SpMax1_Bhm',
            'padelpy:SpMax2_Bhm',
            'padelpy:SpMax3_Bhm',
            'padelpy:SpMax4_Bhm',
            'padelpy:SpMax5_Bhm',
            'padelpy:SpMax6_Bhm',
            'padelpy:SpMax7_Bhm',
            'padelpy:SpMax8_Bhm',
            'padelpy:SpMin1_Bhm',
            'padelpy:SpMin2_Bhm',
            'padelpy:SpMin3_Bhm',
            'padelpy:SpMin4_Bhm',
            'padelpy:SpMin5_Bhm',
            'padelpy:SpMin6_Bhm',
            'padelpy:SpMin7_Bhm',
            'padelpy:SpMin8_Bhm',
            'padelpy:SpMax1_Bhv',
            'padelpy:SpMax2_Bhv',
            'padelpy:SpMax3_Bhv',
            'padelpy:SpMax4_Bhv',
            'padelpy:SpMax5_Bhv',
            'padelpy:SpMax6_Bhv',
            'padelpy:SpMax7_Bhv',
            'padelpy:SpMax8_Bhv',
            'padelpy:SpMin1_Bhv',
            'padelpy:SpMin2_Bhv',
            'padelpy:SpMin3_Bhv',
            'padelpy:SpMin4_Bhv',
            'padelpy:SpMin5_Bhv',
            'padelpy:SpMin6_Bhv',
            'padelpy:SpMin7_Bhv',
            'padelpy:SpMin8_Bhv',
            'padelpy:SpMax1_Bhe',
            'padelpy:SpMax2_Bhe',
            'padelpy:SpMax3_Bhe',
            'padelpy:SpMax4_Bhe',
            'padelpy:SpMax5_Bhe',
            'padelpy:SpMax6_Bhe',
            'padelpy:SpMax7_Bhe',
            'padelpy:SpMax8_Bhe',
            'padelpy:SpMin1_Bhe',
            'padelpy:SpMin2_Bhe',
            'padelpy:SpMin3_Bhe',
            'padelpy:SpMin4_Bhe',
            'padelpy:SpMin5_Bhe',
            'padelpy:SpMin6_Bhe',
            'padelpy:SpMin7_Bhe',
            'padelpy:SpMin8_Bhe',
            'padelpy:SpMax1_Bhp',
            'padelpy:SpMax2_Bhp',
            'padelpy:SpMax3_Bhp',
            'padelpy:SpMax4_Bhp',
            'padelpy:SpMax5_Bhp',
            'padelpy:SpMax6_Bhp',
            'padelpy:SpMax7_Bhp',
            'padelpy:SpMax8_Bhp',
            'padelpy:SpMin1_Bhp',
            'padelpy:SpMin2_Bhp',
            'padelpy:SpMin3_Bhp',
            'padelpy:SpMin4_Bhp',
            'padelpy:SpMin5_Bhp',
            'padelpy:SpMin6_Bhp',
            'padelpy:SpMin7_Bhp',
            'padelpy:SpMin8_Bhp',
            'padelpy:SpMax1_Bhi',
            'padelpy:SpMax2_Bhi',
            'padelpy:SpMax3_Bhi',
            'padelpy:SpMax4_Bhi',
            'padelpy:SpMax5_Bhi',
            'padelpy:SpMax6_Bhi',
            'padelpy:SpMax7_Bhi',
            'padelpy:SpMax8_Bhi',
            'padelpy:SpMin1_Bhi',
            'padelpy:SpMin2_Bhi',
            'padelpy:SpMin3_Bhi',
            'padelpy:SpMin4_Bhi',
            'padelpy:SpMin5_Bhi',
            'padelpy:SpMin6_Bhi',
            'padelpy:SpMin7_Bhi',
            'padelpy:SpMin8_Bhi',
            'padelpy:SpMax1_Bhs',
            'padelpy:SpMax2_Bhs',
            'padelpy:SpMax3_Bhs',
            'padelpy:SpMax4_Bhs',
            'padelpy:SpMax5_Bhs',
            'padelpy:SpMax6_Bhs',
            'padelpy:SpMax7_Bhs',
            'padelpy:SpMax8_Bhs',
            'padelpy:SpMin1_Bhs',
            'padelpy:SpMin2_Bhs',
            'padelpy:SpMin3_Bhs',
            'padelpy:SpMin4_Bhs',
            'padelpy:SpMin5_Bhs',
            'padelpy:SpMin6_Bhs',
            'padelpy:SpMin7_Bhs',
            'padelpy:SpMin8_Bhs',
            'padelpy:C1SP1',
            'padelpy:C2SP1',
            'padelpy:C1SP2',
            'padelpy:C2SP2',
            'padelpy:C3SP2',
            'padelpy:C1SP3',
            'padelpy:C2SP3',
            'padelpy:C3SP3',
            'padelpy:C4SP3',
            'padelpy:SCH-3',
            'padelpy:SCH-4',
            'padelpy:SCH-5',
            'padelpy:SCH-6',
            'padelpy:SCH-7',
            'padelpy:VCH-3',
            'padelpy:VCH-4',
            'padelpy:VCH-5',
            'padelpy:VCH-6',
            'padelpy:VCH-7',
            'padelpy:SC-3',
            'padelpy:SC-4',
            'padelpy:SC-5',
            'padelpy:SC-6',
            'padelpy:VC-3',
            'padelpy:VC-4',
            'padelpy:VC-5',
            'padelpy:VC-6',
            'padelpy:SPC-4',
            'padelpy:SPC-5',
            'padelpy:SPC-6',
            'padelpy:VPC-4',
            'padelpy:VPC-5',
            'padelpy:VPC-6',
            'padelpy:SP-0',
            'padelpy:SP-1',
            'padelpy:SP-2',
            'padelpy:SP-3',
            'padelpy:SP-4',
            'padelpy:SP-5',
            'padelpy:SP-6',
            'padelpy:SP-7',
            'padelpy:ASP-0',
            'padelpy:ASP-1',
            'padelpy:ASP-2',
            'padelpy:ASP-3',
            'padelpy:ASP-4',
            'padelpy:ASP-5',
            'padelpy:ASP-6',
            'padelpy:ASP-7',
            'padelpy:VP-0',
            'padelpy:VP-1',
            'padelpy:VP-2',
            'padelpy:VP-3',
            'padelpy:VP-4',
            'padelpy:VP-5',
            'padelpy:VP-6',
            'padelpy:VP-7',
            'padelpy:AVP-0',
            'padelpy:AVP-1',
            'padelpy:AVP-2',
            'padelpy:AVP-3',
            'padelpy:AVP-4',
            'padelpy:AVP-5',
            'padelpy:AVP-6',
            'padelpy:AVP-7',
            'padelpy:Sv',
            'padelpy:Sse',
            'padelpy:Spe',
            'padelpy:Sare',
            'padelpy:Sp',
            'padelpy:Si',
            'padelpy:Mv',
            'padelpy:Mse',
            'padelpy:Mpe',
            'padelpy:Mare',
            'padelpy:Mp',
            'padelpy:Mi',
            'padelpy:CrippenLogP',
            'padelpy:CrippenMR',
            'padelpy:SpMax_Dt',
            'padelpy:SpDiam_Dt',
            'padelpy:SpAD_Dt',
            'padelpy:SpMAD_Dt',
            'padelpy:EE_Dt',
            'padelpy:VE1_Dt',
            'padelpy:VE2_Dt',
            'padelpy:VE3_Dt',
            'padelpy:VR1_Dt',
            'padelpy:VR2_Dt',
            'padelpy:VR3_Dt',
            'padelpy:ECCEN',
            'padelpy:nHBd',
            'padelpy:nwHBd',
            'padelpy:nHBa',
            'padelpy:nwHBa',
            'padelpy:nHBint2',
            'padelpy:nHBint3',
            'padelpy:nHBint4',
            'padelpy:nHBint5',
            'padelpy:nHBint6',
            'padelpy:nHBint7',
            'padelpy:nHBint8',
            'padelpy:nHBint9',
            'padelpy:nHBint10',
            'padelpy:nHsOH',
            'padelpy:nHdNH',
            'padelpy:nHsSH',
            'padelpy:nHsNH2',
            'padelpy:nHssNH',
            'padelpy:nHaaNH',
            'padelpy:nHsNH3p',
            'padelpy:nHssNH2p',
            'padelpy:nHsssNHp',
            'padelpy:nHtCH',
            'padelpy:nHdCH2',
            'padelpy:nHdsCH',
            'padelpy:nHaaCH',
            'padelpy:nHCHnX',
            'padelpy:nHCsats',
            'padelpy:nHCsatu',
            'padelpy:nHAvin',
            'padelpy:nHother',
            'padelpy:nHmisc',
            'padelpy:nsLi',
            'padelpy:nssBe',
            'padelpy:nssssBem',
            'padelpy:nsBH2',
            'padelpy:nssBH',
            'padelpy:nsssB',
            'padelpy:nssssBm',
            'padelpy:nsCH3',
            'padelpy:ndCH2',
            'padelpy:nssCH2',
            'padelpy:ntCH',
            'padelpy:ndsCH',
            'padelpy:naaCH',
            'padelpy:nsssCH',
            'padelpy:nddC',
            'padelpy:ntsC',
            'padelpy:ndssC',
            'padelpy:naasC',
            'padelpy:naaaC',
            'padelpy:nssssC',
            'padelpy:nsNH3p',
            'padelpy:nsNH2',
            'padelpy:nssNH2p',
            'padelpy:ndNH',
            'padelpy:nssNH',
            'padelpy:naaNH',
            'padelpy:ntN',
            'padelpy:nsssNHp',
            'padelpy:ndsN',
            'padelpy:naaN',
            'padelpy:nsssN',
            'padelpy:nddsN',
            'padelpy:naasN',
            'padelpy:nssssNp',
            'padelpy:nsOH',
            'padelpy:ndO',
            'padelpy:nssO',
            'padelpy:naaO',
            'padelpy:naOm',
            'padelpy:nsOm',
            'padelpy:nsF',
            'padelpy:nsSiH3',
            'padelpy:nssSiH2',
            'padelpy:nsssSiH',
            'padelpy:nssssSi',
            'padelpy:nsPH2',
            'padelpy:nssPH',
            'padelpy:nsssP',
            'padelpy:ndsssP',
            'padelpy:nddsP',
            'padelpy:nsssssP',
            'padelpy:nsSH',
            'padelpy:ndS',
            'padelpy:nssS',
            'padelpy:naaS',
            'padelpy:ndssS',
            'padelpy:nddssS',
            'padelpy:nssssssS',
            'padelpy:nSm',
            'padelpy:nsCl',
            'padelpy:nsGeH3',
            'padelpy:nssGeH2',
            'padelpy:nsssGeH',
            'padelpy:nssssGe',
            'padelpy:nsAsH2',
            'padelpy:nssAsH',
            'padelpy:nsssAs',
            'padelpy:ndsssAs',
            'padelpy:nddsAs',
            'padelpy:nsssssAs',
            'padelpy:nsSeH',
            'padelpy:ndSe',
            'padelpy:nssSe',
            'padelpy:naaSe',
            'padelpy:ndssSe',
            'padelpy:nssssssSe',
            'padelpy:nddssSe',
            'padelpy:nsBr',
            'padelpy:nsSnH3',
            'padelpy:nssSnH2',
            'padelpy:nsssSnH',
            'padelpy:nssssSn',
            'padelpy:nsI',
            'padelpy:nsPbH3',
            'padelpy:nssPbH2',
            'padelpy:nsssPbH',
            'padelpy:nssssPb',
            'padelpy:SHBd',
            'padelpy:SwHBd',
            'padelpy:SHBa',
            'padelpy:SwHBa',
            'padelpy:SHBint2',
            'padelpy:SHBint3',
            'padelpy:SHBint4',
            'padelpy:SHBint5',
            'padelpy:SHBint6',
            'padelpy:SHBint7',
            'padelpy:SHBint8',
            'padelpy:SHBint9',
            'padelpy:SHBint10',
            'padelpy:SHsOH',
            'padelpy:SHdNH',
            'padelpy:SHsSH',
            'padelpy:SHsNH2',
            'padelpy:SHssNH',
            'padelpy:SHaaNH',
            'padelpy:SHsNH3p',
            'padelpy:SHssNH2p',
            'padelpy:SHsssNHp',
            'padelpy:SHtCH',
            'padelpy:SHdCH2',
            'padelpy:SHdsCH',
            'padelpy:SHaaCH',
            'padelpy:SHCHnX',
            'padelpy:SHCsats',
            'padelpy:SHCsatu',
            'padelpy:SHAvin',
            'padelpy:SHother',
            'padelpy:SHmisc',
            'padelpy:SsLi',
            'padelpy:SssBe',
            'padelpy:SssssBem',
            'padelpy:SsBH2',
            'padelpy:SssBH',
            'padelpy:SsssB',
            'padelpy:SssssBm',
            'padelpy:SsCH3',
            'padelpy:SdCH2',
            'padelpy:SssCH2',
            'padelpy:StCH',
            'padelpy:SdsCH',
            'padelpy:SaaCH',
            'padelpy:SsssCH',
            'padelpy:SddC',
            'padelpy:StsC',
            'padelpy:SdssC',
            'padelpy:SaasC',
            'padelpy:SaaaC',
            'padelpy:SssssC',
            'padelpy:SsNH3p',
            'padelpy:SsNH2',
            'padelpy:SssNH2p',
            'padelpy:SdNH',
            'padelpy:SssNH',
            'padelpy:SaaNH',
            'padelpy:StN',
            'padelpy:SsssNHp',
            'padelpy:SdsN',
            'padelpy:SaaN',
            'padelpy:SsssN',
            'padelpy:SddsN',
            'padelpy:SaasN',
            'padelpy:SssssNp',
            'padelpy:SsOH',
            'padelpy:SdO',
            'padelpy:SssO',
            'padelpy:SaaO',
            'padelpy:SaOm',
            'padelpy:SsOm',
            'padelpy:SsF',
            'padelpy:SsSiH3',
            'padelpy:SssSiH2',
            'padelpy:SsssSiH',
            'padelpy:SssssSi',
            'padelpy:SsPH2',
            'padelpy:SssPH',
            'padelpy:SsssP',
            'padelpy:SdsssP',
            'padelpy:SddsP',
            'padelpy:SsssssP',
            'padelpy:SsSH',
            'padelpy:SdS',
            'padelpy:SssS',
            'padelpy:SaaS',
            'padelpy:SdssS',
            'padelpy:SddssS',
            'padelpy:SssssssS',
            'padelpy:SSm',
            'padelpy:SsCl',
            'padelpy:SsGeH3',
            'padelpy:SssGeH2',
            'padelpy:SsssGeH',
            'padelpy:SssssGe',
            'padelpy:SsAsH2',
            'padelpy:SssAsH',
            'padelpy:SsssAs',
            'padelpy:SdsssAs',
            'padelpy:SddsAs',
            'padelpy:SsssssAs',
            'padelpy:SsSeH',
            'padelpy:SdSe',
            'padelpy:SssSe',
            'padelpy:SaaSe',
            'padelpy:SdssSe',
            'padelpy:SssssssSe',
            'padelpy:SddssSe',
            'padelpy:SsBr',
            'padelpy:SsSnH3',
            'padelpy:SssSnH2',
            'padelpy:SsssSnH',
            'padelpy:SssssSn',
            'padelpy:SsI',
            'padelpy:SsPbH3',
            'padelpy:SssPbH2',
            'padelpy:SsssPbH',
            'padelpy:SssssPb',
            'padelpy:minHBd',
            'padelpy:minwHBd',
            'padelpy:minHBa',
            'padelpy:minwHBa',
            'padelpy:minHBint2',
            'padelpy:minHBint3',
            'padelpy:minHBint4',
            'padelpy:minHBint5',
            'padelpy:minHBint6',
            'padelpy:minHBint7',
            'padelpy:minHBint8',
            'padelpy:minHBint9',
            'padelpy:minHBint10',
            'padelpy:minHsOH',
            'padelpy:minHdNH',
            'padelpy:minHsSH',
            'padelpy:minHsNH2',
            'padelpy:minHssNH',
            'padelpy:minHaaNH',
            'padelpy:minHsNH3p',
            'padelpy:minHssNH2p',
            'padelpy:minHsssNHp',
            'padelpy:minHtCH',
            'padelpy:minHdCH2',
            'padelpy:minHdsCH',
            'padelpy:minHaaCH',
            'padelpy:minHCHnX',
            'padelpy:minHCsats',
            'padelpy:minHCsatu',
            'padelpy:minHAvin',
            'padelpy:minHother',
            'padelpy:minHmisc',
            'padelpy:minsLi',
            'padelpy:minssBe',
            'padelpy:minssssBem',
            'padelpy:minsBH2',
            'padelpy:minssBH',
            'padelpy:minsssB',
            'padelpy:minssssBm',
            'padelpy:minsCH3',
            'padelpy:mindCH2',
            'padelpy:minssCH2',
            'padelpy:mintCH',
            'padelpy:mindsCH',
            'padelpy:minaaCH',
            'padelpy:minsssCH',
            'padelpy:minddC',
            'padelpy:mintsC',
            'padelpy:mindssC',
            'padelpy:minaasC',
            'padelpy:minaaaC',
            'padelpy:minssssC',
            'padelpy:minsNH3p',
            'padelpy:minsNH2',
            'padelpy:minssNH2p',
            'padelpy:mindNH',
            'padelpy:minssNH',
            'padelpy:minaaNH',
            'padelpy:mintN',
            'padelpy:minsssNHp',
            'padelpy:mindsN',
            'padelpy:minaaN',
            'padelpy:minsssN',
            'padelpy:minddsN',
            'padelpy:minaasN',
            'padelpy:minssssNp',
            'padelpy:minsOH',
            'padelpy:mindO',
            'padelpy:minssO',
            'padelpy:minaaO',
            'padelpy:minaOm',
            'padelpy:minsOm',
            'padelpy:minsF',
            'padelpy:minsSiH3',
            'padelpy:minssSiH2',
            'padelpy:minsssSiH',
            'padelpy:minssssSi',
            'padelpy:minsPH2',
            'padelpy:minssPH',
            'padelpy:minsssP',
            'padelpy:mindsssP',
            'padelpy:minddsP',
            'padelpy:minsssssP',
            'padelpy:minsSH',
            'padelpy:mindS',
            'padelpy:minssS',
            'padelpy:minaaS',
            'padelpy:mindssS',
            'padelpy:minddssS',
            'padelpy:minssssssS',
            'padelpy:minSm',
            'padelpy:minsCl',
            'padelpy:minsGeH3',
            'padelpy:minssGeH2',
            'padelpy:minsssGeH',
            'padelpy:minssssGe',
            'padelpy:minsAsH2',
            'padelpy:minssAsH',
            'padelpy:minsssAs',
            'padelpy:mindsssAs',
            'padelpy:minddsAs',
            'padelpy:minsssssAs',
            'padelpy:minsSeH',
            'padelpy:mindSe',
            'padelpy:minssSe',
            'padelpy:minaaSe',
            'padelpy:mindssSe',
            'padelpy:minssssssSe',
            'padelpy:minddssSe',
            'padelpy:minsBr',
            'padelpy:minsSnH3',
            'padelpy:minssSnH2',
            'padelpy:minsssSnH',
            'padelpy:minssssSn',
            'padelpy:minsI',
            'padelpy:minsPbH3',
            'padelpy:minssPbH2',
            'padelpy:minsssPbH',
            'padelpy:minssssPb',
            'padelpy:maxHBd',
            'padelpy:maxwHBd',
            'padelpy:maxHBa',
            'padelpy:maxwHBa',
            'padelpy:maxHBint2',
            'padelpy:maxHBint3',
            'padelpy:maxHBint4',
            'padelpy:maxHBint5',
            'padelpy:maxHBint6',
            'padelpy:maxHBint7',
            'padelpy:maxHBint8',
            'padelpy:maxHBint9',
            'padelpy:maxHBint10',
            'padelpy:maxHsOH',
            'padelpy:maxHdNH',
            'padelpy:maxHsSH',
            'padelpy:maxHsNH2',
            'padelpy:maxHssNH',
            'padelpy:maxHaaNH',
            'padelpy:maxHsNH3p',
            'padelpy:maxHssNH2p',
            'padelpy:maxHsssNHp',
            'padelpy:maxHtCH',
            'padelpy:maxHdCH2',
            'padelpy:maxHdsCH',
            'padelpy:maxHaaCH',
            'padelpy:maxHCHnX',
            'padelpy:maxHCsats',
            'padelpy:maxHCsatu',
            'padelpy:maxHAvin',
            'padelpy:maxHother',
            'padelpy:maxHmisc',
            'padelpy:maxsLi',
            'padelpy:maxssBe',
            'padelpy:maxssssBem',
            'padelpy:maxsBH2',
            'padelpy:maxssBH',
            'padelpy:maxsssB',
            'padelpy:maxssssBm',
            'padelpy:maxsCH3',
            'padelpy:maxdCH2',
            'padelpy:maxssCH2',
            'padelpy:maxtCH',
            'padelpy:maxdsCH',
            'padelpy:maxaaCH',
            'padelpy:maxsssCH',
            'padelpy:maxddC',
            'padelpy:maxtsC',
            'padelpy:maxdssC',
            'padelpy:maxaasC',
            'padelpy:maxaaaC',
            'padelpy:maxssssC',
            'padelpy:maxsNH3p',
            'padelpy:maxsNH2',
            'padelpy:maxssNH2p',
            'padelpy:maxdNH',
            'padelpy:maxssNH',
            'padelpy:maxaaNH',
            'padelpy:maxtN',
            'padelpy:maxsssNHp',
            'padelpy:maxdsN',
            'padelpy:maxaaN',
            'padelpy:maxsssN',
            'padelpy:maxddsN',
            'padelpy:maxaasN',
            'padelpy:maxssssNp',
            'padelpy:maxsOH',
            'padelpy:maxdO',
            'padelpy:maxssO',
            'padelpy:maxaaO',
            'padelpy:maxaOm',
            'padelpy:maxsOm',
            'padelpy:maxsF',
            'padelpy:maxsSiH3',
            'padelpy:maxssSiH2',
            'padelpy:maxsssSiH',
            'padelpy:maxssssSi',
            'padelpy:maxsPH2',
            'padelpy:maxssPH',
            'padelpy:maxsssP',
            'padelpy:maxdsssP',
            'padelpy:maxddsP',
            'padelpy:maxsssssP',
            'padelpy:maxsSH',
            'padelpy:maxdS',
            'padelpy:maxssS',
            'padelpy:maxaaS',
            'padelpy:maxdssS',
            'padelpy:maxddssS',
            'padelpy:maxssssssS',
            'padelpy:maxSm',
            'padelpy:maxsCl',
            'padelpy:maxsGeH3',
            'padelpy:maxssGeH2',
            'padelpy:maxsssGeH',
            'padelpy:maxssssGe',
            'padelpy:maxsAsH2',
            'padelpy:maxssAsH',
            'padelpy:maxsssAs',
            'padelpy:maxdsssAs',
            'padelpy:maxddsAs',
            'padelpy:maxsssssAs',
            'padelpy:maxsSeH',
            'padelpy:maxdSe',
            'padelpy:maxssSe',
            'padelpy:maxaaSe',
            'padelpy:maxdssSe',
            'padelpy:maxssssssSe',
            'padelpy:maxddssSe',
            'padelpy:maxsBr',
            'padelpy:maxsSnH3',
            'padelpy:maxssSnH2',
            'padelpy:maxsssSnH',
            'padelpy:maxssssSn',
            'padelpy:maxsI',
            'padelpy:maxsPbH3',
            'padelpy:maxssPbH2',
            'padelpy:maxsssPbH',
            'padelpy:maxssssPb',
            'padelpy:sumI',
            'padelpy:meanI',
            'padelpy:hmax',
            'padelpy:gmax',
            'padelpy:hmin',
            'padelpy:gmin',
            'padelpy:LipoaffinityIndex',
            'padelpy:MAXDN',
            'padelpy:MAXDP',
            'padelpy:DELS',
            'padelpy:MAXDN2',
            'padelpy:MAXDP2',
            'padelpy:DELS2',
            'padelpy:ETA_Alpha',
            'padelpy:ETA_AlphaP',
            'padelpy:ETA_dAlpha_A',
            'padelpy:ETA_dAlpha_B',
            'padelpy:ETA_Epsilon_1',
            'padelpy:ETA_Epsilon_2',
            'padelpy:ETA_Epsilon_3',
            'padelpy:ETA_Epsilon_4',
            'padelpy:ETA_Epsilon_5',
            'padelpy:ETA_dEpsilon_A',
            'padelpy:ETA_dEpsilon_B',
            'padelpy:ETA_dEpsilon_C',
            'padelpy:ETA_dEpsilon_D',
            'padelpy:ETA_Psi_1',
            'padelpy:ETA_dPsi_A',
            'padelpy:ETA_dPsi_B',
            'padelpy:ETA_Shape_P',
            'padelpy:ETA_Shape_Y',
            'padelpy:ETA_Shape_X',
            'padelpy:ETA_Beta',
            'padelpy:ETA_BetaP',
            'padelpy:ETA_Beta_s',
            'padelpy:ETA_BetaP_s',
            'padelpy:ETA_Beta_ns',
            'padelpy:ETA_BetaP_ns',
            'padelpy:ETA_dBeta',
            'padelpy:ETA_dBetaP',
            'padelpy:ETA_Beta_ns_d',
            'padelpy:ETA_BetaP_ns_d',
            'padelpy:ETA_Eta',
            'padelpy:ETA_EtaP',
            'padelpy:ETA_Eta_R',
            'padelpy:ETA_Eta_F',
            'padelpy:ETA_EtaP_F',
            'padelpy:ETA_Eta_L',
            'padelpy:ETA_EtaP_L',
            'padelpy:ETA_Eta_R_L',
            'padelpy:ETA_Eta_F_L',
            'padelpy:ETA_EtaP_F_L',
            'padelpy:ETA_Eta_B',
            'padelpy:ETA_EtaP_B',
            'padelpy:ETA_Eta_B_RC',
            'padelpy:ETA_EtaP_B_RC',
            'padelpy:FMF',
            'padelpy:fragC',
            'padelpy:nHBAcc',
            'padelpy:nHBAcc2',
            'padelpy:nHBAcc3',
            'padelpy:nHBAcc_Lipinski',
            'padelpy:nHBDon',
            'padelpy:nHBDon_Lipinski',
            'padelpy:HybRatio',
            'padelpy:IC0',
            'padelpy:IC1',
            'padelpy:IC2',
            'padelpy:IC3',
            'padelpy:IC4',
            'padelpy:IC5',
            'padelpy:TIC0',
            'padelpy:TIC1',
            'padelpy:TIC2',
            'padelpy:TIC3',
            'padelpy:TIC4',
            'padelpy:TIC5',
            'padelpy:SIC0',
            'padelpy:SIC1',
            'padelpy:SIC2',
            'padelpy:SIC3',
            'padelpy:SIC4',
            'padelpy:SIC5',
            'padelpy:CIC0',
            'padelpy:CIC1',
            'padelpy:CIC2',
            'padelpy:CIC3',
            'padelpy:CIC4',
            'padelpy:CIC5',
            'padelpy:BIC0',
            'padelpy:BIC1',
            'padelpy:BIC2',
            'padelpy:BIC3',
            'padelpy:BIC4',
            'padelpy:BIC5',
            'padelpy:MIC0',
            'padelpy:MIC1',
            'padelpy:MIC2',
            'padelpy:MIC3',
            'padelpy:MIC4',
            'padelpy:MIC5',
            'padelpy:ZMIC0',
            'padelpy:ZMIC1',
            'padelpy:ZMIC2',
            'padelpy:ZMIC3',
            'padelpy:ZMIC4',
            'padelpy:ZMIC5',
            'padelpy:Kier1',
            'padelpy:Kier2',
            'padelpy:Kier3',
            'padelpy:nAtomLC',
            'padelpy:nAtomP',
            'padelpy:nAtomLAC',
            'padelpy:MLogP',
            'padelpy:McGowan_Volume',
            'padelpy:MDEC-11',
            'padelpy:MDEC-12',
            'padelpy:MDEC-13',
            'padelpy:MDEC-14',
            'padelpy:MDEC-22',
            'padelpy:MDEC-23',
            'padelpy:MDEC-24',
            'padelpy:MDEC-33',
            'padelpy:MDEC-34',
            'padelpy:MDEC-44',
            'padelpy:MDEO-11',
            'padelpy:MDEO-12',
            'padelpy:MDEO-22',
            'padelpy:MDEN-11',
            'padelpy:MDEN-12',
            'padelpy:MDEN-13',
            'padelpy:MDEN-22',
            'padelpy:MDEN-23',
            'padelpy:MDEN-33',
            'padelpy:MLFER_A',
            'padelpy:MLFER_BH',
            'padelpy:MLFER_BO',
            'padelpy:MLFER_S',
            'padelpy:MLFER_E',
            'padelpy:MLFER_L',
            'padelpy:MPC2',
            'padelpy:MPC3',
            'padelpy:MPC4',
            'padelpy:MPC5',
            'padelpy:MPC6',
            'padelpy:MPC7',
            'padelpy:MPC8',
            'padelpy:MPC9',
            'padelpy:MPC10',
            'padelpy:TPC',
            'padelpy:piPC1',
            'padelpy:piPC2',
            'padelpy:piPC3',
            'padelpy:piPC4',
            'padelpy:piPC5',
            'padelpy:piPC6',
            'padelpy:piPC7',
            'padelpy:piPC8',
            'padelpy:piPC9',
            'padelpy:piPC10',
            'padelpy:TpiPC',
            'padelpy:R_TpiPCTPC',
            'padelpy:PetitjeanNumber',
            'padelpy:nRing',
            'padelpy:n3Ring',
            'padelpy:n4Ring',
            'padelpy:n5Ring',
            'padelpy:n6Ring',
            'padelpy:n7Ring',
            'padelpy:n8Ring',
            'padelpy:n9Ring',
            'padelpy:n10Ring',
            'padelpy:n11Ring',
            'padelpy:n12Ring',
            'padelpy:nG12Ring',
            'padelpy:nFRing',
            'padelpy:nF4Ring',
            'padelpy:nF5Ring',
            'padelpy:nF6Ring',
            'padelpy:nF7Ring',
            'padelpy:nF8Ring',
            'padelpy:nF9Ring',
            'padelpy:nF10Ring',
            'padelpy:nF11Ring',
            'padelpy:nF12Ring',
            'padelpy:nFG12Ring',
            'padelpy:nTRing',
            'padelpy:nT4Ring',
            'padelpy:nT5Ring',
            'padelpy:nT6Ring',
            'padelpy:nT7Ring',
            'padelpy:nT8Ring',
            'padelpy:nT9Ring',
            'padelpy:nT10Ring',
            'padelpy:nT11Ring',
            'padelpy:nT12Ring',
            'padelpy:nTG12Ring',
            'padelpy:nHeteroRing',
            'padelpy:n3HeteroRing',
            'padelpy:n4HeteroRing',
            'padelpy:n5HeteroRing',
            'padelpy:n6HeteroRing',
            'padelpy:n7HeteroRing',
            'padelpy:n8HeteroRing',
            'padelpy:n9HeteroRing',
            'padelpy:n10HeteroRing',
            'padelpy:n11HeteroRing',
            'padelpy:n12HeteroRing',
            'padelpy:nG12HeteroRing',
            'padelpy:nFHeteroRing',
            'padelpy:nF4HeteroRing',
            'padelpy:nF5HeteroRing',
            'padelpy:nF6HeteroRing',
            'padelpy:nF7HeteroRing',
            'padelpy:nF8HeteroRing',
            'padelpy:nF9HeteroRing',
            'padelpy:nF10HeteroRing',
            'padelpy:nF11HeteroRing',
            'padelpy:nF12HeteroRing',
            'padelpy:nFG12HeteroRing',
            'padelpy:nTHeteroRing',
            'padelpy:nT4HeteroRing',
            'padelpy:nT5HeteroRing',
            'padelpy:nT6HeteroRing',
            'padelpy:nT7HeteroRing',
            'padelpy:nT8HeteroRing',
            'padelpy:nT9HeteroRing',
            'padelpy:nT10HeteroRing',
            'padelpy:nT11HeteroRing',
            'padelpy:nT12HeteroRing',
            'padelpy:nTG12HeteroRing',
            'padelpy:nRotB',
            'padelpy:RotBFrac',
            'padelpy:nRotBt',
            'padelpy:RotBtFrac',
            'padelpy:LipinskiFailures',
            'padelpy:topoRadius',
            'padelpy:topoDiameter',
            'padelpy:topoShape',
            'padelpy:GGI1',
            'padelpy:GGI2',
            'padelpy:GGI3',
            'padelpy:GGI4',
            'padelpy:GGI5',
            'padelpy:GGI6',
            'padelpy:GGI7',
            'padelpy:GGI8',
            'padelpy:GGI9',
            'padelpy:GGI10',
            'padelpy:JGI1',
            'padelpy:JGI2',
            'padelpy:JGI3',
            'padelpy:JGI4',
            'padelpy:JGI5',
            'padelpy:JGI6',
            'padelpy:JGI7',
            'padelpy:JGI8',
            'padelpy:JGI9',
            'padelpy:JGI10',
            'padelpy:JGT',
            'padelpy:SpMax_D',
            'padelpy:SpDiam_D',
            'padelpy:SpAD_D',
            'padelpy:SpMAD_D',
            'padelpy:EE_D',
            'padelpy:VE1_D',
            'padelpy:VE2_D',
            'padelpy:VE3_D',
            'padelpy:VR1_D',
            'padelpy:VR2_D',
            'padelpy:VR3_D',
            'padelpy:TopoPSA',
            'padelpy:VABC',
            'padelpy:VAdjMat',
            'padelpy:MWC2',
            'padelpy:MWC3',
            'padelpy:MWC4',
            'padelpy:MWC5',
            'padelpy:MWC6',
            'padelpy:MWC7',
            'padelpy:MWC8',
            'padelpy:MWC9',
            'padelpy:MWC10',
            'padelpy:TWC',
            'padelpy:SRW2',
            'padelpy:SRW3',
            'padelpy:SRW4',
            'padelpy:SRW5',
            'padelpy:SRW6',
            'padelpy:SRW7',
            'padelpy:SRW8',
            'padelpy:SRW9',
            'padelpy:SRW10',
            'padelpy:TSRW',
            'padelpy:MW',
            'padelpy:AMW',
            'padelpy:WTPT-1',
            'padelpy:WTPT-2',
            'padelpy:WTPT-3',
            'padelpy:WTPT-4',
            'padelpy:WTPT-5',
            'padelpy:WPATH',
            'padelpy:WPOL',
            'padelpy:XLogP',
            'padelpy:Zagreb',
            'padelpy:TDB1u',
            'padelpy:TDB2u',
            'padelpy:TDB3u',
            'padelpy:TDB4u',
            'padelpy:TDB5u',
            'padelpy:TDB6u',
            'padelpy:TDB7u',
            'padelpy:TDB8u',
            'padelpy:TDB9u',
            'padelpy:TDB10u',
            'padelpy:TDB1m',
            'padelpy:TDB2m',
            'padelpy:TDB3m',
            'padelpy:TDB4m',
            'padelpy:TDB5m',
            'padelpy:TDB6m',
            'padelpy:TDB7m',
            'padelpy:TDB8m',
            'padelpy:TDB9m',
            'padelpy:TDB10m',
            'padelpy:TDB1v',
            'padelpy:TDB2v',
            'padelpy:TDB3v',
            'padelpy:TDB4v',
            'padelpy:TDB5v',
            'padelpy:TDB6v',
            'padelpy:TDB7v',
            'padelpy:TDB8v',
            'padelpy:TDB9v',
            'padelpy:TDB10v',
            'padelpy:TDB1e',
            'padelpy:TDB2e',
            'padelpy:TDB3e',
            'padelpy:TDB4e',
            'padelpy:TDB5e',
            'padelpy:TDB6e',
            'padelpy:TDB7e',
            'padelpy:TDB8e',
            'padelpy:TDB9e',
            'padelpy:TDB10e',
            'padelpy:TDB1p',
            'padelpy:TDB2p',
            'padelpy:TDB3p',
            'padelpy:TDB4p',
            'padelpy:TDB5p',
            'padelpy:TDB6p',
            'padelpy:TDB7p',
            'padelpy:TDB8p',
            'padelpy:TDB9p',
            'padelpy:TDB10p',
            'padelpy:TDB1i',
            'padelpy:TDB2i',
            'padelpy:TDB3i',
            'padelpy:TDB4i',
            'padelpy:TDB5i',
            'padelpy:TDB6i',
            'padelpy:TDB7i',
            'padelpy:TDB8i',
            'padelpy:TDB9i',
            'padelpy:TDB10i',
            'padelpy:TDB1s',
            'padelpy:TDB2s',
            'padelpy:TDB3s',
            'padelpy:TDB4s',
            'padelpy:TDB5s',
            'padelpy:TDB6s',
            'padelpy:TDB7s',
            'padelpy:TDB8s',
            'padelpy:TDB9s',
            'padelpy:TDB10s',
            'padelpy:TDB1r',
            'padelpy:TDB2r',
            'padelpy:TDB3r',
            'padelpy:TDB4r',
            'padelpy:TDB5r',
            'padelpy:TDB6r',
            'padelpy:TDB7r',
            'padelpy:TDB8r',
            'padelpy:TDB9r',
            'padelpy:TDB10r',
            'padelpy:PPSA-1',
            'padelpy:PPSA-2',
            'padelpy:PPSA-3',
            'padelpy:PNSA-1',
            'padelpy:PNSA-2',
            'padelpy:PNSA-3',
            'padelpy:DPSA-1',
            'padelpy:DPSA-2',
            'padelpy:DPSA-3',
            'padelpy:FPSA-1',
            'padelpy:FPSA-2',
            'padelpy:FPSA-3',
            'padelpy:FNSA-1',
            'padelpy:FNSA-2',
            'padelpy:FNSA-3',
            'padelpy:WPSA-1',
            'padelpy:WPSA-2',
            'padelpy:WPSA-3',
            'padelpy:WNSA-1',
            'padelpy:WNSA-2',
            'padelpy:WNSA-3',
            'padelpy:RPCG',
            'padelpy:RNCG',
            'padelpy:RPCS',
            'padelpy:RNCS',
            'padelpy:THSA',
            'padelpy:TPSA',
            'padelpy:RHSA',
            'padelpy:RPSA',
            'padelpy:GRAV-1',
            'padelpy:GRAV-2',
            'padelpy:GRAV-3',
            'padelpy:GRAVH-1',
            'padelpy:GRAVH-2',
            'padelpy:GRAVH-3',
            'padelpy:GRAV-4',
            'padelpy:GRAV-5',
            'padelpy:GRAV-6',
            'padelpy:LOBMAX',
            'padelpy:LOBMIN',
            'padelpy:MOMI-X',
            'padelpy:MOMI-Y',
            'padelpy:MOMI-Z',
            'padelpy:MOMI-XY',
            'padelpy:MOMI-XZ',
            'padelpy:MOMI-YZ',
            'padelpy:MOMI-R',
            'padelpy:geomRadius',
            'padelpy:geomDiameter',
            'padelpy:geomShape',
            'padelpy:RDF10u',
            'padelpy:RDF15u',
            'padelpy:RDF20u',
            'padelpy:RDF25u',
            'padelpy:RDF30u',
            'padelpy:RDF35u',
            'padelpy:RDF40u',
            'padelpy:RDF45u',
            'padelpy:RDF50u',
            'padelpy:RDF55u',
            'padelpy:RDF60u',
            'padelpy:RDF65u',
            'padelpy:RDF70u',
            'padelpy:RDF75u',
            'padelpy:RDF80u',
            'padelpy:RDF85u',
            'padelpy:RDF90u',
            'padelpy:RDF95u',
            'padelpy:RDF100u',
            'padelpy:RDF105u',
            'padelpy:RDF110u',
            'padelpy:RDF115u',
            'padelpy:RDF120u',
            'padelpy:RDF125u',
            'padelpy:RDF130u',
            'padelpy:RDF135u',
            'padelpy:RDF140u',
            'padelpy:RDF145u',
            'padelpy:RDF150u',
            'padelpy:RDF155u',
            'padelpy:RDF10m',
            'padelpy:RDF15m',
            'padelpy:RDF20m',
            'padelpy:RDF25m',
            'padelpy:RDF30m',
            'padelpy:RDF35m',
            'padelpy:RDF40m',
            'padelpy:RDF45m',
            'padelpy:RDF50m',
            'padelpy:RDF55m',
            'padelpy:RDF60m',
            'padelpy:RDF65m',
            'padelpy:RDF70m',
            'padelpy:RDF75m',
            'padelpy:RDF80m',
            'padelpy:RDF85m',
            'padelpy:RDF90m',
            'padelpy:RDF95m',
            'padelpy:RDF100m',
            'padelpy:RDF105m',
            'padelpy:RDF110m',
            'padelpy:RDF115m',
            'padelpy:RDF120m',
            'padelpy:RDF125m',
            'padelpy:RDF130m',
            'padelpy:RDF135m',
            'padelpy:RDF140m',
            'padelpy:RDF145m',
            'padelpy:RDF150m',
            'padelpy:RDF155m',
            'padelpy:RDF10v',
            'padelpy:RDF15v',
            'padelpy:RDF20v',
            'padelpy:RDF25v',
            'padelpy:RDF30v',
            'padelpy:RDF35v',
            'padelpy:RDF40v',
            'padelpy:RDF45v',
            'padelpy:RDF50v',
            'padelpy:RDF55v',
            'padelpy:RDF60v',
            'padelpy:RDF65v',
            'padelpy:RDF70v',
            'padelpy:RDF75v',
            'padelpy:RDF80v',
            'padelpy:RDF85v',
            'padelpy:RDF90v',
            'padelpy:RDF95v',
            'padelpy:RDF100v',
            'padelpy:RDF105v',
            'padelpy:RDF110v',
            'padelpy:RDF115v',
            'padelpy:RDF120v',
            'padelpy:RDF125v',
            'padelpy:RDF130v',
            'padelpy:RDF135v',
            'padelpy:RDF140v',
            'padelpy:RDF145v',
            'padelpy:RDF150v',
            'padelpy:RDF155v',
            'padelpy:RDF10e',
            'padelpy:RDF15e',
            'padelpy:RDF20e',
            'padelpy:RDF25e',
            'padelpy:RDF30e',
            'padelpy:RDF35e',
            'padelpy:RDF40e',
            'padelpy:RDF45e',
            'padelpy:RDF50e',
            'padelpy:RDF55e',
            'padelpy:RDF60e',
            'padelpy:RDF65e',
            'padelpy:RDF70e',
            'padelpy:RDF75e',
            'padelpy:RDF80e',
            'padelpy:RDF85e',
            'padelpy:RDF90e',
            'padelpy:RDF95e',
            'padelpy:RDF100e',
            'padelpy:RDF105e',
            'padelpy:RDF110e',
            'padelpy:RDF115e',
            'padelpy:RDF120e',
            'padelpy:RDF125e',
            'padelpy:RDF130e',
            'padelpy:RDF135e',
            'padelpy:RDF140e',
            'padelpy:RDF145e',
            'padelpy:RDF150e',
            'padelpy:RDF155e',
            'padelpy:RDF10p',
            'padelpy:RDF15p',
            'padelpy:RDF20p',
            'padelpy:RDF25p',
            'padelpy:RDF30p',
            'padelpy:RDF35p',
            'padelpy:RDF40p',
            'padelpy:RDF45p',
            'padelpy:RDF50p',
            'padelpy:RDF55p',
            'padelpy:RDF60p',
            'padelpy:RDF65p',
            'padelpy:RDF70p',
            'padelpy:RDF75p',
            'padelpy:RDF80p',
            'padelpy:RDF85p',
            'padelpy:RDF90p',
            'padelpy:RDF95p',
            'padelpy:RDF100p',
            'padelpy:RDF105p',
            'padelpy:RDF110p',
            'padelpy:RDF115p',
            'padelpy:RDF120p',
            'padelpy:RDF125p',
            'padelpy:RDF130p',
            'padelpy:RDF135p',
            'padelpy:RDF140p',
            'padelpy:RDF145p',
            'padelpy:RDF150p',
            'padelpy:RDF155p',
            'padelpy:RDF10i',
            'padelpy:RDF15i',
            'padelpy:RDF20i',
            'padelpy:RDF25i',
            'padelpy:RDF30i',
            'padelpy:RDF35i',
            'padelpy:RDF40i',
            'padelpy:RDF45i',
            'padelpy:RDF50i',
            'padelpy:RDF55i',
            'padelpy:RDF60i',
            'padelpy:RDF65i',
            'padelpy:RDF70i',
            'padelpy:RDF75i',
            'padelpy:RDF80i',
            'padelpy:RDF85i',
            'padelpy:RDF90i',
            'padelpy:RDF95i',
            'padelpy:RDF100i',
            'padelpy:RDF105i',
            'padelpy:RDF110i',
            'padelpy:RDF115i',
            'padelpy:RDF120i',
            'padelpy:RDF125i',
            'padelpy:RDF130i',
            'padelpy:RDF135i',
            'padelpy:RDF140i',
            'padelpy:RDF145i',
            'padelpy:RDF150i',
            'padelpy:RDF155i',
            'padelpy:RDF10s',
            'padelpy:RDF15s',
            'padelpy:RDF20s',
            'padelpy:RDF25s',
            'padelpy:RDF30s',
            'padelpy:RDF35s',
            'padelpy:RDF40s',
            'padelpy:RDF45s',
            'padelpy:RDF50s',
            'padelpy:RDF55s',
            'padelpy:RDF60s',
            'padelpy:RDF65s',
            'padelpy:RDF70s',
            'padelpy:RDF75s',
            'padelpy:RDF80s',
            'padelpy:RDF85s',
            'padelpy:RDF90s',
            'padelpy:RDF95s',
            'padelpy:RDF100s',
            'padelpy:RDF105s',
            'padelpy:RDF110s',
            'padelpy:RDF115s',
            'padelpy:RDF120s',
            'padelpy:RDF125s',
            'padelpy:RDF130s',
            'padelpy:RDF135s',
            'padelpy:RDF140s',
            'padelpy:RDF145s',
            'padelpy:RDF150s',
            'padelpy:RDF155s',
            'padelpy:L1u',
            'padelpy:L2u',
            'padelpy:L3u',
            'padelpy:P1u',
            'padelpy:P2u',
            'padelpy:E1u',
            'padelpy:E2u',
            'padelpy:E3u',
            'padelpy:Tu',
            'padelpy:Au',
            'padelpy:Vu',
            'padelpy:Ku',
            'padelpy:Du',
            'padelpy:L1m',
            'padelpy:L2m',
            'padelpy:L3m',
            'padelpy:P1m',
            'padelpy:P2m',
            'padelpy:E1m',
            'padelpy:E2m',
            'padelpy:E3m',
            'padelpy:Tm',
            'padelpy:Am',
            'padelpy:Vm',
            'padelpy:Km',
            'padelpy:Dm',
            'padelpy:L1v',
            'padelpy:L2v',
            'padelpy:L3v',
            'padelpy:P1v',
            'padelpy:P2v',
            'padelpy:E1v',
            'padelpy:E2v',
            'padelpy:E3v',
            'padelpy:Tv',
            'padelpy:Av',
            'padelpy:Vv',
            'padelpy:Kv',
            'padelpy:Dv',
            'padelpy:L1e',
            'padelpy:L2e',
            'padelpy:L3e',
            'padelpy:P1e',
            'padelpy:P2e',
            'padelpy:E1e',
            'padelpy:E2e',
            'padelpy:E3e',
            'padelpy:Te',
            'padelpy:Ae',
            'padelpy:Ve',
            'padelpy:Ke',
            'padelpy:De',
            'padelpy:L1p',
            'padelpy:L2p',
            'padelpy:L3p',
            'padelpy:P1p',
            'padelpy:P2p',
            'padelpy:E1p',
            'padelpy:E2p',
            'padelpy:E3p',
            'padelpy:Tp',
            'padelpy:Ap',
            'padelpy:Vp',
            'padelpy:Kp',
            'padelpy:Dp',
            'padelpy:L1i',
            'padelpy:L2i',
            'padelpy:L3i',
            'padelpy:P1i',
            'padelpy:P2i',
            'padelpy:E1i',
            'padelpy:E2i',
            'padelpy:E3i',
            'padelpy:Ti',
            'padelpy:Ai',
            'padelpy:Vi',
            'padelpy:Ki',
            'padelpy:Di',
            'padelpy:L1s',
            'padelpy:L2s',
            'padelpy:L3s',
            'padelpy:P1s',
            'padelpy:P2s',
            'padelpy:E1s',
            'padelpy:E2s',
            'padelpy:E3s',
            'padelpy:Ts',
            'padelpy:As',
            'padelpy:Vs',
            'padelpy:Ks',
            'padelpy:Ds',
        ]
